/******************************************************************

            Title  : prothandler.c
            Package: PC/SC Lite
            Author : David Corcoran
            Date   : 10/06/99
            Purpose: This handles protocol defaults, PTS, etc.
            LICENSE: See LICENSE

********************************************************************/

#include <prothandler.h>
#include <atrhandler.h>
#include <ifdwrapper.h>
#include <debuglog.h>

/* Function: PHGetDefaultProtocol
   Purpose : To get the default protocol used immediately after reset.
   This protocol is returned from the function.
*/

UCHAR PHGetDefaultProtocol( PUCHAR pucAtr, DWORD dwLength ) {

  SMARTCARD_EXTENSION sSmartCard;
  
  if ( ATRDecodeAtr( &sSmartCard, pucAtr, dwLength ) ) {
    return sSmartCard.CardCapabilities.CurrentProtocol;
  } else {
    return 0x00; 
  }
 
}

/* Function: PHGetAvailableProtocols
   Purpose : To get the protocols supported by the card.
   These protocols are returned from the function as bit masks.
*/

UCHAR PHGetAvailableProtocols( PUCHAR pucAtr, DWORD dwLength ) {

  SMARTCARD_EXTENSION sSmartCard;
  
  if ( ATRDecodeAtr( &sSmartCard, pucAtr, dwLength ) ) {
    return sSmartCard.CardCapabilities.AvailableProtocols;
  } else {
    return 0x00; 
  }
  
}

/* Function: PHSetProtocol
   Purpose : To determine which protocol to use.  SCardConnect has a
   DWORD dwPreferredProtocols that is a bitmask of what protocols to
   use.  Basically, if T=N where N is not zero will be used first if
   it is available in ucAvailable.  Otherwise it will always default
   to T=0.  Do nothing if the default rContext->dwProtocol is OK.
*/

UCHAR PHSetProtocol( struct ReaderContext *rContext, 
		     DWORD dwPreferred, UCHAR ucAvailable ) {
  

  UCHAR ucPTS[6];
  LONG rv;

  if ( (rContext->dwProtocol == SCARD_PROTOCOL_T1) &&
       ((dwPreferred & SCARD_PROTOCOL_T1) == 0) &&
       (dwPreferred & SCARD_PROTOCOL_T0) ) {

#ifdef PCSC_DEBUG
    DebugLogB("PHSetProtocol: Attempting PTS to T=%d\n", 0, 
	     __FILE__, __LINE__);
#endif

    /* Case 1: T1 is default but is not preferred   */
    /* Soln  : Change to T=0 protocol.              */
    rv = IFDSetPTS( rContext, SCARD_PROTOCOL_T0, 0x00, 
		    0x00, 0x00, 0x00 );
    
    if ( rv != SCARD_S_SUCCESS ) {
      return SCARD_PROTOCOL_T1;
    }

  } else if ( (rContext->dwProtocol == SCARD_PROTOCOL_T0) &&
              ((dwPreferred & SCARD_PROTOCOL_T0) == 0) &&
	      (dwPreferred & SCARD_PROTOCOL_T1) ) {

#ifdef PCSC_DEBUG
    DebugLogB("PHSetProtocol: Attempting PTS to T=%d\n", 1, 
	     __FILE__, __LINE__);
#endif

    /* Case 2: T=0 is default but T=1 is preferred  */
    /* Soln  : Change to T=1 only if supported      */
    
    if ( ucAvailable & SCARD_PROTOCOL_T1 ) {
      rv = IFDSetPTS( rContext, SCARD_PROTOCOL_T1, 0x00, 
		      0x00, 0x00, 0x00 );
      
      if ( rv != SCARD_S_SUCCESS ) {
	return SCARD_PROTOCOL_T0;
      } else {
        return SCARD_PROTOCOL_T1;
      }
      
    } else {
      /* Case 2a: T=1 not supported but preferred   */
      /* Soln   : Keep default T=0 protocol         */
      return SCARD_PROTOCOL_T0;
    }    
    
  } else {
    /* Case 3: Default protocol is preferred        */
    /* Soln  : No need to change protocols          */
    
    return rContext->dwProtocol;
  }                                     
  
}
