/******************************************************************

            Title  : ifdwrapper.c
            Package: PC/SC Lite
            Author : David Corcoran
            Date   : 10/08/99
            Purpose: This wraps the dynamic ifdhandler functions.
            LICENSE: See LICENSE

********************************************************************/

#include <ifdwrapper.h>
#include <atrhandler.h>

/* Function: IFDSetPTS
   Purpose : To set the protocol type selection (PTS).  This function
   sets the appropriate protocol to be used on the card.
*/

LONG IFDSetPTS( PREADER_CONTEXT rContext, DWORD dwProtocol, UCHAR ucFlags, 
		UCHAR ucPTS1, UCHAR ucPTS2, UCHAR ucPTS3 ) {

  RESPONSECODE rv;
  LPVOID vHandle, vFunction;
  UCHAR ucValue[1];

#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IFD_set_protocol_parameters)( DWORD, UCHAR, UCHAR,
					       UCHAR, UCHAR );  
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle                     = rContext->vHandle;
  rv = DYN_GetAddress( vHandle, &vFunction, "IFD_Set_Protocol_Parameters" );

  /* Make sure the symbol exists in the driver */
  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }

  IFD_set_protocol_parameters = (RESPONSECODE (*)(DWORD, UCHAR, UCHAR,
						  UCHAR, UCHAR)) vFunction; 
#endif
  
  ucValue[0] = rContext->dwSlot;
  IFDSetCapabilities( rContext, TAG_IFD_SLOTNUM, ucValue );

#ifndef PCSCLITE_STATIC_DRIVER
  rv = (*IFD_set_protocol_parameters)( dwProtocol, 
				       ucFlags, ucPTS1, 
				       ucPTS2, ucPTS3 );
#else
  rv = IFD_Set_Protocol_Parameters( dwProtocol, ucFlags, ucPTS1, 
				    ucPTS2, ucPTS3 );
#endif

  return rv;
}

/* Function: IFDOpenIFD
   Purpose : This function opens a communication channel 
   to the IFD.
*/

LONG IFDOpenIFD( PREADER_CONTEXT rContext, DWORD dwChannelId ) {

  RESPONSECODE rv;
  LPVOID vHandle, vFunction;

#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IO_create_channel)(DWORD dwChannelID);  
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle                     = rContext->vHandle;
  rv = DYN_GetAddress( vHandle, &vFunction, "IO_Create_Channel" );
  
  /* Make sure the symbol exists in the driver */
  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }
  
  IO_create_channel  = (RESPONSECODE(*)(DWORD)) vFunction;
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  rv = (*IO_create_channel)(dwChannelId);
#else
  rv = IO_Create_Channel(dwChannelId);
#endif

  return rv;
}

/* Function: IFDCloseIFD
   Purpose : This function closes a communication channel 
   to the IFD.
*/

LONG IFDCloseIFD( PREADER_CONTEXT rContext ) {

  RESPONSECODE rv;
  LPVOID vHandle, vFunction;

#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IO_close_channel)();  
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle                     = rContext->vHandle;
  rv = DYN_GetAddress( vHandle, &vFunction, "IO_Close_Channel" );
  
  /* Make sure the symbol exists in the driver */
  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }
  
  IO_close_channel  = (RESPONSECODE(*)()) vFunction;
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  rv = (*IO_close_channel)();
#else
  rv = IO_Close_Channel();
#endif

  return rv;
}

/* Function: IFDSetCapabilites
   Purpose : This function set's capabilities in the reader.
*/

LONG IFDSetCapabilities( PREADER_CONTEXT rContext, DWORD dwTag, 
			 PUCHAR pucValue ) {

  LONG rv;
  LPVOID vHandle, vFunctionA;

#ifndef PCSCLITE_STATIC_DRIVER  
  RESPONSECODE (*IFD_set_capabilities)( DWORD, BYTE[] );  
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle = rContext->vHandle;  
  rv = DYN_GetAddress( vHandle, &vFunctionA, "IFD_Set_Capabilities" );

  /* Make sure the symbol exists in the driver */
  if ( rv != SCARD_S_SUCCESS ) { 
    return SCARD_E_UNSUPPORTED_FEATURE;
  }

  IFD_set_capabilities = (RESPONSECODE (*)(DWORD, BYTE[])) vFunctionA; 
#endif

#ifndef PCSCLITE_STATIC_DRIVER
  rv = (*IFD_set_capabilities)( dwTag, pucValue );  
#else
  rv = IFD_Set_Capabilities( dwTag, pucValue );
#endif  

  return rv;
}


/* Function: IFDPowerICC
   Purpose : This function powers up/down or reset's an
   ICC located in the IFD.
*/

LONG IFDPowerICC(  PREADER_CONTEXT rContext, DWORD dwAction, 
		   PUCHAR pucAtr, PDWORD pdwAtrLen ) {

  RESPONSECODE rv;
  LPVOID vHandle, vFunction;
  DWORD dwStatus, dwProtocol;
  UCHAR ucValue[1];

#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IFD_power_icc)(DWORD dwAction);
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle                     = rContext->vHandle;  
  rv = DYN_GetAddress( vHandle, &vFunction, "IFD_Power_ICC" );

  /* Make sure the symbol exists in the driver */
  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }

  IFD_power_icc      = (RESPONSECODE(*)(DWORD)) vFunction;
#endif

  ucValue[0] = rContext->dwSlot;
  IFDSetCapabilities( rContext, TAG_IFD_SLOTNUM, ucValue );

#ifndef PCSCLITE_STATIC_DRIVER
  rv = (*IFD_power_icc)(dwAction); 
#else
  rv = IFD_Power_ICC(dwAction);
#endif

  /* Get the ATR and it's length */

  IFDStatusICC( rContext, &dwStatus, &dwProtocol, 
		pucAtr, pdwAtrLen );
  
  return rv;
}

/* Function: IFDStatusICC
   Purpose : This function provides statistical information about
   the IFD and ICC including insertions, atr, powering status/etc.
*/

LONG IFDStatusICC( PREADER_CONTEXT rContext, PDWORD pdwStatus,
		   PDWORD pdwProtocol, PUCHAR pucAtr, PDWORD pdwAtrLen ) {

  RESPONSECODE rv, rv1;
  LPVOID vHandle, vFunctionA, vFunctionB;
  DWORD dwAtrLength, dwTag;
  SMARTCARD_EXTENSION sSmartCard;
  UCHAR ucValue[1];

#ifndef PCSCLITE_STATIC_DRIVER 
  RESPONSECODE (*IFD_is_icc_present)(); 
  RESPONSECODE (*IFD_get_capabilities)( DWORD, BYTE[] );  
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle = rContext->vHandle;  
  rv  = DYN_GetAddress( vHandle, &vFunctionA, "IFD_Is_ICC_Present" );
  rv1 = DYN_GetAddress( vHandle, &vFunctionB, "IFD_Get_Capabilities" );

  /* Make sure the symbol exists in the driver */
  if ( rv != SCARD_S_SUCCESS || rv1 != SCARD_S_SUCCESS ) 
    {
      return SCARD_E_UNSUPPORTED_FEATURE;
    }

  IFD_is_icc_present = (RESPONSECODE (*)()) vFunctionA; 
  IFD_get_capabilities = (RESPONSECODE (*)(DWORD, BYTE[])) vFunctionB; 
#endif

  ucValue[0] = rContext->dwSlot;
  IFDSetCapabilities( rContext, TAG_IFD_SLOTNUM, ucValue );

#ifndef PCSCLITE_STATIC_DRIVER  
  rv = (*IFD_is_icc_present)();
#else
  rv = IFD_Is_ICC_Present();
#endif

  if ( rv == IFD_SUCCESS || rv == IFD_ICC_PRESENT ) {
    rContext->dwStatus |= SCARD_PRESENT;
    rContext->dwStatus &= ~SCARD_ABSENT;
  } else {
    rContext->dwStatus |= SCARD_ABSENT;
    rContext->dwStatus &= ~SCARD_PRESENT;
  }
  
  /* Now lets get the ATR and process it */

  if ( rContext->dwStatus & SCARD_PRESENT ) { 
    dwTag      = TAG_IFD_ATR;
#ifndef PCSCLITE_STATIC_DRIVER
    rv = (*IFD_get_capabilities)( dwTag, pucAtr );  
#else
    rv = IFD_Get_Capabilities( dwTag, pucAtr );
#endif
    
    /* FIX :: This is a temporary way to return
       the correct size of the ATR since most of
       the drivers return MAX_ATR_SIZE          */
    
    rv = ATRDecodeAtr( &sSmartCard, pucAtr, MAX_ATR_SIZE );
    
    /* Might be a memory card without an ATR */
    if ( rv == 0 ) { 
      *pdwAtrLen = 0; 
    } else {
      *pdwAtrLen = sSmartCard.ATR.Length;
    }
  } else {
    *pdwAtrLen = 0;
  }


  /* End of FIX                               */

  *pdwStatus   = rContext->dwStatus;
  *pdwProtocol = rContext->dwProtocol;

  return SCARD_S_SUCCESS;
}

/* Function: IFDActionICC
   Purpose : This function provides a means for toggling
   a specific action on the reader such as swallow or eject.
*/

LONG IFDActionICC( PREADER_CONTEXT rContext, DWORD dwAction ) {

  RESPONSECODE rv;
  LPVOID vHandle, vFunction;
  UCHAR ucValue[1];

  /* FIX :: Include more actions */

#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IFD_eject_icc)();
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle          = rContext->vHandle;
  rv = DYN_GetAddress( vHandle, &vFunction, "IFD_Eject_ICC" );

  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }

  IFD_eject_icc    = (RESPONSECODE (*)()) vFunction;
#endif

  ucValue[0] = rContext->dwSlot;
  IFDSetCapabilities( rContext, TAG_IFD_SLOTNUM, ucValue );

  /* FIX :: Do something here */

  return rv;
}

/* Function: IFDTransmit
   Purpose : This function transmits an APDU to the ICC.
*/

LONG IFDTransmit(  PREADER_CONTEXT rContext, struct SCARD_IO_HEADER pioTxPci,
		   PUCHAR pucTxBuffer, DWORD dwTxLength, PUCHAR pucRxBuffer, 
		   PDWORD pdwRxLength, struct SCARD_IO_HEADER* pioRxPci ) {

  RESPONSECODE rv;
  LPVOID vHandle, vFunction;
  UCHAR ucValue[1];
  
#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IFD_transmit_to_icc)(struct SCARD_IO_HEADER, BYTE[], DWORD,
                                      BYTE[], DWORD*, struct SCARD_IO_HEADER*);
#endif

#ifndef PCSCLITE_STATIC_DRIVER
  vHandle              = rContext->vHandle;
  rv = DYN_GetAddress( vHandle, &vFunction, "IFD_Transmit_to_ICC" );

  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }

  IFD_transmit_to_icc  = (RESPONSECODE (*)()) vFunction; 
#endif  

  ucValue[0] = rContext->dwSlot;
  IFDSetCapabilities( rContext, TAG_IFD_SLOTNUM, ucValue );

#ifndef PCSCLITE_STATIC_DRIVER
  rv = (*IFD_transmit_to_icc)( pioTxPci, (LPBYTE)pucTxBuffer, dwTxLength,
			       pucRxBuffer, pdwRxLength, pioRxPci );  
#else
  rv = IFD_Transmit_to_ICC( pioTxPci, (LPBYTE)pucTxBuffer, dwTxLength,
			    pucRxBuffer, pdwRxLength, pioRxPci ); 
#endif

  return rv;
}

/* Function: IFDReadICC
   Purpose : This function reads from the specified memory 
   card type ICC.
*/

LONG IFDReadICC( PREADER_CONTEXT rContext, DWORD dwCardType, DWORD dwCardSize,
		 DWORD dwAddress, PUCHAR pbBuffer, DWORD dwLength ) {

  RESPONSECODE rv;
  LPVOID vHandle, vFunction;
  UCHAR ucValue[1];  

#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IFD_read_memory)( DWORD, DWORD, DWORD, BYTE[], DWORD );
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle              = rContext->vHandle;
  rv = DYN_GetAddress( vHandle, &vFunction, "IFD_Read_Memory" );

  /* Just in case memory cards aren't supported */

  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }

  IFD_read_memory      = (RESPONSECODE (*)()) vFunction; 
#endif

  ucValue[0] = rContext->dwSlot;
  IFDSetCapabilities( rContext, TAG_IFD_SLOTNUM, ucValue );

#ifndef PCSCLITE_STATIC_DRIVER  
  rv = (*IFD_read_memory)( dwCardType, dwCardSize, dwAddress,
                           pbBuffer, dwLength ); 
#else
  rv = IFD_Read_Memory( dwCardType, dwCardSize, dwAddress, 
                        pbBuffer, dwLength );
#endif

  return rv;
}

/* Function: IFDWriteICC
   Purpose : This function writes to the specified memory 
   card type ICC.
*/

LONG IFDWriteICC( PREADER_CONTEXT rContext, DWORD dwCardType, DWORD dwCardSize,
		  DWORD dwAddress, PUCHAR pbBuffer, DWORD dwLength ) {
  RESPONSECODE rv;
  LPVOID vHandle, vFunction;
  UCHAR ucValue[1];  

#ifndef PCSCLITE_STATIC_DRIVER
  RESPONSECODE (*IFD_write_memory)( DWORD, DWORD, DWORD, BYTE[], DWORD );
#endif
  
#ifndef PCSCLITE_STATIC_DRIVER
  vHandle              = rContext->vHandle;
  rv = DYN_GetAddress( vHandle, &vFunction, "IFD_Write_Memory" );

  /* Just in case memory cards aren't supported */

  if ( rv != SCARD_S_SUCCESS ) {
    return SCARD_E_UNSUPPORTED_FEATURE;
  }

  IFD_write_memory      = (RESPONSECODE (*)()) vFunction; 
#endif

  ucValue[0] = rContext->dwSlot;
  IFDSetCapabilities( rContext, TAG_IFD_SLOTNUM, ucValue );

#ifndef PCSCLITE_STATIC_DRIVER  
  rv = (*IFD_write_memory)( dwCardType, dwCardSize, dwAddress,
                           pbBuffer, dwLength ); 
#else
  rv = IFD_Write_Memory( dwCardType, dwCardSize, dwAddress, 
                        pbBuffer, dwLength );
#endif

  return rv;
}
