/*
 * Secure beam, using smart card for encryption.
 *
 * Jim Rees, University of Michigan, July 2000
 */

#pragma pack(2)

#if (__GNUC_MINOR__ > 7)
#include <PalmOS.h>
#else
#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#endif
#include <System/DateTime.h>
#include <System/MemoryMgr.h>
#include <System/DataMgr.h>
#include <System/SysEvtMgr.h>

#include <System/Unix/sys_types.h>
#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#if (__GNUC_MINOR__ > 7)
#include <System/Unix/unix_string.h>
#else
#include <string.h>
#endif

#include "field.h"
#include "resource.h"
#include "addr.h"
#include "secb.h"

FieldPtr printfield;
int currecno, curreclen;
struct addr *currecptr;

static Boolean secb(EventPtr event);
static Boolean rec(EventPtr event);

#if (__GNUC_MINOR__ > 7)
UInt32 PilotMain (UInt16 cmd, void *cmdPBP, UInt16  launchFlags)
#else
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
#endif
{
#if (__GNUC_MINOR__ > 7)
    ExgSocketPtr exgsock;
#endif
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	StopApplication();
	return 0;
#if (__GNUC_MINOR__ > 7)
    } else if (cmd == sysAppLaunchCmdExgReceiveData) {
	exgsock = (ExgSocketPtr) cmdPBP;
#endif
    } else
	return sysErrParamErr;
}

StartApplication()
{
    FrmGotoForm(formID_secb);
    return 0;
}

StopApplication()
{
    closereader();
    if (currecptr)
	free(currecptr);
    FrmCloseAllForms();
    return 0;
}

EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, evtWaitForever);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_secb:
		FrmSetEventHandler(form, (FormEventHandlerPtr) secb);
		break;
	    case formID_rec:
		FrmSetEventHandler(form, (FormEventHandlerPtr) rec);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
secb(EventPtr event)
{
    FormPtr form = FrmGetActiveForm();
    FieldPtr field;
    int handled = 0;

    switch (event->eType) {
    case frmOpenEvent:
	printfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_idx));
	listaddrs();
	FrmDrawForm(form);
	handled = 1;
	break;

    case ctlSelectEvent:
	handled = 1;

	switch (event->data.ctlEnter.controlID) {
	case buttonID_open:
	    currecno = getrecno();
	    FrmGotoForm(formID_rec);
	    break;
	case buttonID_delete:
	    currecno = getrecno();
	    delrec();
	    listaddrs();
	    break;
	default:
	    handled = 0;
	    break;
	}
	break;

    case keyDownEvent:
	handled = scrollfield(printfield, event->data.keyDown.chr);
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	case menuitemID_copy:
	    FldCopy(printfield);
	    break;
	}
	handled = 1;
	break;
    }
    return handled;
}

listaddrs()
{
    DmOpenRef AddrDB;
    int nrec, n, i;
    void *hp;
    struct addr *ap;
    char *cp, *d[19];

    AddrDB = DmOpenDatabaseByTypeCreator('DATA', 'addr', dmModeReadOnly);
    if (!AddrDB)
	return -1;

    nrec = DmNumRecords(AddrDB);
    if (nrec > MAXADDR)
	nrec = MAXADDR;

    hidefield(fieldID_idx);
    setfield(printfield, NULL);

    for (n = 0; n < nrec; n++) {
	hp = DmQueryRecord(AddrDB, n);
	if (!hp)
	    continue;
	ap = (struct addr *) MemHandleLock(hp);
	cp = ap->s + 1;

	for (i = 0; i < 19; i++) {
	    if (ap->f & (1 << i)) {
		d[i] = cp;
		cp += strlen(cp) + 1;
	    } else
		d[i] = NULL;
	}
	if (d[AFfirst] && d[AFlast])
	    printf("%d %s %s\n", n+1, d[AFfirst], d[AFlast]);
	else if (d[AFfirst])
	    printf("%d %s\n", n+1, d[AFfirst]);
	else if (d[AFlast])
	    printf("%d %s\n", n+1, d[AFlast]);
	MemPtrUnlock((void *) ap);
    }
    FldSetScrollPosition(printfield, 0);
    showfield(fieldID_idx);
    DmCloseDatabase(AddrDB);
    return 0;
}

int
getrecno()
{
    char *cp0, *cp;
    int offset;

    /* Get pointer to text at insertion point */
    cp0 = FldGetTextPtr(printfield);
    offset = FldGetInsPtPosition(printfield);

    /* Scan back to preceeding newline */
    for (cp = cp0 + offset - 1; cp >= cp0; cp--)
	if (*cp == '\n')
	    break;

    return atoi(cp+1) - 1;
}

static Boolean
rec(EventPtr event)
{
    FormPtr form = FrmGetActiveForm();
    FieldPtr field;
    int handled = 0;

    switch (event->eType) {
    case frmOpenEvent:
	printfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_rec));
	openrec();
	displayrec();
	FrmDrawForm(form);
	handled = 1;
	break;

    case ctlSelectEvent:
	handled = 1;

	switch (event->data.ctlEnter.controlID) {
	case buttonID_index:
	    FrmGotoForm(formID_secb);
	    break;
	case buttonID_encrypt:
	    encrec();
	    break;
	case buttonID_decrypt:
	    decrec();
	    break;
	case buttonID_beam:
	    beamrec();
	    break;
	case buttonID_add:
	    addrec();
	    break;
	default:
	    handled = 0;
	    break;
	}
	break;

    case keyDownEvent:
    case menuEvent:
	handled = secb(event);
	break;
    }
    return handled;
}

openrec()
{
    DmOpenRef AddrDB;
    void *hp;
    unsigned char *p0, *p1;
    int len;

    AddrDB = DmOpenDatabaseByTypeCreator('DATA', 'addr', dmModeReadOnly);
    if (!AddrDB)
	return -1;

    hp = DmQueryRecord(AddrDB, currecno);
    if (!hp) {
	DmCloseDatabase(AddrDB);
	return -1;
    }

    if (currecptr)
	free(currecptr);

    p0 = (unsigned char *) MemHandleLock(hp);
    len = MemHandleSize(hp);
    p1 = (unsigned char *) malloc(((len + 7) & ~7) + SECHDRLEN);
    bcopy(p0, p1, len);
    curreclen = len;
    currecptr = (struct addr *) p1;

    MemHandleUnlock(hp);
    DmCloseDatabase(AddrDB);
    return 0;
}

beamrec()
{
}

Int
compar(void *r1, void *r2, Int other, SortRecordInfoPtr r1s, SortRecordInfoPtr r2s, VoidHand ah)
{
    return strcasecmp((char *) r1 + 9, (char *) r2 + 9);
}

addrec()
{
    DmOpenRef DB;
    UInt idx;
    VoidHand RecHandle;
    Ptr RecPointer;

    if (!currecptr) {
	printf("nothing to add\n");
	return;
    }

    /* Open address database */
    DB = DmOpenDatabaseByTypeCreator('DATA', 'addr', dmModeReadWrite);
    if (!DB)
	return;

    /* Figure out where to put it */
    idx = DmFindSortPosition(DB, currecptr, NULL, compar, 0);

    /* Allocate new record */
    RecHandle = DmNewRecord(DB, &idx, curreclen);
    RecPointer = MemHandleLock(RecHandle);

    /* Write record */
    DmWrite(RecPointer, 0, currecptr, curreclen);

    /* Release and close */
    MemPtrUnlock(RecPointer);
    DmReleaseRecord(DB, idx, true);
    DmCloseDatabase(DB);
}

delrec()
{
    DmOpenRef DB;

    if (FrmAlert(alertID_confirm_delete) != 0)
	return;

    DB = DmOpenDatabaseByTypeCreator('DATA', 'addr', dmModeReadWrite);
    if (!DB)
	return;

    DmDeleteRecord(DB, currecno);
    DmCloseDatabase(DB);
}

/*
copyright 2000
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
