/*
 * OS dependent part, PalmPilot version
 *
 * Jim Rees, University of Michigan, March 1998
 * Original by Nate Binkerton
 */
static char *rcsid = "$Id: scioPP.c,v 1.2 2000/08/04 03:46:10 rees Exp $";

#include <SystemMgr.h>
#include <SerialMgr.h>

#include "scrw.h"

static UInt refNum;
static SerSettingsType settings;

int
scopen(int ttyn, int flags, int *ep)
{
    Err err;

    if (err = SysLibFind("Serial Library", &refNum))
	goto bad;

#ifdef DEBUG
    printf("SerOpen\n");
#endif
    if (err = SerOpen(refNum, ttyn, 9600L))
	goto bad;

    if (err = SerGetSettings(refNum, &settings))
	goto bad;

    settings.flags = (serSettingsFlagStopBits2
		    | serSettingsFlagParityOnM
		    | serSettingsFlagParityEvenM
		    | serSettingsFlagBitsPerChar8);

    if (err = SerSetSettings(refNum, &settings))
	goto bad;

    /* The open may or may not have reset the card.  Wait a while then flush
       anything that came in on the port. */
    SerReceiveFlush(refNum, SysTicksPerSecond() / 4L);

    return 0;

 bad:
    if (ep)
	*ep = SCENOTTY;
    return -1;
}

int
scsetspeed(int ttyn, int speed)
{
    settings.baudRate = speed;
    return (SerSetSettings(refNum, &settings) ? -1 : 0);
}

int
scclose(int ttyn)
{
    if (!SerClose(refNum))
	return -1;
    return 0;
}

/*
 * get one byte from the card.
 * wait at most ms msec.  0 for poll, -1 for infinite.
 * return byte in *cp.
 * return 0 or error.
 */

int
scgetc(int ttyn, unsigned char *cp, int ms)
{
    Long ticks;
    Err err;
    int code;

    if (ms <= 0)
	ticks = ms;
    else
	ticks = ((long) ms * SysTicksPerSecond() + 999) / 1000L;

    err = SerReceiveWait(refNum, 1L, ticks);
    if (!err)
	SerReceive(refNum, cp, 1L, ticks, &err);
    if (err == serErrLineErr)
	SerReceiveFlush(refNum, 1);
    code = (err ? SCTIMEO : SCEOK);

    return code;
}

/* write one byte to the card */

int
scputc(int ttyn, int ic)
{
    static unsigned char buf[2];
    Err err;

    buf[0] = ic;

    SerSend(refNum, buf, 1L, &err);
    if (err == serErrLineErr)
	SerReceiveFlush(refNum, 1);
    else if (err)
	return SCENOTTY;

    SerSendWait(refNum, -1);

    return SCEOK;
}

void
scsleep(int ms)
{
    long ticks;

    if (ms <= 0)
	return;
    ticks = ((long) ms * SysTicksPerSecond()) / 1000L;
    if (!ticks)
	ticks = 1;
    SysTaskDelay(ticks);
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
