/*
 * Smartcard snooper.
 * Written by Jim Rees <rees@umich.edu>.
 */

#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <System/DataMgr.h>
#include <System/SysEvtMgr.h>
#include <UI/UIAll.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/sys_types.h>

#include "dogg.h"
#include "scrw.h"
#include "field.h"

#define MAXCOLLECT 400
#define MAXMEMO 3000
#define DFLTSPD 9600

/* This will pull in the clib sprintf instead of using the crippled Pilot rom one */
#undef sprintf
#undef vsprintf

static char monthname[][4] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

extern unsigned char *gbuf;
extern int gi, gn, gmax;

DmOpenRef MemoDB;
FieldPtr ansfield, statefield;
int port = 0;
int timeo = 20;
unsigned char bigbuf[2000];
char *memop;
int speed, memolen;
ULong mtime;

static Boolean dogg(EventPtr event);
int myprintf(int d);

DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    int err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
	err = StartApplication();
	if (err)
	    return err;
	EventLoop();
	StopApplication();
	return 0;
    } else
	return sysErrParamErr;
}

StartApplication()
{
    FrmGotoForm(formID_dogg);
    return 0;
}

StopApplication()
{
    if (memop)
	MemPtrFree((VoidPtr) memop);
    return 0;
}

EventLoop()
{
    short err;
    int formID;
    FormPtr form;
    EventType event;

    do {
	EvtGetEvent(&event, 200);

	if (SysHandleEvent(&event))
	    continue;
	if (MenuHandleEvent((void *)0, &event, &err))
	    continue;

	if (event.eType == frmLoadEvent) {
	    formID = event.data.frmLoad.formID;
	    form = FrmInitForm(formID);
	    FrmSetActiveForm(form);
	    switch (formID) {
	    case formID_dogg:
		FrmSetEventHandler(form, (FormEventHandlerPtr) dogg);
		break;
	    }
	}
	FrmDispatchEvent(&event);
    } while (event.eType != appStopEvent);
}

static Boolean
dogg(EventPtr event)
{
    FormPtr form = FrmGetActiveForm();
    Word item;
    FieldPtr field;
    int id, handled = 0;

    item = FrmGetFocus(form);
    if (item == noFocus)
	field = printfield;
    else
	field = FrmGetObjectPtr(form, item);

    switch (event->eType) {
    case frmOpenEvent:
	CtlSetValue(FrmGetObjectPtr(form, FrmGetObjectIndex(form, 2000 + DFLTSPD / 100)), 1);
	speed = DFLTSPD;
	printfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_log));
	ansfield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_ans));
	statefield = FrmGetObjectPtr(form, FrmGetObjectIndex(form, fieldID_state));
	FrmDrawForm(form);
	handled = 1;
	break;

    case ctlSelectEvent:  // A control button was pressed and released.
	id = event->data.ctlEnter.controlID;
	if (id == buttonID_start) {
	    doggydo();
	    handled = 1;
	} else if (id == buttonID_stop) {
	    handled = 1;
	} else if (id == buttonID_compost) {
	    hidefield(fieldID_log);
	    setfield(printfield, NULL);
	    compost();
	    FldSetScrollPosition(printfield, 0);
	    showfield(fieldID_log);
	    handled = 1;
	} else if (id == buttonID_save) {
	    doggydump();
	    handled = 1;
	} else if (id / 1000 == 2) {
	    /* speed select pushbutton */
	    speed = (id % 1000) * 100;
	    handled = 1;
	}
	break;

    case keyDownEvent:
	handled = scrollfield(field, event->data.keyDown.chr);
	break;

    case menuEvent:
	switch (event->data.menu.itemID) {
	case menuitemID_about:
	    FrmAlert(alertID_about);
	    break;
	case menuitemID_cut:
	    FldCut(field);
	    break;
	case menuitemID_copy:
	    FldCopy(field);
	    break;
	case menuitemID_paste:
	    FldPaste(field);
	    break;
	}
	handled = 1;
	break;

    }
    return handled;
}

doggydo()
{
    int fd, n, i, err;
    unsigned char buf[MAXCOLLECT];

    setfield(statefield, "opening");

    fd = scopen(port, SCODCD, NULL);
    if (fd < 0) {
	printf("can't open serial port\n");
	return;
    }

    if (speed != 9600)
	scsetspeed(fd, speed);

    gbuf = bigbuf;
    gn = gi = 0;
    gmax = sizeof bigbuf;

    if (!memop)
	memop = (char *) MemPtrNew((long) MAXMEMO);
    *memop = '\0';
    memolen = 0;

    setfield(statefield, "listening...");
    mtime = TimGetSeconds();

    while (1) {
	err = scgetc(fd, &buf[0], 200);
	if (EvtSysEventAvail(true))
	    break;
	if (err == SCTIMEO)
	    continue;
#ifdef STOP_ON_ERRORS
	if (err != SCEOK)
	    break;
#endif

	n = 1;
	while (scgetc(fd, &buf[n], timeo) == SCEOK && n < MAXCOLLECT)
	    n++;

	if (!n)
	    continue;

	for (i = 0; i < n; i++)
	    myprintf(buf[i]);
	myprintf(-1);
    }

    if (err == SCTIMEO)
	printf("stopped on event\n");
    else
	printf("%s\n", scerrtab[err]);
    scclose(fd);
    setfield(statefield, "closed");
}

doggydump()
{
    int i, clen;
    UInt index = 1000;
    VoidHand RecHandle;
    Ptr RecPointer;
    char title[28];
    DateTimeType dt;

    TimSecondsToDateTime(mtime, &dt);

    /* Save raw trace */

    if (memop && memolen) {
	MemoDB = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
	if (!MemoDB) {
	    printf("save failed\n");
	    return;
	}

	/* Allocate new memo */
	RecHandle = DmNewRecord(MemoDB, &index, sizeof title + memolen + 1);
	RecPointer = MemHandleLock(RecHandle);

	/* Write the title */
	sprintf(title, "trace %d-%s %d:%02d", dt.day, monthname[dt.month - 1], dt.hour, dt.minute);
	for (i = strlen(title); i < sizeof title - 1; i++)
	    title[i] = ' ';
	title[sizeof title - 1] = '\n';
	DmWrite(RecPointer, 0, title, sizeof title);

	/* Write the raw trace data */
	DmWrite(RecPointer, sizeof title, memop, memolen + 1);

	/* Release the record and close the database */
	MemPtrUnlock(RecPointer);
	DmReleaseRecord(MemoDB, index, true);
	DmCloseDatabase(MemoDB);
    }

    /* Save the compost too */

    clen = FldGetTextLength(printfield);
    if (clen) {
	sprintf(title, "compost %d:%02d\n", dt.hour, dt.minute);
	memofield(printfield, title);
    }

    if (memolen || clen)
	printf("saved %d trace and %d compost\n", memolen, i);

    *memop = '\0';
    memolen = 0;
}

/* Format and append to linebuf.  At end-of-line write to ansfield and to memop. */

int
myprintf(int d)
{
    static int linei;
    static char linebuf[200];

    if (d >= 0) {
	/* append to raw buf */
	if (gn < gmax)
	    gbuf[gn++] = d;
	/* append to linebuf */
	if (linei < sizeof linebuf - 2) {
	    linebuf[linei++] = itox(d >> 4);
	    linebuf[linei++] = itox(d & 0xf);
	    linebuf[linei] = '\0';
	}
    }
    if (d < 0 || linei >= sizeof linebuf - 2) {
	/* end of line or out of space; write it out */
	linebuf[linei] = '\0';
	setfield(ansfield, linebuf);
	if (memolen + linei < MAXMEMO) {
	    strcpy(memop + memolen, linebuf);
	    memolen += linei;
	    memop[memolen++] = '\n';
	    memop[memolen] = '\0';
	}
	linei = 0;
    }
}

itox(int d)
{
    return (d > 9 ? d - 10 + 'a' : d + '0');
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
