/*
 * Routines for printing to fields.
 * Jim Rees, University of Michigan CITI, September 1999
 */

#pragma pack(2)

#if (__GNUC_MINOR__ > 7)
#include <PalmOS.h>
#else
#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#define winUp up
#define winDown down
#define WinDirectionType DirectionType
#endif
#include <System/DataMgr.h>
#include <System/MemoryMgr.h>

#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/sys_types.h>

#include "field.h"

#define MAXMEMO 3000


/* Print in to a field */

palmprintf(const char *fmt, ...)
{
    va_list args;
    char buf[1024];

    va_start(args, fmt);
    vsprintf(buf, fmt, args);
    va_end(args);

    return putscrollfield(printfield, buf, FLDFLUSH);
}

prfield(FieldPtr fp, const char *fmt, ...)
{
    va_list args;
    char buf[1024];

    va_start(args, fmt);
    vsprintf(buf, fmt, args);
    va_end(args);

    return putfield(fp, buf);
}

/* Replace existing text with new text */

setfield(FieldPtr fp, char *s)
{
    FieldAttrType attr;
    int len0, len1;

    FldGetAttributes(fp, &attr);
    if (attr.editable) {
	len0 = FldGetTextLength(fp);
	len1 = s ? strlen(s) : 0;
	if (len1) {
	    FldSetSelection(fp, 0, len0);
	    FldInsert(fp, s, strlen(s));
	} else
	    FldDelete(fp, 0, len0);
    } else
	FldSetTextPtr(fp, s);
    FldRecalculateField(fp, true);
    if (!attr.editable)
	FrmDrawForm(FrmGetActiveForm());
    return 0;
}

putscrollfield(FieldPtr fp, char *s, int flags)
{
    FieldAttrType attr;
    int len0, len1, n, maxchars;

    FldGetAttributes(fp, &attr);
    if (!attr.editable || attr.singleLine)
	return setfield(fp, s);

    len0 = FldGetTextLength(fp);
    len1 = strlen(s);

    maxchars = FldGetMaxChars(fp);
    if (len0 + len1 + 1 > maxchars) {
	/* Too much text, delete some */
	n = maxchars / 3;
	FldDelete(fp, 0, n);
	len0 -= n;
    }
    FldSetInsertionPoint(fp, len0);
    if ((flags & FLDADDNL) && len0) {
	FldInsert(fp, "\n", 1);
	FldSetInsertionPoint(fp, ++len0);
    }
    FldInsert(fp, s, len1);
    if (flags & FLDFLUSH) {
	if (len0 + len1 >= maxchars / 2) {
	    /* Enough text to fill the field, make sure that it does */
	    n = FldGetNumberOfBlankLines(fp);
	    if (n)
		FldScrollField(fp, n, winUp);
	}
	FldRecalculateField(fp, true);
    }
    return 0;
}

/* Add text to the end of a multi-line editable field */

putfield(FieldPtr fp, char *s)
{
    return putscrollfield(fp, s, (FLDFLUSH | FLDADDNL));
}

scrollfield(FieldPtr fp, int dirchr)
{
    WinDirectionType dir;

    if (dirchr == pageUpChr)
	dir = winUp;
    else if (dirchr == pageDownChr)
	dir = winDown;
    else
	return 0;
    FldScrollField(fp, (FldGetVisibleLines(fp) + 1) / 2, dir);
    return 1;
}

memofield(FieldPtr fp, char *title)
{
    DmOpenRef MemoDB;
    UInt16 index = dmMaxRecordIndex;
    void *RecHandle;
    void *RecPointer;
    UInt32 tlen, len;

    len = FldGetTextLength(fp);
    if (!len)
	return -1;
    tlen = title ? strlen(title) : 0;
    if (tlen + len > MAXMEMO)
	len = MAXMEMO - tlen;

    /* Open memo database */
    MemoDB = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
    if (!MemoDB)
	return -1;

    /* Allocate new memo */
    RecHandle = DmNewRecord(MemoDB, &index, tlen + len);
    RecPointer = MemHandleLock(RecHandle);

    /* Write title */
    if (tlen)
	DmWrite(RecPointer, 0, title, tlen);

    /* Write body */
    DmWrite(RecPointer, tlen, FldGetTextPtr(fp), len);

    /* Release and close */
    MemPtrUnlock(RecPointer);
    DmReleaseRecord(MemoDB, index, true);
    DmCloseDatabase(MemoDB);
    return 0;
}

hidefield(UInt16 r)
{
    FormPtr form = FrmGetActiveForm();
    UInt16 fldidx = FrmGetObjectIndex(form, r);

    FrmHideObject(form, fldidx);
}

showfield(UInt16 r)
{
    FormPtr form = FrmGetActiveForm();
    UInt16 fldidx = FrmGetObjectIndex(form, r);
    FieldPtr idxfield = FrmGetObjectPtr(form, fldidx);

    FldRecalculateField(idxfield, true);
    FrmShowObject(form, fldidx);
    FrmSetFocus(form, fldidx);
    FrmDrawForm(form);
}

/*
copyright 2000
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
