/*
 * balance
 * check the balance on an Mcard
 *
 * Jim Rees, University of Michigan, April 1997
 */
static char *rcsid = "$Id: balance.c,v 1.29 2000/08/04 21:33:30 rees Exp $";

#ifdef __palmos__
#pragma pack(2)
#include <Common.h>
#include <System/SysAll.h>
#include <System/Unix/unix_stdio.h>
#include <System/Unix/unix_stdlib.h>
#include <System/Unix/sys_types.h>
#else
#ifdef __unix__
#include <unistd.h>
#include <time.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#endif
#include <scrw.h>

#define RESERVE 125

#ifdef __palmos__
#define printf palmprintf
#undef sprintf
extern char *strdup();
#endif

#ifdef WIN32
/* stuff for getopt() */
char *__progname = "balance";
extern char	*optarg;
#endif

char *fmt_location(int n, int code);
char *fmt_date(unsigned char *buf);
char *fmt_pennies(unsigned char *buf);
int dump_msg(char *msg, unsigned char *p, int n);

int Dflag, aflag, vflag;

struct {
    int l;
    char *s;
} loctab[] = {
    0x16, "Argus 535 candy machine",
    0x18, "CITI snack machine",
    0x24, "Wolverine Towers",	/* money sucker on ground floor */
    0x36, "EECS pop machine",
    0x37, "EECS snack machine",
    0x54, "East Quad MCard Machine",
    0x5d, "NC Commons cash machine",
    0x71, "First of America, Liberty St.",
    0x77, "Fishbowl",
    0x79, "SAB",
    0x7d, "NC Commons",
    0x7e, "Union basement",
    0x9b, "Space Research",
    0, NULL
};

char *monthtab[] = {
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec",
    "???",
    NULL
};

#ifndef __palmos__
usage()
{
    printf("usage: balance [ -1 | -2 ] [-t] [-a] [-v] [-m xx] [-h]\n");
    printf("\t-1: use com 1 (i.e., tty00)\n");
    printf("\t-2: use com 2 (i.e., tty01)\n");
    printf("\t-D: demo\n");
    printf("\t-a: show all transactions\n");
    printf("\t-v: verbose mode\n");
    printf("\t-m xx: debit card xx cents\n");
    printf("\t-h: print this message\n");
}


main(ac, av)
int ac;
char *av[];
{
    unsigned char buf[100];
    int port = 0, fd, flags = SCODSR, i, m = 0, n, err;

    while ((i = getopt(ac, av, "12Davm:h")) != -1) {
	switch (i) {
	case '1':
	case '2':
	    port = i - '1';
	    break;
	case 'D':
	    Dflag = 1;
	    break;
	case 'a':
	    aflag = 1;
	    break;
	case 'v':
	    vflag = 1;
	    break;
	case 'm':
	    m = atoi(optarg);
	    break;
	case 'h':
	    usage();
	    exit(0);
	}
    }

    if (Dflag) {
	demo();
	exit(0);
    }

    /* open reader */
    fd = scopen(port, flags, NULL);
    if (fd < 0) {
	printf("can't open reader\n");
	exit(1);
    }

    /* reset */
    n = screset(fd, buf, &err);
    if (!n) {
	printf("%s\n", scerrtab[err]);
	exit(2);
    }
    if (vflag)
	dump_msg("atr: ", buf, n);

    TryThemAll(fd, buf, n, m);

    scclose(fd);
#ifdef WIN32
    printf("exit code %d; press <enter>: ", rtn);
    fgets(buf, sizeof buf, stdin);
#endif
    exit(0);
}
#endif

TryThemAll(fd, buf, n, m)
int fd, n, m;
unsigned char *buf;
{
    cardtech(fd, buf, n);
    mondex(fd, buf, n);
    visa(fd, buf, n);
    payflex(fd, buf, n, m);
}

static char *demolog[] = {
    "17 Mar 2000 18:50  5.30   -$0.50",
    "16 Mar 2000 11:28  5.80   +$1.00",
    "16 Mar 2000 11:28  4.80   +$1.00",
    "14 Mar 2000 13:06  3.80   -$0.50",
    "14 Mar 2000 13:06  4.30   S$1.25",
    "13 Mar 2000 14:00  5.55   -$0.65",
};

demo()
{
    int i;

    printf("Demo $18.23\n");
    if (aflag) {
	printf("Date        Time  Balance Transaction\n");
	for (i = 0; i < 6; i++)
#ifdef __palmos__
	    printf("%s\n-%s\n", demolog[i], fmt_location(-1, 0x50));
#else
	    printf("%s  %s\n", demolog[i], fmt_location(-1, 0x50));
#endif
    }
    return 0;
}

cardtech(fd, buf, n)
int fd, n;
unsigned char *buf;
{
    int nhb, nr, r1, r2;
    unsigned char *hbp;
    static unsigned char mhb[] = {0x80, 0x65, 0xa2};
    static unsigned char aid[] = {0xa0, 0x00, 0x00, 0x00, 0x40, 0x01, 0x01,
				      0, 0, 0, 0, 0, 0, 0, 0, 0};
    static unsigned char tbuf[16];

    /* Find the historical bytes */
    nhb = buf[1] & 0xf;
    hbp = &buf[n - nhb];

    /* Identify Cardtech by first three HB */
    if (memcmp(hbp, mhb, sizeof mhb))
	return -1;

    /* Select aid */
    nr = scwrite(fd, 0, 0xa4, 4, 0, sizeof aid, aid, &r1, &r2);
    if (nr < 0) {
	printf("Cardtech aid scwrite failed\n");
	return -1;
    }
    if (r1 != 0x90 && r1 != 0x61) {
	if (vflag)
	    printf("Cardtech aid select: %s\n", get_r1r2s(r1, r2));
	return -1;
    }

    /* Select fid 4.24 */
    tbuf[0] = 0x04;
    tbuf[1] = 0x24;
    nr = scwrite(fd, 0, 0xa4, 2, 0, 2, tbuf, &r1, &r2);
    if (nr < 0) {
	printf("Cardtech fid scwrite failed\n");
	return -1;
    }
    if (r1 != 0x90 && r1 != 0x61) {
	if (vflag)
	    printf("Cardtech fid select: %s\n", get_r1r2s(r1, r2));
	return -1;
    }

    /* Read record containing balance */
    nr = scread(fd, 0, 0xb0, 0, 8, 16, tbuf, &r1, &r2);
    if (nr < 0) {
	printf("Cardtech scread failed\n");
	return -1;
    }
    if (r1 != 0x90 && r1 != 0x61) {
	if (vflag)
	    printf("Cardtech purse read: %s\n", get_r1r2s(r1, r2));
	return -1;
    }
    printf("Cardtech $%s\n", fmt_pennies(&tbuf[10]));
    return 0;
}

mondex(fd, buf, n)
int fd, n;
unsigned char *buf;
{
    int nhb;
    unsigned char *hbp;
    static unsigned char mhb[] = {0x80, 0x31, 0xe0};

    /* Find the historical bytes */
    nhb = buf[1] & 0xf;
    hbp = &buf[n - nhb];

    /* Identify Mondex by first three HB */
    if (memcmp(hbp, mhb, sizeof mhb) && memcmp(--hbp, mhb, sizeof mhb))
	return -1;

    printf("Mondex ");
    if (nhb >= 13)
	printf("%.3s %s\n", &hbp[4], fmt_pennies(&hbp[11]));
    else
	printf("? (short atr)\n");
    return 0;
}

visa(fd, buf, n)
int fd, n;
unsigned char *buf;
{
    int nhb, nr, r1, r2;
    unsigned char *hbp;
    static unsigned char mhb[] = {0x80, 0x31, 0xc0};
    static unsigned char aid[] = {0xa0, 0x00, 0x00, 0x00, 0x03, 0x60, 0x10};

    /* Find the historical bytes */
    nhb = buf[1] & 0xf;
    hbp = &buf[n - nhb];

    /* Identify Visacash by first three HB */
    if (memcmp(hbp, mhb, sizeof mhb) && memcmp(--hbp, mhb, sizeof mhb))
	return -1;

    if (nhb >= 11) {
	if (vflag)
	    printf("using atr method\n");
	printf("Visa Cash $%s\n", fmt_pennies(&hbp[9]));
	return 0;
    }

    if (vflag)
	printf("short atr; using purse method\n");

    nr = scwrite(fd, 0, 0xa4, 4, 0, 7, aid, &r1, &r2);
    if (nr < 0) {
	printf("visa scwrite failed\n");
	return -1;
    }
    if (r1 != 0x90) {
	if (vflag)
	    printf("visa aid select: %s\n", get_r1r2s(r1, r2));
	return -1;
    }
    nr = scread(fd, 0, 0xb0, 0x93, 0, 11, buf, &r1, &r2);
    if (nr < 0) {
	printf("visa scread failed\n");
	return -1;
    }
    if (r1 != 0x90) {
	if (vflag)
	    printf("visa purse read: %s\n", get_r1r2s(r1, r2));
	return -1;
    }
    printf("Visa Cash $%s\n", fmt_pennies(&buf[2]));
    return 0;
}

payflex(fd, buf, n, m)
int fd, n, m;
unsigned char *buf;
{
    int i, cash;

    /* open 0.2 */
    if (open_file(fd, 0, 2) < 0)
	return -1;
    else {
	if (read_rec(fd, -1, buf, 0x8) < 0)
	    printf("can't read 0.2\n", n);
	else {
	    if (vflag) {
		printf("Card serial no. %u\n", (buf[2] << 8 | buf[3]));
		dump_msg("0.2: ", buf, 0x8);
	    }
	}
    }

    if (vflag) {
	/* open 1.0/1.1 */
	open_file(fd, 0x3f, 0);
	if (open_file(fd, 1, 0) < 0)
	    return -1;
	if (open_file(fd, 1, 1) < 0)
	    return -1;
	else {
	    if (read_rec(fd, 0, buf, 0xa) < 0)
		printf("can't read 1.1\n", n);
	    else
		dump_msg("1.1: ", buf, 0xa);
	}
    }

    open_file(fd, 0x3f, 0);
    if (open_file(fd, 2, 0) < 0)
	return -1;

    /* open purse file */
    if (open_file(fd, 2, 1) < 0)
	return -1;
    else {
	if (read_rec(fd, -1, buf, 0x6) < 0)
	    printf("can't read purse file\n", n);
	else {
	    printf("Payflex $%s\n", fmt_pennies(&buf[4]));
	    if (vflag)
		printf("transaction %d\n", hex(buf));
	    cash = hex(&buf[4]);
	}
    }

    if (aflag) {
	/* open transaction log */
	if (open_file(fd, 2, 2) < 0) {
	    printf("can't open transaction log\n");
	    return -1;
	}

	printf("Date        Time  Balance Transaction\n");
	for (i = 0; i < 10; i++)
	    if (print_rec(fd, buf, i) < 0)
		break;
    }

    if (m)
	payflex_debit(fd, buf, m, cash);

    return 0;
}

payflex_debit(fd, buf, m, cash)
int fd, m, cash;
unsigned char *buf;
{
#ifdef __unix__
    int n, r1, r2;

    if (!m)
	return 0;

    /* Pretend we're a vending machine, and debit the card */

    /* update log 1 */
    if (update_log(fd, buf, RESERVE, cash - RESERVE, 0x76) < 0) {
	printf("Jackpot!\n");
	return 0;
    }

    /* delay for a bit */
    printf("making selection...\n");
    sleep(2);

    /* open purse */
    open_file(fd, 0x3f, 0);
    open_file(fd, 2, 0);
    open_file(fd, 2, 1);

    /* get challenge, generate auth data */
    get_challenge(fd, buf);
    buf[0] = buf[1] = 0;
    buf[2] = m >> 8;
    buf[3] = m & 0xff;
    gen_authdata(fd, &buf[4], 6);

    /* debit */
    n = scwrite(fd, 0, 0x54, 0, 0, 10, buf, &r1, &r2);
    if (n != 10 || r1 != 144 || r2 != 0)
	printf("debit failed %d %s\n", n, get_r1r2s(r1, r2));

    /* update log 2 */
    update_log(fd, buf, m, cash - m, 0x50);
#endif
    return 0;
}

read_rec(fd, rn, buf, n)
int fd, rn, n;
unsigned char *buf;
{
    int len, r1, r2;

    len = scread(fd, 0, 0xb2, rn + 1, (rn < 0) ? 0 : 4, n, buf, &r1, &r2);
    if (len != n || r1 != 144 || r2 != 0) {
	printf("read %d: %s\n", len, get_r1r2s(r1, r2));
	return -1;
    }
    return 0;
}

print_rec(fd, buf, n)
int fd, n;
unsigned char *buf;
{
    int c;

    if (read_rec(fd, n, buf, 0xf) < 0) {
	printf("can't read transaction %d\n", n);
	return -1;
    }

    if (buf[0] == 0)
	/* empty record */
	return -1;

    switch (buf[14]) {
    case 0x4c:
	c = '+';
	break;
    case 0x49:
	c = 'I';
	break;
    case 0x50:
	c = '-';
	break;
    case 0x45:
    case 0x76:
	c = '>';
	break;
    case 0x44:
	c = 'S';
	break;
    default:
	c = '?';
	break;
    }

    /* don't bother with uninteresting records unless -v */
    if (!vflag && c == '>')
	return 0;

    printf(
#ifdef __palmos__
	   "%s  %s   %c$%s\n-%s\n",
#else
	   "%s  %s   %c$%s  %s\n",
#endif
	   fmt_date(buf),
	   fmt_pennies(&buf[11]), c,
	   fmt_pennies(&buf[6]),
	   fmt_location(buf[9], buf[14]));
    if (vflag && c == '?')
	printf("  code 0x%2x\n", (int) buf[14]);

    return 0;
}

char *
fmt_location(int n, int code)
{
    int i;
    static char buf[32];
    static int r;

    if (n == -1) {
	/* Pick one for demo purposes */
	return loctab[semirandom() % 13].s;
    }

    if (n == 0)
	return (code == 0x50) ? "Wendy's" : "ISSUED";

    for (i = 0; loctab[i].s; i++)
	if (loctab[i].l == n)
	    break;

    if (loctab[i].s)
	return loctab[i].s;

    sprintf(buf, "unknown location 0x%x", n);
    return buf;
}

open_file(fd, f0, f1)
int fd, f0, f1;
{
    char buf[2];
    int len, r1, r2;

    buf[0] = f0;
    buf[1] = f1;
    len = scwrite(fd, 0, 0xa4, 0, 0, 2, buf, &r1, &r2);
    if (len != 2 || r1 != 144 || r2 != 0) {
	if (vflag)
	    printf("open->scwrite %d %s\n", len, get_r1r2s(r1, r2));
	return -1;
    }
    return 0;
}

get_challenge(int fd, unsigned char *buf)
{
    int n, r1, r2;

    n = scread(fd, 0, 0x84, 0, 0, 8, buf, &r1, &r2);
    if (n != 8 || r1 != 144 || r2 != 0)
	printf("get challenge returns %d %s\n", n, get_r1r2s(r1, r2));
    else if (vflag)
	dump_msg("challenge: ", buf, n);
}

#ifdef __unix__
gen_authdata(int fd, unsigned char *buf, int len)
{
    int i;

    for (i = 0; i < len; i++)
	buf[i] = random() & 0xff;
}

update_log(fd, buf, m1, m2, type)
int fd;
unsigned char *buf;
int m1, m2, type;
{
    int n, r1, r2;
    struct tm *tm;
    time_t now;

    open_file(fd, 0x3f, 0);
    open_file(fd, 2, 0);

    /* open transaction log */
    if (open_file(fd, 2, 2) < 0) {
	printf("can't open transaction log\n");
	return -1;
    }

    /* external authentication, key no. 0 */
    get_challenge(fd, buf);
    gen_authdata(fd, buf, 6);
    n = scwrite(fd, 0, 0x82, 0, 0, 6, buf, &r1, &r2);
    if (n != 6 || r1 != 144 || r2 != 0)
	printf("external authentication failed %d %s\n", n, get_r1r2s(r1, r2));

    time(&now);
    tm = localtime(&now);

    bzero(buf, 15);
    buf[0] = dcb(tm->tm_mon + 1);
    buf[1] = dcb(tm->tm_mday);
    buf[2] = dcb(tm->tm_year % 100);
    buf[3] = dcb(tm->tm_hour);
    buf[4] = dcb(tm->tm_min);
    buf[6] = m1 >> 8;
    buf[7] = m1 & 0xff;
    buf[9] = 0x18; /* CITI snack machine */
    buf[11] = m2 >> 8;
    buf[12] = m2 & 0xff;
    buf[14] = type;

    n = scwrite(fd, 0, 0xdc, 0, 2, 15, buf, &r1, &r2);
    if (n != 15 || r1 != 144 || r2 != 0) {
	printf("card write failed %d %s\n", n, get_r1r2s(r1, r2));
	return -1;
    }

    return n;
}
#endif

char *
fmt_date(unsigned char *buf)
{
    int month, year;
    static char s[32];

    year = bcd(buf[2]) + 1900;
    if (year < 1950)
	year += 100;

    month = bcd(buf[0]) - 1;
    if (month < 0 || month >= 12)
	month = 12;

    sprintf(s, "%2d %s %d %2d:%02d",
	   bcd(buf[1]),
	   monthtab[month],
	   year,
	   bcd(buf[3]),
	   bcd(buf[4]));
    return s;
}

char *
fmt_pennies(unsigned char *buf)
{
    int pennies;
    char s[32];

    pennies = hex(buf);
    sprintf(s, "%d.%02d", pennies / 100, pennies % 100);
    return strdup(s);
}

bcd(int x)
{
    return (((x >> 4) & 0xf) * 10 + (x & 0xf));
}

dcb(int x)
{
    return (((x / 10) << 4) | (x % 10));
}

hex(buf)
unsigned char *buf;
{
    return((buf[0] << 8) + buf[1]);
}

dump_msg(char *msg, unsigned char *p, int n)
{
    int i;

    printf("%s", msg);
    for (i = 0; i < n; i++)
	printf("%02x ", p[i]);
    printf("\n");
}

semirandom()
{
    static int n = 12345;

    n = n * 47 + 7;
    return ((n >> 4) & 0xff);
}

/*
copyright 1997, 2000
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
