/*
 * Smartcard snooper
 * Jim Rees, University of Michigan
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <termios.h>

#define MAXCOLLECT 400

#ifdef __linux
static char ttynametmpl[] = "/dev/cua%01d";
#else
static char ttynametmpl[] = "/dev/cua%02d";
#endif

int port = 0;
int timeo = 20;
int speed = 9600;
int vflag;
struct termios tio0, tio1;

int scfdopen(int fd, int speed);
int scgetc(int fd, unsigned char *cp, int ms);

main(ac, av)
int ac;
char *av[];
{
    int fd, i, n;
    char buf[MAXCOLLECT];
    struct timeval tv0, tv1, tvd;

    while ((i = getopt(ac, av, "1234s:t:v")) != -1) {
	switch (i) {
	case '1':
	case '2':
	case '3':
	case '4':
	    port = i - '1';
	    break;
	case 's':
	    speed = atoi(optarg);
	    break;
	case 't':
	    timeo = atoi(optarg);
	    break;
	case 'v':
	    vflag = 1;
	    break;
	}
    }

    setlinebuf(stdout);

    sprintf(buf, ttynametmpl, port);
    fd = open(buf, (O_RDWR | O_NONBLOCK), 0);
    if (fd < 0) {
	fprintf(stderr, "can't open port %d\n", port);
	exit(1);
    }

#if (B9600 != 9600)
    speed = B9600;
#endif
    if (scfdopen(fd, speed) < 0) {
	fprintf(stderr, "can't make tty happy\n");
	exit(1);
    }

    tv0.tv_sec = 0;

    while (1) {
	if (scgetc(fd, &buf[0], -1) < 0)
	    break;
	if (tv0.tv_sec == 0)
	    gettimeofday(&tv0, NULL);

	n = 1;
	while (scgetc(fd, &buf[n], timeo) == 1 && n < MAXCOLLECT)
	    n++;
	gettimeofday(&tv1, NULL);
	timersub(&tv1, &tv0, &tvd);

	if (vflag)
	    printf("%02d%03dms ", tvd.tv_sec, tvd.tv_usec / 1000);

	for (i = 0; i < n; i++)
	    printf("%02x ", buf[i] & 0xff);
	printf("\n");
    }

    tcsetattr(fd, TCSANOW, &tio0);
    close(fd);
    exit(0);
}

int
scfdopen(int fd, int speed)
{
    struct termios t;

    /* Get and save the tty state */

    if (tcgetattr(fd, &t) < 0)
	return -1;

    tio0 = t;

    /* Now put the tty in a happy ISO state */

    cfsetispeed(&t, speed);
    cfsetospeed(&t, speed);

    /* raw 8/E/2 */
    t.c_iflag &= ~(ISTRIP|INLCR|IGNCR|ICRNL|IXON);
    t.c_iflag |= (IGNBRK|IGNPAR);
    t.c_oflag &= ~OPOST;
    t.c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
#ifdef CHWFLOW
    t.c_cflag &= ~CHWFLOW;
#endif
#ifdef CRTSCTS
    t.c_cflag &= ~CRTSCTS;
#endif
#ifdef CRTSXOFF
    t.c_cflag &= ~CRTSXOFF;
#endif
    t.c_cflag |= CLOCAL;

    /* 8/E/2 */
    t.c_cflag &= ~(CSIZE | PARODD);
    t.c_cflag |= (CS8 | PARENB | CSTOPB);

    t.c_cc[VMIN] = 1;
    t.c_cc[VTIME] = 0;

    if (tcsetattr(fd, TCSANOW, &t) < 0)
	return -1;
    tio1 = t;

    usleep(250 * 1000);
    tcflush(fd, TCIFLUSH);

    return fd;
}

int
scgetc(int fd, unsigned char *cp, int ms)
{
    fd_set fdset;
    struct timeval tv, *tvp;

    FD_ZERO(&fdset);
    FD_SET(fd, &fdset);

    if (ms == -1)
	tvp = NULL;
    else {
	tv.tv_sec = (ms + 1) / 1000;
	tv.tv_usec = (ms % 1000) * 1000;
	tvp = &tv;
    }

    if (select(fd + 1, &fdset, NULL, NULL, tvp) != 1)
	return -1;

    if (read(fd, cp, 1) != 1)
	return -1;

    return 1;
}

/*
copyright 1998
the regents of the university of michigan
all rights reserved

permission is granted to use, copy, create derivative works 
and redistribute this software and such derivative works 
for any purpose, so long as the name of the university of 
michigan is not used in any advertising or publicity 
pertaining to the use or distribution of this software 
without specific, written prior authorization.  if the 
above copyright notice or any other identification of the 
university of michigan is included in any copy of any 
portion of this software, then the disclaimer below must 
also be included.

this software is provided as is, without representation 
from the university of michigan as to its fitness for any 
purpose, and without warranty by the university of 
michigan of any kind, either express or implied, including 
without limitation the implied warranties of 
merchantability and fitness for a particular purpose. the 
regents of the university of michigan shall not be liable 
for any damages, including special, indirect, incidental, or 
consequential damages, with respect to any claim arising 
out of or in connection with the use of the software, even 
if it has been or is hereafter advised of the possibility of 
such damages.
*/
