/*
 * Copyright (c) 2002-2003, Network Appliance, Inc. All rights reserved.
 *
 * This Software is licensed under both of the following two licenses:
 *
 * 1) under the terms of the "Common Public License 1.0" a copy of which is
 *    in the file LICENSE.txt in the root directory. The license is also
 *    available from the Open Source Initiative, see
 *    http://www.opensource.org/licenses/cpl.php.
 * OR
 *
 * 2) under the terms of the "The BSD License" a copy of which is in the file
 *    LICENSE2.txt in the root directory. The license is also available from
 *    the Open Source Initiative, see
 *    http://www.opensource.org/licenses/bsd-license.php.
 *
 * Licensee has the right to choose either one of the above two licenses.
 *
 * Redistributions of source code must retain both the above copyright
 * notice and either one of the license notices.
 *
 * Redistributions in binary form must reproduce both the above copyright
 * notice, either one of the license notices in the documentation
 * and/or other materials provided with the distribution.
 */

/**********************************************************************
 *
 * HEADER: kdat.h
 *
 * PURPOSE: defines the KDAT API
 *
 * Description:
 *
 * Mapping rules:
 *      All global symbols are prepended with "DAT_" or "dat_"
 *      All DAT objects have an 'api' tag which, such as 'ep' or 'lmr'
 *      The method table is in the provider definition structure.
 *
 *
 *
 **********************************************************************/

#ifndef _KDAT_H_
#define _KDAT_H_

#include <dat/kdat_config.h>

#include <dat/dat_platform_specific.h>

typedef enum dat_mem_type
{
	/* Shared between udat and kdat */
    DAT_MEM_TYPE_VIRTUAL        = 0x00,
    DAT_MEM_TYPE_LMR            = 0x01,
	/* kdat specific */
    DAT_MEM_TYPE_PHYSICAL       = 0x10,
    DAT_MEM_TYPE_PLATFORM       = 0x20,
    DAT_MEM_TYPE_IA             = 0x40,
    DAT_MEM_TYPE_BYPASS         = 0x80
} DAT_MEM_TYPE;

/* dat handle types */
typedef enum dat_handle_type
{
    DAT_HANDLE_TYPE_CR,
    DAT_HANDLE_TYPE_EP,
    DAT_HANDLE_TYPE_EVD,
    DAT_HANDLE_TYPE_IA,
    DAT_HANDLE_TYPE_LMR,
    DAT_HANDLE_TYPE_PSP,
    DAT_HANDLE_TYPE_PZ,
    DAT_HANDLE_TYPE_RMR,
    DAT_HANDLE_TYPE_RSP
} DAT_HANDLE_TYPE;

typedef enum dat_evd_param_mask
{
    DAT_EVD_FIELD_IA_HANDLE             = 0x01,
    DAT_EVD_FIELD_EVD_QLEN              = 0x02,
    DAT_EVD_FIELD_UPCALL_POLICY = 0x04,
    DAT_EVD_FIELD_UPCALL                = 0x08,
    DAT_EVD_FIELD_EVD_FLAGS             = 0x10,

    DAT_EVD_FIELD_ALL                   = 0x1F

} DAT_EVD_PARAM_MASK;


#include <dat/dat.h>

typedef DAT_HANDLE      DAT_CNO_HANDLE;

/* Upcall support */

typedef enum dat_upcall_policy
{
    DAT_UPCALL_DISABLE          = 0,	/* support no_upcalls           */
    DAT_UPCALL_SINGLE_INSTANCE  = 1,	/* support only one upcall      */
    DAT_UPCALL_MANY             = 100,	/* support multiple upcalls     */
    DAT_UPCALL_TEARDOWN         = -1	/* support no upcalls & return  *
					     * after all in rogress UpCalls *
					     * return                       */
} DAT_UPCALL_POLICY;

typedef void (*DAT_UPCALL_FUNC) (
	DAT_PVOID,		/* instance_data        */
	const DAT_EVENT *,	/* event                */
	DAT_BOOLEAN);		/* more_events          */

typedef struct dat_upcall_object
{
    DAT_PVOID       instance_data;
    DAT_UPCALL_FUNC upcall_func;
} DAT_UPCALL_OBJECT;

/* Define  NULL upcall */

#define DAT_UPCALL_NULL \
	(DAT_UPCALL_OBJECT) { (DAT_PVOID) NULL, \
        (DAT_UPCALL_FUNC) NULL}

typedef struct dat_evd_param
{
    DAT_IA_HANDLE       ia_handle;
    DAT_COUNT           evd_qlen;
    DAT_UPCALL_POLICY   upcall_policy;
    DAT_UPCALL_OBJECT   upcall;
    DAT_EVD_FLAGS       evd_flags;
} DAT_EVD_PARAM;

#include <dat/kdat_vendor_specific.h>

/* General Provider attributes. kdat specific. */
typedef struct dat_provider_attr
{
    char                                provider_name[DAT_NAME_MAX_LENGTH];
    DAT_UINT32                          provider_version_major;
    DAT_UINT32                          provider_version_minor;
    DAT_UINT32                          dapl_version_major;
    DAT_UINT32                          dapl_version_minor;
    DAT_MEM_TYPE                        lmr_mem_types_supported;
    DAT_IOV_OWNERSHIP                   iov_ownership_on_return;
    DAT_QOS                             dat_qos_supported;
    DAT_COMPLETION_FLAGS                completion_flags_supported;
    DAT_BOOLEAN                         is_thread_safe;
    DAT_COUNT                           max_private_data_size;
    DAT_BOOLEAN                         supports_multipath;
    DAT_EP_CREATOR_FOR_PSP              ep_creator;
    DAT_UINT32                          optimal_buffer_alignment;
    const DAT_BOOLEAN                   evd_stream_merging_supported[6][6];
    DAT_UPCALL_POLICY                   upcall_policy;
    DAT_COUNT                           num_provider_specific_attr;
    DAT_NAMED_ATTR *                    provider_specific_attr;
} DAT_PROVIDER_ATTR;

typedef enum dat_provider_attr_mask
{
    DAT_PROVIDER_FIELD_PROVIDER_NAME                    = 0x00001,
    DAT_PROVIDER_FIELD_PROVIDER_VERSION_MAJOR           = 0x00002,
    DAT_PROVIDER_FIELD_PROVIDER_VERSION_MINOR           = 0x00004,
    DAT_PROVIDER_FIELD_DAPL_VERSION_MAJOR               = 0x00008,
    DAT_PROVIDER_FIELD_DAPL_VERSION_MINOR               = 0x00010,
    DAT_PROVIDER_FIELD_LMR_MEM_TYPE_SUPPORTED           = 0x00020,
    DAT_PROVIDER_FIELD_IOV_OWNERSHIP                    = 0x00040,
    DAT_PROVIDER_FIELD_DAT_QOS_SUPPORTED                = 0x00080,
    DAT_PROVIDER_FIELD_COMPLETION_FLAGS_SUPPORTED       = 0x00100,
    DAT_PROVIDER_FIELD_IS_THREAD_SAFE                   = 0x00200,
    DAT_PROVIDER_FIELD_MAX_PRIVATE_DATA_SIZE            = 0x00400,
    DAT_PROVIDER_FIELD_SUPPORTS_MULTIPATH               = 0x00800,
    DAT_PROVIDER_FIELD_EP_CREATOR                       = 0x01000,
    DAT_PROVIDER_FIELD_UPCALL_POLICY                    = 0x02000,
    DAT_PROVIDER_FIELD_OPTIMAL_BUFFER_ALIGNMENT         = 0x04000,
    DAT_PROVIDER_FIELD_EVD_STREAM_MERGING_SUPPORTED     = 0x08000,
    DAT_PROVIDER_FIELD_NUM_PROVIDER_SPECIFIC_ATTR       = 0x10000,
    DAT_PROVIDER_FIELD_PROVIDER_SPECIFIC_ATTR           = 0x20000,

    DAT_PROVIDER_FIELD_ALL                              = 0x3FFFF
} DAT_PROVIDER_ATTR_MASK;


/*
 * Memory types
 *
 * Specifing memory type for LMR create. A consumer must use a single
 * value when registering memory. The union of any of these
 * flags is used in the provider parameter to indicate what memory
 *  type provider supports for LMR memory creation.
 */

/* memory data types */

#define DAT_MEM_OPT DAT_MEM_OPTIMIZE_FLAGS

typedef enum dat_mem_optimize_flags
{
    DAT_MEM_OPTIMIZE_DONT_CARE          = 0x00,
    DAT_MEM_OPTIMIZE_IA                 = 0x01,
    DAT_MEM_OPTIMIZE_MIN_EXPOSURE       = 0x02,
    DAT_MEM_OPTIMIZE_EXACT_EXPOSURE     = 0x04
} DAT_MEM_OPTIMIZE_FLAGS;


typedef union dat_region_description
{
    DAT_PVOID                   for_va;
    DAT_LMR_HANDLE              for_lmr_handle;
    void *                      for_pointer;	/* For kdapl only */
    void *                      for_array;	/* For kdapl only */
    DAT_PADDR                   for_pa;		/* For kdapl only */
} DAT_REGION_DESCRIPTION;

typedef struct dat_lmr_param
{
    DAT_IA_HANDLE                       ia_handle;
    DAT_MEM_TYPE                        mem_type;
    DAT_REGION_DESCRIPTION              region_desc;
    DAT_VLEN                            length;
    DAT_PZ_HANDLE                       pz_handle;
    DAT_MEM_PRIV_FLAGS                  mem_priv;
    DAT_LMR_CONTEXT                     lmr_context;
    DAT_RMR_CONTEXT                     rmr_context;
    DAT_VLEN                            registered_size;
    DAT_VADDR                           registered_address;
} DAT_LMR_PARAM;


/****************************************************************************/

/*
 * Kernel DAT function call definitions,
 */

typedef DAT_RETURN (*DAT_LMR_KCREATE_FUNC) (
	IN      DAT_IA_HANDLE,			/* ia_handle            */
	IN      DAT_MEM_TYPE,			/* mem_type             */
	IN      DAT_REGION_DESCRIPTION,		/* region_description   */
	IN      DAT_VLEN,			/* length               */
	IN      DAT_PZ_HANDLE,			/* pz_handle            */
	IN      DAT_MEM_PRIV_FLAGS,		/* privileges           */
	IN      DAT_MEM_OPTIMIZE_FLAGS,		/* mem_optimization     */
	OUT     DAT_LMR_HANDLE *,		/* lmr_handle           */
	OUT     DAT_LMR_CONTEXT *,		/* lmr_context          */
	OUT     DAT_RMR_CONTEXT *,		/* rmr_context          */
	OUT     DAT_VLEN *,			/* registered_length    */
	OUT     DAT_VADDR * );			/* registered_address   */


typedef DAT_RETURN (*DAT_IA_MEMTYPE_HINT_FUNC) (
	IN      DAT_IA_HANDLE,			/* ia_handle            */
	IN      DAT_MEM_TYPE,			/* mem_type             */
	IN      DAT_VLEN,			/* length               */
	IN      DAT_MEM_OPTIMIZE_FLAGS,		/* mem_optimization     */
	OUT     DAT_VLEN *,			/* preferred_length     */
	OUT     DAT_VADDR * );			/* preferred_alignment  */

typedef DAT_RETURN (*DAT_EVD_KCREATE_FUNC) (
	IN      DAT_IA_HANDLE,			/* ia_handle            */
	IN      DAT_COUNT,			/* evd_min_qlen         */
	IN      DAT_UPCALL_POLICY,		/* upcall_policy        */
	IN      const DAT_UPCALL_OBJECT *,	/* upcall               */
	IN      DAT_EVD_FLAGS,			/* evd_flags            */
	OUT     DAT_EVD_HANDLE * );		/* evd_handle           */

typedef DAT_RETURN (*DAT_EVD_MODIFY_UPCALL_FUNC) (
	IN      DAT_EVD_HANDLE,			/* evd_handle           */
	IN      DAT_UPCALL_POLICY,		/* upcall_policy        */
	IN      const DAT_UPCALL_OBJECT * );	/* upcall               */

/* The following two DAT function calls are also found in udat.h.
 * They were removed from dat.h to remove dependancy issues with
 * dat.h file.  There may be a better way to fix the dependancy
 *
 */

typedef DAT_RETURN (*DAT_IA_QUERY_FUNC) (
	IN      DAT_IA_HANDLE,			/* ia handle            */
	OUT     DAT_EVD_HANDLE *,		/* async_evd_handle     */
	IN      DAT_IA_ATTR_MASK,		/* ia_attr_mask         */
	OUT     DAT_IA_ATTR *,			/* ia_attr              */
	IN      DAT_PROVIDER_ATTR_MASK,		/* provider_attr_mask   */
	OUT     DAT_PROVIDER_ATTR * );		/* provider_attr        */

typedef DAT_RETURN (*DAT_EVD_QUERY_FUNC) (
	IN      DAT_EVD_HANDLE,			/* evd_handle           */
	IN      DAT_EVD_PARAM_MASK,		/* evd_param_mask       */
	OUT     DAT_EVD_PARAM * );		/* evd_param            */

typedef DAT_RETURN (*DAT_LMR_QUERY_FUNC) (
	IN      DAT_LMR_HANDLE,
	IN      DAT_LMR_PARAM_MASK,
	OUT     DAT_LMR_PARAM *);

#include <dat/kdat_redirection.h>

#endif /* _KDAT_H_ */


