/*
 * Copyright (c) 2002-2003, Network Appliance, Inc. All rights reserved.
 *
 * This Software is licensed under both of the following two licenses:
 *
 * 1) under the terms of the "Common Public License 1.0" a copy of which is
 *    in the file LICENSE.txt in the root directory. The license is also
 *    available from the Open Source Initiative, see
 *    http://www.opensource.org/licenses/cpl.php.
 * OR
 *
 * 2) under the terms of the "The BSD License" a copy of which is in the file
 *    LICENSE2.txt in the root directory. The license is also available from
 *    the Open Source Initiative, see
 *    http://www.opensource.org/licenses/bsd-license.php.
 *
 * Licensee has the right to choose either one of the above two licenses.
 *
 * Redistributions of source code must retain both the above copyright
 * notice and either one of the license notices.
 *
 * Redistributions in binary form must reproduce both the above copyright
 * notice, either one of the license notices in the documentation
 * and/or other materials provided with the distribution.
 */

/**********************************************************************
 *
 * HEADER: dat_redirection.h
 *
 * PURPOSE: Defines the common redirection macros
 *
 * Description: Header file for "uDAPL: User Direct Access Programming
 *		Library, Version: 1.1"
 *
 * Mapping rules:
 *      All global symbols are prepended with "DAT_" or "dat_"
 *      All DAT objects have an 'api' tag which, such as 'ep' or 'lmr'
 *      The method table is in the provider definition structure.
 *
 *
 *
 **********************************************************************/

#ifndef _DAT_REDIRECTION_H_
#define _DAT_REDIRECTION_H_

typedef struct dat_provider DAT_PROVIDER;

#ifndef DAT_HANDLE_TO_PROVIDER

/* A utility macro to fetch the Provider Library for any object
 *
 * An alternate version could be defined for single library systems.
 * it would look something like:
 *      extern const struct dat_ia my_single_ia_provider;
 *      #define DAT_HANDLE_TO_PROVIDER(ignore) &my_single_ia_provider
 *
 * This would allow a good compiler to avoid indirection overhead when
 * making function calls.
 */

#define DAT_HANDLE_TO_PROVIDER(handle) (*(DAT_PROVIDER **)(handle))
#endif

#define dat_ia_query(ia, evd, ia_msk, ia_ptr, p_msk, p_ptr) \
        (*DAT_HANDLE_TO_PROVIDER(ia)->ia_query_func)(\
                (ia), \
                (evd), \
                (ia_msk), \
                (ia_ptr), \
                (p_msk), \
                (p_ptr))

#define dat_set_consumer_context(handle, context) \
        (*DAT_HANDLE_TO_PROVIDER(handle)->set_consumer_context_func)(\
                (handle), \
                (context))

#define dat_get_consumer_context(handle, context) \
        (*DAT_HANDLE_TO_PROVIDER(handle)->get_consumer_context_func)(\
                (handle), \
                (context))

#define dat_get_handle_type(handle, handle_type) \
        (*DAT_HANDLE_TO_PROVIDER(handle)->get_handle_type_func)(\
                (handle), \
                (handle_type))

#define dat_cr_query(cr, mask, param) \
        (*DAT_HANDLE_TO_PROVIDER(cr)->cr_query_func)(\
                (cr), \
                (mask), \
                (param))

#define dat_cr_accept(cr, ep, size, pdata) \
        (*DAT_HANDLE_TO_PROVIDER(cr)->cr_accept_func)(\
                (cr), \
                (ep), \
                (size), \
                (pdata))

#define dat_cr_reject(cr) \
        (*DAT_HANDLE_TO_PROVIDER(cr)->cr_reject_func)(\
                (cr))

#define dat_evd_query(evd, mask, param) \
        (*DAT_HANDLE_TO_PROVIDER(evd)->evd_query_func)(\
                (evd), \
                (mask), \
                (param))

#define dat_evd_resize(evd, qsize) \
        (*DAT_HANDLE_TO_PROVIDER(evd)->evd_resize_func)(\
                (evd), \
                (qsize))

#define dat_evd_post_se(evd, event) \
        (*DAT_HANDLE_TO_PROVIDER(evd)->evd_post_se_func)(\
                (evd), \
                (event))

#define dat_evd_dequeue(evd, event) \
        (*DAT_HANDLE_TO_PROVIDER(evd)->evd_dequeue_func)(\
                (evd), \
                (event))

#define dat_evd_free(evd)\
        (*DAT_HANDLE_TO_PROVIDER(evd)->evd_free_func)(\
              (evd))

#define dat_ep_create(ia, pz, in_evd, out_evd, connect_evd, attr, ep) \
         (*DAT_HANDLE_TO_PROVIDER(ia)->ep_create_func)(\
                (ia), \
                (pz), \
                (in_evd), \
                (out_evd), \
                (connect_evd), \
                (attr), \
                (ep))

#define dat_ep_query(ep, mask, param) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_query_func)(\
                (ep), \
                (mask), \
                (param))

#define dat_ep_modify(ep, mask, param) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_modify_func)(\
                (ep), \
                (mask), \
                (param))

#define dat_ep_connect(ep, ia_addr, conn_qual, timeout, psize, pdata, qos, flags) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_connect_func)(\
                (ep), \
                (ia_addr), \
                (conn_qual), \
                (timeout), \
                (psize), \
                (pdata), \
                (qos), \
                (flags))

#define dat_ep_dup_connect(ep, dup, timeout, psize, pdata, qos) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_dup_connect_func)(\
                (ep), \
                (dup), \
                (timeout), \
                (psize), \
                (pdata), \
                (qos))

#define dat_ep_disconnect(ep, flags) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_disconnect_func)(\
                 (ep), \
                 (flags))

#define dat_ep_post_send(ep, size, lbuf, cookie, flags) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_post_send_func)(\
                (ep), \
                (size), \
                (lbuf), \
                (cookie), \
                (flags))

#define dat_ep_post_recv(ep, size, lbuf, cookie, flags) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_post_recv_func)(\
                (ep), \
                (size), \
                (lbuf), \
                (cookie), \
                (flags))

#define dat_ep_post_rdma_read(ep, size, lbuf, cookie, rbuf, flags) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_post_rdma_read_func)(\
                (ep), \
                (size), \
                (lbuf), \
                (cookie), \
                (rbuf), \
                (flags))

#define dat_ep_post_rdma_write(ep, size, lbuf, cookie, rbuf, flags) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_post_rdma_write_func)(\
                (ep), \
                (size), \
                (lbuf), \
                (cookie), \
                (rbuf), \
                (flags))

#define dat_ep_get_status(ep, ep_state, recv_idle, request_idle) \
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_get_status_func)(\
                (ep), \
                (ep_state), \
                (recv_idle), \
                (request_idle))

#define dat_ep_free(ep)\
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_free_func)(\
                (ep))

#define dat_ep_reset(ep)\
        (*DAT_HANDLE_TO_PROVIDER(ep)->ep_reset_func)(\
                (ep))

#define dat_lmr_query(lmr, mask, param)\
        (*DAT_HANDLE_TO_PROVIDER(lmr)->lmr_query_func)(\
                (lmr), \
                (mask), \
                (param))

#define dat_lmr_free(lmr)\
        (*DAT_HANDLE_TO_PROVIDER(lmr)->lmr_free_func)(\
                (lmr))

#define dat_rmr_create(pz, rmr) \
         (*DAT_HANDLE_TO_PROVIDER(pz)->rmr_create_func)(\
                (pz), \
                (rmr))

#define dat_rmr_query(rmr, mask, param) \
         (*DAT_HANDLE_TO_PROVIDER(rmr)->rmr_query_func)(\
                (rmr), \
                (mask), \
                (param))

#define dat_rmr_bind(rmr, lmr, mem_priv, ep, cookie, flags, context) \
        (*DAT_HANDLE_TO_PROVIDER(rmr)->rmr_bind_func)(\
                (rmr), \
                (lmr), \
                (mem_priv), \
                (ep), \
                (cookie), \
                (flags), \
                (context))

#define dat_rmr_free(rmr)\
        (*DAT_HANDLE_TO_PROVIDER(rmr)->rmr_free_func)(\
                (rmr))

#define dat_psp_create(ia, conn_qual, evd, flags, handle) \
        (*DAT_HANDLE_TO_PROVIDER(ia)->psp_create_func)(\
                (ia), \
                (conn_qual), \
                (evd), \
                (flags), \
                (handle))

#define dat_psp_create_any(ia, conn_qual, evd, flags, handle) \
        (*DAT_HANDLE_TO_PROVIDER(ia)->psp_create_any_func)(\
                (ia), \
                (conn_qual), \
                (evd), \
                (flags), \
                (handle))

#define dat_psp_query(psp, mask, param) \
        (*DAT_HANDLE_TO_PROVIDER(psp)->psp_query_func)(\
                (psp), \
                (mask), \
                (param))

#define dat_psp_free(psp)\
        (*DAT_HANDLE_TO_PROVIDER(psp)->psp_free_func)(\
                (psp))

#define dat_rsp_create(ia, conn_qual, ep, evd, handle) \
        (*DAT_HANDLE_TO_PROVIDER(ia)->rsp_create_func)(\
                (ia), \
                (conn_qual), \
                (ep), \
                (evd), \
                (handle))

#define dat_rsp_query(rsp, mask, param) \
        (*DAT_HANDLE_TO_PROVIDER(rsp)->rsp_query_func)(\
                (rsp), \
                (mask), \
                (param))

#define dat_rsp_free(rsp)\
        (*DAT_HANDLE_TO_PROVIDER(rsp)->rsp_free_func)(\
                (rsp))

#define dat_pz_create(ia, pz) \
        (*DAT_HANDLE_TO_PROVIDER(ia)->pz_create_func)(\
                (ia), \
                (pz))

#define dat_pz_query(pz, mask, param) \
        (*DAT_HANDLE_TO_PROVIDER(pz)->pz_query_func)(\
                (pz), \
                (mask), \
                (param))

#define dat_pz_free(pz) \
        (*DAT_HANDLE_TO_PROVIDER(pz)->pz_free_func)(\
                (pz))


#endif /* _DAT_REDIRECTION_H_ */


