/*
  gss_proc.c
  
  Copyright (c) 2000 The Regents of the University of Michigan.
  All rights reserved.

  Copyright (c) 2000 Dug Song <dugsong@UMICH.EDU>.
  All rights reserved, all wrongs reversed.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of the University nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  $Id: gss_proc.c,v 1.15 2001/02/24 22:17:31 andros Exp $
*/

#include <sys/types.h>
#include <rpc/rpc.h>
#include <syslog.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include "gss.h"

/* Default values. */
u_int	nobody_uid;
u_int	nobody_gid;
char   *nobody_user = "nobody";
char   *nobody_group = "nobody";
char	dnsdn[128];

static bool_t
hosts_access(struct svc_req *rqstp)
{
	struct sockaddr_in *sinp;
	SVCXPRT *xprt = rqstp->rq_xprt;
	struct authunix_parms *aup;

	sinp = svc_getcaller(xprt);
	
	/* Require loopback source address, reserved source port. */
	if (ntohl(sinp->sin_addr.s_addr) != INADDR_LOOPBACK ||
	    ntohs(sinp->sin_port) >= IPPORT_RESERVED) {
		goto bad_auth;
	}
	/* Require root AUTH_UNIX credentials. */
	aup = (struct authunix_parms *)rqstp->rq_clntcred;
	if (rqstp->rq_cred.oa_flavor != AUTH_UNIX || aup->aup_uid != 0) {
		syslog(LOG_ERR,"RPC_AUTHA_TOOWEAK: flavor %d uid %d\n",rqstp->rq_cred.oa_flavor,aup->aup_uid);
		goto bad_auth;
	}
	return (TRUE);
	
 bad_auth:
	syslog(LOG_ERR, "connection refused from %s port %d",
	       inet_ntoa(sinp->sin_addr), ntohs(sinp->sin_port));
	svcerr_weakauth(xprt);
	
	return (FALSE);
}

bool_t
gssd_init_defaults(void)
{
	struct passwd *pw;
	struct group *gr;

	if ((pw = getpwnam(nobody_user)) == NULL ||
	    (gr = getgrnam(nobody_group)) == NULL)
		return (FALSE);

	nobody_uid = pw->pw_uid;
	nobody_gid = gr->gr_gid;

	memset(dnsdn, 0, sizeof(dnsdn));
	if (gethostname(dnsdn, sizeof(dnsdn)) < 0)
		return (FALSE);
	dnsdn[sizeof(dnsdn) - 1] = 0;
	if (strlen(dnsdn) == (sizeof(dnsdn) - 1))
		return (FALSE);

	return (TRUE);
}

GSS_BUFFER_T *
uid_to_name_1_svc(u_int *uidp, struct svc_req *rqstp)
{
	static GSS_BUFFER_T name;
	static char user[256];
	struct passwd *pw;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "uid_to_name_svc arg (%d)\n", *uidp);
#endif
	name.GSS_BUFFER_T_val = user;
	
	if ((pw = getpwuid(*uidp)) == NULL) {
		snprintf(user, sizeof(user) - 1, "%s@%s", nobody_user, dnsdn);
		user[sizeof(user) - 1] = '\0';
	}
	else {
		snprintf(user, sizeof(user) - 1, "%s@%s", pw->pw_name, dnsdn);
		user[sizeof(user) - 1] = '\0';
	}
	name.GSS_BUFFER_T_len = strlen(user);
#ifdef DEBUG
	fprintf(stderr, "uid_to_name_svc res (%.*s)\n",
		name.GSS_BUFFER_T_len, name.GSS_BUFFER_T_val);
#endif	
	return (&name);
}

u_int *
name_to_uid_1_svc(GSS_BUFFER_T *namep, struct svc_req *rqstp)
{
	static u_int uid;
	char user[256];
	struct passwd *pw;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "name_to_uid_svc arg (%.*s)\n",
		namep->GSS_BUFFER_T_len, namep->GSS_BUFFER_T_val);
#endif
	uid = nobody_uid;
	
	if (namep->GSS_BUFFER_T_len < sizeof(user) - 1) {
		memcpy(user, namep->GSS_BUFFER_T_val, namep->GSS_BUFFER_T_len);
		user[namep->GSS_BUFFER_T_len] = '\0';
		
		/* XXX - user@REALM handling */
		strtok(user, "@");
		if ((pw = getpwnam(user)) != NULL)
			uid = pw->pw_uid;
	}
#ifdef DEBUG
	fprintf(stderr, "name_to_uid_svc res (%d)\n", uid);
#endif
	return (&uid);
}

GSS_BUFFER_T *
gid_to_name_1_svc(u_int *gidp, struct svc_req *rqstp)
{
	static GSS_BUFFER_T name;
	static char group[256];
	struct group *gr;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "gid_to_name_svc arg (%d)\n", *gidp);
#endif
	name.GSS_BUFFER_T_val = group;
	
	if ((gr = getgrgid(*gidp)) == NULL) {
		snprintf(group, sizeof(group) - 1, "%s@%s", nobody_group, dnsdn);
		group[sizeof(group) - 1] = '\0';
	}
	else {
		snprintf(group, sizeof(group) - 1, "%s@%s", gr->gr_name, dnsdn);
		group[sizeof(group) - 1] = '\0';
	}
	name.GSS_BUFFER_T_len = strlen(group);
#ifdef DEBUG
	fprintf(stderr, "gid_to_name_svc res (%.*s)\n",
		name.GSS_BUFFER_T_len, name.GSS_BUFFER_T_val);
#endif
	return (&name);
}

u_int *
name_to_gid_1_svc(GSS_BUFFER_T *namep, struct svc_req *rqstp)
{
	static u_int gid;
	char group[256];
	struct group *gr;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "name_to_gid_svc arg (%.*s)\n",
		namep->GSS_BUFFER_T_len, namep->GSS_BUFFER_T_val);
#endif
	gid = nobody_gid;
	
	if (namep->GSS_BUFFER_T_len < sizeof(group) - 1) {
		memcpy(group, namep->GSS_BUFFER_T_val,
		       namep->GSS_BUFFER_T_len);
		group[namep->GSS_BUFFER_T_len] = '\0';
		
		if ((gr = getgrnam(group)) != NULL)
			gid = gr->gr_gid;
	}
#ifdef DEBUG
	fprintf(stderr, "name_to_gid_svc res (%d)\n", gid);
#endif
	return (&gid);
}
