/*
  in_gssd_proc.c
  
  Copyright (c) 2000 The Regents of the University of Michigan.
  All rights reserved.

  Copyright (c) 2000 Dug Song <dugsong@UMICH.EDU>.
  Copyright (c) 2001 Andy Adamson <andros@UMICH.EDU>.
  Copyright (c) 2002 Marius Aamodt Eriksen <marius@UMICH.EDU>.
  Copyright (c) 2002 Bruce Fields <bfields@UMICH.EDU>
  All rights reserved, all wrongs reversed.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of the University nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  $Id: gss_proc.c,v 1.26 2002/09/03 20:22:28 marius Exp $
*/

#define _GNU_SOURCE
#include <rpc/rpc.h>
#include <syslog.h>
#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <dirent.h>
#include <poll.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <arpa/inet.h>
#ifdef HAVE_KRB5
#include <gssapi/gssapi.h>
#include <gssapi/gssapi_generic.h>
#define CA_RUN_AS_MACHINE  0x00000200
#elif HAVE_HEIMDAL
#include <gssapi.h>
#define gss_nt_service_name GSS_C_NT_HOSTBASED_SERVICE
#endif
#include <netdb.h>
#include <auth_gss.h>
#include <krb5.h>
#include "in_gssd.h"
#include "util.h"
/* XXX argggg: */

typedef struct _krb5_gss_ctx_id_rec {
	int initiate;
	OM_uint32 gss_flags;
	int seed_init;
	unsigned char seed[16];
	krb5_principal here;
	krb5_principal there;
	krb5_keyblock *subkey;
	int signalg;
	int cksum_size;
	int sealalg;
	krb5_keyblock *enc;
	krb5_keyblock *seq;
	krb5_timestamp endtime;
	krb5_flags krb_flags;
	krb5_ui_4 seq_send;
	krb5_ui_4 seq_recv;
	void *seqstate;
	int established;
	int big_endian;
	krb5_auth_context auth_context;
	gss_buffer_desc *mech_used;
	int nctypes;
	krb5_cksumtype *ctypes;
} krb5_gss_ctx_id_rec, *krb5_gss_ctx_id_t;


/* from our rpcsec_gss auth_gss.c: */

struct rpc_gss_data {
	bool_t			established;
	gss_buffer_desc		gc_wire_verf; 
	CLIENT			*clnt;
	gss_name_t		name;	/* service name */
	struct rpc_gss_sec	sec;	/* security tuple */
	gss_ctx_id_t		ctx;	/* context id */
	struct rpc_gss_cred	gc;	/* client credentials */
	u_int			win;	/* sequence window */
};

#define AUTH_PRIVATE(auth)  ((struct rpc_gss_data *)auth->ah_private)

/* from rpcsec source */
extern gss_OID_desc krb5oid;

/*
 * pollkrrb5
 *      array of struct pollfd suitable to pass to poll. initialized to 
 *      zero - a zero struct is ignored by poll() because the events mask is 0.
 *
 * clnt_list: 
 *      linked list of struct clnt_info which associates a clntXXX directory
 *	with an index into pollkrb5[], and other basic data about that client.
 *
 * Directory structure: created by the kernel nfs client
 *      /RPCSEC_DIR/clntXX             : one per rpc_clnt struct in the kernel
 *      /RPCSEC_DIR/clntXX/krb5        : read uid for which kernel wants
 *      				 a context, write the resulting context
 *      /RPCSEC_DIR/clntXX/info        : stores info such as server name
 *
 * Algorithm:
 *      Poll all RPCSEC_DIR/clntXX/krb5 files.  When ready, data read
 *      is a uid; performs rpcsec_gss context initialization protocol to
 *      get a cred for that user.  Writes result to corresponding krb5 file
 *      in a form the kernel code will understand.
 *      In addition, we make sure we are notified whenever anything is
 *      created or destroyed in RPCSEC_DIR/ or in an of the clntXX directories,
 *      and rescan the whole RPCSEC_DIR when this happens.
 */

struct pollfd * pollkrb5;

int pollsize;  /* the number of FD_ARRAY_BLOCKS */

/* adapted from mit kerberos 5 ../lib/gssapi/mechglue/mglueP.h 
 * this is what gets passed around when the mechglue code is enabled : */
typedef struct gss_union_ctx_id_t {
   gss_OID	mech_type;
   gss_ctx_id_t	internal_ctx_id;
} gss_union_ctx_id_desc, *gss_union_ctx_id_t;


/* GSS server credentials. */
static gss_cred_id_t	gssd_creds;

/* XXX buffer problems: */
static int
read_service_info(char *info_file_name, char **servicename, char **servername,
		  int *prog, int *vers) {
#define INFOBUFLEN 256
	char		buf[INFOBUFLEN];
	int		nbytes;
	static char	dummy[16];
	static char	service[128];
	static char	address[128];
	char		program[16];
	char		version[16];
	in_addr_t	inaddr;
	int		fd = -1;
	struct hostent	*ent;

	if ((fd = open(info_file_name, O_RDONLY)) == -1) {
		syslog(LOG_ERR, "can't open %s", info_file_name);
		goto fail;
	}
	if ((nbytes = read(fd, buf, INFOBUFLEN)) == -1)
		goto fail;
	close(fd);

	if (sscanf(buf,"RPC server: %s\n"
		   "service: %s %s version %s\n"
		   "address: %s",
	           dummy,
	           service, program, version,
	           address) != 5)
		goto fail;

	/* check service, program, and version */
	if(memcmp(service, "nfs", 3)) return -1;
	*prog = atoi(program + 1); /* skip open paren */
	*vers = atoi(version);
	if((*prog != 100003) || ((*vers != 3) && (*vers != 4)))
		goto fail;

	/* create service name */
	inaddr = inet_addr(address);
	if (!(ent = gethostbyaddr(&inaddr, sizeof(inaddr), AF_INET))) {
		syslog(LOG_ERR, "can't resolve server %s name",address);
		goto fail;
	}
	if (!(*servername = calloc(strlen(ent->h_name) + 1, 1)))
		goto fail;
	memcpy(*servername, ent->h_name, strlen(ent->h_name));
	snprintf(buf, INFOBUFLEN, "%s@%s", service, ent->h_name);
	if (!(*servicename = calloc(strlen(buf) + 1, 1)))
		goto fail;
	memcpy(*servicename, buf, strlen(buf));
	return 0;
fail:
	syslog(LOG_ERR, "failed to read service info");
	if (fd != -1) close(fd);
	if (*servername) free(*servername);
	if (*servicename) free(servicename);
	return -1;
}

static void
destroy_client(struct clnt_info *clp)
{
	if (clp->dir_fd != -1) close(clp->dir_fd);
	if (clp->krb5_fd != -1) close(clp->krb5_fd);
	if (clp->dirname) free(clp->dirname);
	if (clp->servicename) free(clp->servicename);
	if (clp->servername) free(clp->servername);
	free(clp);
}

static struct clnt_info *
insert_new_clnt(void)
{
	struct clnt_info	*clp = NULL;

	if (!(clp = (struct clnt_info *)calloc(1,sizeof(struct clnt_info)))) {
		syslog(LOG_ERR, "can't malloc clnt_info");
		goto out;
	}
	clp->krb5_poll_index = -1;
	clp->krb5_fd = -1;
	clp->dir_fd = -1;

	TAILQ_INSERT_HEAD(&clnt_list, clp, list);
out:
	return clp;
}

static int
process_clnt_dir_files(struct clnt_info * clp)
{
	char	kname[32];
	char	info_file_name[32];

	snprintf(kname,sizeof(kname),"%s/krb5", clp->dirname);
	if ((clp->krb5_fd = open(kname, O_RDWR)) == -1)
		return -1;
	snprintf(info_file_name, sizeof(info_file_name), "%s/info",
			clp->dirname);
	if (read_service_info(info_file_name, &clp->servicename,
				&clp->servername, &clp->prog, &clp->vers))
		return -1;
	return 0;
}

static int
get_poll_index(int *ind)
{
	int i;

	*ind = -1;
	for (i=0; i<FD_ALLOC_BLOCK; i++) {
		if (pollkrb5[i].events == 0) {
			*ind = i;
			break;
		}
	}
	if (*ind == -1) {
		syslog(LOG_ERR, "No pollkrb5 slots open");
		return -1;
	}
	return 0;
}

static void
process_clnt_dir(char *dir)
{
	struct clnt_info *	clp;

	if (!(clp = insert_new_clnt()))
		goto fail_destroy_client;

        if (!(clp->dirname = calloc(strlen(dir) + 1, 1))) {
		goto fail_destroy_client;
	}
	memcpy(clp->dirname, dir, strlen(dir));
	if ((clp->dir_fd = open(clp->dirname, O_RDONLY)) == -1) {
		syslog(LOG_ERR, "can't open %s", clp->dirname);
		goto fail_destroy_client;
	}
	fcntl(clp->dir_fd, F_SETSIG, SIGRTMIN);
	fcntl(clp->dir_fd, F_NOTIFY, DN_CREATE | DN_DELETE | DN_MULTISHOT);

	if (process_clnt_dir_files(clp))
		goto fail_keep_client;

	if (get_poll_index(&clp->krb5_poll_index)) {
		syslog(LOG_ERR, "Too many clients");
		goto fail_destroy_client;
	}
	pollkrb5[clp->krb5_poll_index].fd = clp->krb5_fd;
	pollkrb5[clp->krb5_poll_index].events |= POLLIN;

	return;

fail_destroy_client:
	if (clp) {
		TAILQ_REMOVE(&clnt_list, clp, list);
		destroy_client(clp);
	}
fail_keep_client:
	/* We couldn't find some subdirectories, but we keep the client
	 * around in case we get a notification on the directory when the
	 * subdirectories are created. */
	return;
}

void
init_client_list(void)
{
	TAILQ_INIT(&clnt_list);
	/* Eventually plan to grow/shrink poll array: */
	pollsize = FD_ALLOC_BLOCK;
	pollkrb5 = calloc(pollsize, sizeof(struct pollfd));
}

static void
destroy_client_list(void)
{
	struct clnt_info	*clp;

	while (clnt_list.tqh_first != NULL) {
		clp = clnt_list.tqh_first;
		TAILQ_REMOVE(&clnt_list, clp, list);
		destroy_client(clp);
	}
}

/* Used to read (and re-read) list of clients, set up poll array. */
int
update_client_list(void)
{
	struct dirent **namelist;
	int i,j;

	destroy_client_list();

	if (chdir(RPCSEC_DIR) < 0) {
		syslog(LOG_ERR, "can't chdir to " RPCSEC_DIR);
		return -1;
	}

	memset(pollkrb5, 0, pollsize * sizeof(struct pollfd));

	j = scandir(RPCSEC_DIR, &namelist, NULL, alphasort);
	if (j < 0) {
		syslog(LOG_ERR, "can't scandir " RPCSEC_DIR);
		return -1;
	}
	for (i=0; i < j; i++) {
		if (i < FD_ALLOC_BLOCK
				&& !strncmp(namelist[i]->d_name, "clnt", 4))
			process_clnt_dir(namelist[i]->d_name);
		free(namelist[i]);
	}

	free(namelist);
	return 0;
}

static int
do_downcall(int k5_fd, uid_t uid, OM_UINT32 seq_win,
		gss_buffer_desc *srv_ctx_id, gss_buffer_desc *context_token)
{
	char    buf[1024];
	char    *p = buf, *end = buf + 1024;
	unsigned int timeout = 0; /* XXX decide on a reasonable value */

	printf("doing downcall\n");

	if (WRITE_BYTES(&p, end, uid)) return -1;
	/* Not setting any timeout for now: */
	if (WRITE_BYTES(&p, end, timeout)) return -1;
	if (WRITE_BYTES(&p, end, seq_win)) return -1;
	if (write_buffer(&p, end, srv_ctx_id)) return -1;
	if (write_buffer(&p, end, context_token)) return -1;

	if (write(k5_fd, buf, p - buf) < p - buf) return -1;
	return 0;
}

static int
write_keyblock(char **p, char *end, struct _krb5_keyblock *arg)
{
	gss_buffer_desc	tmp;

	if (WRITE_BYTES(p, end, arg->enctype)) return -1;
	tmp.length = arg->length;
	tmp.value = arg->contents;
	if (write_buffer(p, end, &tmp)) return -1;
	return 0;
}

/* Should be overkill; figure out the right number some day: */
#define MAX_CTX_LEN 1024
static int
export_krb5_ctx_to_kernel(gss_ctx_id_t ctx, gss_buffer_desc *buf)
{
	krb5_gss_ctx_id_t	kctx = (krb5_gss_ctx_id_t)ctx;
	char *p, *end;

	if (!(buf->value = calloc(1, MAX_CTX_LEN)))
		goto out_err;
	p = buf->value;
	end = buf->value + MAX_CTX_LEN;

	if (WRITE_BYTES(&p, end, kctx->initiate)) goto out_err;
	if (WRITE_BYTES(&p, end, kctx->seed_init)) goto out_err;
	if (write_bytes(&p, end, &kctx->seed, sizeof(kctx->seed)))
		goto out_err;
	if (WRITE_BYTES(&p, end, kctx->signalg)) goto out_err;
	if (WRITE_BYTES(&p, end, kctx->sealalg)) goto out_err;
	if (WRITE_BYTES(&p, end, kctx->endtime)) goto out_err;
	if (WRITE_BYTES(&p, end, kctx->seq_send)) goto out_err;
	if (write_buffer(&p, end, kctx->mech_used)) goto out_err;
	if (write_keyblock(&p, end, kctx->enc)) goto out_err;
	if (write_keyblock(&p, end, kctx->seq)) goto out_err;

	buf->length = p - (char *)buf->value;
	return 0;
out_err:
	if (buf->value) free(buf->value);
	buf->length = 0;
	return -1;
}

#define g_OID_equal(o1, o2) \
	(((o1)->length == (o2)->length) && \
	 (memcmp((o1)->elements,(o2)->elements,(o1)->length) == 0))

static int
export_ctx_to_kernel(gss_ctx_id_t ctx, gss_buffer_desc *buf)
{
	gss_union_ctx_id_t	uctx = (gss_union_ctx_id_t)ctx;

	if (g_OID_equal(&krb5oid, uctx->mech_type))
		return export_krb5_ctx_to_kernel(uctx->internal_ctx_id, buf);
	else { /* will switch on other mechanisms here */
		syslog(LOG_ERR, "attempting to export context with unknown"
				" mechanism oid\n");
		return -1;
	}
}

/* this code uses the userland rpcsec gss library to create a context on
 * behalf of the kernel */
void
handle_upcall(struct clnt_info *kclp)
{
	static char		buf[MAX_NETOBJ_SZ];
	CLIENT			*rpc_clnt = NULL;
	struct rpc_gss_sec	sec;
	AUTH			*auth = NULL;
	u_int			min_stat;
	uid_t			uid;
	struct rpc_gss_data	*gd;
	gss_buffer_desc		token;

	if (read(kclp->krb5_fd, &uid, sizeof(uid)) < sizeof(uid)) {
		syslog(LOG_ERR, "failed reading uid from upcall pipe\n");
		goto out;
	}

	/* Tell krb5 gss which user to get a context for */

	memset(buf, 0, sizeof(buf));
	if (uid == 0)
		snprintf(buf, sizeof(buf), "FILE:/tmp/krb5cc_machine");
	else
		snprintf(buf, sizeof(buf), "FILE:/tmp/krb5cc_%u", uid);
	gss_krb5_ccache_name(&min_stat, buf, NULL);

	/* create a connection to the nfs server */

	if ((rpc_clnt = clnt_create(kclp->servername, kclp->prog, kclp->vers,
					"udp")) == NULL) {
		syslog(LOG_ERR, "can't create rpc_clnt for server %s\n",
			kclp->servername);
		goto out;
	}

	sec.mech = (gss_OID)&krb5oid;
	sec.qop = GSS_C_QOP_DEFAULT;
	sec.svc = RPCSEC_GSS_SVC_NONE;

	auth = authgss_create_default(rpc_clnt, kclp->servicename, &sec);
	if (!auth) {
		syslog(LOG_ERR, "Failed to initialize context for user with"
				" uid %d\n", uid);
		goto out;
	}

	gd = AUTH_PRIVATE(auth);

	if (export_ctx_to_kernel(gd->ctx, &token)) {
		syslog(LOG_ERR, "Failed to export context for user with"
				" uid %d\n", uid);
		goto out;
	}

	do_downcall(kclp->krb5_fd, uid, gd->win, &gd->gc.gc_ctx, &token);
out:
	if (rpc_clnt) clnt_destroy(rpc_clnt);
	return;
}

int
gssd_acquire_cred(char *server_name)
{
	gss_buffer_desc name;
	gss_name_t target_name;
	OM_uint32 maj_stat, min_stat;
	
	name.value = (void *)server_name;
	name.length = strlen(server_name);
	
	maj_stat = gss_import_name(&min_stat, &name, gss_nt_service_name,
				   &target_name);

	if (maj_stat != GSS_S_COMPLETE) {
		pgsserr("gss_import_name", maj_stat, min_stat);
		return (FALSE);
	}

	maj_stat = gss_acquire_cred(&min_stat, target_name, 0,
				    GSS_C_NULL_OID_SET, GSS_C_ACCEPT,
				    &gssd_creds, NULL, NULL);
	
	(void)gss_release_name(&min_stat, &target_name);

	if (maj_stat != GSS_S_COMPLETE)
		pgsserr("gss_acquire_cred", maj_stat, min_stat);

	return (maj_stat == GSS_S_COMPLETE);
}
