/*	$Id: nfs4_vfsops.c,v 1.36 2003/06/20 17:05:11 rees Exp $	*/
/*	$OpenBSD: nfs_vfsops.c,v 1.48 2002/03/14 01:27:13 millert Exp $	*/
/*	$NetBSD: nfs_vfsops.c,v 1.46.4.1 1996/05/25 22:40:35 fvdl Exp $	*/

/*
 * Copyright (c) 1989, 1993, 1995
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Rick Macklem at The University of Guelph.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)nfs_vfsops.c	8.12 (Berkeley) 5/20/95
 */

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/signal.h>
#include <sys/proc.h>
#include <sys/namei.h>
#include <sys/vnode.h>
#include <sys/kernel.h>
#include <sys/mount.h>
#include <sys/buf.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/systm.h>
#include <sys/sysctl.h>
#include <sys/kthread.h>

#include <net/if.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/in_var.h>

#include <nfs/rpcv2.h>
#include <nfs/nfsproto.h>
#include <nfs/nfs.h>
#include <nfs/xdr_subs.h>
#include <nfs4/list.h>
#include <nfs4/nfs4_prot.h>
#include <nfs4/nfs4_debug.h>
#include <nfs4/nfs4.h>
#include <nfs4/nfs4mount.h>
#include <nfs4/nfs4fs.h>
#include <nfs4/nfs4node.h>
#include <nfs4/nfsm_subs.h>
#include <nfs4/nfs4_var.h>

#include <rpc/rpcclnt.h>

extern int rpcclnt_ticks;

int nfs_sysctl(int *, u_int, void *, size_t *, void *, size_t, struct proc *);
int nfs_checkexp(struct mount *mp, struct mbuf *nam,
	 int *extflagsp, struct ucred **credanonp);
static int nfs4_do_renew(struct nfs4mount *nmp, struct ucred *cred);
void nfs4_daemon(void *arg);
int mountnfs(struct nfs_args *argp, struct mount *mp,
	     struct mbuf *nam, char *pth, char *hst, struct proc *p);
static int nfs4_stuff_fsattr(struct nfs4fs_fattr *fap, struct nfs4mount *nmp, struct statfs *sbp);

/*
 * nfs vfs operations.
 */
struct vfsops nfs_vfsops = {
	nfs_mount,
	nfs_start,
	nfs_unmount,
	nfs_root,
	nfs_quotactl,
	nfs_statfs,
	nfs_sync,
	nfs_vget,
	nfs_fhtovp,
	nfs_vptofh,
	nfs_vfs_init,
	nfs_sysctl,
	nfs_checkexp
};

extern u_int32_t nfs_prog, nfs_vers;

#define TRUE	1
#define	FALSE	0

/*
 * nfs statfs call
 */
int
nfs_statfs(mp, sbp, p)
	struct mount *mp;
	struct statfs *sbp;
	struct proc *p;
{
	struct vnode *vp;
	struct nfs4mount *nmp = VFSTONFS(mp);
	int error = 0;
	struct ucred *cred;
	struct nfsnode *np;
	struct nfs4fs_compound compound;
	struct nfs4fs_getattr getattr;
	struct nfs4fs_putfh putfh;

	error = nfs_nget(mp, (nfsfh_t *)nmp->nm_fh, nmp->nm_fhsize, &np);
	if (error)
		return (error);
	vp = NFSTOV(np);
	cred = crget();
	cred->cr_ngroups = 0;
	nfsstats.rpccnt[NFSPROC_FSSTAT]++;

	/* Setup and do the rpc */
	compound.nmp = nmp;
	compound.procp = p;
	compound.cred = cred;
	getattr.flags = FL_NOREVAL;
	getattr.bm = &fsattr_bm;
	putfh.dentry = &np->n_dentry;

	if ((error = nfs4_setup_compound(&compound, "nfs4_statfs()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_getattr(&compound, &getattr)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_getattr(&compound, &getattr)))
		goto nfsmout;

	nfs4_stuff_fsattr(&getattr.fattr, nmp, sbp);

	if (sbp != &mp->mnt_stat) {
		bcopy(mp->mnt_stat.f_mntonname, sbp->f_mntonname, MNAMELEN);
		bcopy(mp->mnt_stat.f_mntfromname, sbp->f_mntfromname, MNAMELEN);
		bcopy(&mp->mnt_stat.mount_info.nfs_args,
		    &sbp->mount_info.nfs_args, sizeof(struct nfs_args));
	}
	strncpy(&sbp->f_fstypename[0], mp->mnt_vfc->vfc_name, MFSNAMELEN);

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	nfsv4_printk(level2, "nfs4_statfs: returning status %d\n", error);
	vrele(vp);
	crfree(cred);
	if (error < 0)
	    error = -error;
	return (error);
}

static int
nfs4_stuff_fsattr(struct nfs4fs_fattr *fp, struct nfs4mount *nmp, struct statfs *sbp)
{
	sbp->f_flags = nmp->nm_flag;
	sbp->f_bsize = NFS_FABLKSIZE;

	if (fp->fa_valid & FA_FSID) {
		sbp->f_fsid.val[0] = fp->fa_fsid_major;
		sbp->f_fsid.val[1] = fp->fa_fsid_minor;
	}

	sbp->f_ffree = (fp->fa_valid & FA_FFREE) ? fp->fa_ffree : 0;
	sbp->f_files = (fp->fa_valid & FA_FTOTAL) ? fp->fa_ftotal : 0;

	sbp->f_bavail = (fp->fa_valid & FA_SAVAIL) ? (fp->fa_savail / NFS_FABLKSIZE) : 500000;
	sbp->f_bfree = (fp->fa_valid & FA_SFREE) ? (fp->fa_sfree / NFS_FABLKSIZE) : 500000;
	sbp->f_blocks = (fp->fa_valid & FA_STOTAL) ? (fp->fa_stotal / NFS_FABLKSIZE) : 1000000;

	return 0;
}

/* We got fsinfo at mount time, so nothing to do here */
int
nfs_fsinfo(nmp, vp, cred, p)
	struct nfs4mount *nmp;
	struct vnode *vp;
	struct ucred *cred;
	struct proc *p;
{
	return 0;
}

int
nfs_mountroot()
{
	printf("nfs_mountroot\n");
	return 0;
}

void
nfs_decode_args(nmp, argp, nargp)
	struct nfs4mount *nmp;
	struct nfs_args *argp;
	struct nfs_args *nargp;
{
	int s;
	int adjsock = 0;
	int maxio;

	s = splsoftnet();

#if 0
	/* Re-bind if rsrvd port requested and wasn't on one */
	adjsock = !(nmp->nm_flag & NFSMNT_RESVPORT)
		  && (argp->flags & NFSMNT_RESVPORT);
#endif
	/* Also re-bind if we're switching to/from a connected UDP socket */
	adjsock |= ((nmp->nm_flag & NFSMNT_NOCONN) !=
	    (argp->flags & NFSMNT_NOCONN));

	/* Update flags atomically.  Don't change the lock bits. */
	nmp->nm_flag =
	    (argp->flags & ~NFSMNT_INTERNAL) | (nmp->nm_flag & NFSMNT_INTERNAL);
	splx(s);

	if ((argp->flags & NFSMNT_TIMEO) && argp->timeo > 0) {
		nmp->nm_rpcclnt.rc_timeo = (argp->timeo * NFS_HZ + 5) / 10;
		if (nmp->nm_rpcclnt.rc_timeo < NFS_MINTIMEO)
			nmp->nm_rpcclnt.rc_timeo = NFS_MINTIMEO;
		else if (nmp->nm_rpcclnt.rc_timeo > NFS_MAXTIMEO)
			nmp->nm_rpcclnt.rc_timeo = NFS_MAXTIMEO;
	}

	if ((argp->flags & NFSMNT_RETRANS) && argp->retrans > 1) {
		nmp->nm_rpcclnt.rc_retry = argp->retrans;
		if (nmp->nm_rpcclnt.rc_retry > NFS_MAXREXMIT)
			nmp->nm_rpcclnt.rc_retry = NFS_MAXREXMIT;
	}

	maxio = (argp->sotype == SOCK_DGRAM) ? NFS_MAXDGRAMDATA : NFS_MAXDATA;

	if ((argp->flags & NFSMNT_WSIZE) && argp->wsize > 0) {
		int osize = nmp->nm_rpcclnt.rc_wsize;
		nmp->nm_rpcclnt.rc_wsize = argp->wsize;
		/* Round down to multiple of blocksize */
		nmp->nm_rpcclnt.rc_wsize &= ~(NFS_FABLKSIZE - 1);
		if (nmp->nm_rpcclnt.rc_wsize <= 0)
			nmp->nm_rpcclnt.rc_wsize = NFS_FABLKSIZE;
		adjsock |= (nmp->nm_rpcclnt.rc_wsize != osize);
	}
	if (nmp->nm_rpcclnt.rc_wsize > maxio)
		nmp->nm_rpcclnt.rc_wsize = maxio;
	if (nmp->nm_rpcclnt.rc_wsize > MAXBSIZE)
		nmp->nm_rpcclnt.rc_wsize = MAXBSIZE;

	if ((argp->flags & NFSMNT_RSIZE) && argp->rsize > 0) {
		int osize = nmp->nm_rpcclnt.rc_rsize;
		nmp->nm_rpcclnt.rc_rsize = argp->rsize;
		/* Round down to multiple of blocksize */
		nmp->nm_rpcclnt.rc_rsize &= ~(NFS_FABLKSIZE - 1);
		if (nmp->nm_rpcclnt.rc_rsize <= 0)
			nmp->nm_rpcclnt.rc_rsize = NFS_FABLKSIZE;
		adjsock |= (nmp->nm_rpcclnt.rc_rsize != osize);
	}
	if (nmp->nm_rpcclnt.rc_rsize > maxio)
		nmp->nm_rpcclnt.rc_rsize = maxio;
	if (nmp->nm_rpcclnt.rc_rsize > MAXBSIZE)
		nmp->nm_rpcclnt.rc_rsize = MAXBSIZE;

	if ((argp->flags & NFSMNT_READDIRSIZE) && argp->readdirsize > 0) {
		nmp->nm_readdirsize = argp->readdirsize;
		/* Round down to multiple of blocksize */
		nmp->nm_readdirsize &= ~(NFS_DIRBLKSIZ - 1);
		if (nmp->nm_readdirsize < NFS_DIRBLKSIZ)
			nmp->nm_readdirsize = NFS_DIRBLKSIZ;
	} else if (argp->flags & NFSMNT_RSIZE)
		nmp->nm_readdirsize = nmp->nm_rpcclnt.rc_rsize;

	if (nmp->nm_readdirsize > maxio)
		nmp->nm_readdirsize = maxio;

	if ((argp->flags & NFSMNT_MAXGRPS) && argp->maxgrouplist >= 0 &&
		argp->maxgrouplist <= NFS_MAXGRPS)
		nmp->nm_numgrps = argp->maxgrouplist;
	if ((argp->flags & NFSMNT_READAHEAD) && argp->readahead >= 0 &&
		argp->readahead <= NFS_MAXRAHEAD)
		nmp->nm_readahead = argp->readahead;

	if (argp->flags & NFSMNT_ACREGMIN && argp->acregmin >= 0) {
		if (argp->acregmin > 0xffff)
			nmp->nm_acregmin = 0xffff;
		else
			nmp->nm_acregmin = argp->acregmin;
	}
	if (argp->flags & NFSMNT_ACREGMAX && argp->acregmax >= 0) {
		if (argp->acregmax > 0xffff)
			nmp->nm_acregmax = 0xffff;
		else
			nmp->nm_acregmax = argp->acregmax;
	}
	if (nmp->nm_acregmin > nmp->nm_acregmax)
	  nmp->nm_acregmin = nmp->nm_acregmax;

	if (argp->flags & NFSMNT_ACDIRMIN && argp->acdirmin >= 0) {
		if (argp->acdirmin > 0xffff)
			nmp->nm_acdirmin = 0xffff;
		else
			nmp->nm_acdirmin = argp->acdirmin;
	}
	if (argp->flags & NFSMNT_ACDIRMAX && argp->acdirmax >= 0) {
		if (argp->acdirmax > 0xffff)
			nmp->nm_acdirmax = 0xffff;
		else
			nmp->nm_acdirmax = argp->acdirmax;
	}
	if (nmp->nm_acdirmin > nmp->nm_acdirmax)
	  nmp->nm_acdirmin = nmp->nm_acdirmax;

	/* XXX: set deadthresh */
	nmp->nm_rpcclnt.rc_deadthresh = argp->deadthresh;


	if (nmp->nm_rpcclnt.rc_so && adjsock) {
		nfs_disconnect(nmp);
		if (nmp->nm_rpcclnt.rc_sotype == SOCK_DGRAM)
			while (nfs_connect(nmp, NULL)) {
				printf("nfs_args: retrying connect\n");
				(void) tsleep((caddr_t)&lbolt,
					      PSOCK, "nfscon", 0);
			}
	}

	/* Update nargp based on nmp */
	nargp->wsize = nmp->nm_rpcclnt.rc_wsize;
	nargp->rsize = nmp->nm_rpcclnt.rc_rsize;
	nargp->readdirsize = nmp->nm_readdirsize;
	nargp->timeo = nmp->nm_rpcclnt.rc_timeo;
	nargp->retrans = nmp->nm_rpcclnt.rc_retry;
	nargp->maxgrouplist = nmp->nm_numgrps;
	nargp->readahead = nmp->nm_readahead;
	nargp->acregmin = nmp->nm_acregmin;
	nargp->acregmax = nmp->nm_acregmax;
	nargp->acdirmin = nmp->nm_acdirmin;
	nargp->acdirmax = nmp->nm_acdirmax;
	nargp->deadthresh = nmp->nm_rpcclnt.rc_deadthresh;
}

/*
 * VFS Operations.
 *
 * mount system call
 */
/* ARGSUSED */
int
nfs_mount(mp, path, data, ndp, p)
	struct mount *mp;
	const char *path;
	void *data;
	struct nameidata *ndp;
	struct proc *p;
{
	int error;
	struct nfs_args args;
	struct mbuf *nam;
	char pth[MNAMELEN], hst[MNAMELEN];
	size_t len;

	error = copyin (data, &args, sizeof (args.version));
	if (error)
		return (error);
	if (args.version == 3) {
		error = copyin (data, (caddr_t)&args,
				sizeof (struct nfs_args3));
		args.flags &= ~(NFSMNT_INTERNAL|NFSMNT_NOAC);
	}
	else if (args.version == NFS_ARGSVERSION) {
		error = copyin(data, (caddr_t)&args, sizeof (struct nfs_args));
		args.flags &= ~NFSMNT_NOAC; /* XXX - compatibility */
	}
	else
		return (EPROGMISMATCH);
	if (error)
		return (error);

	if (nfs_niothreads < 0) {
		nfs_niothreads = 4;
		nfs_getset_niothreads(TRUE);
	}

	if (mp->mnt_flag & MNT_UPDATE) {
		struct nfs4mount *nmp = VFSTONFS(mp);

		if (nmp == NULL)
			return (EIO);
		nfs_decode_args(nmp, &args, &mp->mnt_stat.mount_info.nfs_args);
		return (0);
	}
	error = copyinstr(path, pth, MNAMELEN-1, &len);
	if (error)
		return (error);
	bzero(&pth[len], MNAMELEN - len);
	error = copyinstr(args.hostname, hst, MNAMELEN-1, &len);
	if (error)
		return (error);
	bzero(&hst[len], MNAMELEN - len);
	/* sockargs() call must be after above copyin() calls */
	error = sockargs(&nam, (caddr_t)args.addr, args.addrlen, MT_SONAME);
	if (error)
		return (error);
	error = mountnfs(&args, mp, nam, pth, hst, p);
	return (error);
}

/*
 * renew should be done async
 * should re-scan mount queue each time
 */
struct proc *nfs4_daemonproc;

static int
nfs4_do_renew(struct nfs4mount *nmp, struct ucred *cred)
{
	struct nfs4fs_compound compound;
	int error;

	compound.nmp = nmp;
	compound.procp = curproc;
	compound.cred = cred;

	if ((error = nfs4_setup_compound(&compound, "nfs4_do_renew()")))
		goto out;
	if ((error = nfs4_setup_renew(&compound)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_simple_op(&compound, OP_RENEW, 0)))
		goto nfsmout;

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		error = -error;
	return (error);
}

void
nfs4_daemon(void *arg)
{
	struct mount *mp;
	struct nfs4mount *nmp;

	while (1) {
		simple_lock(&mountlist_slock);
		CIRCLEQ_FOREACH(mp, &mountlist, mnt_list) {
			if (strcmp(mp->mnt_vfc->vfc_name, MOUNT_NFS) != 0)
				continue;
			nmp = VFSTONFS(mp);
			if (time.tv_sec < nmp->nm_last_renewal + nmp->nm_lease_time - 4)
				continue;
			simple_unlock(&mountlist_slock);
			nfs4_do_renew(nmp, (struct ucred *) arg);
			simple_lock(&mountlist_slock);
		}
		simple_unlock(&mountlist_slock);
		tsleep(&nfs4_daemonproc, PVFS, "nfs4", 2 * hz);
	}
}

/*
 * Common code for mount and mountroot
 */
int
mountnfs(struct nfs_args *argp, struct mount *mp, struct mbuf *nam, char *pth, char *hst, struct proc *p)
{
	struct nfs4mount *nmp;
	char *rpth, *cp1, *cp2;
	int error;
	nfsfh_t *fhp;
	int fhsize;
	struct nfs4fs_compound compound;
	struct nfs4fs_getattr getattr;
	struct nfs4fs_getfh getfh;
	struct nfs4fs_lookup lookup;
	int nlookup = 0;
	struct dentry dentry;
	struct nfs4fs_fattr *fp;
	u32 max;

	nfs4_setup_bitmaps();
	if (mp->mnt_flag & MNT_UPDATE) {
		nmp = VFSTONFS(mp);
		/* update paths, file handles, etc, here	XXX */
		m_freem(nam);
		return (0);
	} else {
		MALLOC(nmp, struct nfs4mount *, sizeof (struct nfs4mount),
		    M_NFSMNT, M_WAITOK);
		bzero((caddr_t)nmp, sizeof (struct nfs4mount));
		mp->mnt_data = (qaddr_t)nmp;
		TAILQ_INIT(&nmp->nm_uidlruhead);
	}

	vfs_getnewfsid(mp);
	nmp->nm_mountp = mp;
	nmp->nm_flag |= NFSMNT_NFSV3;
	nmp->nm_rpcclnt.rc_timeo = NFS_TIMEO;
	nmp->nm_rpcclnt.rc_retry = NFS_RETRANS;
	nmp->nm_rpcclnt.rc_wsize = NFS_WSIZE;
	nmp->nm_rpcclnt.rc_rsize = NFS_RSIZE;
	nmp->nm_readdirsize = NFS_READDIRSIZE;
	nmp->nm_numgrps = NFS_MAXGRPS;
	nmp->nm_readahead = NFS_DEFRAHEAD;
	CIRCLEQ_INIT(&nmp->nm_timerhead);
	nmp->nm_acregmin = NFS_MINATTRTIMO;
	nmp->nm_acregmax = NFS_MAXATTRTIMO;
	nmp->nm_acdirmin = NFS_MINATTRTIMO;
	nmp->nm_acdirmax = NFS_MAXATTRTIMO;
	nmp->nm_clientid = 0;
	strncpy(&mp->mnt_stat.f_fstypename[0], mp->mnt_vfc->vfc_name, MFSNAMELEN);
	bcopy(hst, mp->mnt_stat.f_mntfromname, MNAMELEN);
	bcopy(pth, mp->mnt_stat.f_mntonname, MNAMELEN);
	bcopy(argp, &mp->mnt_stat.mount_info.nfs_args, sizeof(*argp));
	nmp->nm_rpcclnt.rc_name = nam;
	nfs_decode_args(nmp, argp, &mp->mnt_stat.mount_info.nfs_args);

	/* Set up the sockets and per-host congestion */
	nmp->nm_rpcclnt.rc_sotype = argp->sotype;
	nmp->nm_rpcclnt.rc_soproto = argp->proto;

	/*
	 * This is silly, but it has to be set so that vinifod() works.
	 * We do not want to do an nfs_statfs() here since we can get
	 * stuck on a dead server and we are holding a lock on the mount
	 * point.
	 */
	mp->mnt_stat.f_iosize = NFS_MAXDGRAMDATA;

	nfsv4_printk(mount, "mountnfs: hst \"%s\" pth \"%s\"\n", hst, pth);

	error = nfs_connect(nmp, NULL);
	if (error)
		goto out;

	compound.nmp = nmp;
	compound.procp = p;
	compound.cred = p->p_ucred;
	getattr.bm = &fsinfo_bm;

	if ((error = nfs4_setup_compound(&compound, "mountnfs()")))
		goto out;
	if ((error = nfs4_setup_putrootfh(&compound)))
		goto nfsmout;

	/* Get remote path */
	rpth = nfs4_strchr(hst, ':');
	if (rpth)
		rpth++;

	compound.current_fh = lookup.dentry = &dentry;
	for (cp1 = rpth; cp1 && *cp1; cp1 = cp2)  {
		while (*cp1 == '/')
			cp1++;
		if (!*cp1)
			break;
		for (cp2 = cp1; *cp2 && *cp2 != '/'; cp2++)
			;
		lookup.name = cp1;
		lookup.namelen = cp2 - cp1;
		if ((error = nfs4_setup_lookup(&compound, &lookup)))
			goto nfsmout;
		nlookup++;
	}
	if ((error = nfs4_setup_getfh(&compound, &getfh)))
		goto nfsmout;
	if ((error = nfs4_setup_getattr(&compound, &getattr)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putrootfh(&compound)))
		goto nfsmout;
	while (nlookup--) {
		if ((error = nfs4_handle_lookup(&compound)))
			goto nfsmout;
	}
	if ((error = nfs4_handle_getfh(&compound, &getfh)))
		goto nfsmout;
	if ((error = nfs4_handle_getattr(&compound, &getattr)))
		goto nfsmout;

	/* Copy the file handle to the nfs4mount struct */
	fhp = &dentry.fh_val;
	fhsize = dentry.fh_len;
	nmp->nm_fhsize = fhsize;
	bcopy((caddr_t) fhp, (caddr_t) nmp->nm_fh, fhsize);
	nfsv4_printk(mount, "mountnfs fhsize %d\n", fhsize);

	/* Copy fsinfo */
	fp = &getattr.fattr;
	if (fp->fa_valid & FA_FSID) {
		nmp->nm_fsid.val[0] = fp->fa_fsid_major;
		nmp->nm_fsid.val[1] = fp->fa_fsid_minor;
	}
	if (fp->fa_valid & FA_MAXREAD) {
		max = fp->fa_maxread;
		if (max < nmp->nm_rpcclnt.rc_rsize) {
			nmp->nm_rpcclnt.rc_rsize = max & ~(NFS_FABLKSIZE - 1);
			if (nmp->nm_rpcclnt.rc_rsize == 0)
				nmp->nm_rpcclnt.rc_rsize = max;
		}
		if (max < nmp->nm_readdirsize) {
			nmp->nm_readdirsize = max & ~(NFS_DIRBLKSIZ - 1);
			if (nmp->nm_readdirsize == 0)
				nmp->nm_readdirsize = max;
		}
	}
	if (fp->fa_valid & FA_MAXWRITE) {
		max = fp->fa_maxwrite;
		if (max < nmp->nm_rpcclnt.rc_wsize) {
			nmp->nm_rpcclnt.rc_wsize = max & ~(NFS_FABLKSIZE - 1);
			if (nmp->nm_rpcclnt.rc_wsize == 0)
				nmp->nm_rpcclnt.rc_wsize = max;
		}
	}
	if (fp->fa_valid & FA_LEASE_TIME)
		nmp->nm_lease_time = fp->fa_lease_time;

	nmp->nm_flag |= NFSMNT_GOTFSINFO;

	if ((error = nfs4_do_setclientid(nmp, p)) != 0)
		goto nfsmout;

	/* Start renewd if it isn't already running */
	if (nfs4_daemonproc == NULL)
		kthread_create(nfs4_daemon, crdup(p->p_ucred), &nfs4_daemonproc, "nfs4rd");

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error) {
		nfs_disconnect(nmp);
		free((caddr_t)nmp, M_NFSMNT);
		m_freem(nam);
	}
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * unmount system call
 */
int
nfs_unmount(mp, mntflags, p)
	struct mount *mp;
	int mntflags;
	struct proc *p;
{
	struct nfs4mount *nmp = VFSTONFS(mp);

	vflush(mp, NULL, (mntflags & MNT_FORCE) ? FORCECLOSE : 0);
	nfs_disconnect(nmp);
	m_freem(nmp->nm_rpcclnt.rc_name);

	return (0);
}

/*
 * Return root of a filesystem
 */
int
nfs_root(mp, vpp)
	struct mount *mp;
	struct vnode **vpp;
{
	struct nfs4mount *nmp = VFSTONFS(mp);
	struct nfsnode *np;
	int error;

	error = nfs_nget(mp, (nfsfh_t *)nmp->nm_fh, nmp->nm_fhsize, &np);
	if (error)
		return (error);
	*vpp = NFSTOV(np);
	return (0);
}

extern int syncprt;

/*
 * Flush out the buffer cache
 */
/* ARGSUSED */
int
nfs_sync(mp, waitfor, cred, p)
	struct mount *mp;
	int waitfor;
	struct ucred *cred;
	struct proc *p;
{
	struct vnode *vp;
	int error, allerror = 0;

	/*
	 * Force stale buffer cache information to be flushed.
	 */
loop:
	for (vp = LIST_FIRST(&mp->mnt_vnodelist); vp != NULL;
	     vp = LIST_NEXT(vp, v_mntvnodes)) {
		/*
		 * If the vnode that we are about to sync is no longer
		 * associated with this mount point, start over.
		 */
		if (vp->v_mount != mp)
			goto loop;
		if (VOP_ISLOCKED(vp) || LIST_FIRST(&vp->v_dirtyblkhd) == NULL ||
		    waitfor == MNT_LAZY)
			continue;
		if (vget(vp, LK_EXCLUSIVE, p))
			goto loop;
		error = VOP_FSYNC(vp, cred, waitfor, p);
		if (error)
			allerror = error;
		vput(vp);
	}
	return (allerror);
}

/*
 * NFS flat namespace lookup.
 * Currently unsupported.
 */
/* ARGSUSED */
int
nfs_vget(mp, ino, vpp)
	struct mount *mp;
	ino_t ino;
	struct vnode **vpp;
{

	return (EOPNOTSUPP);
}

/*
 * Do that sysctl thang...
 */
int
nfs_sysctl(int *name, u_int namelen, void *oldp, size_t *oldlenp, void *newp,
	   size_t newlen, struct proc *p)
{
	int rv;

	/*
	 * All names at this level are terminal.
	 */
	if(namelen > 1)
		return ENOTDIR;	/* overloaded */

	switch(name[0]) {
	case NFS_NFSSTATS:
		if(!oldp) {
			*oldlenp = sizeof nfsstats;
			return 0;
		}

		if(*oldlenp < sizeof nfsstats) {
			*oldlenp = sizeof nfsstats;
			return ENOMEM;
		}

		rv = copyout(&nfsstats, oldp, sizeof nfsstats);
		if(rv) return rv;

		if(newp && newlen != sizeof nfsstats)
			return EINVAL;

		if(newp) {
			return copyin(newp, &nfsstats, sizeof nfsstats);
		}
		return 0;

	case NFS_NIOTHREADS:
		nfs_getset_niothreads(0);

		rv = sysctl_int(oldp, oldlenp, newp, newlen, &nfs_niothreads);
		if (newp)
			nfs_getset_niothreads(1);

		return rv;

	default:
		return EOPNOTSUPP;
	}
}


/*
 * At this point, this should never happen
 */
/* ARGSUSED */
int
nfs_fhtovp(mp, fhp, vpp)
	struct mount *mp;
	struct fid *fhp;
	struct vnode **vpp;
{

	return (EINVAL);
}

/*
 * Vnode pointer to File handle, should never happen either
 */
/* ARGSUSED */
int
nfs_vptofh(vp, fhp)
	struct vnode *vp;
	struct fid *fhp;
{

	return (EINVAL);
}

/*
 * Vfs start routine, a no-op.
 */
/* ARGSUSED */
int
nfs_start(mp, flags, p)
	struct mount *mp;
	int flags;
	struct proc *p;
{

	return (0);
}

/*
 * Do operations associated with quotas, not supported
 */
/* ARGSUSED */
int
nfs_quotactl(mp, cmd, uid, arg, p)
	struct mount *mp;
	int cmd;
	uid_t uid;
	caddr_t arg;
	struct proc *p;
{

	return (EOPNOTSUPP);
}

/*
 * check export permission, not supported
 */
/* ARGUSED */
int
nfs_checkexp(mp, nam, exflagsp, credanonp)
	struct mount *mp;
	struct mbuf *nam;
	int *exflagsp;
	struct ucred **credanonp;
{
	return (EOPNOTSUPP);
}

int
nfs4_do_setclientid(struct nfs4mount *nmp, struct proc *p)
{
	struct nfs4fs_compound compound;
	struct nfs4fs_setclientid setclientid;
	struct route ro;
	char *ipsrc, uaddr[24], name[24];
	int try = 0;
	int status;

	nfsv4_printk(level2, "nfs4_do_setclientid: starting\n");

	if (nmp->nm_clientid) {
		nfsv4_printk(clientid2, "nfs4_do_setclientid: already have clientid!\n");
		status = 0;
		goto out;
	}

	compound.nmp = nmp;
	compound.procp = p;
	compound.cred = p->p_ucred;

	setclientid.callback_netid = (nmp->nm_rpcclnt.rc_sotype == SOCK_STREAM) ? "tcp" : "udp";

	/* Do a route lookup to find our source address for talking to this server */
	bzero(&ro, sizeof ro);
	ro.ro_dst = *mtod(nmp->nm_rpcclnt.rc_name, struct sockaddr *);
	rtalloc(&ro);
	if (ro.ro_rt == NULL) {
		status = EHOSTUNREACH;
		goto out;
	}
	ipsrc = inet_ntoa(ifatoia(ro.ro_rt->rt_ifa)->ia_addr.sin_addr);
	sprintf(uaddr, "%s.12.48", ipsrc);
	setclientid.callback_universal_address = uaddr;
	RTFREE(ro.ro_rt);

try_again:
	sprintf(name, "%s-%d", ipsrc, try);
	setclientid.nmp = nmp;
	setclientid.namlen = strlen(name);
	setclientid.name = name;

	if ((status = nfs4_setup_compound(&compound, "nfs4_do_setclientid()")))
		goto out;
	if ((status = nfs4_setup_setclientid(&compound, &setclientid)))
		goto out_release;

	if ((status = nfs4fs_call_compound(&compound)))
		goto out_release;
	if ((status = nfs4_handle_setclientid(&compound, &setclientid))) {
		if (setclientid.in_use && (++try < NFS4_SETCLIENTID_MAXTRIES)) {
			nfsv4_printk(clientid2, "do_setclientid: clid in use, retrying\n");
			nfs4fs_release_compound(&compound);
			goto try_again;
		}
		else
			goto out_release;
	}

	nfsv4_printk(clientid, "nfs4_do_setclientid: obtained clientid successfully\n");
	nmp->nm_clientid = setclientid.clientid;

out_release:
	nfs4fs_release_compound(&compound);
out:
	nfsv4_printk(level2, "nfs4_do_setclientid: returning status %d\n", status);
	return status;
}
