/*	$Id: nfs4fs.h,v 1.31 2003/03/02 02:28:06 rees Exp $	*/

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

/*
 *  Kendrick Smith <kmsmith@umich.edu>
 *  Andy Adamson <andros@umich.edu>
 *  Jim Rees <rees@umich.edu>
 */

#ifndef _LINUX_NFS4_NFS4FS_H
#define _LINUX_NFS4_NFS4FS_H

#define NFS4_MINOR_VERSION		0
#define NFS_PORT			2049
#define NFS4_DEF_FILE_IO_BUFFER_SIZE	4096
#define NFS4_MAX_FILE_IO_BUFFER_SIZE	32768
#define NFS4_DEF_MAXFILESIZE		0xffffffff
#define NFS4_SUPER_MAGIC		0xF00BA4

#define NFS4FS_SILLY_RENAME             1
#define NFS4FS_STRICT_LOCKING           1
#define NFS4FS_RETRY_OLD_STATEID	1
#define NFS4FS_MIN_LEASE                (1 * HZ)
#define NFS4FS_DEFAULT_LEASE            (30 * HZ)
#define NFS4FS_MAX_LEASE                (120 * HZ)
#define NFS4FS_RETRY_MIN_DELAY		(HZ >> 4)
#define NFS4FS_RETRY_MAX_DELAY		(HZ << 3)
#define NFS4FS_SEMAPHORE_DELAY		(HZ >> 4)
#define NFS4FS_GRACE_DELAY		(HZ * 5)
#define NFS4FS_OLD_STATEID_DELAY	(HZ >> 3)
#define NFS4FS_OP_MAX			10

#define NFS4_CLNT(sb)			((sb)->u.nfs4_sb.client)
#define NFS4_RPAGES(sb)			((sb)->u.nfs4_sb.rpages)
#define NFS4_WPAGES(sb)			((sb)->u.nfs4_sb.wpages)
#define NFS4_HAVE_CLIENTID(sb)		((sb)->u.nfs4_sb.have_clientid)
#define NFS4_CLIENTID(sb)		((sb)->u.nfs4_sb.clientid)
#define NFS4_PLEN(sb)			((sb)->u.nfs4_sb.plen)
#define NFS4_PVAL(sb)			((sb)->u.nfs4_sb.pval)
#define NFS4_LOCK_MINTIMEOUT(sb)	((sb)->u.nfs4_sb.lock_mintimeout)
#define NFS4_LOCK_MAXTIMEOUT(sb)	((sb)->u.nfs4_sb.lock_maxtimeout)
#define NFS4_STRICT_LOCKING(sb)		((sb)->u.nfs4_sb.strict_locking)
#define NFS4_SILLY_RENAME(sb)		((sb)->u.nfs4_sb.silly_rename)
#define NFS4_LEASE_TIME(sb)             ((sb)->u.nfs4_sb.lease_time)
#define NFS4_PIDOF_RECOVERD(sb)         ((sb)->u.nfs4_sb.pidof_recoverd)
#define NFS4_GRACE_WAIT_QUEUE(sb)       ((sb)->u.nfs4_sb.grace_wait_queue)
#define NFS4_FILE_LIST(sb)		((sb)->u.nfs4_sb.file_list)
#define NFS4_RENEWD(sb)			((sb)->u.nfs4_sb.renewd)
#define NFS4_LAST_RENEWAL(sb)		((sb)->u.nfs4_sb.last_renewal)
#define NFS4_FLUSHD(sb)			((sb)->u.nfs4_sb.flushd)
#define NFS4_FLUSHD_TIMEOUT(sb)		((sb)->u.nfs4_sb.flushd_timeout)
#define NFS4_FLUSHD_INODES(sb)		((sb)->u.nfs4_sb.flushd_inodes)
#define NFS4_FLUSHD_WAIT(sb)		((sb)->u.nfs4_sb.flushd_wait)
#define NFS4_IP_ADDR(sb)		((sb)->u.nfs4_sb.ip_addr)
#define NFS4_CALLBACK_PORT(sb)		((sb)->u.nfs4_sb.callback_port)

#define NFS4_DFSID_MAJOR(dentry)	((dentry)->d_inode->u.nfs4_i.fsid_major)
#define NFS4_DFSID_MINOR(dentry)	((dentry)->d_inode->u.nfs4_i.fsid_minor)

#define NFS4_BUFSIZE			8192
#define NFS4FS_MAX_IOV			10
#define NFS4_SETCLIENTID_MAXTRIES	5
#define NFS4_READDIR_MAXTRIES		5
#define NFS4_MAXIO			4
#define NFS4_MAX_REQUEST_SOFT		192
#define NFS4_MAX_REQUEST_HARD		256
#define NFS4_MAXCOMMIT			64
#define NFS4_READ_DELAY			(2 * HZ)
#define NFS4_WRITEBACK_DELAY		(5 * HZ)
#define NFS4_WRITEBACK_LOCKDELAY	(60 * HZ)
#define NFS4_COMMIT_DELAY		(5 * HZ)
#define RPC_SLACK_SPACE			512

#ifdef __OpenBSD__
struct dentry {
	struct qstr d_name;
	u32	fh_len;
	nfsfh_t	fh_val;
};

#define nfs4_up_write(lock)
#endif /* OpenBSD */

typedef struct {
	int read_count;       /* negative if there are writers */
	struct rw_semaphore sem;
} nfs4_rwlock_t;

#define DDATA_SIZE(len)    (sizeof(struct nfs4fs_dentry_data) + len - 1)
#define NFS4_HAVE_FH(dentry)  ((dentry)->d_fsdata && \
			       ((struct nfs4fs_dentry_data *)(dentry)->d_fsdata)->fh_len)

struct nfs4fs_client {
	atomic_t		cl_count;	/* refcount */
	u64			cl_clientid;	/* constant */
};

struct nfs4fs_flavor_info {
	u32       oid_len;
	char *     oid_data;
	u32       qop;
	u32       service;
};

struct nfs4fs_sec_flavor {
	u32       flavor;
	struct nfs4fs_flavor_info   *f_info;
};

/* According to fs/locks.c, locking state is separated by (files_struct, pid)-pair for each
   process.  In addition, an NFSv4 lockowner must maintain a seperate seqid for each mount.
   Therefore, an NFSv4 lockowner is a (superblock, files_struct, pid)-triple. */

struct nfs4fs_lockowner {
	atomic_t		lo_count;	/* refcount */
	struct nfs4fs_client	*lo_client;	/* holds reference */
	struct list_head	lo_stateids; 	/* list of file_data, linked through fi_list */
	struct list_head	lo_list;	/* links all lockowners on this client */
	u32			lo_id;		/* 32-bit identifier, unique */
	
	/*
	 * The following fields are used for lock_owners only.
	 * Note!  We do not claim a reference in ->lo_inode; this is OK because we never
	 * dereference it.  The pointer itself is used as a hash key.
	 * The hash lists are protected by the state_spinlock.
	 */
	struct list_head	lo_hash;	/* hashed by inode, pid */
	struct list_head	lo_id_hash;	/* hashed by id */
	struct inode *		lo_inode;	/* constant */
	pid_t			lo_pid;		/* constant */

	/*
	 * The ->lo_sema is held during all seqid-mutating operations.
	 * Its purpose is to properly serialize the seqid, as mandated by the protocol.
	 */
	struct semaphore	lo_sema;
	u32			lo_seqid;	/* protected by lo_sema */
};

/*
 * These are in one-to-one correspondence with stateid's on the server.
 * They can represent either open_stateid's or lock_stateid's.
 * An open_stateid can be "delegated"; this means that the OPEN was done locally without
 * sending anything to the server.
 *
 * For open_stateid's:
 *     ->fi_lockowner points to an open_owner
 *     ->fi_read (resp. fi_write) links all OPENs for read (resp. write) on this inode
 *                                or delegation (depending on whether this OPEN is "delegated")
 *     ->fi_linkage starts a list of all lock_stateids associated with the open_stateid
 *     ->fi_open_stateid is NULL
 *     ->fi_delegation is non-NULL iff the OPEN is "delegated", and holds a reference
 *
 * For lock_stateid's:
 *     ->fi_lockowner points to a lock_owner
 *     ->fi_{read,write} are empty
 *     ->fi_linkage links all lock_stateids associciated with the same open_stateid
 *     ->fi_open_stateid points to the associated open_stateid
 *     ->fi_delegation is always NULL
 *
 * Everything here is protected by the state_spinlock, except the fields where
 * protected by 'fi_sema' is noted.
 */
struct nfs4fs_file_data {
	struct nfsnode			*fi_n;
	pid_t				fi_pid;

	atomic_t			fi_count;	/* refcount */
	struct nfs4fs_lockowner	*	fi_lockowner;	/* constant */
	struct list_head		fi_list;	/* all files on this lockowner */

	/*
	 * Linkage for open_stateids and lock_stateids.
	 * There is another subtlety here: it would be nice if the lock_stateid could just
	 * hold a reference to the associated open_stateid.  However, things are not as
	 * simple as that.  We want to leave lock_stateids lying around for the duration
	 * of the open_stateid, and only destroy them when the last reference to the
	 * open_stateid goes away.  If references are held, this will be impossible.
	 *
	 * The solution that I have devised is to have the lock_stateid hold a
	 * reference to the open_stateid, but only if its own refcount is > 0.
	 * This means that when new references to the lock_stateid are acquired,
	 * we must test if a new reference to the open_stateid must be acquired.
	 * When the last reference to the lock_stateid goes away, the reference
	 * to the open_stateid must be dropped.
	 *
	 * TODO: This solution seems awfully delicate; can someone think of another
	 * way which might be more robust?
	 */
	struct list_head		fi_linkage;
	struct nfs4fs_file_data *	fi_open_stateid;

	/*
	 * The ->fi_sema is held during all operations which use the stateid.
	 * Its main purpose is to properly serialize the stateid, as mandated by the protocol.
	 * I have also found that it is the "right" way to protect ->fi_flags, ->fi_delegation.
	 */
	nfs4_rwlock_t			fi_sema;
	stateid4			fi_stateid;	/* protected by fi_sema */
	unsigned int			fi_flags;	/* protected by fi_sema */
	struct nfs4fs_delegation *	fi_delegation;	/* protected by fi_sema, holds ref */
};
#define fi_client			fi_lockowner->lo_client
#define IS_OPEN_STATEID(fp)		(fp->fi_open_stateid == NULL)
#define IS_LOCK_STATEID(fp)		(fp->fi_open_stateid != NULL)

/*
 * fi_flags are as follows:
 *   FI_READ - if this is an open stateid, is the OPEN for read?
 *   FI_WRITE - if this is an open stateid, is the OPEN for write?
 *   FI_FAKE - if set, this is a "fake" stateid, meaning that it does not exist
 *             on the server.  This can happen if:
 *                o The stateid is freshly instantiated and we are waiting for the
 *                  OPEN or LOCK response to instantiate it on the server.
 *                o This is a "delegated" OPEN.
 *                o Server reboot has been detected, and consequently all stateids
 *                  have been marked "fake" pending reboot recovery (not yet implemented).
 */
#define FI_READ				0x00000001    /* same as OPEN4_SHARE_ACCESS_READ */
#define FI_WRITE		       	0x00000002    /* same as OPEN4_SHARE_ACCESS_WRITE */
#define FI_FAKE				0x00000004

/*
 * This macro converts between the open flags used internally by the VFS and the
 * FI_{READ,WRITE} flags above.  (In this kernel, we just have to keep the 2 low bits.)
 */
#define OPEN_FLAGS_TO_FI_FLAGS(f)	((f) & 3)

/*
 * Note on the ->dl_{read,write}_placeholder fields:  If we hold a delegation, and
 * a file descriptor is closed, we might want to keep the file open on the server.
 * This is so that we can re-use the OPEN if another process opens the file (i.e.,
 * "delegating" the new OPEN).  This is implemented by holding a "placeholder" reference
 * to the file_data structure (the CLOSE is not sent until the last reference is dropped).
 * Actually, we might need two placeholders, one for READ and one for WRITE.  The
 * key invariant of the placeholders is: If there is a _unique_ open for READ (resp.
 * write) on the server, ->dl_{read,write}_placeholder is non-NULL and holds a reference.
 * We always maintain this invariant.  Thus if a new OPEN is instantiated, it may
 * become a placeholder (if it is the unique OPEN for read or write), or cause a
 * placeholder to go away (if an old OPEN is no longer the unique one).  When an OPEN
 * is destroyed, it may cause a placeholder to be set (if destroying it means that there
 * is now a unique OPEN for read or write).
 *
 * Everything here is either constant, or protected by the state_spinlock.
 */
struct nfs4fs_delegation {
	atomic_t		dl_count;
	struct super_block	*dl_sb;		/* constant */
	struct inode		*dl_inode;	/* constant */
	struct list_head	dl_hash;	/* hashed by superblock, filehandle */
	struct list_head	dl_read_opens;	/* starts a list of delegated read OPENs */
	struct list_head	dl_write_opens;	/* starts a list of delegated write OPENs */
	struct list_head	dl_list;	/* links a list of delegations on this sb */
	struct nfs4fs_file_data *dl_read_placeholder;	/* holds reference */
	struct nfs4fs_file_data *dl_write_placeholder;	/* holds reference */
	u32			dl_type;	/* constant */
	stateid4		dl_stateid;	/* constant */

	/*
	 * If the delegation is recalled, we might have to "undelegate" OPENs
	 * which have been delegated to the client.  The current version of the
	 * nfsv4 protocol requires the name to be specified in the OPEN request.
	 * (Hopefully, this can be changed.)  For the time being, then, we must
	 * save the following in the delegation:
	 *   - the filename
	 *   - the containing directory's filehandle
	 * Since I anticipate this being a temporary hack, pending change in
	 * the protocol, I've set hardcoded limits here.  If the limits are
	 * exceeded, we refuse to accept the delegation..
	 *
	 * These are constant, so don't worry about protecting them with
	 * a lock.
	 */
	unsigned int		dl_fhlen;   		/* parent filehandle length */
	char			dl_fhval[64];		/* parent filehandle value */
	char			dl_filename_len;
	char			dl_filename[64];
};

/* end of client-side state definitions */

struct nfs4fs_fattr {
	unsigned int	fa_valid;
	nfs_ftype4	fa_type;
	loff_t		fa_size;
	u64		fa_fsid_major;
	u64		fa_fsid_minor;
	u64		fa_fileid;
	umode_t		fa_mode;
	nlink_t		fa_nlink;
	uid_t		fa_uid;
	gid_t		fa_gid;
	u32		fa_rdev_major;
	u32		fa_rdev_minor;
	struct timespec	fa_atime;
	struct timespec	fa_ctime;
	struct timespec	fa_mtime;
	u64		fa_maxread;
	u64		fa_maxwrite;
	u64		fa_ffree;
	u64		fa_ftotal;
	u32		fa_maxname;
	u64		fa_savail;
	u64		fa_sfree;
	u64		fa_stotal;
	u64		fa_changeid;
	u32		fa_lease_time;
	u64		fa_maxfilesize;
#if 0
	unsigned int 	fa_nacl;	/* number of acl's allocated */
	nfs4_acl	*fa_acl;	/* array of size fa_nacl */
#endif
};
/* flags for fa_valid */
#define FA_SIZE		0x00000001
#define FA_FSID		0x00000002
#define FA_FILEID	0x00000004
#define FA_MODE		0x00000008
#define FA_NLINK	0x00000010
#define FA_UID 		0x00000020
#define FA_GID 		0x00000040
#define FA_RDEV		0x00000080
#define FA_ATIME	0x00000100
#define FA_CTIME	0x00000200
#define FA_MTIME	0x00000400
#define FA_MAXREAD	0x00000800
#define FA_MAXWRITE	0x00001000
#define FA_TYPE		0x00002000
#define FA_FFREE       	0x00004000
#define FA_FTOTAL       0x00008000
#define FA_MAXNAME	0x00010000
#define FA_SAVAIL	0x00020000
#define FA_SFREE 	0x00040000
#define FA_STOTAL	0x00080000
#define FA_CHANGEID	0x00100000
#define FA_LEASE_TIME	0x00200000
#define FA_MAXFILESIZE	0x00400000
#define FA_ACL		0x00800000

struct nfs4fs_getattr {
	/* filled in by caller */
	int flags;
	bitmap4 *bm;

	/* filled in by setup routine */
	struct dentry *dentry;

	/* filled in by handler */
	struct nfs4fs_fattr fattr;
};
/* flags for struct nfs4fs_getattr */
#define FL_NONFATAL		0x00000001
#define FL_NOREVAL		0x00000002
#define FL_CLOBBER_OK		0x00000004
#define FL_NOENT_OK             0x00000008

struct nfs4fs_getfh {
	/* filled in by caller */
	int nonfatal;

	/* filled in by setup routine */
	struct dentry *dentry;
};

struct nfs4fs_access {
	u32 mode, supported, rmode;
};

struct nfs4fs_close {
	/* filled in by caller */
	struct nfs4fs_file_data *data;
	int read;
	int write;
	
	/* The CLOSE setup routine will either use a CLOSE or OPEN_DOWNGRADE, as needed.
	   We record the actual operation used here, for the handler's benefit. */
	u32 op;
};

struct nfs4fs_commit {
	/* filled in by caller */
	loff_t start;
	u32 length;

	/* filled in by handler */
	verifier4 verifier;
};

struct nfs4fs_create {
	/* filled in by caller */
	struct dentry *dentry;
	nfs_ftype4 type;
	struct vattr *vap;
	char *linktext;

	/* coalesced into the call by the setup routine */
	struct nfs4fs_getattr getattr;
	struct nfs4fs_getfh getfh;
};

struct nfs4fs_reval {
	/* filled in by caller */
	int force_getfh;
	
	/* filled in by setup routine */
	int did_getattr;
	int did_getfh;
	struct nfs4fs_getattr getattr;
};

struct nfs4fs_link {
	char *name;
	int namelen;
};

struct nfs4fs_lookup {
	/* filled in by caller.  Note: we keep a separate field for the name, rather than
	   always using dentry->d_name, because nfs4_rename() needs to make them different. */
	struct dentry *dentry;
	char *name;
	int namelen;
};

struct nfs4fs_open {
	/* filled in by caller */
	open_claim_type4		claim_type;
	int				flags;
	struct vattr			*vap;		/* used only if O_CREAT is set */
	struct dentry			*dentry;
	struct nfs4fs_file_data		*fp;

	/* coalesced into call */
	struct nfs4fs_getattr		getattr;
	struct nfs4fs_getfh		getfh;
};

struct nfs4fs_putfh {
	/* filled in by caller */
	struct dentry *dentry;

	/* filled in by setup routine */
	nfs_opnum4 op;
	int nlookups;
};

struct nfs4fs_read {
	/* filled in by caller */
	struct nfs4fs_file_data *file_data;
	loff_t offset;
	u32 maxcount;
	struct uio *uiop;

	/* filled in by handler */
	u32 bytes_read;
	int eof;
};

struct nfs4fs_readdir {
	/* filled in by caller */
	u32 count;
	bitmap4 *bm;
	nfs_cookie4 cookie;
	verifier4 verifier;
};

struct nfs4fs_rename {
	/* XXX : should these be dentries? */
	struct qstr *old_name;
	struct qstr *new_name;
};

struct nfs4fs_setclientid {
	/* filled in by caller */
	struct nfs4mount *nmp;
	int namlen;
	char *name;
	u32 callback_prog;
	char *callback_netid;
	char *callback_universal_address;

	/* filled in by handler, used by setclientid_confirm */
	int in_use;
	int old_proto;
	clientid4 clientid;
	verifier4 verifier;
};

struct nfs4fs_write {
	/* filled in by caller */
	struct nfs4fs_file_data *file_data;
	loff_t offset;
	u32 stable;
	u32 count;
	struct uio *uiop;

	/* filled in by handler */
	u32 bytes_written;
	u32 committed;
	verifier4 writeverf;
};

/* need to move this to nfs4fs_xdr.h */
struct nfs4fs_op {
	u32					opnum;
	u32					nfserr;
};

struct nfs4fs_compound {
	struct nfs4mount		*nmp;
	struct proc			*procp;
	struct ucred			*cred;
	struct mbuf			*mreq, *mrep;
	struct mbuf			*mb, *md;
	unsigned int			flags;

	/* If this is a stateful operation, all of our state hangs here */
	struct nfs4fs_client *		client;
	struct nfs4fs_lockowner *	seqid_holder;
	struct nfs4fs_file_data *	stateid_holder;

	/* Leftovers, should go away */
	struct dentry			*current_fh, *save_fh;
	u32 *				nopsp;
	u32 *				seqidp;
	u32 *				stateidp[NFS4_MAXIO];
	unsigned int			nstateids;

	/* used for SECINFO */
	struct rpc_auth *		secinfo_auth;
	
	int				seqid_index;
	int				renew_index;
	int				confirm_index;
	unsigned long			timestamp;
	unsigned long			retry_delay;

	/* Scratch variables for XDR encode/decode */
	int				nops;
	u32 *				p;
	u32 *				end;

	/* request */
	unsigned int			req_nops;
	u32				taglen;
	char *				tag;

	/* response */
	caddr_t				dpos;
	int				resp_nops;
	int				toplevel_status;
};

/*
 * Bits in the "flags" field of a struct nfs4fs_compound.
 */
#define CA_STATEFUL		0x00000001
#define	CA_SEQID_LOCKED		0x00000002	/* is seqid currently locked? */
#define CA_SEQID_RELEASE	0x00000004	/* should handler release seqid? */
#define CA_STATEID_LOCKED	0x00000008	/* is stateid currently locked? */
#define CA_STATEID_RELEASE	0x00000010	/* should handler release stateid? */
#define CA_AUXILIARY		0x00000020
#define CA_AUX_SEQID_LOCKED	0x00000040
#define CA_AUX_STATEID_LOCKED	0x00000080
#define CA_AUX_SEQID_MUTATE	0x00000100      /* should handler increase aux. seqid? */
#define CA_RUN_AS_MACHINE  0x00000200  /* tell rpciod to run with GSS machine creds */

struct ng_compound4args {
	char			*tag;
#ifdef __linux__
	struct rpc_clnt		*clnt;
	int			iov_nr;
	struct iovec		iov[NFS4FS_MAX_IOV];
	int			iov_static[NFS4FS_MAX_IOV];
	int			iov_userspace[NFS4FS_MAX_IOV];
#else
	struct vnode *vp;
	struct proc *procp;
	struct ucred *cred;
	struct mbuf *mreq;
	struct mbuf *mb;
#endif
	u32			*p;        /* buffer pos in current iovec */
	u32			*end;      /* end of buffer in current iovec */
	int			nops;
	u32			*nopsp;
	struct dentry		*current_fh;
	struct dentry		*save_fh;
	struct nfs4fs_lockowner *seqid_holder;
	int			release_seqid;
	int			seqid_index;
	struct nfs4fs_file_data *stateid_holder;
	int			stateid_write_locked;
};

struct ng_compound4res {
	struct vnode		*vp;
	struct proc		*procp;
	struct ucred		*cred;
	struct mbuf		*mrep;
	struct mbuf		*md;
	caddr_t			dpos;
	int			toplevel_status;
	int			nops;
	struct nfs4fs_lockowner *seqid_holder;     /* if not already released */
	struct nfs4fs_file_data *stateid_holder;   /* will be a writelock, if not released */
};

/* encode.c */
extern int nfs4_setup_compound(struct nfs4fs_compound *cp, char *tag);
extern int nfs4_setup_simple_op(struct nfs4fs_compound *cp, nfs_opnum4 op);
extern int nfs4_setup_access(struct nfs4fs_compound *cp, struct nfs4fs_access *access);
extern int nfs4_setup_close(struct nfs4fs_compound *cp, struct nfs4fs_close *close);
extern int nfs4_setup_commit(struct nfs4fs_compound *cp, struct nfs4fs_commit *commit);
extern int nfs4_setup_create(struct nfs4fs_compound *cp, struct nfs4fs_create *create);
extern int nfs4_setup_delegreturn(struct nfs4fs_compound *cp, stateid4 stateid);
extern int nfs4_setup_getattr(struct nfs4fs_compound *cp, struct nfs4fs_getattr *getattr);
extern int nfs4_setup_getfh(struct nfs4fs_compound *cp, struct nfs4fs_getfh *getfh);
extern int nfs4_setup_link(struct nfs4fs_compound *cp, struct nfs4fs_link *link);
extern int nfs4_setup_lookup(struct nfs4fs_compound *cp, struct nfs4fs_lookup *lookup);
extern int nfs4_setup_lookupp(struct nfs4fs_compound *cp, struct dentry *dentry);
extern int nfs4_setup_open(struct nfs4fs_compound *cp, struct nfs4fs_open *open);
extern int nfs4_setup_open_confirm(struct nfs4fs_compound *cp, struct nfs4fs_open *open);
extern int nfs4_setup_putfh(struct nfs4fs_compound *cp, struct nfs4fs_putfh *putfh);
#define    nfs4_setup_putrootfh(cp)    nfs4_setup_simple_op((cp), OP_PUTROOTFH)
extern int nfs4_setup_read(struct nfs4fs_compound *cp, struct nfs4fs_read *read);
extern int nfs4_setup_readdir(struct nfs4fs_compound *cp, struct nfs4fs_readdir *readdir);
#define    nfs4_setup_readlink(cp)     nfs4_setup_simple_op((cp), OP_READLINK)
extern int nfs4_setup_remove(struct nfs4fs_compound *cp, char *name, int namelen);
extern int nfs4_setup_rename(struct nfs4fs_compound *cp, struct nfs4fs_rename *rename);
extern int nfs4_setup_renew(struct nfs4fs_compound *cp);
extern int nfs4_setup_restorefh(struct nfs4fs_compound *cp);
extern int nfs4_setup_reval(struct nfs4fs_compound *cp, struct nfs4fs_getattr *reval);
extern int nfs4_setup_savefh(struct nfs4fs_compound *cp);
extern int nfs4_setup_setattr(struct nfs4fs_compound *cp, struct vattr *vap,
			      struct nfs4fs_file_data *fdata);
extern int nfs4_setup_setattr_mode(struct nfs4fs_compound *cp, umode_t mode);
extern int nfs4_setup_setclientid(struct nfs4fs_compound *cp,
				  struct nfs4fs_setclientid *setclientid);
extern int nfs4_setup_setclientid_confirm(struct nfs4fs_compound *cp,
					  struct nfs4fs_setclientid *setclientid);
extern int nfs4_setup_write(struct nfs4fs_compound *cp, struct nfs4fs_write *write);
extern bitmap4 nfs4_positive_bitmap;
extern bitmap4 nfs4_negative_bitmap;

/* decode.c */
extern int nfs4_read_uio(struct nfs4fs_compound *cp, struct uio *uiop, int len);
extern int nfs4_handle_compound(struct nfs4fs_compound *cp);
extern int nfs4_handle_simple_op(struct nfs4fs_compound *cp, nfs_opnum4 op, int nbytes);
extern int nfs4_handle_access(struct nfs4fs_compound *rp, struct nfs4fs_access *access);
extern int nfs4_handle_close(struct nfs4fs_compound *rp, struct nfs4fs_close *close);
extern int nfs4_handle_commit(struct nfs4fs_compound *rp, struct nfs4fs_commit *commit);
extern int nfs4_handle_create(struct nfs4fs_compound *cp, struct nfs4fs_create *create);
extern int nfs4_handle_getattr(struct nfs4fs_compound *cp, struct nfs4fs_getattr *getattr);
extern int nfs4_handle_getfh(struct nfs4fs_compound *cp, struct nfs4fs_getfh *getfh);
extern int nfs4_handle_link(struct nfs4fs_compound *cp, struct nfs4fs_link *link);
#define    nfs4_handle_lookup(rp)	nfs4_handle_simple_op(rp, OP_LOOKUP, 0)
#define    nfs4_handle_lookupp(rp)	nfs4_handle_simple_op(rp, OP_LOOKUPP, 0)
extern int nfs4_handle_open(struct nfs4fs_compound *cp, struct nfs4fs_open *open);
extern int nfs4_handle_open_confirm(struct nfs4fs_compound *cp, struct nfs4fs_open *open);
extern int nfs4_handle_putfh(struct nfs4fs_compound *cp, struct nfs4fs_putfh *putfh);
#define    nfs4_handle_putrootfh(rp)	nfs4_handle_simple_op(rp, OP_PUTROOTFH, 0)
extern int nfs4_handle_read(struct nfs4fs_compound *cp, struct nfs4fs_read *read);
extern int nfs4_handle_readdir(struct nfs4fs_compound *cp, struct nfs4fs_readdir *readdir);
extern int nfs4_handle_readlink(struct nfs4fs_compound *cp, struct uio *uiop);
#define    nfs4_handle_remove(rp)	nfs4_handle_simple_op(rp, OP_REMOVE, \
						sizeof(change_info4))
#define    nfs4_handle_rename(rp)	nfs4_handle_simple_op(rp, OP_RENAME, \
						2 * sizeof(change_info4))
#define    nfs4_handle_restorefh(rp)	nfs4_handle_simple_op(rp, OP_RESTOREFH, 0)
extern int nfs4_handle_reval(struct nfs4fs_compound *cp, struct nfs4fs_reval *reval);
#define    nfs4_handle_savefh(rp)	nfs4_handle_simple_op(rp, OP_SAVEFH, 0)
extern int nfs4_handle_setattr(struct nfs4fs_compound *cp);
extern int nfs4_handle_setclientid(struct nfs4fs_compound *rp,
				   struct nfs4fs_setclientid *setclientid);
extern int nfs4_handle_setclientid_confirm(struct nfs4fs_compound *rp,
					   struct nfs4fs_setclientid *setclientid);
extern int nfs4_handle_write(struct nfs4fs_compound *cp, struct nfs4fs_write *write);
extern int nfs4_read_attrs(struct nfs4fs_compound *cp, struct nfs4fs_fattr *nfp);
extern u32 nfs4_fileid4_to_fileid(u64 fileid4);

/* inode.c */
extern int nfs4_inode_revalidate(struct dentry *dentry);
extern int nfs4_setattr(struct dentry *dentry, struct iattr *iap);
extern int nfs4_do_setclientid(struct nfs4mount *nmp, struct proc *p);
extern int nfs4fs_start_recoverd(struct super_block *sb);
extern int nfs4fs_wait_on_recoverd(struct super_block *sb);

/* dir.c */
extern struct dentry_operations nfs4_dentry_operations;
extern struct inode_operations nfs4_dir_inode_operations;
extern struct file_operations nfs4_dir_operations;

/* open.c */
struct nfs4fs_lockowner *nfs4_get_lockowner_by_id(unsigned int id);
void nfs4_put_lockowner(struct nfs4fs_lockowner *lockowner);
struct nfs4fs_file_data *nfs4_get_file_data(struct nfsnode *np);
void nfs4_put_file_data(struct nfs4fs_file_data *file_data);
void nfs4fs_init_state(void);
void nfs4fs_cleanup_state(void);
int nfs4fs_do_open(struct nfs4fs_open *open);
int nfs4fs_do_close(struct dentry *dentry, struct nfs4fs_close *close);

/* file.c */
extern struct inode_operations nfs4_file_inode_operations;
extern struct file_operations nfs4_file_operations;
extern struct address_space_operations nfs4_file_aops;

/* symlink.c */
extern struct inode_operations nfs4_symlink_inode_operations;

/* proc.c */
#if NFS4_DEBUG
extern void debug_dump(struct nfs4fs_compound *cp, int nbytes);
#endif
extern int  nfs4fs_call_compound(struct nfs4fs_compound *cp);
extern void nfs4fs_release_compound(struct nfs4fs_compound *cp);
extern struct rpc_program nfs4_program;
extern struct rpc_stat nfs4_rpcstat;

/* subs.c */
extern bitmap4 fsinfo_bm, fsattr_bm, getattr_bm, readdir_bm;
extern int nfs4_loadattrcache(struct vnode *vp, struct nfs4fs_fattr *fp, struct vattr *vaper);
extern void nfs4_setup_bitmaps(void);

#endif
