/*
 *  GSSD client.
 *
 *  Copyright (c) 2000 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Weston Andros Adamson   <muzzle@citi.umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: nfs4_gss.c,v 1.10 2003/05/02 20:27:45 rees Exp $
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/socket.h>
#include <sys/mbuf.h>
#include <sys/socketvar.h>
#include <sys/errno.h>
#include <sys/malloc.h>

#include <netinet/in.h>
#include <netinet/tcp.h>

#include <rpc/rpcclnt.h>
#include <rpc/rpcv2.h>
#include <rpc/rpc_pmap.h>
#include <rpc/rpcm_subs.h>

#include <nfs4/list.h>
#include <nfs4/nfs4_prot.h>
#include <nfs4/nfs4_debug.h>
#include <nfs4/nfs4.h>
#include <nfs4/nfs4_gss.h>

static int gss_fake_string_to_uint32(int type, unsigned int l, char *s, u_int32_t *u);
static int gss_fake_uint32_to_string(int type, u_int32_t u, struct gss_name **s);

struct rpcclnt * gss_rpcclnt = NULL;
int gss_fake;
static char gss_fake_user[] = "rees@citi.umich.edu";
static char gss_fake_group[] = "staff@citi.umich.edu";

static int
gss_fake_string_to_uint32(int type, unsigned int l, char *s, u_int32_t *u)
{
	*u = (type == GSS_GROUP) ? 20 : 4905;
	return 0;
}

static int
gss_fake_uint32_to_string(int type, u_int32_t u, struct gss_name **s)
{
	char *fake;

	fake = (type == GSS_GROUP) ? gss_fake_group : gss_fake_user;

	MALLOC(*s, struct gss_name *, sizeof(struct gss_name), M_TEMP, M_WAITOK);
	MALLOC((*s)->name, char *, strlen(fake) + 1, M_TEMP, M_WAITOK);
	strcpy((*s)->name, fake);

	return 0;
}

int
gss_create(void)
{
	struct sockaddr_in	sin;
	struct rpcclnt * 	rpc;
	struct mbuf *		nam;
	int status;
	int port;

	if (gss_rpcclnt != NULL)
		gss_destroy();

	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_LOOPBACK;
	sin.sin_len = sizeof(struct sockaddr_in);

	if ((status = rpc_pmap_getport(RPC_GSSD_PROGRAM, RPC_GSSD_VERSION,
				       IPPROTO_UDP, &sin, &port)) != 0) {
		RPCDEBUG("no portmap");
#ifdef NFS4_DEBUG
		gss_fake = 1;
#endif
		return status;
	}

	if (port == 0) {
		RPCDEBUG("gssd not registered");
#ifdef NFS4_DEBUG
		gss_fake = 1;
#endif
		return EPROGUNAVAIL;
	}

	sin.sin_port = htons(port);

	if (sin.sin_len > MLEN)
                return EINVAL;

	nam = m_get(M_WAIT, MT_SONAME);

	bcopy(&sin, mtod(nam, void *), sin.sin_len);
	nam->m_len = sin.sin_len;

	rpcclnt_get(rpc);
	bzero(rpc, sizeof(struct rpcclnt));

	RPCDEBUG("CALL SETUP/CONNECT");
	status = rpcclnt_setup(rpc, RPCCLNT_SOFT, nam,
			       SOCK_DGRAM, 0, RPC_GSSD_PROGRAM, RPC_GSSD_VERSION, "gssd",
			       GSSD_MAX_NAMELEN, GSSD_MAX_NAMELEN);

	if (status) {
		RPCDEBUG("rpcclnt_setup returns error %d", status);
		m_free(nam);
		rpcclnt_put(rpc);
		return status;
	}

	gss_rpcclnt = rpc;
	return 0;
}

void
gss_destroy(void)
{
	if (gss_rpcclnt == NULL)
		return;

	m_free(gss_rpcclnt->rc_name);
	rpcclnt_put(gss_rpcclnt);
	gss_rpcclnt = NULL;
	return;
}

int
gss_string_to_uint32(int type, unsigned int l, char *s, u_int32_t *u)
{
	u_int32_t *lp, *up, offp, procid;
	struct rpc_reply rep;
	struct mbuf * m;
	int status;

	if (s == NULL || u == NULL)
		return EFAULT;

	if (gss_fake)
		return gss_fake_string_to_uint32(type, l, s, u);

	if (l > MCLBYTES || l <= 0)
		return EINVAL;

	switch (type) {
	case GSS_OWNER:
		procid = GSSD_NAME_TO_UID;
		break;
	case GSS_GROUP:
		procid = GSSD_NAME_TO_GID;
		break;
	default:
		return EINVAL;
		break;
	}

	/* XXX bad model... should just check to make sure that the gss rpcclnt
	 * struct is initialized */
	if (gss_rpcclnt == NULL) {
		status = gss_create();
		if (status)
#ifdef NFS4_DEBUG
			return gss_fake_string_to_uint32(type, l, s, u);
#else
			return status;
#endif
	}

	/* allocate an mbuf to hold the string... */
	m = m_get(M_WAIT, MT_DATA);
	if (rpcm_rndup(l) + RPCX_UNSIGNED > MLEN) {
		MCLGET(m, M_WAITOK);
		if ((m->m_flags & M_EXT) == 0) {
			m_free(m);
			return ENOBUFS;
		}
	}

	lp = mtod(m, u_int32_t *);
 	*lp = htonl(l);

	bcopy(s, mtod(m, char *) + RPCX_UNSIGNED, l);

	m->m_len = RPCX_UNSIGNED + rpcm_rndup(l);

	rpcclnt_setauth(gss_rpcclnt, RPCAUTH_UNIX, 0, NULL, 0, NULL,
			RPCAUTH_ROOTCREDS);

	if ((status = rpcclnt_request(gss_rpcclnt, m, procid, curproc, &rep)) != 0) {
		RPCDEBUG("error %d in rpcclnt_request", status);
		gss_destroy();
		return status;
	}

	if ((status = rpcclnt_err(&rep)) != 0 || rep.result_md == NULL) {
		RPCDEBUG("error returned from gssd");
		m_freem(rep.mrep);
		return status;
	}

	if ((rep.result_dpos - mtod(rep.result_md, caddr_t)) >=
	    rep.result_md->m_len) {
		RPCDEBUG("no more space in mbuf, get next..");

		if (rep.result_md->m_next == NULL){
		 	RPCDEBUG("no more data!\n");
			m_freem(rep.mrep);
			return EFAULT;
		}
		rep.result_md = rep.result_md->m_next;
		rep.result_dpos = mtod(rep.result_md, caddr_t);
	}

	m = m_pulldown(rep.result_md, rep.result_dpos -
		       mtod(rep.result_md, char *), RPCX_UNSIGNED, &offp);

	if (m == NULL) {
		RPCDEBUG("m_pulldown failed!");
		m_freem(rep.mrep);
		return EFAULT;
	}

	up = (u_int32_t *)(mtod(m, caddr_t)+offp);
	*u = ntohl(*up);
	m_freem(rep.mrep);
	return 0;
}

int
gss_uint32_to_string(int type, u_int32_t u, struct gss_name **s)
{
	u_int32_t len, *up, *lp, off;
	u_int32_t procid;
	int status;
	struct mbuf * m;
	struct rpc_reply rep;

	if (s == NULL)
		return EFAULT;

	if (gss_fake)
		return gss_fake_uint32_to_string(type, u, s);

	if (gss_rpcclnt == NULL) {
		status = gss_create();
		if (status)
#ifdef NFS4_DEBUG
			return gss_fake_uint32_to_string(type, u, s);
#else
			return status;
#endif
	}

	switch (type) {
	case GSS_OWNER:
		procid = GSSD_UID_TO_NAME;
		break;
	case GSS_GROUP:
		procid = GSSD_GID_TO_NAME;
		break;
	default:
		return EINVAL;
		break;
	}

	m = m_get(M_WAIT, MT_DATA);
	up = mtod(m, u_int32_t *);
 	*up = htonl(u);
	m->m_len = RPCX_UNSIGNED;

	rpcclnt_setauth(gss_rpcclnt, RPCAUTH_UNIX, 0, NULL, 0, NULL,
			RPCAUTH_ROOTCREDS);

	if ((status = rpcclnt_request(gss_rpcclnt, m, procid, curproc, &rep))
	    != 0) {
		RPCDEBUG("error %d in rpcclnt_request", status);
		return status;
	}

	if ((status = rpcclnt_err(&rep)) != 0) {
		RPCDEBUG("error returned from gssd");
		return status;
	}

	m = m_pulldown(rep.result_md, rep.result_dpos -
		       mtod(rep.result_md, caddr_t), RPCX_UNSIGNED, &off);

	if (m == NULL) {
		RPCDEBUG("m_pulldown failed!");
		return EFAULT;
	}

	lp = (u_int32_t *)(mtod(m, caddr_t)+off);
	len = ntohl(*lp);

	RPCDEBUG("the length is %d", len);

	/* XXX check for len > MCLBYTES */
	m = m_pulldown(rep.result_md, (rep.result_dpos -
				       mtod(rep.result_md, caddr_t)) + RPCX_UNSIGNED, len, &off);

	if (m == NULL) {
		RPCDEBUG("m_pulldown of %d bytes failed!", len);
		return EFAULT;
	}

	MALLOC(*s, struct gss_name *, sizeof(struct gss_name), M_TEMP, M_WAITOK);
	MALLOC((*s)->name, char *, len + 1, M_TEMP, M_WAITOK);
	bcopy(mtod(m, char *) + off, (*s)->name, len);
	(*s)->name[len] = '\0';
	(*s)->len = len;

	/* free reply */
	m_freem(rep.mrep);

	return 0;
}

void gss_free_name(struct gss_name *n)
{
	FREE(n->name, M_TEMP);
	FREE(n, M_TEMP);
}
