/*	$Id: nfs4fs_encode.c,v 1.4 2002/09/12 17:36:59 rees Exp $	*/

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

/*
 *  Kendrick Smith <kmsmith@umich.edu>
 *  Jim Rees <rees@umich.edu>
 */

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/resourcevar.h>
#include <sys/proc.h>
#include <sys/mount.h>
#include <sys/buf.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/conf.h>
#include <sys/namei.h>
#include <sys/vnode.h>
#include <sys/dirent.h>
#include <sys/fcntl.h>
#include <sys/lockf.h>

#include <nfs/rpcv2.h>
#include <nfs/nfsproto.h>
#include <nfs/nfs.h>
#include <nfs4/list.h>
#include <nfs4/nfs4_prot.h>
#include <nfs4/nfs4_debug.h>
#include <nfs4/nfs4.h>
#include <nfs4/nfs4mount.h>
#include <nfs4/nfs4fs.h>
#include <nfs4/nfs4node.h>
#include <nfs/xdr_subs.h>
#include <nfs4/nfsm_subs.h>
#include <nfs4/nfs4_var.h>

#define nfs4fs_wait_on_recoverd(x) 0

#define RESERVE_SPACE(nbytes)    do {				\
	if (!(p = reserve_space(cp, XDR_QUADLEN(nbytes))))	\
		goto nomem;					\
} while (0)

#define WRITE32(n)	*p++ = htonl(n)

/* This could cause alignment problems on a real 64 bit architecture */
#define WRITE64(n)	*((u64 *)p)++ = htobe64(n)

#define WRITEMEM(ptr, nbytes)     do {				\
	memcpy(p, (ptr), (nbytes));				\
	p += XDR_QUADLEN(nbytes);				\
} while (0)

/*
 *  Primitives for XDR encoding are:
 *    reserve_space() - returns a pointer to a contiguous block of memory to be XDR'ed into,
 *                      starting at the current value of the XDR encode pointer, if such a
 *                      block is available.
 *    adjust_args()   - sets the value of the XDR encode pointer
 *    send_buf()      - puts an opaque buffer into the XDR stream, hopefully without a copy
 *
 *  All XDR encode routines are written in terms of these primitives, without direct use
 *  to the iov_* fields, so if a different internal representation of the output buffer is
 *  needed, it is only necessary to change the primitives.
 */

static u32 *
reserve_space(struct nfs4fs_compound *cp, unsigned int nwords)
{
	int s = sizeof (u32) * nwords;
	struct mbuf *mb2;

	nfsv4_printk(xdr, "reserve_space: want %d, have %d\n", s, M_TRAILINGSPACE(cp->mb));

	if (s > M_TRAILINGSPACE(cp->mb)) {
		MGET(mb2, M_WAIT, MT_DATA);
		if (s > MLEN)
			return NULL;
		cp->mb->m_next = mb2;
		cp->mb = mb2;
		cp->mb->m_len = 0;
		cp->p = mtod(cp->mb, u32 *);
		cp->end = cp->p + M_TRAILINGSPACE(cp->mb) / sizeof (u32);
	}

	bzero(cp->p, s);
	return cp->p;
}

static void
adjust_args(struct nfs4fs_compound *cp, u32 *p)
{
	int s = (caddr_t)p - (caddr_t)cp->p;

#if NFS4_DEBUG
	/* make sure XDR encode pointer is within allowed range */
	if (s < 0 || ((caddr_t)p > (caddr_t)cp->end)) {
		printk("!!! nfs4_adjust_args: out-of-range pointer !!! %x %x %x\n",
		       (caddr_t)cp->p, (caddr_t)p, (caddr_t)cp->end);
		return;
	}
	/* make sure XDR encode pointer is kept "aligned", i.e. always advanced by a number of
	   bytes divisible by 4 */
	if (s & 3) {
		printk("!!! nfs4fs: unaligned pointer in adjust_args !!!\n");
		return;
	}
#endif
	cp->mb->m_len += s;
	cp->p = p;
}

/*
 *  Setup routines
 */

/* type, changeid, size, fsid, fileid */
static u32 positive_bmval[] = { 0x0010011a };
bitmap4 nfs4_positive_bitmap = { 1, positive_bmval };

/* type, changeid, size, fsid, fileid, mode, nlink, owner, group,
   rawdev, ctime, atime, mtime */
static u32 negative_bmval[] = { 0x0010011a, 0x0020823a };
bitmap4 nfs4_negative_bitmap = { 2, negative_bmval };

int
nfs4_setup_compound(struct nfs4fs_compound *cp, char *tag)
{
	u32 *p;
	caddr_t bpos;
	int taglen = strlen(tag), nwords;

	/*
	 * Buffer setup.
	 */
	nwords = 3 + XDR_QUADLEN(taglen);
	cp->mreq = cp->mb = nfsm_reqh(NULL, NFSPROC4_COMPOUND, sizeof (u32) * nwords, &bpos);
	cp->mrep = cp->md = NULL;
	cp->p = (u32 *) bpos;
	cp->end = cp->p + M_TRAILINGSPACE(cp->mb) / sizeof (u32);
	nfsv4_printk(xdr, "setup_compound: %s: got %d word mbuf at %x\n", tag, cp->end - cp->p, cp->p);
	if (!(p = reserve_space(cp, nwords))) {
		nfsv4_printk(xdr2, "setup_compound: out of memory!\n");
		return -ENOMEM;
	}

	/*
	 * General compound setup.
	 */
	cp->flags = 0;
	cp->renew_index = 0;
	cp->req_nops = 0;
	cp->current_fh = NULL;
	cp->save_fh = NULL;
	cp->seqid_holder = NULL;
	cp->stateid_holder = NULL;
	cp->seqidp = NULL;
	cp->nstateids = 0;

	/*
	 * Now the initial XDR encoding, tag first.
	 */
	*p++ = htonl(taglen);
	cp->taglen = taglen;
	cp->tag = tag;
	WRITEMEM(tag, taglen);

	/*
	 * Minor version and argarray length next...
	 */
	*p++ = htonl(NFS4_MINOR_VERSION);
	cp->nopsp = p;
	*p++ = 0;   /* set argarray_len to 0, just to be paranoid */
	adjust_args(cp, p);
	return 0;
}

static int
encode_attrs(struct nfs4fs_compound *cp, struct vattr *vap, fattr4_acl *acl)
{
	u32 *p, *q, *bmval, *attrlenp;
	int len, bmlen = 2;
	struct gss_cacheent *owner = NULL;
	struct gss_cacheent *group = NULL;
	int status;
#ifdef NFS4ACL
	nfsace4 *ace;
	struct list_head *list;
#endif

	/*
	 * We reserve enough space to write the entire attribute buffer at once.
	 * In the worst-case, this would be
	 *   12(bitmap) + 4(attrlen) + 8(size) + 4(mode) + 4(atime) + 4(mtime)
	 *          = 36 bytes, plus any contribution from variable-length fields
	 *            such as owner/group/acl's.
	 */
	len = 36;
	if (vap->va_uid != VNOVAL) {
		if ((status = gss_get_name(GSS_OWNER, vap->va_uid, &owner))) {
			printk("nfs4: couldn't resolve uid %d to string\n", vap->va_uid);
			goto out;
		}
		len += (XDR_LEN(owner->name.len) + 4);
	}
	if (vap->va_gid != VNOVAL) {
		if ((status = gss_get_name(GSS_GROUP, vap->va_gid, &group))) {
			printk("nfs4: couldn't resolve uid %d to string\n", vap->va_gid);
			goto out;
		}
		len += (XDR_LEN(group->name.len) + 4);
	}
#ifdef NFS4ACL
	if (acl != NULL) {
		len += acl->facl_len * 16 + 4;
		list_for_each(list, &acl->facl_val) {
		        ace = list_entry(list, nfsace4, l_acl);
			len += XDR_LEN(ace->wholen);
		}
	}
#endif
	RESERVE_SPACE(len);

	/*
	 * We write the bitmap length now, but leave the
	 * attribute buffer length to be backfilled at the end of
	 * this routine.
	 */
	WRITE32(bmlen);
	bmval = p;
	FATTR4_ZERO(bmval);
	FATTR4_SKIP(p);
	attrlenp = p++;
	q = p;

	if (vap->va_size != VNOVAL) {
		FATTR4_SET(FATTR4_SIZE, bmval);
		WRITE64(vap->va_size);
	}
#ifdef NFS4ACL
	/* acl */
	/* XXX determine when ACLs change. */
	if (acl != NULL) {
		FATTR4_SET(FATTR4_ACL, bmval);
		WRITE32(acl->facl_len);
		list_for_each(list, &acl->facl_val) {
		        ace = list_entry(list, nfsace4, l_acl);
			WRITE32(ace->type);
			WRITE32(ace->flag);
			WRITE32(ace->access_mask);
			WRITE32(ace->wholen);
			WRITEMEM(ace->who, ace->wholen);
		}
	}
#endif
	if (vap->va_mode != (u_short) VNOVAL) {
		FATTR4_SET(FATTR4_MODE, bmval);
		WRITE32(vap->va_mode);
	}
	if (vap->va_uid != VNOVAL) {
		FATTR4_SET(FATTR4_OWNER, bmval);
		WRITE32(owner->name.len);
		WRITEMEM(owner->name.name, owner->name.len);
	}
	if (vap->va_gid != VNOVAL) {
		FATTR4_SET(FATTR4_OWNER_GROUP, bmval);
		WRITE32(group->name.len);
		WRITEMEM(group->name.name, group->name.len);
	}
	if (vap->va_atime.tv_sec != VNOVAL) {
		FATTR4_SET(FATTR4_TIME_ACCESS_SET, bmval);
		WRITE32(SET_TO_CLIENT_TIME4);
		WRITE64(vap->va_atime.tv_sec);
		WRITE32(vap->va_atime.tv_nsec);
	}
	if (vap->va_mtime.tv_sec != VNOVAL) {
		FATTR4_SET(FATTR4_TIME_MODIFY_SET, bmval);
		WRITE32(SET_TO_CLIENT_TIME4);
		WRITE64(vap->va_mtime.tv_sec);
		WRITE32(vap->va_mtime.tv_nsec);
	}
	adjust_args(cp, p);

	/*
	 * Now we backfill the attribute buffer length.
	 */
	FATTR4_XDR_BITMAP(bmval, bmlen);
	*attrlenp = htonl((p - q) * 4);

	status = 0;
out:
	if (owner)
		gss_put(owner);
	if (group)
		gss_put(group);
	return status;
nomem:
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_simple_op(struct nfs4fs_compound *cp, nfs_opnum4 op)
{
	u32 *p;
	int status;

	if (!(p = reserve_space(cp, 1)))
		goto nomem;
	*p++ = htonl(op);
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_simpleop: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_access(struct nfs4fs_compound *cp, struct nfs4fs_access *access)
{
	u32 *p;
	int status;

	if (!(p = reserve_space(cp, 2)))
		goto nomem;
	*p++ = htonl(OP_ACCESS);
	*p++ = htonl(access->mode);
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_access: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_close(struct nfs4fs_compound *cp, struct nfs4fs_close *close)
{
	u32 *p;
	struct nfs4fs_file_data *fdata = close->data;
	int access;
	int status;

#if NFS4_DEBUG
	if (!fdata || !fdata->fi_nopen) {
		printk("!!! nfs4_setup_close: no opened file !!!\n");
		return -EIO;
	}
#endif

	if ((status = nfs4fs_wait_on_recoverd(cp->sb)))
		goto out;
	if (!(p = reserve_space(cp, 10)))
		goto nomem;

	if (fdata->fi_nopen == 1) {
		nfsv4_printk(level2, "    setting up op #%d: CLOSE\n", cp->req_nops);
		*p++ = close->op = htonl(OP_CLOSE);
		cp->seqidp = p;
		p++;
		cp->stateidp[cp->nstateids++] = p;
		p += XDR_QUADLEN(sizeof(stateid4));
	}
	else {
		access = 0;
		if (fdata->fi_nread > close->read)
			access |= OPEN4_SHARE_ACCESS_READ;
		if (fdata->fi_nwrite > close->write)
			access |= OPEN4_SHARE_ACCESS_WRITE;
		nfsv4_printk(level2, "    setting up op #%d: OPEN_DOWNGRADE\n", cp->req_nops);
		*p++ = close->op = htonl(OP_OPEN_DOWNGRADE);
		cp->stateidp[cp->nstateids++] = p;
		p += XDR_QUADLEN(sizeof(stateid4));
		cp->seqidp = p;
		p++;
		*p++ = htonl(access);
		*p++ = htonl(OPEN4_SHARE_DENY_NONE);
	}
	adjust_args(cp, p);
	cp->req_nops++;
	cp->flags = (CA_STATEFUL | CA_SEQID_RELEASE);
	cp->seqid_holder = fdata->fi_lockowner;
	cp->stateid_holder = fdata;
	cp->seqid_index = cp->req_nops;
	cp->renew_index = cp->req_nops;

	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_close: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_commit(struct nfs4fs_compound *cp, struct nfs4fs_commit *commit)
{
	u32 *p;
	int status;

	if (!(p = reserve_space(cp, 4)))
		goto nomem;
	nfsv4_printk(level2, "    setting up op #%d: COMMIT\n", cp->req_nops);
	*p++ = htonl(OP_COMMIT);
	WRITE64(commit->start);
	WRITE32(commit->length);
	cp->req_nops++;
	adjust_args(cp, p);
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_commit: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_create(struct nfs4fs_compound *cp, struct nfs4fs_create *create)
{
	u32 *p;
	int status;
	int len, quadlen;

	nfsv4_printk(level2, "    setting up op #%d: CREATE\n", cp->req_nops);

	if (!(p = reserve_space(cp, 2)))
		goto nomem;
	*p++ = htonl(OP_CREATE);
	*p++ = htonl(create->type);
	adjust_args(cp, p);

	if (create->type == NF4LNK) {
		len = strlen(create->linktext);
		quadlen = XDR_QUADLEN(len);
		if (!(p = reserve_space(cp, quadlen + 1)))
			goto nomem;
		*p++ = htonl(len);
		memcpy(p, create->linktext, len);
		p += quadlen;
		adjust_args(cp, p);
	} else if (create->type == NF4CHR || create->type == NF4BLK) {
		if (!(p = reserve_space(cp, 2)))
			goto nomem;
		*p++ = htonl(major(create->vap->va_rdev));
		*p++ = htonl(minor(create->vap->va_rdev));
		adjust_args(cp, p);
	}

	/* Name */
	len = create->dentry->d_name.len;
	quadlen = XDR_QUADLEN(len);
	if (!(p = reserve_space(cp, quadlen + 4)))
		goto nomem;
	*p++ = htonl(len);
	memcpy(p, create->dentry->d_name.name, len);
	p += quadlen;
	adjust_args(cp, p);

	/* Initial attributes */
	if ((status = encode_attrs(cp, create->vap, NULL)))
		goto out;

	cp->current_fh = create->dentry;
	cp->req_nops++;

	create->getattr.flags = FL_NONFATAL;
	create->getattr.bm = &nfs4_negative_bitmap;
	create->getfh.nonfatal = 1;
	if ((status = nfs4_setup_getattr(cp, &create->getattr)))
		goto out;
	if ((status = nfs4_setup_getfh(cp, &create->getfh)))
		goto out;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_create: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_delegreturn(struct nfs4fs_compound *cp, stateid4 stateid)
{
	u32 *p;
	int status;

	nfsv4_printk(level2, "    setting up op #%d: DELEGRETURN\n", cp->req_nops);
	p = reserve_space(cp, 1 + XDR_QUADLEN(sizeof(stateid4)));
	if (!p)
		goto nomem;
	*p++ = htonl(OP_DELEGRETURN);
	WRITEMEM(stateid, sizeof(stateid4));
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_delegreturn: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_getattr(struct nfs4fs_compound *cp, struct nfs4fs_getattr *getattr)
{
	bitmap4 *bm = getattr->bm;
	u32 *p;
	int i;
	int status;

#if NFS4_DEBUG
	if (!cp->current_fh && !(getattr->flags & FL_NOREVAL)) {
		printk("!!! setup_getattr: no fh !!!\n");
/**		return -EIO;*/
	}
#endif

	if (!(p = reserve_space(cp, 2 + bm->bitmap4_len)))
		goto nomem;
	*p++ = htonl(OP_GETATTR);
	*p++ = htonl(bm->bitmap4_len);
	for (i = 0; i < bm->bitmap4_len; i++)
		*p++ = htonl(bm->bitmap4_val[i]);
	adjust_args(cp, p);
	cp->req_nops++;
	getattr->dentry = cp->current_fh;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_getattr: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_getfh(struct nfs4fs_compound *cp, struct nfs4fs_getfh *getfh)
{
	u32 *p;
	int status;

#if NFS4_DEBUG
	if (!cp->current_fh) {
		printk("!!! setup_getfh: no fh !!!\n");
		return -EIO;
	}
#endif
	nfsv4_printk(level2, "    setting up op #%d: GETFH\n", cp->req_nops);

	getfh->dentry = cp->current_fh;

	if (!(p = reserve_space(cp, 1)))
		goto nomem;
	*p++ = htonl(OP_GETFH);
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_getfh: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_link(struct nfs4fs_compound *cp, struct nfs4fs_link *link)
{
	u32 *p;
	int quadlen = XDR_QUADLEN(link->namelen);
	int status;

#if NFS4_DEBUG
	if (!cp->current_fh || !cp->save_fh) {
		printk("!!! setup_link: missing fh !!!\n");
		return -EIO;
	}
#endif

	if (!(p = reserve_space(cp, quadlen + 2)))
		goto nomem;
	*p++ = htonl(OP_LINK);
	*p++ = htonl(link->namelen);
	memcpy(p, link->name, link->namelen);
	p += quadlen;
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_link: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_lookup(struct nfs4fs_compound *cp, struct nfs4fs_lookup *lookup)
{
	struct dentry *dentry = lookup->dentry;
	u32 *p;
	int len = lookup->namelen;
	int quadlen = XDR_QUADLEN(len);
	int status;

	if (!(p = reserve_space(cp, quadlen + 2)))
		goto nomem;
	cp->current_fh = dentry;

	*p++ = htonl(OP_LOOKUP);
	*p++ = htonl(len);
	memcpy(p, lookup->name, len);
	p += quadlen;
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_lookup: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_lookupp(struct nfs4fs_compound *cp, struct dentry *dentry)
{
	u32 *p;
	int status;

	if (!(p = reserve_space(cp, 1)))
		goto nomem;
	cp->current_fh = dentry;

	*p++ = htonl(OP_LOOKUPP);
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_lookupp: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_open(struct nfs4fs_compound *cp, struct nfs4fs_open *open)
{
	u32 *p;
	struct nfs4fs_lockowner *lockowner = open->fp->fi_lockowner;
	struct dentry *dentry = open->dentry;
	struct vattr *vap = open->vap;
	int len, quadlen;
	int status;

	if (!(open->flags & O_CREAT) && !lockowner)
		panic("nfs4_setup_open: not lock owner");

	/* XXX: Does the OPEN operation renew state? */

	if ((status = nfs4fs_wait_on_recoverd(cp->sb)))
		goto out;

	if (!(p = reserve_space(cp, 10)))
		goto nomem;
	*p++ = htonl(OP_OPEN);
	cp->seqidp = p++;
	*p++ = htonl(open->flags & O_ACCMODE);
	*p++ = htonl(OPEN4_SHARE_DENY_NONE);
	WRITE64(cp->nmp->nm_clientid);
	WRITE32(4);
	WRITE32(lockowner->lo_id);

	if (open->flags & O_CREAT) {
		*p++ = htonl(OPEN4_CREATE);
		*p++ = htonl((open->flags & O_EXCL) ? GUARDED4 : UNCHECKED4);
		if (vap) {
			adjust_args(cp, p);
			if (open->flags & O_TRUNC)
				vap->va_size = 0;
			status = encode_attrs(cp, vap, NULL);
		}
	} else {
		*p++ = htonl(OPEN4_NOCREATE);
		adjust_args(cp, p);
	}

	len = dentry->d_name.len;
	quadlen = XDR_QUADLEN(len);
	if (!(p = reserve_space(cp, quadlen + 2)))
		goto nomem;
	*p++ = htonl(open->claim_type);
	*p++ = htonl(len);
	memcpy(p, dentry->d_name.name, len);
	p += quadlen;
	adjust_args(cp, p);

	cp->flags = CA_STATEFUL;
	cp->current_fh = dentry;
	cp->seqid_holder = lockowner;
	cp->stateid_holder = open->fp;
	cp->req_nops++;
	cp->seqid_index = cp->req_nops;
	cp->renew_index = cp->req_nops;

	open->getattr.flags = FL_CLOBBER_OK;
	open->getattr.bm = &nfs4_negative_bitmap;
	open->getfh.nonfatal = 1;
	if ((status = nfs4_setup_getattr(cp, &open->getattr)))
		goto out;
	status = nfs4_setup_getfh(cp, &open->getfh);

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_open: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_open_confirm(struct nfs4fs_compound *cp, struct nfs4fs_open *open)
{
	u32 *p;
	int status;

	/* The passed in lockowner and file_data already have their semaphores claimed */

	if ((status = nfs4fs_wait_on_recoverd(cp->sb)))
		goto out;
	nfsv4_printk(level2, "    setting up op #%d: OPEN_CONFIRM\n", cp->req_nops);

	if (!(p = reserve_space(cp, 6)))
		goto nomem;
	*p++ = htonl(OP_OPEN_CONFIRM);
	cp->stateidp[cp->nstateids++] = p;
	p += XDR_QUADLEN(sizeof(stateid4));
	cp->seqidp = p;
	p++;
	adjust_args(cp, p);

	cp->req_nops++;
	cp->flags = (CA_STATEFUL | CA_SEQID_LOCKED | CA_SEQID_RELEASE | CA_STATEID_LOCKED);
	cp->stateid_holder = open->fp;
	cp->seqid_holder = open->fp->fi_lockowner;
	cp->seqid_index = cp->req_nops;
	cp->renew_index = cp->req_nops;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_open_confirm: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_putfh(struct nfs4fs_compound *cp, struct nfs4fs_putfh *putfh)
{
	struct dentry *dentry = putfh->dentry;
	int quadlen;
	u32 *p;

	nfsv4_printk(level2, "    setting up op #%d: PUTFH\n", cp->req_nops);
	quadlen = XDR_QUADLEN(putfh->dentry->fh_len);
	if (!(p = reserve_space(cp, quadlen + 2))) {
		nfsv4_printk(xdr2, "setup_putfh: can't reserve %d\n", quadlen + 2);
		return -ENOMEM;
	}
	*p++ = htonl(OP_PUTFH);
	*p++ = htonl(putfh->dentry->fh_len);
	memcpy(p, &putfh->dentry->fh_val, putfh->dentry->fh_len);
	p += quadlen;
	adjust_args(cp, p);
	cp->req_nops++;
	putfh->op = OP_PUTFH;
	putfh->nlookups = 0;
	cp->current_fh = dentry;
	return 0;
}

int
nfs4_setup_read(struct nfs4fs_compound *cp, struct nfs4fs_read *read)
{
	u32 *p;
	int status;

	if (!(p = reserve_space(cp, 4 + XDR_QUADLEN(sizeof(stateid4)))))
		goto nomem;
	*p++ = htonl(OP_READ);
	cp->stateidp[cp->nstateids++] = p;
	p += XDR_QUADLEN(sizeof(stateid4));
	WRITE64(read->offset);
	WRITE32(read->maxcount);
	adjust_args(cp, p);
	cp->req_nops++;
	cp->flags = (CA_STATEFUL | CA_STATEID_RELEASE);
	cp->renew_index = cp->req_nops;
	cp->stateid_holder = read->file_data;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_read: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_readdir(struct nfs4fs_compound *cp, struct nfs4fs_readdir *readdir)
{
	bitmap4 *bm = readdir->bm;
	u32 *p;
	int i;
	int status;

	nfsv4_printk(level2, "    setting up op #%d: READDIR count %d\n", cp->req_nops, (int) readdir->count);

	if (!(p = reserve_space(cp, 8 + bm->bitmap4_len)))
		goto nomem;
	WRITE32(OP_READDIR);
	WRITE64(readdir->cookie);
	WRITEMEM(readdir->verifier, sizeof(readdir->verifier));
	*p++ = htonl(readdir->count >> 4);  /* meaningless "dircount" field */
	*p++ = htonl(readdir->count);
	*p++ = htonl(bm->bitmap4_len);
	for (i = 0; i < bm->bitmap4_len; i++)
		*p++ = htonl(bm->bitmap4_val[i]);
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_readdir: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_remove(struct nfs4fs_compound *cp, char *name, int namelen)
{
	u32 *p;
	int quadlen = XDR_QUADLEN(namelen);
	int status;

	if (!(p = reserve_space(cp, quadlen + 2)))
		goto nomem;
	*p++ = htonl(OP_REMOVE);
	*p++ = htonl(namelen);
	memcpy(p, name, namelen);
	p += quadlen;
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_remove: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_rename(struct nfs4fs_compound *cp, struct nfs4fs_rename *rename)
{
	u32 *p;
	int old_len = rename->old_name->len;
	int old_quadlen = XDR_QUADLEN(old_len);
	int new_len = rename->new_name->len;
	int new_quadlen = XDR_QUADLEN(new_len);
	int status;

	if (!(p = reserve_space(cp, old_quadlen + new_quadlen + 3)))
		goto nomem;
	*p++ = htonl(OP_RENAME);
	*p++ = htonl(old_len);
	memcpy(p, rename->old_name->name, old_len);
	p += old_quadlen;
	*p++ = htonl(new_len);
	memcpy(p, rename->new_name->name, new_len);
	p += new_quadlen;
	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_rename: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_restorefh(struct nfs4fs_compound *cp)
{
	u32 *p;
	int status;

	if (!(p = reserve_space(cp, 1)))
		goto nomem;
	*p++ = htonl(OP_RESTOREFH);
	adjust_args(cp, p);
	cp->req_nops++;
	cp->current_fh = cp->save_fh;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_restorefh: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_savefh(struct nfs4fs_compound *cp)
{
	u32 *p;
	int status;

#if NFS4_DEBUG
	if (!cp->current_fh) {
		printk("!!! setup_savefh: current fh not set !!!\n");
		return -EIO;
	}
#endif

	if (!(p = reserve_space(cp, 1)))
		goto nomem;
	*p++ = htonl(OP_SAVEFH);
	adjust_args(cp, p);
	cp->req_nops++;
	cp->save_fh = cp->current_fh;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_savefh: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_setattr(struct nfs4fs_compound *cp, struct vattr *vap,
		   struct nfs4fs_file_data *fdata)
{
	u32 *p;
       	int status;

	nfsv4_printk(level2, "    setting up op #%d: SETATTR\n", cp->req_nops);

	if (!(p = reserve_space(cp, 1 + XDR_QUADLEN(sizeof(stateid4)))))
		goto nomem;
	*p++ = htonl(OP_SETATTR);
	if (vap->va_size != VNOVAL && fdata) {
		cp->stateidp[cp->nstateids++] = p;
		cp->stateid_holder = fdata;
		cp->flags = (CA_STATEFUL | CA_STATEID_RELEASE);
		cp->renew_index = cp->req_nops + 1;
	} else
		bzero(p, sizeof(stateid4));
	p += XDR_QUADLEN(sizeof(stateid4));
	adjust_args(cp, p);

	cp->req_nops++;
	status = encode_attrs(cp, vap, NULL);

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_setattr: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_setclientid(struct nfs4fs_compound *cp, struct nfs4fs_setclientid *setclientid)
{
	u32 *p;
	struct timeval tv;
	int len1, len2, len3, quadlen1, quadlen2, quadlen3, status;

	nfsv4_printk(level2, "    setting up op #%d: SETCLIENTID\n", cp->req_nops);
	
	get_fast_time(&tv);

	len1 = setclientid->namlen;
	len2 = strlen(setclientid->callback_netid);
	len3 = strlen(setclientid->callback_universal_address);
	quadlen1 = XDR_QUADLEN(len1);
	quadlen2 = XDR_QUADLEN(len2);
	quadlen3 = XDR_QUADLEN(len3);

	if (!(p = reserve_space(cp, quadlen1 + quadlen2 + quadlen3 + 8)))
		goto nomem;

	WRITE32(OP_SETCLIENTID);

	/* verifier */
	WRITE32(tv.tv_sec);
	WRITE32(tv.tv_usec);

	/* name */
	WRITE32(len1);
	memcpy(p, setclientid->name, len1);
	p += quadlen1;

	/* callback prog */
	WRITE32(setclientid->callback_prog);

	/* callback netid */
	WRITE32(len2);
	memcpy(p, setclientid->callback_netid, len2);
	p += quadlen2;

	/* callback address */
	WRITE32(len3);
	memcpy(p, setclientid->callback_universal_address, len3);
	p += quadlen3;

	/* callback ident */
	WRITE32(0xca11bacc);

	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_setclientid: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_setclientid_confirm(struct nfs4fs_compound *cp,
			       struct nfs4fs_setclientid *setclientid)
{
	u32 *p;
	int status;

	nfsv4_printk(level2, "    setting up op #%d: SETCLIENTID_CONFIRM\n", cp->req_nops);

	if (!(p = reserve_space(cp, 5)))
		goto nomem;
	WRITE32(OP_SETCLIENTID_CONFIRM);
	WRITE64(setclientid->clientid);
	if (!setclientid->old_proto)
		WRITEMEM(&setclientid->verifier, sizeof setclientid->verifier);

	adjust_args(cp, p);
	cp->req_nops++;
	status = 0;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "nfs4_setup_setclientid_confirm: out of memory!\n");
	status = -ENOMEM;
	goto out;
}

int
nfs4_setup_write(struct nfs4fs_compound *cp, struct nfs4fs_write *write)
{
	u32 *p;
	int status;

	if (!(p = reserve_space(cp, 5 + XDR_QUADLEN(sizeof(stateid4)))))
		goto nomem;
	WRITE32(OP_WRITE);
	cp->stateidp[cp->nstateids++] = p;
	p += XDR_QUADLEN(sizeof(stateid4));
	WRITE64(write->offset);
	WRITE32(write->stable);
	WRITE32(write->count);
	adjust_args(cp, p);

	if ((status = nfsm_uiotombuf(write->uiop, &cp->mb, write->count, (caddr_t *) &cp->p)))
		goto out;

	cp->req_nops++;
	cp->flags = (CA_STATEFUL | CA_STATEID_RELEASE);
	cp->stateid_holder = write->file_data;
	if (!cp->renew_index)
		cp->renew_index = cp->req_nops;

out:
	return status;
nomem:
	nfsv4_printk(xdr2, "setup_write: out of memory!\n");
	status = -ENOMEM;
	goto out;

}
