/*	$Id: nfs4_vnops.c,v 1.39 2002/09/12 19:47:24 rees Exp $	*/
/*	$OpenBSD: nfs_vnops.c,v 1.50 2002/03/14 01:27:13 millert Exp $	*/
/*	$NetBSD: nfs_vnops.c,v 1.62.4.1 1996/07/08 20:26:52 jtc Exp $	*/

/*
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Rick Macklem at The University of Guelph.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)nfs_vnops.c	8.16 (Berkeley) 5/27/95
 */

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

/*
 * vnode op calls for Sun NFS version 4
 */

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/resourcevar.h>
#include <sys/proc.h>
#include <sys/mount.h>
#include <sys/buf.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/conf.h>
#include <sys/namei.h>
#include <sys/vnode.h>
#include <sys/dirent.h>
#include <sys/fcntl.h>
#include <sys/lockf.h>

#include <uvm/uvm_extern.h>

#include <miscfs/specfs/specdev.h>
#include <miscfs/fifofs/fifo.h>

#include <nfs/rpcv2.h>
#include <nfs/nfsproto.h>
#include <nfs/nfs.h>
#include <nfs4/list.h>
#include <nfs4/nfs4_prot.h>
#include <nfs4/nfs4_debug.h>
#include <nfs4/nfs4.h>
#include <nfs4/nfs4mount.h>
#include <nfs4/nfs4fs.h>
#include <nfs4/nfs4node.h>
#include <nfs/xdr_subs.h>
#include <nfs4/nfsm_subs.h>
#include <nfs4/nfs4_var.h>

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_var.h>

#include <rpc/rpcclnt.h>

int nfs_openrpc(struct vnode *, struct vnode **, struct componentname *, int, struct vattr *);
int nfs_createrpc(struct vnode *, struct vnode **, struct componentname *,
		  nfs_ftype4, struct vattr *, char *);
int nfs_closerpc(struct vnode *, struct ucred *, struct proc *);

/*
 * Global vfs data structures for nfs
 */
int (**nfsv2_vnodeop_p)(void *);
struct vnodeopv_entry_desc nfsv2_vnodeop_entries[] = {
	{ &vop_default_desc, vn_default_error },
	{ &vop_lookup_desc, nfs_lookup },	/* lookup */
	{ &vop_create_desc, nfs_create },	/* create */
	{ &vop_mknod_desc, nfs_mknod },		/* mknod */
	{ &vop_open_desc, nfs_open },		/* open */
	{ &vop_close_desc, nfs_close },		/* close */
	{ &vop_access_desc, nfs_access },	/* access */
	{ &vop_getattr_desc, nfs_getattr },	/* getattr */
	{ &vop_setattr_desc, nfs_setattr },	/* setattr */
	{ &vop_read_desc, nfs_read },		/* read */
	{ &vop_write_desc, nfs_write },		/* write */
	{ &vop_lease_desc, nfs_lease_check },	/* lease */
	{ &vop_ioctl_desc, nfs_ioctl },		/* ioctl */
	{ &vop_select_desc, nfs_select },	/* select */
	{ &vop_revoke_desc, nfs_revoke },	/* revoke */
	{ &vop_fsync_desc, nfs_fsync },		/* fsync */
	{ &vop_remove_desc, nfs_remove },	/* remove */
	{ &vop_link_desc, nfs_link },		/* link */
	{ &vop_rename_desc, nfs_rename },	/* rename */
	{ &vop_mkdir_desc, nfs_mkdir },		/* mkdir */
	{ &vop_rmdir_desc, nfs_rmdir },		/* rmdir */
	{ &vop_symlink_desc, nfs_symlink },	/* symlink */
	{ &vop_readdir_desc, nfs_readdir },	/* readdir */
	{ &vop_readlink_desc, nfs_readlink },	/* readlink */
	{ &vop_abortop_desc, vop_generic_abortop },	/* abortop */
	{ &vop_inactive_desc, nfs_inactive },	/* inactive */
	{ &vop_reclaim_desc, nfs_reclaim },	/* reclaim */
	{ &vop_lock_desc, nfs_lock },		/* lock */
	{ &vop_unlock_desc, nfs_unlock },	/* unlock */
	{ &vop_bmap_desc, nfs_bmap },		/* bmap */
	{ &vop_strategy_desc, nfs_strategy },	/* strategy */
	{ &vop_print_desc, nfs_print },		/* print */
	{ &vop_islocked_desc, nfs_islocked },	/* islocked */
	{ &vop_pathconf_desc, nfs_pathconf },	/* pathconf */
	{ &vop_advlock_desc, nfs_advlock },	/* advlock */
	{ &vop_reallocblks_desc, nfs_reallocblks },	/* reallocblks */
	{ &vop_bwrite_desc, nfs_bwrite },
	{ (struct vnodeop_desc*)NULL, (int(*)(void *))NULL }
};
struct vnodeopv_desc nfsv2_vnodeop_opv_desc =
	{ &nfsv2_vnodeop_p, nfsv2_vnodeop_entries };

/*
 * Special device vnode ops
 */
int (**spec_nfsv2nodeop_p)(void *);
struct vnodeopv_entry_desc spec_nfsv2nodeop_entries[] = {
	{ &vop_default_desc, vn_default_error },
	{ &vop_lookup_desc, spec_lookup },	/* lookup */
	{ &vop_create_desc, spec_create },	/* create */
	{ &vop_mknod_desc, spec_mknod },	/* mknod */
	{ &vop_open_desc, spec_open },		/* open */
	{ &vop_close_desc, nfsspec_close },	/* close */
	{ &vop_access_desc, nfsspec_access },	/* access */
	{ &vop_getattr_desc, nfs_getattr },	/* getattr */
	{ &vop_setattr_desc, nfs_setattr },	/* setattr */
	{ &vop_read_desc, nfsspec_read },	/* read */
	{ &vop_write_desc, nfsspec_write },	/* write */
	{ &vop_lease_desc, spec_lease_check },	/* lease */
	{ &vop_ioctl_desc, spec_ioctl },	/* ioctl */
	{ &vop_select_desc, spec_select },	/* select */
	{ &vop_revoke_desc, spec_revoke },	/* revoke */
	{ &vop_fsync_desc, nfs_fsync },		/* fsync */
	{ &vop_remove_desc, spec_remove },	/* remove */
	{ &vop_link_desc, spec_link },		/* link */
	{ &vop_rename_desc, spec_rename },	/* rename */
	{ &vop_mkdir_desc, spec_mkdir },	/* mkdir */
	{ &vop_rmdir_desc, spec_rmdir },	/* rmdir */
	{ &vop_symlink_desc, spec_symlink },	/* symlink */
	{ &vop_readdir_desc, spec_readdir },	/* readdir */
	{ &vop_readlink_desc, spec_readlink },	/* readlink */
	{ &vop_abortop_desc, spec_abortop },	/* abortop */
	{ &vop_inactive_desc, nfs_inactive },	/* inactive */
	{ &vop_reclaim_desc, nfs_reclaim },	/* reclaim */
	{ &vop_lock_desc, nfs_lock },		/* lock */
	{ &vop_unlock_desc, nfs_unlock },	/* unlock */
	{ &vop_bmap_desc, spec_bmap },		/* bmap */
	{ &vop_strategy_desc, spec_strategy },	/* strategy */
	{ &vop_print_desc, nfs_print },		/* print */
	{ &vop_islocked_desc, nfs_islocked },	/* islocked */
	{ &vop_pathconf_desc, spec_pathconf },	/* pathconf */
	{ &vop_advlock_desc, spec_advlock },	/* advlock */
	{ &vop_reallocblks_desc, spec_reallocblks },	/* reallocblks */
	{ &vop_bwrite_desc, vop_generic_bwrite },
	{ (struct vnodeop_desc*)NULL, (int(*)(void *))NULL }
};
struct vnodeopv_desc spec_nfsv2nodeop_opv_desc =
	{ &spec_nfsv2nodeop_p, spec_nfsv2nodeop_entries };

#ifdef FIFO
int (**fifo_nfsv2nodeop_p)(void *);
struct vnodeopv_entry_desc fifo_nfsv2nodeop_entries[] = {
	{ &vop_default_desc, vn_default_error },
	{ &vop_lookup_desc, fifo_lookup },	/* lookup */
	{ &vop_create_desc, fifo_create },	/* create */
	{ &vop_mknod_desc, fifo_mknod },	/* mknod */
	{ &vop_open_desc, fifo_open },		/* open */
	{ &vop_close_desc, nfsfifo_close },	/* close */
	{ &vop_access_desc, nfsspec_access },	/* access */
	{ &vop_getattr_desc, nfs_getattr },	/* getattr */
	{ &vop_setattr_desc, nfs_setattr },	/* setattr */
	{ &vop_read_desc, nfsfifo_read },	/* read */
	{ &vop_write_desc, nfsfifo_write },	/* write */
	{ &vop_lease_desc, fifo_lease_check },	/* lease */
	{ &vop_ioctl_desc, fifo_ioctl },	/* ioctl */
	{ &vop_select_desc, fifo_select },	/* select */
	{ &vop_revoke_desc, fifo_revoke },	/* revoke */
	{ &vop_fsync_desc, nfs_fsync },		/* fsync */
	{ &vop_remove_desc, fifo_remove },	/* remove */
	{ &vop_link_desc, fifo_link },		/* link */
	{ &vop_rename_desc, fifo_rename },	/* rename */
	{ &vop_mkdir_desc, fifo_mkdir },	/* mkdir */
	{ &vop_rmdir_desc, fifo_rmdir },	/* rmdir */
	{ &vop_symlink_desc, fifo_symlink },	/* symlink */
	{ &vop_readdir_desc, fifo_readdir },	/* readdir */
	{ &vop_readlink_desc, fifo_readlink },	/* readlink */
	{ &vop_abortop_desc, fifo_abortop },	/* abortop */
	{ &vop_inactive_desc, nfs_inactive },	/* inactive */
	{ &vop_reclaim_desc, nfs_reclaim },	/* reclaim */
	{ &vop_lock_desc, nfs_lock },		/* lock */
	{ &vop_unlock_desc, nfs_unlock },	/* unlock */
	{ &vop_bmap_desc, fifo_bmap },		/* bmap */
	{ &vop_strategy_desc, fifo_badop },	/* strategy */
	{ &vop_print_desc, nfs_print },		/* print */
	{ &vop_islocked_desc, nfs_islocked },	/* islocked */
	{ &vop_pathconf_desc, fifo_pathconf },	/* pathconf */
	{ &vop_advlock_desc, fifo_advlock },	/* advlock */
	{ &vop_reallocblks_desc, fifo_reallocblks },	/* reallocblks */
	{ &vop_bwrite_desc, vop_generic_bwrite },
	{ (struct vnodeop_desc*)NULL, (int(*)(void *))NULL }
};
struct vnodeopv_desc fifo_nfsv2nodeop_opv_desc =
	{ &fifo_nfsv2nodeop_p, fifo_nfsv2nodeop_entries };
#endif /* FIFO */

/*
 * Global variables
 */
extern struct nfsstats nfsstats;
struct proc *nfs_iodwant[NFS_MAXASYNCDAEMON];
int nfs_numasync = 0;

static uint32_t __getattr_bm[] = { 0x20d00112, 0x0024be3a };
bitmap4 getattr_bm = { 2, __getattr_bm };
static uint32_t  __no_owner_bm[]      = { 0x00100112, 0x0024a20a };
bitmap4          no_owner_bm          = { 2, __no_owner_bm };

/*
 * nfs null call from vfs.
 */
int
nfs_null(vp, cred, procp)
	struct vnode *vp;
	struct ucred *cred;
	struct proc *procp;
{
	caddr_t bpos;
	int error = 0;
	struct mbuf *mreq, *mrep, *mb;

	nfsm_reqhead(vp, NFSPROC_NULL, 0);
	nfsm_request(vp, NFSPROC_NULL, procp, cred);
	nfsm_reqdone;
	return (error);
}

/*
 * nfs access vnode op.
 * For nfs version 2, just return ok. File accesses may fail later.
 * For nfs version 3, use the access rpc to check accessibility. If file modes
 * are changed on the server, accesses might still fail later.
 */
int
nfs_access(v)
	void *v;
{
	struct vop_access_args /* {
		struct vnode *a_vp;
		int  a_mode;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_access access;
	int error = 0;
	u_int32_t mode;

	/*
	 * Disallow write attempts on filesystems mounted read-only;
	 * unless the file is a socket, fifo, or a block or character
	 * device resident on the filesystem.
	 */
	if ((ap->a_mode & VWRITE) && (vp->v_mount->mnt_flag & MNT_RDONLY)) {
		switch (vp->v_type) {
		case VREG:
		case VDIR:
		case VLNK:
			return (EROFS);
		default:
			break;
		}
	}

	if (ap->a_mode & VREAD)
		mode = ACCESS4_READ;
	else
		mode = 0;
	if (vp->v_type == VDIR) {
		if (ap->a_mode & VWRITE)
			mode |= (ACCESS4_MODIFY | ACCESS4_EXTEND | ACCESS4_DELETE);
		if (ap->a_mode & VEXEC)
			mode |= ACCESS4_LOOKUP;
	} else {
		if (ap->a_mode & VWRITE)
			mode |= (ACCESS4_MODIFY | ACCESS4_EXTEND);
		if (ap->a_mode & VEXEC)
			mode |= ACCESS4_EXECUTE;
	}

	nfsstats.rpccnt[NFSPROC_ACCESS]++;

	compound.nmp = VFSTONFS(vp->v_mount);
	compound.procp = ap->a_p;
	compound.cred = ap->a_cred;
	putfh.dentry = &VTONFS(vp)->n_dentry;
	access.mode = mode;

	if ((error = nfs4_setup_compound(&compound, "nfs_access()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_access(&compound, &access)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_access(&compound, &access)))
		goto nfsmout;

	if ((access.rmode & mode) != mode)
		error = EACCES;

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		error = -error;
	if (error == EOPNOTSUPP)
		error = nfsspec_access(ap);
	return (error);
}

struct nfs4fs_lockowner master_lockowner;

/*
 * nfs open rpc
 */
int
nfs_openrpc(dvp, vpp, cnp, flags, vap)
	struct vnode *dvp;
	struct vnode **vpp;
	struct componentname *cnp;
	int flags;
	struct vattr *vap;
{
	struct vnode *vp = *vpp;
	struct nfsnode *np;
	struct nfs4fs_file_data file_data, *fp;
	struct nfs4fs_compound compound;
	struct nfs4fs_open open;
	struct nfs4fs_putfh putfh;
	struct dentry dentry;
	int error = 0;

	if (vp == NULL) {
		/* Creating a new file */
		np = NULL;
		fp = &file_data;
		bzero(fp, sizeof *fp);
	} else {
		/* Opening an existing file */
		np = VTONFS(vp);
		fp = &np->n_file_data;
	}

	/*
	 * We only open the file once.
	 * This is a bug and must be fixed.
	 */
	if (fp->fi_count++) {
		*vpp = vp;
		goto out;
	}
	flags |= FWRITE;

	/*
	 * Set up file data.
	 * This should be attached to the file table entry, like in linux,
	 * but we have no way of finding the filp, so we attach it to the vnode.
	 * Also, for now the entire machine is a single lockowner.
	 */
	fp->fi_lockowner = &master_lockowner;
	fp->fi_n = np;
	fp->fi_pid = cnp->cn_proc->p_pid;

	compound.nmp = VFSTONFS(dvp->v_mount);
	compound.procp = cnp->cn_proc;
	compound.cred = cnp->cn_cred;
	open.claim_type = CLAIM_NULL;
	open.flags = flags;
	open.vap = vap;
	open.dentry = &dentry;
	open.fp = fp;
	putfh.dentry = &(VTONFS(dvp)->n_dentry);
	dentry.d_name.name = cnp->cn_nameptr;
	dentry.d_name.len = cnp->cn_namelen;

	if ((error = nfs4_setup_compound(&compound, "nfs_openrpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_open(&compound, &open)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_open(&compound, &open)))
		goto nfsmout;

	if (vp == NULL) {
		/* New file; get a vnode and fill it in */
		error = nfs_nget(dvp->v_mount, &dentry.fh_val, dentry.fh_len, &np);
		if (error)
			goto nfsmout;

		vp = NFSTOV(np);
		np->n_dvp = dvp;
		np->n_namelen = cnp->cn_namelen;
		np->n_name = NFS4_ALLOC(np->n_namelen + 1);
		memcpy(np->n_name, cnp->cn_nameptr, cnp->cn_namelen);
		np->n_name[np->n_namelen] = '\0';
		if (np->n_file_data.fi_count)
			printf("nfs_openrpc(): already open %d\n", np->n_file_data.fi_count);
		else
			np->n_file_data = file_data;
		error = nfs4_loadattrcache(vp, &open.getattr.fattr, NULL);
		if (error)
			goto nfsmout;
		*vpp = vp;
	} else {
		if ((flags & O_TRUNC) && np->n_size != 0) {
			struct vattr va;

			VATTR_NULL(&va);
			va.va_size = 0;

			error = nfs_setattrrpc(vp, &va, cnp->cn_cred, cnp->cn_proc);
			np->n_size = np->n_vattr.va_size = 0;
		}
		np->n_attrstamp = 0; /* For Open/Close consistency */
	}

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		return (-error);
	return (error);
}

/*
 * nfs open vnode op
 */
/* ARGSUSED */
int
nfs_open(v)
	void *v;
{
	struct vop_open_args /* {
		struct vnode *a_vp;
		int  a_mode;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	struct componentname cn;
	enum vtype vtype = vp->v_type;

	if (vtype != VREG) {
		if (vtype != VDIR && vtype != VLNK) {
			printf("nfs4_open eacces vtyp=%d\n", vtype);
			return (EACCES);
		} else
			return 0;
	}

	/* See comment above nfs_create() */
	if (np->n_flag & NCREATED) {
		np->n_flag &= ~NCREATED;
		return 0;
	}

#ifdef DIAGNOSTIC
	if (ap->a_mode & O_CREAT) {
		printf("nfs_open(): attempted create\n");
		ap->a_mode &= ~O_CREAT;
	}
#endif

	cn.cn_nameptr = np->n_name;
	cn.cn_namelen = np->n_namelen;
	cn.cn_cred = ap->a_cred;
	cn.cn_proc = ap->a_p;

	return nfs_openrpc(np->n_dvp, &vp, &cn, ap->a_mode, NULL);
}

/*
 * nfs file create call
 *
 * There is no way to create a file without also opening it.
 * So we create and open the file, mark it as having already been opened, then
 * avoid re-opening it in nfs_open().
 * I'm sure there's a nasty race in there somewhere.
 */
int
nfs_create(v)
	void *v;
{
	struct vop_create_args /* {
		struct vnode *a_dvp;
		struct vnode **a_vpp;
		struct componentname *a_cnp;
		struct vattr *a_vap;
	} */ *ap = v;
	struct vnode *dvp = ap->a_dvp;
	struct nfsnode *dnp = VTONFS(dvp);
	struct componentname *cnp = ap->a_cnp;
	struct vattr *vap = ap->a_vap;
	struct vnode *newvp = NULL;
	int error = 0, fmode = (O_CREAT | FREAD | FWRITE);

	if (vap->va_vaflags & VA_EXCLUSIVE)
		fmode |= O_EXCL;

	/* We'll have to get a new vnode here. See nfs4_vfsops.c:mountnfs(). */
	if ((error = nfs_openrpc(dvp, &newvp, cnp, fmode, vap)))
		goto out;

	VTONFS(newvp)->n_flag |= NCREATED;

	if (cnp->cn_flags & MAKEENTRY)
		cache_enter(dvp, newvp, cnp);
	*ap->a_vpp = newvp;

 out:
	FREE(cnp->cn_pnbuf, M_NAMEI);
	dnp->n_flag |= NMODIFIED;
	dnp->n_attrstamp = 0;
	vrele(dvp);
	return (error);
}

/*
 * nfs close rpc
 */
int
nfs_closerpc(vp, cred, proc)
	struct vnode *vp;
	struct ucred *cred;
	struct proc *proc;
{
	struct nfs4fs_file_data *fp;
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_close close;
	struct nfsnode *np = VTONFS(vp);
	int error = 0;

	fp = &np->n_file_data;

	if (--fp->fi_count)
		goto out;

	if (fp->fi_nopen != 1)
		printf("nfs_close(): fp->fi_nopen == %d\n", fp->fi_nopen);

	compound.nmp = VFSTONFS(vp->v_mount);
	compound.procp = proc;
	compound.cred = cred;
	putfh.dentry = &np->n_dentry;
	close.data = fp;

	if ((error = nfs4_setup_compound(&compound, "nfs_closerpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_close(&compound, &close)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_close(&compound, &close)))
		goto nfsmout;

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * nfs close vnode op
 */
int
nfs_close(v)
	void *v;
{
	struct vop_close_args /* {
		struct vnodeop_desc *a_desc;
		struct vnode *a_vp;
		int  a_fflag;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	int error = 0;

	if (vp->v_type != VREG)
		return 0;

	if (np->n_flag & NMODIFIED) {
		error = nfs_flush(vp, ap->a_cred, MNT_WAIT, ap->a_p, 1);
		np->n_flag &= ~NMODIFIED;
		np->n_attrstamp = 0;
	}

	error = nfs_closerpc(vp, ap->a_cred, ap->a_p);

	if (!error && np->n_flag & NWRITEERR) {
		np->n_flag &= ~NWRITEERR;
		error = np->n_error;
	}
	return (error);
}

/*
 * nfs getattr call from vfs.
 */
int
nfs_getattr(v)
	void *v;
{
	struct vop_getattr_args /* {
		struct vnode *a_vp;
		struct vattr *a_vap;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_getattr getattr;
	int error;

	/*
	 * Update local times for special files.
	 */
	if (np->n_flag & (NACC | NUPD))
		np->n_flag |= NCHG;
	/*
	 * First look in the cache.
	 */
	if (nfs_getattrcache(vp, ap->a_vap) == 0)
		return (0);
	nfsstats.rpccnt[NFSPROC_GETATTR]++;

	compound.nmp = VFSTONFS(vp->v_mount);
	compound.procp = ap->a_p;
	compound.cred = ap->a_cred;
	putfh.dentry = &np->n_dentry;
	getattr.dentry = &np->n_dentry;
	getattr.bm = &getattr_bm;

	if ((error = nfs4_setup_compound(&compound, "nfs_getattr()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_getattr(&compound, &getattr)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_getattr(&compound, &getattr)))
		goto nfsmout;

	if ((error = nfs4_loadattrcache(vp, &getattr.fattr, ap->a_vap)))
		goto nfsmout;

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * nfs setattr call.
 */
int
nfs_setattr(v)
	void *v;
{
	struct vop_setattr_args /* {
		struct vnodeop_desc *a_desc;
		struct vnode *a_vp;
		struct vattr *a_vap;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	struct vattr *vap = ap->a_vap;
	struct ucred *cred = ap->a_cred;
	struct proc *p = ap->a_p;
	int error;
	u_quad_t tsize = 0;

	/*
	 * Setting of flags is not supported.
	 */
	if (vap->va_flags != VNOVAL)
		return (EOPNOTSUPP);

	/*
	 * Disallow write attempts if the filesystem is mounted read-only.
	 */
  	if ((vap->va_uid != (uid_t)VNOVAL ||
	    vap->va_gid != (gid_t)VNOVAL || vap->va_atime.tv_sec != VNOVAL ||
	    vap->va_mtime.tv_sec != VNOVAL || vap->va_mode != (mode_t)VNOVAL) &&
	    (vp->v_mount->mnt_flag & MNT_RDONLY))
		return (EROFS);
	if (vap->va_size != VNOVAL) {
 		switch (vp->v_type) {
 		case VDIR:
 			return (EISDIR);
 		case VCHR:
 		case VBLK:
 		case VSOCK:
 		case VFIFO:
			if (vap->va_mtime.tv_sec == VNOVAL &&
			    vap->va_atime.tv_sec == VNOVAL &&
			    vap->va_mode == (mode_t)VNOVAL &&
			    vap->va_uid == (uid_t)VNOVAL &&
			    vap->va_gid == (gid_t)VNOVAL)
				return (0);
 			vap->va_size = VNOVAL;
 			break;
 		default:
			/*
			 * Disallow write attempts if the filesystem is
			 * mounted read-only.
			 */
			if (vp->v_mount->mnt_flag & MNT_RDONLY)
				return (EROFS);
			error = nfs_vinvalbuf(vp, vap->va_size ? V_SAVE : 0, cred, p, 1);
			if (error)
				return (error);
 			tsize = np->n_size;
 			np->n_size = np->n_vattr.va_size = vap->va_size;
			uvm_vnp_setsize(vp, np->n_size);
  		};
  	} else if ((vap->va_mtime.tv_sec != VNOVAL ||
		    vap->va_atime.tv_sec != VNOVAL) &&
		   vp->v_type == VREG &&
		   (error = nfs_vinvalbuf(vp, V_SAVE, cred, p, 1)) == EINTR)
		return (error);

	if (vap->va_size != VNOVAL && np->n_file_data.fi_nopen == 0) {
		/* Have to open the file before we can truncate it */
		struct componentname cn;

		cn.cn_nameptr = np->n_name;
		cn.cn_namelen = np->n_namelen;
		cn.cn_cred = cred;
		cn.cn_proc = p;
		error = nfs_openrpc(np->n_dvp, &vp, &cn, FWRITE, NULL);
		if (error)
			goto out;
		np->n_flag |= NTRUNCATE;
	}

	error = nfs_setattrrpc(vp, vap, ap->a_cred, ap->a_p);
	if (error && vap->va_size != VNOVAL) {
		np->n_size = np->n_vattr.va_size = tsize;
		uvm_vnp_setsize(vp, np->n_size);
	}

 out:
	return (error);
}

/*
 * Do an nfs setattr rpc.
 */
int
nfs_setattrrpc(vp, vap, cred, procp)
	struct vnode *vp;
	struct vattr *vap;
	struct ucred *cred;
	struct proc *procp;
{
	struct nfsnode *np = VTONFS(vp);
	struct nfs4fs_compound compound;
	struct nfs4fs_file_data *fp = &np->n_file_data;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_getattr getattr;
	int error;

	nfsstats.rpccnt[NFSPROC_SETATTR]++;

	compound.nmp = VFSTONFS(vp->v_mount);
	compound.procp = procp;
	compound.cred = cred;
	putfh.dentry = &np->n_dentry;
	getattr.bm = &getattr_bm;

	if ((error = nfs4_setup_compound(&compound, "nfs_setattrrpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_setattr(&compound, vap, fp)))
		goto nfsmout;
	if ((error = nfs4_setup_getattr(&compound, &getattr)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_setattr(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_getattr(&compound, &getattr)))
		goto nfsmout;

	nfs4_loadattrcache(vp, &getattr.fattr, NULL);

	/* TODO: do the settatr and close in a single compound rpc */
	if (np->n_flag & NTRUNCATE) {
		error = nfs_closerpc(vp, cred, procp);
		np->n_flag &= ~NTRUNCATE;
	}

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * nfs lookup call, one step at a time...
 * First look in cache
 * If not found, unlock the directory nfsnode and do the rpc
 */
int
nfs_lookup(v)
	void *v;
{
	struct vop_lookup_args /* {
		struct vnodeop_desc *a_desc;
		struct vnode *a_dvp;
		struct vnode **a_vpp;
		struct componentname *a_cnp;
	} */ *ap = v;
	struct componentname *cnp = ap->a_cnp;
	struct vnode *dvp = ap->a_dvp;
	struct vnode **vpp = ap->a_vpp;
	struct proc *p = cnp->cn_proc;
	int flags = cnp->cn_flags;
	struct vnode *newvp;
	long len;
	nfsfh_t *fhp;
	struct nfsnode *dnp, *np;
	struct nfs4mount *nmp;
	int lockparent, wantparent, error = 0, fhsize;
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_getattr dgetattr, getattr;
	struct dentry dentry;
	struct nfs4fs_lookup lookup;
	struct nfs4fs_getfh getfh;
	int didlookup;

	*vpp = NULLVP;
	if ((flags & ISLASTCN) && (dvp->v_mount->mnt_flag & MNT_RDONLY) &&
	    (cnp->cn_nameiop == DELETE || cnp->cn_nameiop == RENAME))
		return (EROFS);
	if (dvp->v_type != VDIR)
		return (ENOTDIR);
	lockparent = flags & LOCKPARENT;
	wantparent = flags & (LOCKPARENT|WANTPARENT);
	nmp = VFSTONFS(dvp->v_mount);
	dnp = VTONFS(dvp);
	if ((error = cache_lookup(dvp, vpp, cnp)) != 0 && error != ENOENT) {
		struct vattr vattr;
		int vpid;

		newvp = *vpp;
		vpid = newvp->v_id;
		/*
		 * See the comment starting `Step through' in ufs/ufs_lookup.c
		 * for an explanation of the locking protocol
		 */
		if (dvp == newvp) {
			VREF(newvp);
			error = 0;
		} else
			error = vget(newvp, LK_EXCLUSIVE, p);

		if (!error) {
			if (vpid == newvp->v_id) {
			   if (!VOP_GETATTR(newvp, &vattr, cnp->cn_cred, cnp->cn_proc)
			    && vattr.va_ctime.tv_sec == VTONFS(newvp)->n_ctime) {
				nfsstats.lookupcache_hits++;
				if (cnp->cn_nameiop != LOOKUP &&
				    (flags & ISLASTCN))
					cnp->cn_flags |= SAVENAME;
				return (0);
			   }
			   cache_purge(newvp);
			}
			vrele(newvp);
		}
		*vpp = NULLVP;
	}
	error = 0;
	newvp = NULLVP;
	nfsstats.lookupcache_misses++;
	nfsstats.rpccnt[NFSPROC_LOOKUP]++;
	len = cnp->cn_namelen;

	compound.nmp = nmp;
	compound.procp = cnp->cn_proc;
	compound.cred = cnp->cn_cred;
	dgetattr.bm = &no_owner_bm;
	getattr.bm = &no_owner_bm;
	putfh.dentry = &dnp->n_dentry;

	if ((error = nfs4_setup_compound(&compound, "nfs_lookup()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_getattr(&compound, &dgetattr)))
		goto nfsmout;
	if (len == 1 && cnp->cn_nameptr[0] == '.') {
		/* Fake up a "." lookup; nothing to do */
		nfsv4_printk(lookup, "looking up dot\n");
		didlookup = 0;
	} else {
		if (len == 2 && !strncmp(cnp->cn_nameptr, "..", 2)) {
			nfsv4_printk(lookup, "looking up dotdot\n");
			if ((error = nfs4_setup_lookupp(&compound, &dentry)))
				goto nfsmout;
			didlookup = 1;
		} else {
			lookup.dentry = &dentry;
			lookup.name = cnp->cn_nameptr;
			lookup.namelen = len;
			if ((error = nfs4_setup_lookup(&compound, &lookup)))
				goto nfsmout;
			didlookup = 2;
		}
		if ((error = nfs4_setup_getattr(&compound, &getattr)))
			goto nfsmout;
	}
	if ((error = nfs4_setup_getfh(&compound, &getfh)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_getattr(&compound, &dgetattr)))
		goto nfsmout;
	if ((error = nfs4_loadattrcache(dvp, &dgetattr.fattr, NULL)))
		goto nfsmout;
	if (didlookup) {
		if (didlookup == 1) {
			if ((error = nfs4_handle_lookupp(&compound)))
				goto nfsmout;
		} else if (didlookup == 2) {
			if ((error = nfs4_handle_lookup(&compound)))
				goto nfsmout;
		}
		if ((error = nfs4_handle_getattr(&compound, &getattr)))
			goto nfsmout;
	}
	if ((error = nfs4_handle_getfh(&compound, &getfh)))
		goto nfsmout;
	fhp = &getfh.dentry->fh_val;
	fhsize = getfh.dentry->fh_len;
	nfsv4_printk(lookup, "nfs_lookup %s: fhsize %d\n", cnp->cn_nameptr, fhsize);

	if (NFS_CMPFH(dnp, fhp, fhsize)) {
		if (cnp->cn_nameiop == RENAME && wantparent && (flags & ISLASTCN)) {
			error = EISDIR;
			goto nfsmout;
		}
		VREF(dvp);
		newvp = dvp;
	} else {
		error = nfs_nget(dvp->v_mount, fhp, fhsize, &np);
		if (error)
			goto nfsmout;
		newvp = NFSTOV(np);

		/* Fill in parent vnode pointer and name (used by nfs_open) */
		np->n_dvp = dvp;
		np->n_namelen = cnp->cn_namelen;
		np->n_name = NFS4_ALLOC(np->n_namelen + 1);
		memcpy(np->n_name, cnp->cn_nameptr, cnp->cn_namelen);
		np->n_name[np->n_namelen] = '\0';
		nfs4_loadattrcache(newvp, &getattr.fattr, NULL);
	}
	if (cnp->cn_nameiop != LOOKUP && (flags & ISLASTCN))
		cnp->cn_flags |= SAVENAME;
	if ((cnp->cn_flags & MAKEENTRY) &&
	    (cnp->cn_nameiop != DELETE || !(flags & ISLASTCN))) {
		dnp->n_ctime = dnp->n_vattr.va_ctime.tv_sec;
		cache_enter(dvp, newvp, cnp);
	}
	*vpp = newvp;

    nfsmout:
	nfs4fs_release_compound(&compound);
    out:
	if (error < 0)
		error = -error;
	if (error) {
		if (newvp != NULLVP)
			vrele(newvp);
		if ((cnp->cn_nameiop == CREATE || cnp->cn_nameiop == RENAME) &&
		    (flags & ISLASTCN) && error == ENOENT) {
			if (dvp->v_mount->mnt_flag & MNT_RDONLY)
				error = EROFS;
			else
				error = EJUSTRETURN;
		}
		if (cnp->cn_nameiop != LOOKUP && (flags & ISLASTCN))
			cnp->cn_flags |= SAVENAME;
	}
	return (error);
}

/*
 * nfs read call.
 * Just call nfs_bioread() to do the work.
 */
int
nfs_read(v)
	void *v;
{
	struct vop_read_args /* {
		struct vnode *a_vp;
		struct uio *a_uio;
		int  a_ioflag;
		struct ucred *a_cred;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;

	if (vp->v_type != VREG)
		return (EPERM);
	return (nfs_bioread(vp, ap->a_uio, ap->a_ioflag, ap->a_cred));
}

/*
 * nfs readlink call
 */
int
nfs_readlink(v)
	void *v;
{
	struct vop_readlink_args /* {
		struct vnode *a_vp;
		struct uio *a_uio;
		struct ucred *a_cred;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;

	if (vp->v_type != VLNK)
		return (EPERM);
	return (nfs_bioread(vp, ap->a_uio, 0, ap->a_cred));
}

/*
 * Do a readlink rpc.
 * Called by nfs_doio() from below the buffer cache.
 */
int
nfs_readlinkrpc(vp, uiop, cred)
	struct vnode *vp;
	struct uio *uiop;
	struct ucred *cred;
{
	struct nfsnode *np = VTONFS(vp);
	int error;
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;

	nfsstats.rpccnt[NFSPROC_READLINK]++;

	compound.nmp = VFSTONFS(vp->v_mount);
	compound.procp = uiop->uio_procp;
	compound.cred = cred;
	putfh.dentry = &np->n_dentry;

	if ((error = nfs4_setup_compound(&compound, "nfs_readlinkrpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_readlink(&compound)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_readlink(&compound, uiop)))
		goto nfsmout;

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		return (-error);
	return (error);
}

/*
 * nfs read rpc call
 * Ditto above
 */
int
nfs_readrpc(vp, uiop, cred)
	struct vnode *vp;
	struct uio *uiop;
	struct ucred *cred;
{
	struct nfs4mount *nmp = VFSTONFS(vp->v_mount);
	int error = 0, len, tsiz;
	struct nfsnode *np = VTONFS(vp);
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_read read;

	tsiz = uiop->uio_resid;

	compound.nmp = nmp;
	compound.procp = uiop->uio_procp;
	compound.cred = cred;
	compound.mreq = compound.mrep = NULL;
	putfh.dentry = &np->n_dentry;
	read.file_data = &np->n_file_data;

	while (tsiz > 0) {
		nfsstats.rpccnt[NFSPROC_READ]++;
		len = (tsiz > nmp->nm_rsize) ? nmp->nm_rsize : tsiz;
		read.offset = uiop->uio_offset;
		read.maxcount = len;
		read.uiop = uiop;

		if ((error = nfs4_setup_compound(&compound, "nfs_readrpc()")))
			break;
		if ((error = nfs4_setup_putfh(&compound, &putfh)))
			break;
		if ((error = nfs4_setup_read(&compound, &read)))
			break;

		if ((error = nfs4fs_call_compound(&compound)))
			break;
		if ((error = nfs4_handle_putfh(&compound, &putfh)))
			break;
		if ((error = nfs4_handle_read(&compound, &read)))
			break;

		if (read.eof || read.bytes_read == 0)
			tsiz = 0;
		else
			tsiz -= read.bytes_read;

		nfs4fs_release_compound(&compound);
	}

	nfs4fs_release_compound(&compound);
	if (error < 0)
		return (-error);
	return (error);
}

/*
 * nfs write call
 */
int
nfs_writerpc(vp, uiop, cred, iomode, must_commit)
	struct vnode *vp;
	struct uio *uiop;
	struct ucred *cred;
	int *iomode, *must_commit;
{
	struct nfs4mount *nmp = VFSTONFS(vp->v_mount);
	int error = 0, len, tsiz, wccflag = NFSV3_WCCRATTR, rlen, commit;
	int committed = NFSV3WRITE_FILESYNC;
	int32_t backup;
	struct nfsnode *np = VTONFS(vp);
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_write write;
	caddr_t verf;

#ifndef DIAGNOSTIC
	if (uiop->uio_iovcnt != 1)
		panic("nfs: writerpc iovcnt > 1");
#endif
	*must_commit = 0;
	tsiz = uiop->uio_resid;

	compound.nmp = nmp;
	compound.procp = uiop->uio_procp;
	compound.cred = cred;
	compound.mreq = compound.mrep = NULL;
	putfh.dentry = &np->n_dentry;
	write.file_data = &np->n_file_data;
	write.stable = (stable_how4) *iomode;
	write.uiop = uiop;

	while (tsiz > 0) {
		nfsstats.rpccnt[NFSPROC_WRITE]++;
		len = (tsiz > nmp->nm_wsize) ? nmp->nm_wsize : tsiz;
		write.offset = uiop->uio_offset;
		write.count = len;

		if ((error = nfs4_setup_compound(&compound, "nfs_writerpc()")))
			goto out;
		if ((error = nfs4_setup_putfh(&compound, &putfh)))
			break;
		if ((error = nfs4_setup_write(&compound, &write)))
			break;

		if ((error = nfs4fs_call_compound(&compound)))
			break;
		if ((error = nfs4_handle_putfh(&compound, &putfh)))
			break;
		if ((error = nfs4_handle_write(&compound, &write)))
			break;

		wccflag = 0;
		rlen = write.bytes_written;
		if (rlen == 0) {
			error = NFSERR_IO;
			break;
		} else if (rlen < len) {
			backup = len - rlen;
			uiop->uio_iov->iov_base -= backup;
			uiop->uio_iov->iov_len += backup;
			uiop->uio_offset -= backup;
			uiop->uio_resid += backup;
			len = rlen;
		}
		commit = write.committed;

		/*
		 * Return the lowest committment level
		 * obtained by any of the RPCs.
		 */
		if (committed == NFSV3WRITE_FILESYNC)
			committed = commit;
		else if (committed == NFSV3WRITE_DATASYNC &&
			 commit == NFSV3WRITE_UNSTABLE)
			committed = commit;
		verf = (caddr_t) write.writeverf;
		if ((nmp->nm_flag & NFSMNT_HASWRITEVERF) == 0) {
			bcopy(verf, (caddr_t)nmp->nm_verf, NFSX_V3WRITEVERF);
			nmp->nm_flag |= NFSMNT_HASWRITEVERF;
		} else if (bcmp(verf, (caddr_t)nmp->nm_verf, NFSX_V3WRITEVERF)) {
			*must_commit = 1;
			bcopy(verf, (caddr_t)nmp->nm_verf, NFSX_V3WRITEVERF);
		}
#if 0
		if (wccflag)
			VTONFS(vp)->n_mtime = VTONFS(vp)->n_vattr.va_mtime.tv_sec;
#endif
		nfs4fs_release_compound(&compound);
		tsiz -= len;
	}

	nfs4fs_release_compound(&compound);
	*iomode = committed;

 out:
	if (error)
		uiop->uio_resid = tsiz;
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * nfs create rpc
 * Used by nfs_symlink, nfs_mknod, and nfs_mkdir
 */
int
nfs_createrpc(dvp, vpp, cnp, ftype, vap, linktarget)
	struct vnode *dvp;
	struct vnode **vpp;
	struct componentname *cnp;
	nfs_ftype4 ftype;
	struct vattr *vap;
	char *linktarget;
{
	struct nfsnode *dnp = VTONFS(dvp);
	struct nfsnode *np = NULL;
	struct vnode *newvp = NULL;
	struct nfs4fs_compound compound;
	struct nfs4fs_create create;
	struct nfs4fs_putfh putfh;
	struct dentry dentry;
	int len = cnp->cn_namelen;
	int error = 0;

	nfsstats.rpccnt[NFSPROC_CREATE]++;

	compound.nmp = VFSTONFS(dvp->v_mount);
	compound.procp = cnp->cn_proc;
	compound.cred = cnp->cn_cred;
	putfh.dentry = &dnp->n_dentry;
	create.type = ftype;
	create.dentry = &dentry;
	create.vap = vap;
	create.linktext = linktarget;
	dentry.d_name.name = cnp->cn_nameptr;
	dentry.d_name.len = len;

	if ((error = nfs4_setup_compound(&compound, "nfs_createrpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_create(&compound, &create)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_create(&compound, &create)))
		goto nfsmout;
	if ((error = nfs4_handle_getattr(&compound, &create.getattr)))
		goto nfsmout;
	if ((error = nfs4_handle_getfh(&compound, &create.getfh)))
		goto nfsmout;

	error = nfs_nget(dvp->v_mount, &create.getfh.dentry->fh_val, create.getfh.dentry->fh_len, &np);
	if (error)
		goto nfsmout;

	newvp = NFSTOV(np);
	if ((error = nfs4_loadattrcache(newvp, &create.getattr.fattr, NULL)))
		goto nfsmout;

	if (cnp->cn_flags & MAKEENTRY)
		cache_enter(dvp, newvp, cnp);

	dnp->n_flag |= NMODIFIED;
	dnp->n_attrstamp = 0;

    nfsmout:
	nfs4fs_release_compound(&compound);
    out:
	FREE(cnp->cn_pnbuf, M_NAMEI);
	vrele(dvp);
	if (error) {
		if (newvp)
			vrele(newvp);
	} else
		*vpp = newvp;
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * nfs mknod vop
 * just call nfs_createrpc() to do the work.
 */
/* ARGSUSED */
int
nfs_mknod(v)
	void *v;
{
	struct vop_create_args /* {
		struct vnode *a_dvp;
		struct vnode **a_vpp;
		struct componentname *a_cnp;
		struct vattr *a_vap;
	} */ *ap = v;
	struct vnode *dvp = ap->a_dvp;
	struct componentname *cnp = ap->a_cnp;
	struct vattr *vap = ap->a_vap;
	struct vnode *newvp = NULL;
	int error = 0;

	error = nfs_createrpc(dvp, &newvp, cnp, (nfs_ftype4) vap->va_type, vap, NULL);

	if (!error) {
		*ap->a_vpp = newvp;
		vrele(newvp);
	}

	return (error);
}

/*
 * nfs file remove call
 * To try and make nfs semantics closer to ufs semantics, a file that has
 * other processes using the vnode is renamed instead of removed and then
 * removed later on the last close.
 * - If v_usecount > 1
 *	  If a rename is not already in the works
 *	     call nfs_sillyrename() to set it up
 *     else
 *	  do the remove rpc
 */
int
nfs_remove(v)
	void *v;
{
	struct vop_remove_args /* {
		struct vnodeop_desc *a_desc;
		struct vnode * a_dvp;
		struct vnode * a_vp;
		struct componentname * a_cnp;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct vnode *dvp = ap->a_dvp;
	struct componentname *cnp = ap->a_cnp;
	struct nfsnode *np = VTONFS(vp);
	int error = 0;
	struct vattr vattr;

#ifndef DIAGNOSTIC
	if ((cnp->cn_flags & HASBUF) == 0)
		panic("nfs_remove: no name");
	if (vp->v_usecount < 1)
		panic("nfs_remove: bad v_usecount");
#endif
	if (vp->v_type == VDIR)
		error = EPERM;
	else if (vp->v_usecount == 1 || (np->n_sillyrename &&
	    VOP_GETATTR(vp, &vattr, cnp->cn_cred, cnp->cn_proc) == 0 &&
	    vattr.va_nlink > 1)) {
		/*
		 * Purge the name cache so that the chance of a lookup for
		 * the name succeeding while the remove is in progress is
		 * minimized. Without node locking it can still happen, such
		 * that an I/O op returns ESTALE, but since you get this if
		 * another host removes the file..
		 */
		cache_purge(vp);
		/*
		 * throw away biocache buffers, mainly to avoid
		 * unnecessary delayed writes later.
		 */
		error = nfs_vinvalbuf(vp, 0, cnp->cn_cred, cnp->cn_proc, 1);
		/* Do the rpc */
		if (error != EINTR)
			error = nfs_removerpc(dvp, cnp->cn_nameptr,
				cnp->cn_namelen, cnp->cn_cred, cnp->cn_proc);
		/*
		 * Kludge City: If the first reply to the remove rpc is lost..
		 *   the reply to the retransmitted request will be ENOENT
		 *   since the file was in fact removed
		 *   Therefore, we cheat and return success.
		 */
		if (error == ENOENT)
			error = 0;
	} else if (!np->n_sillyrename)
		error = nfs_sillyrename(dvp, vp, cnp);
	FREE(cnp->cn_pnbuf, M_NAMEI);
	np->n_attrstamp = 0;
	vrele(dvp);
	vrele(vp);
	return (error);
}

/*
 * nfs file remove rpc called from nfs_inactive
 */
int
nfs_removeit(sp, p)
	struct sillyrename *sp;
	struct proc *p;
{

	return (nfs_removerpc(sp->s_dvp, sp->s_name, sp->s_namlen, sp->s_cred, p));
}

/*
 * Nfs remove rpc, called from nfs_remove() and nfs_removeit().
 */
int
nfs_removerpc(dvp, name, namelen, cred, proc)
	struct vnode *dvp;
	char *name;
	int namelen;
	struct ucred *cred;
	struct proc *proc;
{
	struct nfsnode *np = VTONFS(dvp);
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	int error = 0;

	nfsstats.rpccnt[NFSPROC_REMOVE]++;

	compound.nmp = VFSTONFS(dvp->v_mount);
	compound.procp = proc;
	compound.cred = cred;
	putfh.dentry = &np->n_dentry;

	if ((error = nfs4_setup_compound(&compound, "nfs_removerpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_remove(&compound, name, namelen)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_remove(&compound)))
		goto nfsmout;

	np->n_flag |= NMODIFIED;
	np->n_attrstamp = 0;

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		return (-error);
	return (error);
}

/*
 * nfs file rename call
 */
int
nfs_rename(v)
	void *v;
{
	struct vop_rename_args  /* {
		struct vnode *a_fdvp;
		struct vnode *a_fvp;
		struct componentname *a_fcnp;
		struct vnode *a_tdvp;
		struct vnode *a_tvp;
		struct componentname *a_tcnp;
	} */ *ap = v;
	struct vnode *fvp = ap->a_fvp;
	struct vnode *tvp = ap->a_tvp;
	struct vnode *fdvp = ap->a_fdvp;
	struct vnode *tdvp = ap->a_tdvp;
	struct componentname *tcnp = ap->a_tcnp;
	struct componentname *fcnp = ap->a_fcnp;
	int error;

#ifndef DIAGNOSTIC
	if ((tcnp->cn_flags & HASBUF) == 0 ||
	    (fcnp->cn_flags & HASBUF) == 0)
		panic("nfs_rename: no name");
#endif
	/* Check for cross-device rename */
	if ((fvp->v_mount != tdvp->v_mount) ||
	    (tvp && (fvp->v_mount != tvp->v_mount))) {
		error = EXDEV;
		goto out;
	}

	/*
	 * If the tvp exists and is in use, sillyrename it before doing the
	 * rename of the new file over it.
	 */
	if (tvp && tvp->v_usecount > 1 && !VTONFS(tvp)->n_sillyrename &&
	    tvp->v_type != VDIR && !nfs_sillyrename(tdvp, tvp, tcnp)) {
		vrele(tvp);
		tvp = NULL;
	}

	error = nfs_renamerpc(fdvp, fcnp->cn_nameptr, fcnp->cn_namelen,
		tdvp, tcnp->cn_nameptr, tcnp->cn_namelen, tcnp->cn_cred,
		tcnp->cn_proc);

	if (fvp->v_type == VDIR) {
		if (tvp != NULL && tvp->v_type == VDIR)
			cache_purge(tdvp);
		cache_purge(fdvp);
	}
out:
	if (tdvp == tvp)
		vrele(tdvp);
	else
		vput(tdvp);
	if (tvp)
		vput(tvp);
	vrele(fdvp);
	vrele(fvp);
	/*
	 * Kludge: Map ENOENT => 0 assuming that it is a reply to a retry.
	 */
	if (error == ENOENT)
		error = 0;
	return (error);
}

/*
 * nfs file rename rpc called from nfs_remove() above
 */
int
nfs_renameit(sdvp, scnp, sp)
	struct vnode *sdvp;
	struct componentname *scnp;
	struct sillyrename *sp;
{
	return (nfs_renamerpc(sdvp, scnp->cn_nameptr, scnp->cn_namelen,
		sdvp, sp->s_name, sp->s_namlen, scnp->cn_cred, scnp->cn_proc));
}

/*
 * Do an nfs rename rpc. Called from nfs_rename() and nfs_renameit().
 */
int
nfs_renamerpc(fdvp, fnameptr, fnamelen, tdvp, tnameptr, tnamelen, cred, proc)
	struct vnode *fdvp;
	char *fnameptr;
	int fnamelen;
	struct vnode *tdvp;
	char *tnameptr;
	int tnamelen;
	struct ucred *cred;
	struct proc *proc;
{
	struct nfsnode *fnp = VTONFS(fdvp), *tnp = VTONFS(tdvp);
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh fputfh, tputfh;
	struct nfs4fs_rename rename;
	struct qstr fqstr, tqstr;
	int error = 0;

	nfsstats.rpccnt[NFSPROC_RENAME]++;

	compound.nmp = VFSTONFS(fdvp->v_mount);
	compound.procp = proc;
	compound.cred = cred;
	fputfh.dentry = &fnp->n_dentry;
	tputfh.dentry = &tnp->n_dentry;
	rename.old_name = &fqstr;
	rename.new_name = &tqstr;
	fqstr.name = fnameptr;
	fqstr.len = fnamelen;
	tqstr.name = tnameptr;
	tqstr.len = tnamelen;

	if ((error = nfs4_setup_compound(&compound, "nfs_renamerpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &fputfh)))
		goto nfsmout;
	if ((error = nfs4_setup_savefh(&compound)))
		goto nfsmout;
	if ((error = nfs4_setup_putfh(&compound, &tputfh)))
		goto nfsmout;
	if ((error = nfs4_setup_rename(&compound, &rename)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &fputfh)))
		goto nfsmout;
	if ((error = nfs4_handle_savefh(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &tputfh)))
		goto nfsmout;
	if ((error = nfs4_handle_rename(&compound)))
		goto nfsmout;

    nfsmout:
	nfs4fs_release_compound(&compound);
    out:
	if (error < 0)
		error = -error;

	fnp->n_flag |= NMODIFIED;
	tnp->n_flag |= NMODIFIED;
	fnp->n_attrstamp = 0;
	tnp->n_attrstamp = 0;

	return (error);
}

/*
 * nfs hard link create call
 */
int
nfs_link(v)
	void *v;
{
	struct vop_link_args /* {
		struct vnode *a_dvp;
		struct vnode *a_vp;
		struct componentname *a_cnp;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct vnode *dvp = ap->a_dvp;
	struct componentname *cnp = ap->a_cnp;
	struct nfsnode *np = VTONFS(dvp);
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh dputfh, putfh;
	struct nfs4fs_link link;
	int error = 0;

	if (dvp->v_mount != vp->v_mount) {
		FREE(cnp->cn_pnbuf, M_NAMEI);
		if (vp == dvp)
			vrele(dvp);
		else
			vput(dvp);
		return (EXDEV);
	}

	/*
	 * Push all writes to the server, so that the attribute cache
	 * doesn't get "out of sync" with the server.
	 * XXX There should be a better way!
	 */
	VOP_FSYNC(vp, cnp->cn_cred, MNT_WAIT, cnp->cn_proc);

	nfsstats.rpccnt[NFSPROC_LINK]++;

	compound.nmp = VFSTONFS(vp->v_mount);
	compound.procp = cnp->cn_proc;
	compound.cred = cnp->cn_cred;
	putfh.dentry = &VTONFS(vp)->n_dentry;
	dputfh.dentry = &np->n_dentry;
	link.name = cnp->cn_nameptr;
	link.namelen = cnp->cn_namelen;

	if ((error = nfs4_setup_compound(&compound, "nfs_link()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_savefh(&compound)))
		goto nfsmout;
	if ((error = nfs4_setup_putfh(&compound, &dputfh)))
		goto nfsmout;
	if ((error = nfs4_setup_link(&compound, &link)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_savefh(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &dputfh)))
		goto nfsmout;
	if ((error = nfs4_handle_link(&compound, &link)))
		goto nfsmout;

    nfsmout:
	nfs4fs_release_compound(&compound);
    out:
	if (error < 0)
		error = -error;

	FREE(cnp->cn_pnbuf, M_NAMEI);
	VTONFS(dvp)->n_flag |= NMODIFIED;
	VTONFS(vp)->n_attrstamp = 0;
	VTONFS(dvp)->n_attrstamp = 0;
	vput(dvp);

	/*
	 * Kludge: Map EEXIST => 0 assuming that it is a reply to a retry.
	 */
	if (error == EEXIST)
		error = 0;
	return (error);
}

/*
 * nfs symbolic link create call
 */
int
nfs_symlink(v)
	void *v;
{
	struct vop_symlink_args /* {
		struct vnode *a_dvp;
		struct vnode **a_vpp;
		struct componentname *a_cnp;
		struct vattr *a_vap;
		char *a_target;
	} */ *ap = v;
	struct vnode *newvp = NULL;
	int error = 0;

	error = nfs_createrpc(ap->a_dvp, &newvp, ap->a_cnp, NF4LNK, ap->a_vap, ap->a_target);

	if (!error) {
		*ap->a_vpp = newvp;
		vrele(newvp);
	}

	return (error);
}

/*
 * nfs make dir call
 */
int
nfs_mkdir(v)
	void *v;
{
	struct vop_mkdir_args /* {
		struct vnode *a_dvp;
		struct vnode **a_vpp;
		struct componentname *a_cnp;
		struct vattr *a_vap;
	} */ *ap = v;
	int error = 0;

	error = nfs_createrpc(ap->a_dvp, ap->a_vpp, ap->a_cnp, NF4DIR, ap->a_vap, NULL);

	return (error);
}

/*
 * nfs remove directory call
 */
int
nfs_rmdir(v)
	void *v;
{
	struct vop_rmdir_args /* {
		struct vnode *a_dvp;
		struct vnode *a_vp;
		struct componentname *a_cnp;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct vnode *dvp = ap->a_dvp;
	struct nfsnode *dnp = VTONFS(dvp);
	struct componentname *cnp = ap->a_cnp;
	int error = 0;

	if (dvp == vp) {
		vrele(dvp);
		vrele(dvp);
		FREE(cnp->cn_pnbuf, M_NAMEI);
		return (EINVAL);
	}

	error = nfs_removerpc(dvp, cnp->cn_nameptr, cnp->cn_namelen, cnp->cn_cred, cnp->cn_proc);
	FREE(cnp->cn_pnbuf, M_NAMEI);
	if (error)
		return (error);

	dnp->n_flag |= NMODIFIED;
	dnp->n_attrstamp = 0;
	cache_purge(dvp);
	cache_purge(vp);
	vrele(vp);
	vrele(dvp);

	return (0);
}


/*
 * The readdir logic below has a big design bug. It stores the NFS cookie in
 * the returned uio->uio_offset but does not store the verifier (it cannot).
 * Instead, the code stores the verifier in the nfsnode and applies that
 * verifies to all cookies, no matter what verifier was originally with
 * the cookie.
 *
 * From a practical standpoint, this is not a problem since almost all
 * NFS servers do not change the validity of cookies across deletes
 * and inserts.
 */

struct nfs_dirent {
	nfs_cookie4 cookie;
	struct dirent dirent;
};

#define	NFS_DIRHDSIZ    (sizeof (struct nfs_dirent) - (MAXNAMLEN + 1))
#define NFS_DIRENT_OVERHEAD  offsetof(struct nfs_dirent, dirent)

/*
 * nfs readdir call
 */
int
nfs_readdir(v)
	void *v;
{
	struct vop_readdir_args /* {
		struct vnode *a_vp;
		struct uio *a_uio;
		struct ucred *a_cred;
		int *a_eofflag;
		u_long **a_cookies;
		int *a_ncookies;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	struct uio *uio = ap->a_uio;
	int tresid, error;
	struct vattr vattr;
	u_long *cookies = NULL;
	int ncookies = 0, cnt;
	u_int64_t  newoff = uio->uio_offset;
	struct nfs4mount *nmp = VFSTONFS(vp->v_mount);
	struct uio readdir_uio;
	struct iovec readdir_iovec;
	struct proc *p = uio->uio_procp;
	int done = 0, eof = 0;
	struct ucred *cred = ap->a_cred;
	void *data;

	if (vp->v_type != VDIR)
		return (EPERM);
	/*
	 * First, check for hit on the EOF offset cache
	 */
	if (np->n_direofoffset != 0 &&
	    uio->uio_offset == np->n_direofoffset) {
		if (VOP_GETATTR(vp, &vattr, ap->a_cred, uio->uio_procp) == 0 &&
		    np->n_mtime == vattr.va_mtime.tv_sec) {
			nfsstats.direofcache_hits++;
			*ap->a_eofflag = 1;
			return (0);
		}
	}

	if (uio->uio_resid < NFS_FABLKSIZE)
		return (EINVAL);

	tresid = uio->uio_resid;

	if (uio->uio_rw != UIO_READ)
		return (EINVAL);

	if (ap->a_cookies) {
		ncookies = uio->uio_resid / 20;

		MALLOC(cookies, u_long *, sizeof(*cookies) * ncookies,
		       M_TEMP, M_WAITOK);
		*ap->a_ncookies = ncookies;
		*ap->a_cookies = cookies;
	}

	if ((nmp->nm_flag & (NFSMNT_NFSV3 | NFSMNT_GOTFSINFO)) == NFSMNT_NFSV3)
		(void)nfs_fsinfo(nmp, vp, cred, p);

	cnt = 5;

	MALLOC(data, void *, NFS_DIRBLKSIZ, M_TEMP,
	    M_WAITOK);

	do {
		struct nfs_dirent *ndp = data;

		readdir_iovec.iov_len = NFS_DIRBLKSIZ;
		readdir_iovec.iov_base = data;
		readdir_uio.uio_offset = newoff;
		readdir_uio.uio_iov = &readdir_iovec;
		readdir_uio.uio_iovcnt = 1;
		readdir_uio.uio_segflg = UIO_SYSSPACE;
		readdir_uio.uio_rw = UIO_READ;
		readdir_uio.uio_resid = NFS_DIRBLKSIZ;
		readdir_uio.uio_procp = curproc;

		error = nfs_readdirrpc(vp, &readdir_uio, cred, &eof);

		if (error == NFSERR_BAD_COOKIE)
			error = EINVAL;

		while (error == 0 &&
		    (ap->a_cookies == NULL || ncookies != 0) &&
		    ndp < (struct nfs_dirent *)readdir_iovec.iov_base) {
			struct dirent *dp = &ndp->dirent;
			int reclen = dp->d_reclen;

			dp->d_reclen -= NFS_DIRENT_OVERHEAD;

			if (uio->uio_resid < dp->d_reclen) {
				done = 1;
				break;
			}

			error = uiomove((caddr_t)dp, dp->d_reclen, uio);
			if (error)
				break;

			newoff = ndp->cookie;

			if (ap->a_cookies != NULL) {
				*cookies = newoff;
				cookies++;
				ncookies--;
			}

			ndp = (struct nfs_dirent *)((u_int8_t *)ndp + reclen);
		}
	} while (!error && !done && !eof && cnt--);

	FREE(data, M_TEMP);
	data = NULL;

	if (ap->a_cookies) {
		if (error) {
			FREE(*ap->a_cookies, M_TEMP);
			*ap->a_cookies = NULL;
			*ap->a_ncookies = 0;
		} else {
			*ap->a_ncookies -= ncookies;
		}
	}

	if (!error)
		uio->uio_offset = newoff;

	if (!error && (eof || uio->uio_resid == tresid)) {
		nfsstats.direofcache_misses++;
		*ap->a_eofflag = 1;
		return (0);
	}

	*ap->a_eofflag = 0;
	return (error);
}


static unsigned char fty_to_dty[] = {
	DT_UNKNOWN,
	DT_REG,
	DT_DIR,
	DT_BLK,
	DT_CHR,
	DT_LNK,
	DT_SOCK,
	DT_FIFO,
	DT_UNKNOWN,
	DT_UNKNOWN,
	DT_UNKNOWN,
};

/*
 * The function below stuff the cookies in after the name
 */

/*
 * Readdir rpc call.
 */
int
nfs_readdirrpc(struct vnode *vp,
	struct uio *uiop,
	struct ucred *cred,
	int *end_of_directory)
{
	int len, left;
	struct nfs_dirent *ndp = NULL;
	struct dirent *dp = NULL;
	u_int32_t *tl;
	caddr_t cp;
	int32_t t1;
	caddr_t dpos, cp2;
	struct mbuf *mrep, *md;
	nfs_cookie4 cookie;
	struct nfsnode *dnp = VTONFS(vp);
	int error = 0, status, tlen, more_dirs = 1, blksiz = 0, bigenough = 1;
	unsigned int fty;
	static u32 readdir_bmval[] = { 0x00100002 };
	static bitmap4 readdir_bm = { 1, readdir_bmval };
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_readdir readdir;
	struct nfs4fs_fattr fattr;

#ifndef DIAGNOSTIC
	if (uiop->uio_iovcnt != 1 ||
		(uiop->uio_resid & (NFS_DIRBLKSIZ - 1)))
		panic("nfs readdirrpc bad uio");
#endif

	cookie = uiop->uio_offset;

	while ((cookie == 0 || cookie == 1) && bigenough) {
		/* Fake up "." and "..". */
		if ((4 + NFS_DIRHDSIZ) > uiop->uio_resid) {
			bigenough = 0;
			break;
		}
		ndp = (struct nfs_dirent *) uiop->uio_iov->iov_base;
		ndp->cookie = ++cookie;
		dp = &ndp->dirent;
		len = 4 + NFS_DIRHDSIZ;
		if (cookie == 1)
			dp->d_fileno = dnp->n_vattr.va_fileid;
		else {
			/* n_dvp will be null in the root */
			struct vnode *ddvp = dnp->n_dvp;
			dp->d_fileno = ddvp ? VTONFS(ddvp)->n_vattr.va_fileid : 1;
		}
		dp->d_namlen = (int) cookie;
		dp->d_reclen = len;
		dp->d_type = DT_DIR;
		cp = dp->d_name;
		*cp++ = '.';
		if (cookie == 2)
			*cp++ = '.';
		*cp = '\0';
		blksiz += len;
		uiop->uio_iov->iov_base += len;
		uiop->uio_resid -= len;
		uiop->uio_iov->iov_len -= len;
		nfsv4_printk(readdir, "faked up \"%s\"\n", dp->d_name);
	}
	if (cookie == 2)
		cookie = 0;

	compound.nmp = VFSTONFS(vp->v_mount);
	compound.procp = uiop->uio_procp;
	compound.cred = cred;
	putfh.dentry = &dnp->n_dentry;
	readdir.count = VFSTONFS(vp->v_mount)->nm_readdirsize;
	readdir.bm = &readdir_bm;

	/*
	 * Loop around doing readdir rpc's of size nm_readdirsize
	 * truncated to a multiple of NFS_READDIRBLKSIZ.
	 * The stopping criteria is EOF or buffer full.
	 */
	while (more_dirs && bigenough) {
		nfsstats.rpccnt[NFSPROC_READDIR]++;
		if ((error = nfs4_setup_compound(&compound, "nfs_readdirrpc()")))
			goto out;
		if ((error = nfs4_setup_putfh(&compound, &putfh)))
			goto nfsmout;

		/* cookie */
		readdir.cookie = cookie;

		/* verifier */
		if (cookie == 0)
			bzero(&readdir.verifier, sizeof readdir.verifier);
		else
			bcopy(&dnp->n_cookieverf, &readdir.verifier, sizeof readdir.verifier);

		if ((error = nfs4_setup_readdir(&compound, &readdir)))
			goto nfsmout;

		if ((error = nfs4fs_call_compound(&compound)))
			goto nfsmout;
		if ((error = nfs4_handle_putfh(&compound, &putfh)))
			goto nfsmout;

		/* Handle readdir directly, the old-fashioned way */
		mrep = compound.mrep;
		md = compound.md;
		dpos = compound.dpos;

		nfsm_dissect(tl, u_int32_t *, 5 * NFSX_UNSIGNED);
		if (fxdr_unsigned(int, *tl++) != OP_READDIR)
			goto nfsmout;
		status = fxdr_unsigned(int, *tl++);
		if (status) {
			nfsv4_printk(readdir2, "nfs4_handle_readdir: nfserr=%d!\n", status);
			error = kernerrno(status);
			goto nfsmout;
		}
		bcopy(tl, &dnp->n_cookieverf, 8);
		tl += 2;
		more_dirs = fxdr_unsigned(int, *tl);

		/* loop thru the dir entries, doctoring them to 4bsd form */
		while (more_dirs && bigenough) {
			nfsm_dissect(tl, u_int32_t *,
				     3 * NFSX_UNSIGNED);
			cookie = fxdr_hyper(tl);
			nfsv4_printk(readdir, "-- got cookie %d\n", (int) cookie);
			if (cookie < 3)
				printf("nfs_readdirrpc: reserved cookie %d\n", (int) cookie);
			len = fxdr_unsigned(int, *(tl + 2));
			if (len <= 0 || len > NFS_MAXNAMLEN) {
				error = EBADRPC;
				goto nfsmout;
			}
			tlen = nfsm_rndup(len + 1);
			left = NFS_READDIRBLKSIZ - blksiz;
			if ((tlen + NFS_DIRHDSIZ) > left) {
				dp->d_reclen += left;
				uiop->uio_iov->iov_base += left;
				uiop->uio_iov->iov_len -= left;
				uiop->uio_resid -= left;
				blksiz = 0;
			}
			if ((tlen + NFS_DIRHDSIZ) > uiop->uio_resid)
				bigenough = 0;
			if (bigenough) {
				ndp = (struct nfs_dirent *)
				    uiop->uio_iov->iov_base;
				ndp->cookie = cookie;
				dp = &ndp->dirent;
				dp->d_namlen = len;
				dp->d_reclen = tlen + NFS_DIRHDSIZ;
				blksiz += dp->d_reclen;
				if (blksiz == NFS_READDIRBLKSIZ)
					blksiz = 0;

				/* Skip header */
				uiop->uio_iov->iov_base += NFS_DIRHDSIZ;
				uiop->uio_resid -= NFS_DIRHDSIZ;
				uiop->uio_iov->iov_len -= NFS_DIRHDSIZ;

				/* Copy name */
				nfsm_mtouio(uiop, len);
				cp = uiop->uio_iov->iov_base;
				tlen -= len;
				*cp = '\0';	/* null terminate */
				nfsv4_printk(readdir, "nfs_readdirrpc: name \"%s\" cookie %d\n", cp - len, (int) cookie);

				/* Skip padding */
				uiop->uio_iov->iov_base += tlen;
				uiop->uio_iov->iov_len -= tlen;
				uiop->uio_resid -= tlen;

				/* Copy attributes */
				compound.md = md;
				compound.dpos = dpos;
				if ((error = nfs4_read_attrs(&compound, &fattr)) != 0)
					goto nfsmout;
				md = compound.md;
				dpos = compound.dpos;

				/* fileid and type */
				if (fattr.fa_valid & FA_FILEID && fattr.fa_fileid)
					dp->d_fileno = nfs4_fileid4_to_fileid(fattr.fa_fileid);
				else
					dp->d_fileno = cookie;

				fty = (unsigned int) fattr.fa_type;
				dp->d_type = (fty < sizeof fty_to_dty) ? fty_to_dty[fty] : DT_UNKNOWN;
			} else
				nfsm_adv(nfsm_rndup(len));
			nfsm_dissect(tl, u_int32_t *, NFSX_UNSIGNED);
			more_dirs = fxdr_unsigned(int, *tl);
		}
		/*
		 * Get the eof boolean
		 */
		nfsm_dissect(tl, u_int32_t *, NFSX_UNSIGNED);
		more_dirs = (fxdr_unsigned(int, *tl) == 0);

		nfs4fs_release_compound(&compound);
	}
	/*
	 * Fill last record, iff any, out to a multiple of NFS_READDIRBLKSIZ
	 * by increasing d_reclen for the last record.
	 */
	if (blksiz > 0) {
		left = NFS_READDIRBLKSIZ - blksiz;
		dp->d_reclen += left;
		uiop->uio_iov->iov_base += left;
		uiop->uio_iov->iov_len -= left;
		uiop->uio_resid -= left;
	}

	/*
	 * We are now either at the end of the directory or have filled the
	 * block.
	 */
	if (bigenough) {
		dnp->n_direofoffset = cookie;
		if (end_of_directory) *end_of_directory = 1;
	} else {
		if (uiop->uio_resid > 0)
			printf("EEK! readdirrpc resid > 0\n");
	}

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * Silly rename. To make the NFS filesystem that is stateless look a little
 * more like the "ufs" a remove of an active vnode is translated to a rename
 * to a funny looking filename that is removed by nfs_inactive on the
 * nfsnode. There is the potential for another process on a different client
 * to create the same funny name between the nfs_lookitup() fails and the
 * nfs_rename() completes, but...
 */
int
nfs_sillyrename(dvp, vp, cnp)
	struct vnode *dvp, *vp;
	struct componentname *cnp;
{
	struct sillyrename *sp;
	struct nfsnode *np;
	int error;

	cache_purge(dvp);
	np = VTONFS(vp);
	MALLOC(sp, struct sillyrename *, sizeof (struct sillyrename),
		M_NFSREQ, M_WAITOK);
	sp->s_cred = crdup(cnp->cn_cred);
	sp->s_dvp = dvp;
	VREF(dvp);

	if (vp->v_type == VDIR) {
#ifdef DIAGNOSTIC
		printf("nfs: sillyrename dir\n");
#endif
		error = EINVAL;
		goto bad;
	}

	/* Fudge together a funny name */
	sp->s_namlen = sprintf(sp->s_name, ".nfsA%05x4.4", cnp->cn_proc->p_pid);

	/* Try lookitups until we get one that isn't there */
	while (nfs_lookitup(dvp, sp->s_name, sp->s_namlen, sp->s_cred,
		cnp->cn_proc, (struct nfsnode **)0) == 0) {
		sp->s_name[4]++;
		if (sp->s_name[4] > 'z') {
			error = EINVAL;
			goto bad;
		}
	}
	error = nfs_renameit(dvp, cnp, sp);
	if (error)
		goto bad;
	error = nfs_lookitup(dvp, sp->s_name, sp->s_namlen, sp->s_cred,
		cnp->cn_proc, &np);
	np->n_sillyrename = sp;
	return (0);
bad:
	vrele(sp->s_dvp);
	crfree(sp->s_cred);
	FREE((caddr_t)sp, M_NFSREQ);
	return (error);
}

/*
 * Look up a file name and optionally either update the file handle or
 * allocate an nfsnode, depending on the value of npp.
 * npp == NULL	--> just do the lookup
 * *npp == NULL --> allocate a new nfsnode and make sure attributes are
 *			handled too
 * *npp != NULL --> update the file handle in the vnode
 */
int
nfs_lookitup(dvp, name, len, cred, procp, npp)
	struct vnode *dvp;
	char *name;
	int len;
	struct ucred *cred;
	struct proc *procp;
	struct nfsnode **npp;
{
	struct vnode *newvp = NULL;
	struct nfsnode *np, *dnp = VTONFS(dvp);
	int error = 0, fhlen;
	nfsfh_t *nfhp;
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct dentry dentry;
	struct nfs4fs_lookup lookup;
	struct nfs4fs_getfh getfh;
	struct nfs4fs_getattr getattr;

	nfsstats.rpccnt[NFSPROC_LOOKUP]++;

	compound.nmp = VFSTONFS(dvp->v_mount);
	compound.procp = procp;
	compound.cred = cred;
	putfh.dentry = &dnp->n_dentry;
	lookup.name = name;
	lookup.namelen = len;
	lookup.dentry = &dentry;
	getattr.bm = &no_owner_bm;

	if ((error = nfs4_setup_compound(&compound, "nfs_lookitup()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_lookup(&compound, &lookup)))
		goto nfsmout;
	if ((error = nfs4_setup_getfh(&compound, &getfh)))
		goto nfsmout;
	if ((error = nfs4_setup_getattr(&compound, &getattr)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_lookup(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_getfh(&compound, &getfh)))
		goto nfsmout;
	if ((error = nfs4_handle_getattr(&compound, &getattr)))
		goto nfsmout;

	if (npp && !error) {
		nfhp = &getfh.dentry->fh_val;
		fhlen = getfh.dentry->fh_len;
		if (*npp) {
			np = *npp;
			if (np->n_fhsize > NFS_SMALLFH && fhlen <= NFS_SMALLFH) {
				free((caddr_t)np->n_fhp, M_NFSBIGFH);
				np->n_fhp = &np->n_fh;
			} else if (np->n_fhsize <= NFS_SMALLFH && fhlen>NFS_SMALLFH)
				np->n_fhp =(nfsfh_t *)malloc(fhlen,M_NFSBIGFH,M_WAITOK);
			bcopy((caddr_t)nfhp, (caddr_t)np->n_fhp, fhlen);
			np->n_fhsize = fhlen;
			newvp = NFSTOV(np);
		} else if (NFS_CMPFH(dnp, nfhp, fhlen)) {
			VREF(dvp);
			newvp = dvp;
		} else {
			error = nfs_nget(dvp->v_mount, nfhp, fhlen, &np);
			if (error)
				goto nfsmout;
			newvp = NFSTOV(np);
		}
		if (newvp != dvp) {
			np->n_dvp = dvp;
			np->n_namelen = len;
			np->n_name = NFS4_ALLOC(len + 1);
			memcpy(np->n_name, name, len);
			np->n_name[len] = '\0';
		}
		nfs4_loadattrcache(newvp, &getattr.fattr, NULL);
	}

    nfsmout:
	nfs4fs_release_compound(&compound);
    out:
	if (error < 0)
		error = -error;
	if (npp && *npp == NULL) {
		if (error) {
			if (newvp)
				vrele(newvp);
		} else
			*npp = np;
	}
	return (error);
}

/*
 * Nfs Version 3 commit rpc
 */
int
nfs_commit(vp, offset, cnt, procp)
	struct vnode *vp;
	u_quad_t offset;
	int cnt;
	struct proc *procp;
{
	struct nfs4mount *nmp = VFSTONFS(vp->v_mount);
	struct nfs4fs_compound compound;
	struct nfs4fs_putfh putfh;
	struct nfs4fs_commit commit;
	int error = 0;
	
	if ((nmp->nm_flag & NFSMNT_HASWRITEVERF) == 0)
		return (0);
	nfsstats.rpccnt[NFSPROC_COMMIT]++;

	compound.nmp = nmp;
	compound.procp = procp;
	compound.cred = procp->p_ucred;
	putfh.dentry = &(VTONFS(vp)->n_dentry);
	commit.start = offset;
	commit.length = cnt;

	if ((error = nfs4_setup_compound(&compound, "nfs_commitrpc()")))
		goto out;
	if ((error = nfs4_setup_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_setup_commit(&compound, &commit)))
		goto nfsmout;

	if ((error = nfs4fs_call_compound(&compound)))
		goto nfsmout;
	if ((error = nfs4_handle_putfh(&compound, &putfh)))
		goto nfsmout;
	if ((error = nfs4_handle_commit(&compound, &commit)))
		goto nfsmout;

	if (bcmp((caddr_t) nmp->nm_verf, commit.verifier, NFS4_VERIFIER_SIZE)) {
		bcopy(commit.verifier, (caddr_t) nmp->nm_verf, NFS4_VERIFIER_SIZE);
		error = NFSERR_STALEWRITEVERF;
	}

 nfsmout:
	nfs4fs_release_compound(&compound);
 out:
	if (error < 0)
		error = -error;
	return (error);
}

/*
 * Kludge City..
 * - make nfs_bmap() essentially a no-op that does no translation
 * - do nfs_strategy() by doing I/O with nfs_readrpc/nfs_writerpc
 *   (Maybe I could use the process's page mapping, but I was concerned that
 *    Kernel Write might not be enabled and also figured copyout() would do
 *    a lot more work than bcopy() and also it currently happens in the
 *    context of the swapper process (2).
 */
int
nfs_bmap(v)
	void *v;
{
	struct vop_bmap_args /* {
		struct vnode *a_vp;
		daddr_t  a_bn;
		struct vnode **a_vpp;
		daddr_t *a_bnp;
		int *a_runp;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;

	if (ap->a_vpp != NULL)
		*ap->a_vpp = vp;
	if (ap->a_bnp != NULL)
		*ap->a_bnp = ap->a_bn * btodb(vp->v_mount->mnt_stat.f_iosize);
	return (0);
}

/*
 * Strategy routine.
 * For async requests when nfsiod(s) are running, queue the request by
 * calling nfs_asyncio(), otherwise just all nfs_doio() to do the
 * request.
 */
int
nfs_strategy(v)
	void *v;
{
	struct vop_strategy_args *ap = v;
	struct buf *bp = ap->a_bp;
	struct proc *p;
	int error = 0;

	if ((bp->b_flags & (B_PHYS|B_ASYNC)) == (B_PHYS|B_ASYNC))
		panic("nfs physio/async");
	if (bp->b_flags & B_ASYNC)
		p = NULL;
	else
		p = curproc;	/* XXX */
	/*
	 * If the op is asynchronous and an i/o daemon is waiting
	 * queue the request, wake it up and wait for completion
	 * otherwise just do it ourselves.
	 */
	if ((bp->b_flags & B_ASYNC) == 0 || nfs_asyncio(bp))
		error = nfs_doio(bp, p);
	return (error);
}

/*
 * fsync vnode op. Just call nfs_flush() with commit == 1.
 */
/* ARGSUSED */
int
nfs_fsync(v)
	void *v;
{
	struct vop_fsync_args /* {
		struct vnodeop_desc *a_desc;
		struct vnode * a_vp;
		struct ucred * a_cred;
		int  a_waitfor;
		struct proc * a_p;
	} */ *ap = v;

	return (nfs_flush(ap->a_vp, ap->a_cred, ap->a_waitfor, ap->a_p, 1));
}

/*
 * Flush all the blocks associated with a vnode.
 * 	Walk through the buffer pool and push any dirty pages
 *	associated with the vnode.
 */
int
nfs_flush(vp, cred, waitfor, p, commit)
	struct vnode *vp;
	struct ucred *cred;
	int waitfor;
	struct proc *p;
	int commit;
{
	struct nfsnode *np = VTONFS(vp);
	struct buf *bp;
	int i;
	struct buf *nbp;
	struct nfs4mount *nmp = VFSTONFS(vp->v_mount);
	int s, error = 0, slptimeo = 0, slpflag = 0, retv, bvecpos;
	int passone = 1;
	u_quad_t off = (u_quad_t)-1, endoff = 0, toff;
#ifndef NFS_COMMITBVECSIZ
#define NFS_COMMITBVECSIZ	20
#endif
	struct buf *bvec[NFS_COMMITBVECSIZ];

	if (nmp->nm_flag & NFSMNT_INT)
		slpflag = PCATCH;
	if (!commit)
		passone = 0;
	/*
	 * A b_flags == (B_DELWRI | B_NEEDCOMMIT) block has been written to the
	 * server, but nas not been committed to stable storage on the server
	 * yet. On the first pass, the byte range is worked out and the commit
	 * rpc is done. On the second pass, nfs_writebp() is called to do the
	 * job.
	 */
again:
	bvecpos = 0;
	if (NFS_ISV3(vp) && commit) {
		s = splbio();
		for (bp = LIST_FIRST(&vp->v_dirtyblkhd); bp != NULL; bp = nbp) {
			nbp = LIST_NEXT(bp, b_vnbufs);
			if (bvecpos >= NFS_COMMITBVECSIZ)
				break;
			if ((bp->b_flags & (B_BUSY | B_DELWRI | B_NEEDCOMMIT))
				!= (B_DELWRI | B_NEEDCOMMIT))
				continue;
			bremfree(bp);
			bp->b_flags |= (B_BUSY | B_WRITEINPROG);
			/*
			 * A list of these buffers is kept so that the
			 * second loop knows which buffers have actually
			 * been committed. This is necessary, since there
			 * may be a race between the commit rpc and new
			 * uncommitted writes on the file.
			 */
			bvec[bvecpos++] = bp;
			toff = ((u_quad_t)bp->b_blkno) * DEV_BSIZE +
				bp->b_dirtyoff;
			if (toff < off)
				off = toff;
			toff += (u_quad_t)(bp->b_dirtyend - bp->b_dirtyoff);
			if (toff > endoff)
				endoff = toff;
		}
		splx(s);
	}
	if (bvecpos > 0) {
		/*
		 * Commit data on the server, as required.
		 */
		retv = nfs_commit(vp, off, (int)(endoff - off), p);
		if (retv == NFSERR_STALEWRITEVERF)
			nfs_clearcommit(vp->v_mount);
		/*
		 * Now, either mark the blocks I/O done or mark the
		 * blocks dirty, depending on whether the commit
		 * succeeded.
		 */
		for (i = 0; i < bvecpos; i++) {
			bp = bvec[i];
			bp->b_flags &= ~(B_NEEDCOMMIT | B_WRITEINPROG);
			if (retv)
			    brelse(bp);
			else {
			    s = splbio();
			    buf_undirty(bp);
			    vp->v_numoutput++;
			    bp->b_flags |= B_ASYNC;
			    bp->b_flags &= ~(B_READ|B_DONE|B_ERROR);
			    bp->b_dirtyoff = bp->b_dirtyend = 0;
			    splx(s);
			    biodone(bp);
			}
		}
	}

	/*
	 * Start/do any write(s) that are required.
	 */
loop:
	s = splbio();
	for (bp = LIST_FIRST(&vp->v_dirtyblkhd); bp != NULL; bp = nbp) {
		nbp = LIST_NEXT(bp, b_vnbufs);
		if (bp->b_flags & B_BUSY) {
			if (waitfor != MNT_WAIT || passone)
				continue;
			bp->b_flags |= B_WANTED;
			error = tsleep((caddr_t)bp, slpflag | (PRIBIO + 1),
				       "nfsfsync", slptimeo);
			splx(s);
			if (error) {
				if (nfs_sigintr(nmp, NULL, p))
					return (EINTR);
				if (slpflag == PCATCH) {
					slpflag = 0;
					slptimeo = 2 * hz;
				}
			}
			goto loop;
		}
		if ((bp->b_flags & B_DELWRI) == 0)
			panic("nfs_fsync: not dirty");
		if ((passone || !commit) && (bp->b_flags & B_NEEDCOMMIT))
			continue;
		bremfree(bp);
		if (passone || !commit)
			bp->b_flags |= (B_BUSY|B_ASYNC);
		else
			bp->b_flags |= (B_BUSY|B_ASYNC|B_WRITEINPROG|B_NEEDCOMMIT);
		splx(s);
		VOP_BWRITE(bp);
		goto loop;
	}
	splx(s);
	if (passone) {
		passone = 0;
		goto again;
	}
	if (waitfor == MNT_WAIT) {
 loop2:
	        s = splbio();
		error = vwaitforio(vp, slpflag, "nfs_fsync", slptimeo);
		splx(s);
		if (error) {
			if (nfs_sigintr(nmp, NULL, p))
				return (EINTR);
			if (slpflag == PCATCH) {
				slpflag = 0;
				slptimeo = 2 * hz;
			}
			goto loop2;
		}

		if (LIST_FIRST(&vp->v_dirtyblkhd) && commit) {
#if 0
			vprint("nfs_fsync: dirty", vp);
#endif
			goto loop;
		}
	}
	if (np->n_flag & NWRITEERR) {
		error = np->n_error;
		np->n_flag &= ~NWRITEERR;
	}
	return (error);
}

/*
 * Return POSIX pathconf information applicable to nfs.
 *
 * The NFS V2 protocol doesn't support this, so just return EINVAL
 * for V2.
 */
/* ARGSUSED */
int
nfs_pathconf(v)
	void *v;
{
#if 0
	struct vop_pathconf_args /* {
		struct vnode *a_vp;
		int a_name;
		register_t *a_retval;
	} */ *ap = v;
#endif

	return (EINVAL);
}

/*
 * NFS advisory byte-level locks.
 */
int
nfs_advlock(v)
	void *v;
{
	struct vop_advlock_args /* {
		struct vnode *a_vp;
		caddr_t  a_id;
		int  a_op;
		struct flock *a_fl;
		int  a_flags;
	} */ *ap = v;
	struct nfsnode *np = VTONFS(ap->a_vp);

	return (lf_advlock(&np->n_lockf, np->n_size, ap->a_id, ap->a_op,
	    ap->a_fl, ap->a_flags));
}

/*
 * Print out the contents of an nfsnode.
 */
int
nfs_print(v)
	void *v;
{
	struct vop_print_args /* {
		struct vnode *a_vp;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);

	printf("tag VT_NFS, fileid %ld fsid 0x%lx",
		np->n_vattr.va_fileid, np->n_vattr.va_fsid);
#ifdef FIFO
	if (vp->v_type == VFIFO)
		fifo_printinfo(vp);
#endif
	printf("\n");
	return (0);
}

/*
 * Just call nfs_writebp() with the force argument set to 1.
 */
int
nfs_bwrite(v)
	void *v;
{
	struct vop_bwrite_args /* {
		struct buf *a_bp;
	} */ *ap = v;

	return (nfs_writebp(ap->a_bp, 1));
}

/*
 * This is a clone of vop_generic_bwrite(), except that B_WRITEINPROG isn't set unless
 * the force flag is one and it also handles the B_NEEDCOMMIT flag.
 */
int
nfs_writebp(bp, force)
	struct buf *bp;
	int force;
{
	int oldflags = bp->b_flags, retv = 1;
	struct proc *p = curproc;	/* XXX */
	off_t off;
	int   s;

	if(!(bp->b_flags & B_BUSY))
		panic("bwrite: buffer is not busy???");

#ifdef fvdl_debug
	printf("nfs_writebp(%x): vp %x voff %d vend %d doff %d dend %d\n",
	    bp, bp->b_vp, bp->b_validoff, bp->b_validend, bp->b_dirtyoff,
	    bp->b_dirtyend);
#endif
	bp->b_flags &= ~(B_READ|B_DONE|B_ERROR);

	s = splbio();
	buf_undirty(bp);

	if ((oldflags & B_ASYNC) && !(oldflags & B_DELWRI) && p)
		++p->p_stats->p_ru.ru_oublock;

	bp->b_vp->v_numoutput++;
	splx(s);

	/*
	 * If B_NEEDCOMMIT is set, a commit rpc may do the trick. If not
	 * an actual write will have to be scheduled via. VOP_STRATEGY().
	 * If B_WRITEINPROG is already set, then push it with a write anyhow.
	 */
	if ((oldflags & (B_NEEDCOMMIT | B_WRITEINPROG)) == B_NEEDCOMMIT) {
		off = ((u_quad_t)bp->b_blkno) * DEV_BSIZE + bp->b_dirtyoff;
		bp->b_flags |= B_WRITEINPROG;
		retv = nfs_commit(bp->b_vp, off, bp->b_dirtyend-bp->b_dirtyoff,
			bp->b_proc);
		bp->b_flags &= ~B_WRITEINPROG;
		if (!retv) {
			bp->b_dirtyoff = bp->b_dirtyend = 0;
			bp->b_flags &= ~B_NEEDCOMMIT;
			biodone(bp);
		} else if (retv == NFSERR_STALEWRITEVERF)
			nfs_clearcommit(bp->b_vp->v_mount);
	}
	if (retv) {
		if (force)
			bp->b_flags |= B_WRITEINPROG;
		VOP_STRATEGY(bp);
	}

	if( (oldflags & B_ASYNC) == 0) {
		int rtval = biowait(bp);
		if (!(oldflags & B_DELWRI) && p) {
			++p->p_stats->p_ru.ru_oublock;
		}
		brelse(bp);
		return (rtval);
	} 

	return (0);
}

/*
 * nfs special file access vnode op.
 * Essentially just get vattr and then imitate iaccess() since the device is
 * local to the client.
 */
int
nfsspec_access(v)
	void *v;
{
	struct vop_access_args /* {
		struct vnode *a_vp;
		int  a_mode;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vattr va;
	struct vnode *vp = ap->a_vp;
	int error;

        /*
	 * Disallow write attempts on filesystems mounted read-only;
	 * unless the file is a socket, fifo, or a block or character
	 * device resident on the filesystem.
	 */
	if ((ap->a_mode & VWRITE) && (vp->v_mount->mnt_flag & MNT_RDONLY)) {
		switch (vp->v_type) {
		case VREG:
		case VDIR:
		case VLNK:
			return (EROFS);
		default:
			break;
		}
	}

	error = VOP_GETATTR(vp, &va, ap->a_cred, ap->a_p);
	if (error)
		return (error);

	return (vaccess(va.va_mode, va.va_uid, va.va_gid, ap->a_mode,
	    ap->a_cred));
}

/*
 * Read wrapper for special devices.
 */
int
nfsspec_read(v)
	void *v;
{
	struct vop_read_args /* {
		struct vnode *a_vp;
		struct uio *a_uio;
		int  a_ioflag;
		struct ucred *a_cred;
	} */ *ap = v;
	struct nfsnode *np = VTONFS(ap->a_vp);

	/*
	 * Set access flag.
	 */
	np->n_flag |= NACC;
	np->n_atim.tv_sec = time.tv_sec;
	np->n_atim.tv_nsec = time.tv_usec * 1000;
	return (VOCALL(spec_vnodeop_p, VOFFSET(vop_read), ap));
}

/*
 * Write wrapper for special devices.
 */
int
nfsspec_write(v)
	void *v;
{
	struct vop_write_args /* {
		struct vnode *a_vp;
		struct uio *a_uio;
		int  a_ioflag;
		struct ucred *a_cred;
	} */ *ap = v;
	struct nfsnode *np = VTONFS(ap->a_vp);

	/*
	 * Set update flag.
	 */
	np->n_flag |= NUPD;
	np->n_mtim.tv_sec = time.tv_sec;
	np->n_mtim.tv_nsec = time.tv_usec * 1000;
	return (VOCALL(spec_vnodeop_p, VOFFSET(vop_write), ap));
}

/*
 * Close wrapper for special devices.
 *
 * Update the times on the nfsnode then do device close.
 */
int
nfsspec_close(v)
	void *v;
{
	struct vop_close_args /* {
		struct vnode *a_vp;
		int  a_fflag;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	struct vattr vattr;

	if (np->n_flag & (NACC | NUPD)) {
		np->n_flag |= NCHG;
		if (vp->v_usecount == 1 &&
		    (vp->v_mount->mnt_flag & MNT_RDONLY) == 0) {
			VATTR_NULL(&vattr);
			if (np->n_flag & NACC)
				vattr.va_atime = np->n_atim;
			if (np->n_flag & NUPD)
				vattr.va_mtime = np->n_mtim;
			(void)VOP_SETATTR(vp, &vattr, ap->a_cred, ap->a_p);
		}
	}
	return (VOCALL(spec_vnodeop_p, VOFFSET(vop_close), ap));
}

#ifdef FIFO
/*
 * Read wrapper for fifos.
 */
int
nfsfifo_read(v)
	void *v;
{
	struct vop_read_args /* {
		struct vnode *a_vp;
		struct uio *a_uio;
		int  a_ioflag;
		struct ucred *a_cred;
	} */ *ap = v;
	extern int (**fifo_vnodeop_p)(void *);
	struct nfsnode *np = VTONFS(ap->a_vp);

	/*
	 * Set access flag.
	 */
	np->n_flag |= NACC;
	np->n_atim.tv_sec = time.tv_sec;
	np->n_atim.tv_nsec = time.tv_usec * 1000;
	return (VOCALL(fifo_vnodeop_p, VOFFSET(vop_read), ap));
}

/*
 * Write wrapper for fifos.
 */
int
nfsfifo_write(v)
	void *v;
{
	struct vop_write_args /* {
		struct vnode *a_vp;
		struct uio *a_uio;
		int  a_ioflag;
		struct ucred *a_cred;
	} */ *ap = v;
	extern int (**fifo_vnodeop_p)(void *);
	struct nfsnode *np = VTONFS(ap->a_vp);

	/*
	 * Set update flag.
	 */
	np->n_flag |= NUPD;
	np->n_mtim.tv_sec = time.tv_sec;
	np->n_mtim.tv_nsec = time.tv_usec * 1000;
	return (VOCALL(fifo_vnodeop_p, VOFFSET(vop_write), ap));
}

/*
 * Close wrapper for fifos.
 *
 * Update the times on the nfsnode then do fifo close.
 */
int
nfsfifo_close(v)
	void *v;
{
	struct vop_close_args /* {
		struct vnode *a_vp;
		int  a_fflag;
		struct ucred *a_cred;
		struct proc *a_p;
	} */ *ap = v;
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	struct vattr vattr;
	extern int (**fifo_vnodeop_p)(void *);

	if (np->n_flag & (NACC | NUPD)) {
		if (np->n_flag & NACC) {
			np->n_atim.tv_sec = time.tv_sec;
			np->n_atim.tv_nsec = time.tv_usec * 1000;
		}
		if (np->n_flag & NUPD) {
			np->n_mtim.tv_sec = time.tv_sec;
			np->n_mtim.tv_nsec = time.tv_usec * 1000;
		}
		np->n_flag |= NCHG;
		if (vp->v_usecount == 1 &&
		    (vp->v_mount->mnt_flag & MNT_RDONLY) == 0) {
			VATTR_NULL(&vattr);
			if (np->n_flag & NACC)
				vattr.va_atime = np->n_atim;
			if (np->n_flag & NUPD)
				vattr.va_mtime = np->n_mtim;
			(void)VOP_SETATTR(vp, &vattr, ap->a_cred, ap->a_p);
		}
	}
	return (VOCALL(fifo_vnodeop_p, VOFFSET(vop_close), ap));
}
#endif /* ! FIFO */
