/*	$Id: auth_gss.h,v 1.5 2002/08/20 21:49:19 rees Exp $	*/

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

/*
 * Declarations for RPCSEC_GSS
 *
 * Dug Song <dugsong@monkey.org>
 * Andy Adamson <andros@umich.edu>
 */

#ifndef _LINUX_SUNRPC_AUTH_GSS_H
#define _LINUX_SUNRPC_AUTH_GSS_H

#ifdef __KERNEL__
#ifdef __linux__
#include <linux/sunrpc/auth.h>
#include <linux/sunrpc/svc.h>
#endif

/*
 * GSS-API definitions.
 */

struct xdr_netobj {
	unsigned int		len;
	u8 *			data;
};

typedef struct xdr_netobj	GSS_CTX_ID_T;
typedef struct xdr_netobj	GSS_CRED_ID_T;
typedef struct xdr_netobj	GSS_OID;
typedef struct xdr_netobj	GSS_BUFFER_T;

#define GSS_C_NO_BUFFER		((GSS_BUFFER_T) 0)
#define GSS_C_NO_CONTEXT	((GSS_CTX_ID_T) 0)
#define GSS_C_NULL_OID		((GSS_OID) 0)

#define GSS_C_QOP_DEFAULT	0

#define GSS_S_COMPLETE		       0
#define GSS_S_CONTINUE_NEEDED	   1
#define GSS_S_DUPLICATE_TOKEN    2
#define GSS_S_OLD_TOKEN          4
#define GSS_S_UNSEQ_TOKEN        8
#define GSS_S_GAP_TOKEN          16

/* NOTE: the following were taken from MIT kerberos code, 
 * ../lib/gssapi/generic/gssapi.h
 */
#define GSS_S_BAD_SIG           393216
#define GSS_S_NO_CRED           458752
#define GSS_S_NO_CONTEXT        524288
#define GSS_S_DEFECTIVE_TOKEN   589824        
#define GSS_S_FAILURE           851968
#define GSS_S_CONTEXT_EXPIRED   786432

extern GSS_OID gss_mech_krb5_oid;

/*
 * RPCSEC_GSS definitions.
 */

#define RPC_GSS_VERSION		1

enum rpc_gss_proc {
	RPC_GSS_PROC_DATA = 0,
	RPC_GSS_PROC_INIT = 1,
	RPC_GSS_PROC_CONTINUE_INIT = 2,
	RPC_GSS_PROC_DESTROY = 3
};

#ifdef RPC_GSS_SVC_T_H
enum rpc_gss_svc {
	RPC_GSS_SVC_NONE = 1,
	RPC_GSS_SVC_INTEGRITY = 2,
	RPC_GSS_SVC_PRIVACY = 3
};
#endif

/* on the wire gss cred */
struct rpc_gss_cred {
	u32		gc_v;		/* version */
	u32		gc_proc;	/* control procedure */
	u32		gc_seq;		/* sequence number */
	u32		gc_svc;		/* service */
	GSS_BUFFER_T	gc_ctx;		/* context handle */
};

/* return from gss NULL PROC init sec context */
struct rpc_gss_init_res {
	GSS_BUFFER_T	gr_ctx;		/* context handle */
	u32		gr_major;	/* major status */
	u32		gr_minor;	/* minor status */
	u32		gr_win;		/* sequence window */
	GSS_BUFFER_T	gr_token;	/* token */
};

/* supported security triples */
struct sup_sec_triple {
	GSS_OID *oid;
	u32     qop;
	u32     service;
};

/*
 * GSS Context Cache
 */


#define GSS_CTX_MAX     2048 /* arbitrary limit on context size */

/* client gss context cache */
struct gss_ctx_mapping {
        rwlock_t                lock;
        struct list_head        gss_ctx_cache;
};


/* chuck suggested that this should be a dynamically allocated 
 * hash table size, dependant on the amount of memory on the machine. 
 */
#define GSS_HASH_BITS   8
#define GSS_HASH_SIZE   (1 << GSS_HASH_BITS)
#define GSS_HASH_MASK   (GSS_HASH_SIZE - 1)

/* server gss context cache */
struct gss_svc_ctx_mapping {
        rwlock_t                lock;
        struct list_head        gss_ctx_cache[GSS_HASH_SIZE];
};

struct gss_ctx_cacheent {
        GSS_OID			gcc_mech;
        u32			gcc_qop;
        void		*gcc_ctx; /* pointer to mechanism specific context */

/* fields after this point are private, for use by the gss cache */ 
        atomic_t		gcc_refcount;
        struct list_head	gss_ctx_cache;
};

#ifndef __OpenBSD__
/* Client cred structure. 
 *     gc_ctx_id is really a type gss_union_ctx_id_t 
 *     (see mit kerberos v5 ../lib/gssapi/mechglue code)
 *     that contains the mechanism OID as well as a pointer to
 *     the current client context which before gssd_import_sec_context() 
 *     points to gssd's per mech userland context store 
 *     and after gssd_import_sec_context () points to the kernel
 *     context stored in a gss_ctx_cacheent in the gss_ctx_map cache.
 *
 *     The clients reference to the SERVER's corresponding 
 *     context is stored in gc_cred.gc_ctx, and is what is 
 *     placed on the wire.
 */
struct gss_cred {
        struct rpc_cred         gc_base;
        u32                     gc_established;
        GSS_BUFFER_T            gc_wire_verf;
        GSS_BUFFER_T		        gc_service_name;
        GSS_CTX_ID_T		        gc_ctx_id;
        struct gss_ctx_mapping  gc_ctx_map;
        u32                     gc_win;
        struct rpc_gss_cred     gc_cred;
};
#define gc_uid                  gc_base.cr_uid
#define gc_count                gc_base.cr_count
#define gc_flags                gc_base.cr_flags
#define gc_expire               gc_base.cr_expire
#endif

/* gss cache function declarations*/

struct gss_ctx_cacheent * gss_ctx_alloc(void);
void gss_ctx_free(struct gss_ctx_cacheent *p);
struct gss_ctx_cacheent * gss_create_cacheent(struct gss_ctx_mapping *map);
u32 gss_get_context(u32 *maj_stat, u32* min_stat,struct gss_ctx_mapping *map, GSS_CTX_ID_T *ctx);
u32 gss_svc_get_context(u32 *maj_stat, u32* min_stat, GSS_CTX_ID_T *ctx);
u32 gss_get_ctx_mech(GSS_OID *mech,GSS_CTX_ID_T *ctx_id);
u32 gss_set_ctx(GSS_CTX_ID_T *ctx_id,struct gss_ctx_cacheent *cep);
u32 gss_validate_context(GSS_CTX_ID_T *ctx_id,struct gss_ctx_mapping *map);
u32 gss_svc_validate_context(GSS_CTX_ID_T *ctx_id);


static inline void
gss_ctx_put(struct gss_ctx_cacheent *p)
{
        if (--(p->gcc_refcount))
                gss_ctx_free(p);
}


/* Crypto interface */
int gss_cmp_triples(u32 oid_len, char *oid_data, u32 qop, u32 service);

void print_digest(unsigned char *md,int len);                                                
int gss_get_mic(u32                  *minor_status, 
                GSS_CTX_ID_T         *contextid, 
                u32                  qop, 
		GSS_BUFFER_T         *message_buffer, 
		GSS_BUFFER_T         *message_token);                                      
int
gss_verify_mic(u32                    *minor_status, 
               GSS_CTX_ID_T           *contextid, 
               GSS_BUFFER_T           *signbuf,
               GSS_BUFFER_T           *checksum, 
               u32                    *qstate);                                             
/*
 * GSSD gss_sec interface.
 */

int gssd_client_test(struct xdr_netobj *name, u32 *uidp);

int gssd_init_sec_context(u32			uid,
			  u32		       *maj_statp,
			  u32		       *min_statp,
			  GSS_CTX_ID_T	       *ctx_id,
			  GSS_BUFFER_T	       *service_name,
			  GSS_OID	       *mech,
			  GSS_BUFFER_T	       *input_tokenp,
			  GSS_BUFFER_T	       *output_token,
			  u32		       *ret_flagsp);
     
int gssd_accept_sec_context(u32		       *maj_statp,
			    u32		       *min_statp,
			    GSS_CTX_ID_T       *ctx_id,
			    GSS_BUFFER_T       *input_token,
			    GSS_BUFFER_T       *client_namep,
			    GSS_BUFFER_T       *output_tokenp,
			    u32		       *ret_flagsp);

int
gssd_import_sec_context(u32             *maj_statp,
                        u32             *min_statp,
                        GSS_CTX_ID_T    *ctx_id,
                        GSS_BUFFER_T    *context_token); 

#endif /* __KERNEL__ */
#endif /* _LINUX_SUNRPC_AUTH_GSS_H */

