/*	$Id: nfs4fs_decode.c,v 1.1 2002/08/20 21:49:27 rees Exp $	*/

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

/*
 *  Kendrick Smith <kmsmith@umich.edu>
 *  Jim Rees <rees@umich.edu>
 */

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/resourcevar.h>
#include <sys/proc.h>
#include <sys/mount.h>
#include <sys/buf.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/conf.h>
#include <sys/namei.h>
#include <sys/vnode.h>
#include <sys/dirent.h>
#include <sys/fcntl.h>
#include <sys/lockf.h>

#include <nfs/rpcv2.h>
#include <nfs/nfsproto.h>
#include <nfs/nfs.h>
#include <nfs4/list.h>
#include <nfs4/nfs4_prot.h>
#include <nfs4/nfs4_debug.h>
#include <nfs4/nfs4.h>
#include <nfs4/nfs4mount.h>
#include <nfs4/nfs4fs.h>
#include <nfs4/nfs4node.h>
#include <nfs/xdr_subs.h>
#include <nfs4/nfsm_subs.h>
#include <nfs4/nfs4_var.h>

static int open_confirm(struct nfs4fs_compound *opencp, struct nfs4fs_open *open);

/*
 *  Primitives for XDR decoding are:
 *    copy_buf() - copies the next n bytes out of the XDR stream into a preallocated buffer
 *    read_buf() - returns a pointer to a block of memory containing the next n bytes from
 *                 the XDR stream.  The caller must pass in a "backup buffer" of size n.
 *                 If feasible, the return value of read_buf() will return a pointer which
 *                 points directly into the XDR stream.  If this is not feasible, the data
 *                 will be copied into the backup buffer, and a pointer to the backup buffer
 *                 returned.
 *    skip_buf() - skips over the next n bytes in the XDR decode stream
 *
 *  All XDR decode routines are written in terms of these primitives, without direct use
 *  to the iov_* fields, so if a different internal representation of the output buffer is
 *  needed, it is only necessary to change the primitives.
 */

/* OpenBSD version doesn't use backup buffer */

static u32 *
read_buf(struct nfs4fs_compound *cp, int nbytes)
{
	int t1;
	u32 *p;

	nbytes = nfsm_rndup(nbytes);

	/* loosely modeled after nfsm_dissect */
	t1 = mtod(cp->md, caddr_t) + cp->md->m_len - cp->dpos;
	if (t1 >= nbytes) {
		p = (u32 *) cp->dpos;
		cp->dpos += nbytes;
	} else {
		if (nfsm_disct(&cp->md, &cp->dpos, nbytes, t1, (caddr_t *) &p) != 0)
			return NULL;
	}

	return p;
}

static int
copy_buf(struct nfs4fs_compound *cp, int nbytes, u_char *buf)
{
	u32 *p;

	if (nbytes <= 0)
		return 0;

	if ((p = read_buf(cp, nbytes)) == NULL)
	    return -ENOMEM;
	if (buf)
	    bcopy((caddr_t) p, (caddr_t) buf, nbytes);
	return 0;
}

static int
skip_buf(struct nfs4fs_compound *cp, int nbytes)
{
#if NFS4_DEBUG
	if (nbytes & 3) {
		printk("!!! skip_buf: nbytes not divisible by 4 !!!\n");
		return -EIO;
	}
#endif

	return copy_buf(cp, nbytes, NULL);
}

static int
read_uio(struct nfs4fs_compound *cp, struct uio *uiop, int len)
{
	if (len <= 0)
		return 0;
	return -(nfsm_mbuftouio(&cp->md, uiop, len, &cp->dpos));
}

/*
 * Inconvenient shorthand for XDR decode routines...
 */
#define DECODE_HEAD				\
	u32 *p;					\
	int status

#define READ_BUF(nbytes)  do {			\
	if (!(p = read_buf(cp, (nbytes)))) {	\
		nfsv4_printk(level1, "xdr error! (%s:%d)\n", __FILE__, __LINE__); \
		goto xdr_error;			\
	}					\
} while (0)

#define READ32(x)         (x) = ntohl(*p++)

#define READ64(x)         do {			\
	(x) = (u64)ntohl(*p++) << 32;		\
	(x) |= ntohl(*p++);			\
} while (0)

#define READTIME(x) do {		\
	READ64(x.tv_sec);		\
	(x).tv_nsec = ntohl(*p++);		\
} while (0)

#define COPYMEM(x,nbytes) do {			\
	memcpy((x), p, nbytes);			\
	p += XDR_QUADLEN(nbytes);		\
} while (0)

#define DECODE_TAIL				\
	status = 0;				\
out:						\
	return status;				\
xdr_error:					\
	nfsv4_printk(level1, "xdr error! (%s:%d)\n", __FILE__, __LINE__); \
	status = -EIO;				\
	goto out

/*
 * The actual XDR decode routines start here...
 */
static int
decode_change_info(struct nfs4fs_compound *cp, struct change_info4 *cinfo)
{
	DECODE_HEAD;

	READ_BUF(20);
	READ32(cinfo->atomic);
	READ64(cinfo->before);
	READ64(cinfo->after);

	DECODE_TAIL;
}

/*
 *  read_attrs: a routine to read the result of a GETATTR operation
 */

#define OVERFLOW_CHECK(n)  do { if ((attrlen -= (n)) < 0) goto xdr_error; } while (0)

int
nfs4_read_attrs(struct nfs4fs_compound *cp, struct nfs4fs_fattr *nfp)
{
	int bmlen;
        u32 *bmval;
	u32 *p;
	int status;
	int attrlen, len, xdrlen;

	/* XXX : need more sanity checking on attributes read! */
	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	bmlen = ntohl(*p++);
	nfsv4_printk(xdr, "read_attrs: read bitmap len %d\n", bmlen);

	if (!(bmval = read_buf(cp, bmlen << 2)))
		goto xdr_error;
	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	attrlen = ntohl(*p++);
	nfsv4_printk(xdr, "read_attrs: read attribute len %d\n", attrlen);

	FATTR4_XDR_BITMAP(bmval, bmlen);
	nfp->fa_valid = 0;

	/*
	 * parse first bitmap block
	 */
	if (bmlen < 1)
		goto done;
	if (FATTR4_ISSET(FATTR4_TYPE, bmval)) {
		OVERFLOW_CHECK(4);
		READ_BUF(4);
		nfp->fa_type = ntohl(*p++);
		nfsv4_printk(xdr, "read_attrs: type=%d\n", (u32)nfp->fa_type);
		nfp->fa_valid |= FA_TYPE;
	}
	if (FATTR4_ISSET(FATTR4_CHANGE, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_changeid);
		nfsv4_printk(getattr, "read_attrs: changeid=%d\n", (u32)nfp->fa_changeid);
		nfp->fa_valid |= FA_CHANGEID;
	}
	if (FATTR4_ISSET(FATTR4_SIZE, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_size);
		nfsv4_printk(xdr, "read_attrs: size=%d\n", (u32)nfp->fa_size);
		nfp->fa_valid |= FA_SIZE;
	}
	if (FATTR4_ISSET(FATTR4_FSID, bmval)) {
		OVERFLOW_CHECK(16);
		READ_BUF(16);
		READ64(nfp->fa_fsid_major);
		READ64(nfp->fa_fsid_minor);
		nfsv4_printk(xdr, "read_attrs: fsid=0x%Lx/0x%Lx\n",
			     nfp->fa_fsid_major, nfp->fa_fsid_minor);
		nfp->fa_valid |= FA_FSID;
	}
	if (FATTR4_ISSET(FATTR4_LEASE_TIME, bmval)) {
		OVERFLOW_CHECK(4);
		READ_BUF(4);
		nfp->fa_lease_time = ntohl(*p++);
		nfp->fa_valid |= FA_LEASE_TIME;
	}
	if (FATTR4_ISSET(FATTR4_FILEID, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_fileid);
		nfsv4_printk(xdr, "read_attrs: fileid=0x%x\n", (int) nfp->fa_fileid);
		nfp->fa_valid |= FA_FILEID;
	}
	if (FATTR4_ISSET(FATTR4_FILES_FREE, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_ffree);
		nfsv4_printk(xdr, "read_attrs: files_free=0x%Lx\n", nfp->fa_ffree);
		nfp->fa_valid |= FA_FFREE;
	}
	if (FATTR4_ISSET(FATTR4_FILES_TOTAL, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_ftotal);
		nfsv4_printk(xdr, "read_attrs: files_tot=0x%Lx\n", nfp->fa_ftotal);
		nfp->fa_valid |= FA_FTOTAL;
	}
	if (FATTR4_ISSET(FATTR4_MAXFILESIZE, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_maxfilesize);
		nfsv4_printk(getattr, "read_attrs: maxfilesize=0x%Lx\n", nfp->fa_maxfilesize);
		nfp->fa_valid |= FA_MAXFILESIZE;
	}
	if (FATTR4_ISSET(FATTR4_MAXNAME, bmval)) {
		OVERFLOW_CHECK(4);
		READ_BUF(4);
		nfp->fa_maxname = ntohl(*p++);
		nfsv4_printk(xdr, "read_attrs: maxname=%d\n", nfp->fa_maxname);
		nfp->fa_valid |= FA_MAXNAME;
	}
	if (FATTR4_ISSET(FATTR4_MAXREAD, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_maxread);
		nfsv4_printk(xdr, "read_attrs: maxread=0x%Lx\n", nfp->fa_maxread);
		nfp->fa_valid |= FA_MAXREAD;
	}
	if (FATTR4_ISSET(FATTR4_MAXWRITE, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_maxwrite);
		nfsv4_printk(xdr, "read_attrs: maxwrite=0x%Lx\n", nfp->fa_maxwrite);
		nfp->fa_valid |= FA_MAXWRITE;
	}

	/*
	 *  parse second bitmap block
	 */
	if (bmlen < 2)
		goto done;
	if (FATTR4_ISSET(FATTR4_MODE, bmval)) {
		OVERFLOW_CHECK(4);
		READ_BUF(4);
		nfp->fa_mode = ntohl(*p++);
		nfsv4_printk(xdr, "read_attrs: mode=0%o\n", nfp->fa_mode);
		nfp->fa_valid |= FA_MODE;
	}
	if (FATTR4_ISSET(FATTR4_NUMLINKS, bmval)) {
		OVERFLOW_CHECK(4);
		READ_BUF(4);
		nfp->fa_nlink = ntohl(*p++);
		nfsv4_printk(xdr, "read_attrs: nlink=%d\n", nfp->fa_nlink);
		nfp->fa_valid |= FA_NLINK;
	}
	if (FATTR4_ISSET(FATTR4_OWNER, bmval)) {
		OVERFLOW_CHECK(4);
		READ_BUF(4);
		len = ntohl(*p++);
		nfsv4_printk(xdr, "read_attrs: owner len=%d\n", len);
		if (len > GSS_NAME_MAX)
			goto xdr_error;
		xdrlen = XDR_QUADLEN(len) << 2;
		OVERFLOW_CHECK(xdrlen);
		READ_BUF(xdrlen);
		if ((status = gss_get_num(GSS_OWNER, len, (char *)p, &nfp->fa_uid)))
			goto out;
		nfsv4_printk(xdr, "read_attrs: uid=%d\n", (int)nfp->fa_uid);
		nfp->fa_valid |= FA_UID;
	}
	if (FATTR4_ISSET(FATTR4_OWNER_GROUP, bmval)) {
		OVERFLOW_CHECK(4);
		READ_BUF(4);
		len = ntohl(*p++);
		nfsv4_printk(xdr, "read_attrs: group len=%d\n", len);
		if (len > GSS_NAME_MAX)
			goto xdr_error;
		xdrlen = XDR_QUADLEN(len) << 2;
		OVERFLOW_CHECK(xdrlen);
		READ_BUF(xdrlen);
		if ((status = gss_get_num(GSS_GROUP, len, (char *)p, &nfp->fa_gid)))
			goto out;
		nfsv4_printk(xdr, "read_attrs: gid=%d\n", (int)nfp->fa_gid);
		nfp->fa_valid |= FA_GID;
	}
	if (FATTR4_ISSET(FATTR4_RAWDEV, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		nfp->fa_rdev_major = ntohl(*p++);
		nfp->fa_rdev_minor = ntohl(*p++);
		nfsv4_printk(xdr, "read_attrs: rdev=%d %d\n",
			     nfp->fa_rdev_major, nfp->fa_rdev_minor);
		nfp->fa_valid |= FA_RDEV;
	}
	if (FATTR4_ISSET(FATTR4_SPACE_AVAIL, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_savail);
		nfsv4_printk(xdr, "read_attrs: savail=0x%Lx\n", nfp->fa_savail);
		nfp->fa_valid |= FA_SAVAIL;
	}
	if (FATTR4_ISSET(FATTR4_SPACE_FREE, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_sfree);
		nfsv4_printk(xdr, "read_attrs: sfree=0x%Lx\n", nfp->fa_sfree);
		nfp->fa_valid |= FA_SFREE;
	}
	if (FATTR4_ISSET(FATTR4_SPACE_TOTAL, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		READ64(nfp->fa_stotal);
		nfsv4_printk(xdr, "read_attrs: stotal=0x%Lx\n", nfp->fa_stotal);
		nfp->fa_valid |= FA_STOTAL;
	}
	if (FATTR4_ISSET(FATTR4_SPACE_USED, bmval)) {
		OVERFLOW_CHECK(8);
		READ_BUF(8);
		p += 2;
	}
	if (FATTR4_ISSET(FATTR4_TIME_ACCESS, bmval)) {
		OVERFLOW_CHECK(12);
		READ_BUF(12);
		READTIME(nfp->fa_atime);
		nfsv4_printk(xdr, "read_attrs: atime=%d\n", (int)nfp->fa_atime.tv_sec);
		nfp->fa_valid |= FA_ATIME;
	}
	if (FATTR4_ISSET(FATTR4_TIME_CREATE, bmval)) {
		OVERFLOW_CHECK(12);
		READ_BUF(12);
		READTIME(nfp->fa_ctime);
		nfsv4_printk(xdr, "read_attrs: ctime=%d\n", (int)nfp->fa_ctime.tv_sec);
		nfp->fa_valid |= FA_CTIME;
	}
	if (FATTR4_ISSET(FATTR4_TIME_MODIFY, bmval)) {
		OVERFLOW_CHECK(12);
		READ_BUF(12);
		READTIME(nfp->fa_mtime);
		nfsv4_printk(xdr, "read_attrs: mtime=%d\n", (int)nfp->fa_mtime.tv_sec);
		nfp->fa_valid |= FA_MTIME;
	}

done:
	status = 0;
out:
	nfsv4_printk(getattr, "read_attrs returning %d\n", status);
	return status;
xdr_error:
	nfsv4_printk(getattr2, "read_attrs: xdr error!\n");
	nfp->fa_valid = 0;
	return -EIO;
}

u32
nfs4_fileid4_to_fileid(u64 fileid4)
{
	return (u32) ((fileid4 >>32) | fileid4);
}

/*
 *  The actual XDR handler routines
 */

int
nfs4_handle_compound(struct nfs4fs_compound *cp)
{
	u32 *p;
	int taglen;
	int status = 0;

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	cp->toplevel_status = ntohl(*p++);

	/* Error Handler Calls */
	switch (cp->toplevel_status){
	case NFS4ERR_WRONGSEC:
		nfsv4_printk(auth,"nfs4_handle_compound: NFS4ERR_WRONGSEC\n");
#ifdef CTX
		if(ctx && ctx->pt_sb && ctx->pt_dentry)
			status = nfs4_do_secinfo(ctx->pt_sb,ctx->pt_dentry);
		else
			goto ctx_error;
		nfsv4_printk(auth,"nfs4_handle_compound: nfs4_do_secinfo returns %d\n",status);
		/* secinfo call succeeded, retry compound with correct creds */
		if(status == NFS4_OK)
			status = NFS4ERR_RETRY_COMPOUND;
#endif
		break;
	case NFS4ERR_STALE_CLIENTID:
		nfsv4_printk(auth,"nfs4_handle_compound: NFS4ERR_STALE_CLIENTID\n");
#ifdef CTX
		if(ctx && ctx->pt_sb)
			status = nfs4_do_setclientid(ctx->pt_sb);
		else
			goto ctx_error;
		nfsv4_printk(xdr,"nfs4_handle_compound: nfs4_do_setclientid returns %d\n",
			     status);
		/* setclientid call succeeded, retry compound with correct clientid */
		if(status == NFS4_OK)
			status = NFS4ERR_RETRY_COMPOUND;
#endif
		break;
	default:		/* fall through */
		nfsv4_printk(xdr,"nfs4_handle_compound: DEFAULT toplevel_status %d\n",
			     cp->toplevel_status);
		break;
	}
	/* Error Handler Call Failed */
	if(status)
		goto out;

	taglen = ntohl(*p++);
	nfsv4_printk(xdr, "nfs4_handle_compound: taglen=%d\n", taglen);

	if (skip_buf(cp, XDR_LEN(taglen)))
		goto xdr_error;
	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	cp->resp_nops = ntohl(*p++);
	status = 0;
	nfsv4_printk(xdr, "nfs4_handle_compound: nfserr=%d, nops=%d\n",
		     cp->toplevel_status, cp->resp_nops);

 out:
	return status;
 xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_compound: xdr error!\n");
	status = -EIO;
	goto out;
#ifdef CTX
 ctx_error:
	nfsv4_printk(xdr,"nfs4_handle_compound: invalid nfs4fs_ctx \n");
	status = -EINVAL;
	goto out;
#endif
}

int
nfs4_handle_simple_op(struct nfs4fs_compound *cp, nfs_opnum4 op, int nbytes)
{
	u32 *p;
	int status;

	nfsv4_printk(xdr, "nfs4_handle_simple_op: called with op=%d, nbytes=%d\n", op, nbytes);

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(op))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(level2, "nfs4_handle_simple_op(%d): nfserr=%d\n",
			     op, ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}
	if (nbytes && (status = skip_buf(cp, nbytes)))
		goto xdr_error;

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_simpleop(%d): xdr error!\n", op);
	status = -EIO;
	goto out;
}

int
nfs4_handle_access(struct nfs4fs_compound *rp, struct nfs4fs_access *access)
{
	u32 *p;
	int status;

	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_ACCESS))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(open2, "nfs4_handle_access: nfserr=%d\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	access->supported = ntohl(*p++);
	access->rmode = ntohl(*p++);

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_access: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_close(struct nfs4fs_compound *rp, struct nfs4fs_close *close)
{
	u32 *p;
	struct nfs4fs_file_data *file_data = close->data;
	int status;

	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	if (*p++ != close->op)
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(open2, "nfs4_handle_close: CLOSE/OPEN_DOWNGRADE failed, "
			     "nfserr=%d\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(rp, sizeof(stateid4))))
		goto xdr_error;
	memcpy(file_data->fi_stateid, p, sizeof(stateid4));
	p += XDR_QUADLEN(sizeof(stateid4));
	file_data->fi_nopen--;

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_close: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_commit(struct nfs4fs_compound *rp, struct nfs4fs_commit *commit)
{
	u32 *p;
	int status;
	
	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_COMMIT))
		goto xdr_error;
	if ((status = ntohl(*p++))) {
		nfsv4_printk(write2, "nfs4_handle_commit: nfserr=%d!\n", status);
		status = kernerrno(status);
		goto out;
	}

	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	memcpy(commit->verifier, p, 8);
	status = 0;
	
out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_commit: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_create(struct nfs4fs_compound *cp, struct nfs4fs_create *create)
{
	u32 *p;
	int bmlen;
	int status;

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_CREATE))
		goto xdr_error;
	if ((status = ntohl(*p++))) {
		nfsv4_printk(create2, "nfs4_handle_create: nfserr=%d!\n", status);
		status = kernerrno(status);
		goto out;
	}

	/* Change info */
	if (skip_buf(cp, sizeof(change_info4)))
		goto xdr_error;

	/* Attributes mask */
	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	bmlen = ntohl(*p++);
	if (skip_buf(cp, bmlen << 2))
		goto xdr_error;


out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_create: xdr error!\n");
	status = -EIO;
	goto out;
}

/* attributes which are required on a GETATTR.
   XXX : right now, we require a lot of attributes to be returned.. later, think about how
   to relax this requirement. */
#define POSITIVE_MASK	(FA_TYPE | FA_FSID | FA_FILEID | FA_SIZE | FA_MODE | FA_NLINK | \
			 FA_UID | FA_GID | FA_RDEV | FA_ATIME | FA_MTIME)
#define NEGATIVE_MASK	(FA_TYPE | FA_FSID | FA_FILEID | FA_SIZE | FA_MODE | FA_NLINK | \
			 FA_UID | FA_GID | FA_RDEV | FA_ATIME | FA_CTIME | FA_MTIME)

int
nfs4_handle_getattr(struct nfs4fs_compound *cp, struct nfs4fs_getattr *getattr)
{
	u32 *p;
	int status;

	nfsv4_printk(xdr, "nfs4_handle_getattr: starting\n");

	if (!(p = read_buf(cp, 8)) || *p++ != htonl(OP_GETATTR)) {
		nfsv4_printk(xdr2, "nfs4_handle_getattr: xdr error!\n");
		return -EIO;
	}

	if ((status = *p++)) {
		nfsv4_printk(getattr2, "nfs4_handle_getattr: nfserr=%d\n", ntohl(status));
		return kernerrno(ntohl(status));
	}

	return nfs4_read_attrs(cp, &getattr->fattr);
}

int
nfs4_handle_getfh(struct nfs4fs_compound *cp, struct nfs4fs_getfh *getfh)
{
	u32 *p;
	struct dentry *dentry = getfh->dentry;
	unsigned int len, xdrlen;
	int status;

	/* XXX : need to add a NONFATAL flag which will return error only if there is an
	   XDR error.  handle_getfh() should be nonfatal if called from reval, fatal if
	   called from mount or force_getfh() */

	nfsv4_printk(xdr, "nfs4_handle_getfh: starting\n");

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_GETFH))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(fh2, "nfs4_handle_getfh: nfserr=%d!\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	len = ntohl(*p++);
	nfsv4_printk(xdr, "nfs4_handle_getfh: fhlen=%d\n", len);

	if (len > NFS4_FHSIZE) {
		nfsv4_printk(xdr2, "nfs4_handle_getfh: length too long!\n");
		status = -EIO;
		goto out;
	}
	xdrlen = XDR_QUADLEN(len) << 2;
	dentry->fh_len = len;
	if (copy_buf(cp, xdrlen, (u_char *) &dentry->fh_val))
		goto xdr_error;

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_getfh: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_link(struct nfs4fs_compound *cp, struct nfs4fs_link *link)
{
	u32 *p;
	int status;

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_LINK))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(link2, "nfs4_handle_link: LINK failed, nfserr=%d\n",
			     ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}
	if (skip_buf(cp, sizeof(change_info4)))
		goto xdr_error;
	status = 0;

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_link: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_open(struct nfs4fs_compound *cp, struct nfs4fs_open *open)
{
	u32 *p;
	u32 rflags;
	struct nfs4fs_file_data *fp = open->fp;
	stateid4 stateid;
	u32 bmlen;
	struct change_info4 cinfo;
	u32 delegation_type = OPEN_DELEGATE_NONE;
	stateid4 delegation_stateid;
	int delegation_recall;
	int delegation_acelen;
	int status;

#if 0
	/* At the top of handle_open(), we should have the seqid semaphore and (if file_data
	 * is non-NULL), the the stateid semaphore in file_data.
	 */
	if (!(cp->flags & CA_SEQID_LOCKED))
		printf("nfs4_handle_open: !CA_SEQID_LOCKED\n");
	if (fp && !(cp->flags & CA_STATEID_LOCKED))
		printf("nfs4_handle_open: fp && !CA_STATEID_LOCKED\n");
#endif

	nfsv4_printk(open, "nfs4_handle_open: starting\n");

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_OPEN))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(open2, "nfs4_handle_open: OPEN failed, nfserr=%d\n",
			     ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(cp, sizeof(stateid4))))
		goto xdr_error;
	memcpy(stateid, p, sizeof(stateid4));
	p += XDR_QUADLEN(sizeof(stateid4));

	decode_change_info(cp, &cinfo);

	if (!(p = read_buf(cp, 8)))   /* rflags, bitmap len */
		goto xdr_error;
	rflags = ntohl(*p++);
	bmlen = ntohl(*p++);
	if (bmlen > 2)
		goto xdr_error;

	if (!(p = read_buf(cp, (bmlen << 2) + 4)))    /* bitmap val, delegation type */
		goto xdr_error;
	p += bmlen;

	delegation_type = ntohl(*p++);
	switch (delegation_type) {
	case OPEN_DELEGATE_NONE:
		break;
	case OPEN_DELEGATE_READ:
		/*
		 * Get stateid, recall flag, and the first four words of the ACE
		 * (this gets the ACE principal's length, but leaves the principal).
		 * XXX: For now, the ACE is ignored.
		 */
		printk("AAA handle_open: OPEN_DELEGATE_READ xdr starting\n");
		if (!(p = read_buf(cp, sizeof(stateid4) + 20)))
			goto xdr_error;
		memcpy(delegation_stateid, p, sizeof(stateid4));
		p += XDR_QUADLEN(sizeof(stateid4));
		delegation_recall = ntohl(*p++);
		p += 3;
		delegation_acelen = ntohl(*p++);

		/*
		 * Get the ACE principal.
		 */
		if (!read_buf(cp, XDR_LEN(delegation_acelen)))
			goto xdr_error;
		printk("AAA handle_open: OPEN_DELEGATE_READ xdr done\n");
		break;

	case OPEN_DELEGATE_WRITE:
		/*
		 * Get stateid, recall flag, limit, and the first four words of the ACE
		 * (this gets the ACE principal's length, but leaves the principal).
		 * XXX: For now, the limit and ACE are ignored.
		 */
		printk("AAA handle_open: OPEN_DELEGATE_WRITE xdr starting\n");
		if (!(p = read_buf(cp, sizeof(stateid4) + 32)))
			goto xdr_error;
		memcpy(delegation_stateid, p, sizeof(stateid4));
		p += XDR_QUADLEN(sizeof(stateid4));
		delegation_recall = ntohl(*p++);
		p += 6;
		delegation_acelen = ntohl(*p++);

		/*
		 * Get the ACE principal.
		 */
		if (!read_buf(cp, XDR_LEN(delegation_acelen)))
			goto xdr_error;
		printk("AAA handle_open: OPEN_DELEGATE_WRITE xdr done\n");
		break;

	default:
		goto xdr_error;
	}

	if ((status = nfs4_handle_getattr(cp, &open->getattr))) {
		nfsv4_printk(open, "nfs4_handle_open: nfs4_handle_getattr failed!\n");
		goto out;
	}
	if ((status = nfs4_handle_getfh(cp, &open->getfh))) {
		nfsv4_printk(open, "nfs4_handle_open: nfs4_handle_getfh failed!\n");
		goto out;
	}

	memcpy(fp->fi_stateid, stateid, sizeof(stateid4));
	cp->flags &= ~(CA_SEQID_LOCKED | CA_STATEID_LOCKED);       /* "take over" */
	if (!(rflags & OPEN4_RESULT_CONFIRM)) {
		nfsv4_printk(open, "nfs4_handle_open: confirm flag is not set\n");
		nfs4_up_write(&fp->fi_sema);
		up(&open->lockowner->lo_sema);
		fp->fi_nopen++;
		if (open->flags & FREAD)
			fp->fi_nread++;
		if (open->flags & FWRITE)
			fp->fi_nwrite++;
	}
	else {
		/*
		 * XXX: Should we return the delegation if OPEN_CONFIRM fails?
		 * If we decide "no", need to change the out_deleg branch to
		 * be conditional on the RESULT_CONFIRM flag.
		 */
		nfsv4_printk(open, "nfs4_handle_open: confirm flag is set\n");
		if ((status = open_confirm(cp, open)))
			goto out;
	}

	status = 0;
	if (delegation_type != OPEN_DELEGATE_NONE)
		printf("got unwanted delegation on %*s\n",
		       open->dentry->d_name.len, open->dentry->d_name.name);

out:
	nfsv4_printk(open, "nfs4_handle_open: returning status %d\n", status);
	return status;

xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_open: xdr error!\n");
	status = -EIO;
	goto out;
}

static int
open_confirm(struct nfs4fs_compound *opencp, struct nfs4fs_open *open)
{
	struct nfs4fs_compound *cp = NFS4_ALLOC(sizeof (struct nfs4fs_compound));
	struct nfs4fs_putfh putfh;
	int status;

#if NFS4_DEBUG
	if (!open->fp->fi_lockowner) {
		printk("!!! open_confirm(): no lockowner !!!\n");
		return -EIO;
	}
#endif

	nfsv4_printk(open, "open_confirm: starting\n");

	cp->nmp = opencp->nmp;
	cp->procp = opencp->procp;
	cp->cred = opencp->cred;
	putfh.dentry = open->dentry;

	if ((status = nfs4_setup_compound(cp, "open_confirm()")))
		goto out_release;
	if ((status = nfs4_setup_putfh(cp, &putfh)))
		goto out_release_compound;
	if ((status = nfs4_setup_open_confirm(cp, open)))
		goto out_release_compound;

	status = nfs4fs_call_compound(cp);
#if 0
	if (status == -EREBOOT) {
		/* XXX: Need to wait for recoverd, then check whether it succeeded in
		 * renewing the OPEN.
		 */
		status = -EIO;
		printk("!!! open_confirm(): server reboot, can't handle this case yet !!!\n");
		goto out_args;
	}
#endif
	if (status)
		goto out_compound;
	if ((status = nfs4_handle_putfh(cp, &putfh)))
		goto out_compound;
	if ((status = nfs4_handle_open_confirm(cp, open)))
		goto out_compound;

out_compound:
	nfs4fs_release_compound(cp);
out:
	NFS4_FREE(cp);
	nfsv4_printk(open, "open_confirm: returning status %d\n", status);
	return status;
out_release_compound:
	nfs4fs_release_compound(cp);
out_release:
	up(&open->lockowner->lo_sema);
	nfs4_up_write(&open->post_file_data->fi_sema);
	goto out;
}

int
nfs4_handle_open_confirm(struct nfs4fs_compound *rp, struct nfs4fs_open *open)
{
	u32 *p;
	struct nfs4fs_file_data *fp = open->fp;
	int status;

	nfsv4_printk(open, "nfs4_handle_open_confirm: starting\n");

	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_OPEN_CONFIRM))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(open2, "nfs4_handle_open_confirm: OPEN_CONFIRM failed, "
			     "nfserr=%d\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(rp, sizeof(stateid4))))
		goto xdr_error;

	/* The OPEN is finished; update the stateid and open counts */
	memcpy(fp->fi_stateid, p, sizeof(stateid4));
	p += XDR_QUADLEN(sizeof(stateid4));

	fp->fi_nopen++;
	if (open->flags & FREAD)
		fp->fi_nread++;
	if (open->flags & FWRITE)
		fp->fi_nwrite++;

	status = 0;

out:
	nfsv4_printk(open, "nfs4_handle_open_confirm: returning status %d\n", status);
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_open_confirm: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_putfh(struct nfs4fs_compound *cp, struct nfs4fs_putfh *putfh)
{
	u32 *p;
	int i;
	int status;

	nfsv4_printk(xdr, "nfs4_handle_putfh: called with op=%d, nlookups=%d\n", putfh->op,
		     putfh->nlookups);

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(putfh->op))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(xdr2, "nfs4_handle_putfh: nfserr=%d!\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	for (i = 0; i < putfh->nlookups; i++) {
		if (!(p = read_buf(cp, 8)))
			goto xdr_error;
		if (*p++ != htonl(OP_LOOKUP))
			goto xdr_error;
		if ((status = *p++)) {
			nfsv4_printk(xdr2, "nfs4_handle_putfh: nfserr=%d!\n", ntohl(status));
			status = kernerrno(ntohl(status));
			goto out;
		}
	}

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_putfh: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_read(struct nfs4fs_compound *cp, struct nfs4fs_read *read)
{
	u32 *p;
	int status;

	nfsv4_printk(xdr, "nfs4_handle_read: starting\n");

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_READ))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(read2, "nfs4_handle_read: READ failed, nfserr=%d\n",
			     ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}
	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	read->eof = ntohl(*p++);
	read->bytes_read = ntohl(*p++);

	status = read_uio(cp, read->uiop, read->bytes_read);

	nfsv4_printk(xdr, "nfs4_handle_read: eof=%d, bytes_read=%d\n", read->eof,
		     read->bytes_read);

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_read: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_readlink(struct nfs4fs_compound *cp, struct uio *uiop)
{
	u32 *p;
	int linklen, copylen;
	int status;

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_READLINK))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(link2, "nfs4_handle_readlink: LINK failed, nfserr=%d\n",
			     ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	linklen = ntohl(*p++);
	if (linklen <= 0)
		goto xdr_error;
	copylen = MIN(linklen, uiop->uio_resid);
	if ((status = read_uio(cp, uiop, copylen)))
		goto out;

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_readlink: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_setattr(struct nfs4fs_compound *cp)
{
	u32 *p;
	u32 bmlen;
	int status;

	/* the SETATTR op returns a bitmap even if it fails (is this a bug in the spec?) */

	nfsv4_printk(xdr, "nfs4_handle_setattr: starting\n");

	if (!(p = read_buf(cp, 12)))
		goto xdr_error;
	if (*p++ != htonl(OP_SETATTR))
		goto xdr_error;
	status = *p++;
	bmlen = ntohl(*p++);
	if (skip_buf(cp, bmlen << 2))
		goto xdr_error;
	if (status) {
		nfsv4_printk(setattr2, "nfs4_handle_setattr: nfserr=%d!\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

out:
	nfsv4_printk(xdr, "nfs4_handle_setattr: returning status %d\n", status);
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_setattr: xdr error!\n");
	status = -EIO;
	goto out;
}

#if 0
int
nfs4_handle_secinfo(struct nfs4fs_compound *cp, struct nfs4fs_secinfo *secinfo)
{
	u32 *p;
	int len,flavorlen,i,flavor=0;
	int status;

	nfsv4_printk(level2, "nfs4_handle_secinfo\n");
	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != ntohl(OP_SECINFO))
		goto xdr_error;
	status = *p++;
	if (status) {
		nfsv4_printk(secinfo, "nfs4_handle_secinfo: SECINFO failed, "
			     "nfserr=%d\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}
	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	len = ntohl(*p++);
	nfsv4_printk(secinfo,"nfs4_handle_secinfo num flavors %d\n",len);
	for(i=0;i<len;i++){
		if (!(p = read_buf(cp, 4)))
			goto xdr_error;
		flavor = ntohl(*p++);
		switch(flavor){
		case RPC_AUTH_NULL:
			nfsv4_printk(secinfo,"nfs4_handle_secinfo RPC_AUTH_NULL flavor \n");
			break;
		case RPC_AUTH_UNIX:
			nfsv4_printk(secinfo,"nfs4_handle_secinfo RPC_AUTH_UNIX flavor \n");
			break;
		case RPC_AUTH_GSS:
			nfsv4_printk(secinfo,"nfs4_handle_secinfo RPC_AUTH_GSS flavor \n");
			/* opaque data - not xdr'ed */
			/* sec_oid4 */
			if (!(p = read_buf(cp, 4)))
				goto xdr_error;
			flavorlen = ntohl(*p++);
			nfsv4_printk(secinfo,"nfs4_handle_secinfo flavorlen %d \n",flavorlen);

			/* for now, don't parse, use hardcoded default so skip oid ,qop4,
			 * and service.  eventually, we will compare the QOP values
			 * to the static client state struct that
			 * contains the QOP supported by this client.
			 */
			if (!(p = read_buf(cp, flavorlen)))
				goto xdr_error;
			p+= flavorlen;

			nfsv4_printk(secinfo,"nfs4_handle_secinfo rpcsec_gss_info skipped \n");
			break;
		default:
			nfsv4_printk(secinfo,"NFSV4: do_secinfo INVALID flavor %d\n",flavor);
			break;
		}
	}
	/* XXX  the following  assumes the hard_coded default qop */
	if (cp->clnt != NULL &&
	    cp->clnt->cl_auth->au_ops->au_flavor != flavor) {
		nfsv4_printk(secinfo, "nfs4_handle_secinfo: changing to "
			     "flavor %d\n", flavor);
		/* Set the mech, service, QOP, based on negotiated secinfo. */
		rpcauth_destroy(cp->clnt->cl_auth);
		rpcauth_create(flavor, cp->clnt);
	}

 out:
	return status;
 xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_secinfo: xdr error!\n");
	status = -EIO;
	goto out;
}
#endif

int
nfs4_handle_setclientid(struct nfs4fs_compound *rp, struct nfs4fs_setclientid *setclientid)
{
	struct nfs4fs_compound compound;
	u32 *p;
	int len;
	clientid4 clientid;
	int status;

	setclientid->in_use = 0;
	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_SETCLIENTID))
		goto xdr_error;
	status = *p++;
	if (status == htonl(NFS4ERR_CLID_INUSE)) {
		if (!(p = read_buf(rp, 4)))     /* skip first string in clientaddr */
			goto xdr_error;
		len = XDR_LEN(ntohl(*p++));
		if (skip_buf(rp, len))
			goto xdr_error;
		if (!(p = read_buf(rp, 4)))     /* skip second string */
			goto xdr_error;
		len = XDR_LEN(ntohl(*p++));
		if (skip_buf(rp, len))
			goto xdr_error;
		setclientid->in_use = 1;
		status = -EIO;
		goto out;
	}
	if (status) {
		nfsv4_printk(clientid2, "nfs4_handle_setclientid: SETCLIENTID failed, "
			     "nfserr=%d\n", ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	clientid = *((clientid4 *) p);
	p += 2;

	compound.procp = rp->procp;
	compound.cred = rp->cred;
	compound.nmp = rp->nmp;

	if ((status = nfs4_setup_compound(&compound, "handle_setclientid()")))
		goto out;
	if ((status = nfs4_setup_setclientid_confirm(&compound, clientid)))
		goto out_release;
	if ((status = nfs4fs_call_compound(&compound)))
		goto out_release;
	if ((status = nfs4_handle_setclientid_confirm(&compound, setclientid)))
		goto out_release;

	setclientid->clientid = betoh64(clientid);

out_release:
	nfs4fs_release_compound(&compound);
out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_setclientid: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_setclientid_confirm(struct nfs4fs_compound *rp,
				struct nfs4fs_setclientid *setclientid)
{
	u32 *p;
	int status;

	if (!(p = read_buf(rp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_SETCLIENTID_CONFIRM))
		goto xdr_error;
	if ((status = *p++)) {
		if (status == NFS4ERR_CLID_INUSE) {
			nfsv4_printk(clientid2, "nfs4_handle_setclientid: clid in use!\n");
			setclientid->in_use = 1;
			status = -EIO;
			goto out;
		}
		nfsv4_printk(clientid2, "nfs4_handle_setclientid: nfserr=%d!\n",
			     ntohl(status));
		status = kernerrno(ntohl(status));
	}

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_setclientid_confirm: xdr error!\n");
	status = -EIO;
	goto out;
}

int
nfs4_handle_write(struct nfs4fs_compound *cp, struct nfs4fs_write *write)
{
	u32 *p;
	int status;

	if (!(p = read_buf(cp, 8)))
		goto xdr_error;
	if (*p++ != htonl(OP_WRITE))
		goto xdr_error;
	if ((status = *p++)) {
		nfsv4_printk(write2, "nfs4_handle_write: WRITE failed, nfserr=%d\n",
			     ntohl(status));
		status = kernerrno(ntohl(status));
		goto out;
	}

	if (!(p = read_buf(cp, 4)))
		goto xdr_error;
	write->bytes_written = ntohl(*p++);
	write->committed = ntohl(*p++);
	if (copy_buf(cp, NFS4_VERIFIER_SIZE, write->writeverf))
		goto xdr_error;

out:
	return status;
xdr_error:
	nfsv4_printk(xdr2, "nfs4_handle_write: xdr error!\n");
	status = -EIO;
	goto out;
}
