/*	$Id: nfs4_util.c,v 1.1 2002/08/20 21:49:21 rees Exp $	*/

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/ioctl.h>
#include <sys/signal.h>
#include <sys/proc.h>
#include <sys/namei.h>
#include <sys/vnode.h>
#include <sys/kernel.h>
#include <sys/mount.h>
#include <sys/buf.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/systm.h>
#include <sys/sysctl.h>

#include <net/if.h>
#include <net/route.h>
#include <netinet/in.h>

#include <nfs/rpcv2.h>
#include <nfs/nfsproto.h>
#include <nfs/nfs.h>
#include <nfs/xdr_subs.h>
#include <nfs/nfsdiskless.h>
#include <nfs4/list.h>
#include <nfs4/nfs4_var.h>
#include <nfs4/nfs4mount.h>
#include <nfs4/nfs4_prot.h>
#include <nfs4/nfs4_debug.h>
#include <nfs4/nfs4.h>
#include <nfs4/nfs4fs.h>
#include <nfs4/nfs4node.h>
#include <nfs4/nfsm_subs.h>

#if NFS4_DEBUG
int nfsv4_debug_level1		= 0;
int nfsv4_debug_level2		= 0;
int nfsv4_debug_level3		= 0;

int nfsv4_debug_mount		= 0;
int nfsv4_debug_ctl		= 0;
int nfsv4_debug_dispatcher	= 0;
int nfsv4_debug_fh		= 0;
int nfsv4_debug_xdr		= 0;
int nfsv4_debug_gss		= 0;

int nfsv4_debug_getattr		= 0;
int nfsv4_debug_setattr		= 0;
int nfsv4_debug_lookup		= 0;
int nfsv4_debug_readdir		= 0;
int nfsv4_debug_create		= 0;
int nfsv4_debug_remove		= 0;
int nfsv4_debug_rename		= 0;
int nfsv4_debug_link		= 0;
int nfsv4_debug_readlink	= 0;
int nfsv4_debug_open		= 0;
int nfsv4_debug_read		= 0;
int nfsv4_debug_write		= 0;
int nfsv4_debug_lock		= 0;

int nfsv4_debug_seqid		= 0;
int nfsv4_debug_stateid		= 0;
int nfsv4_debug_lockowner	= 0;
int nfsv4_debug_clientid	= 0;
int nfsv4_debug_reval		= 0;
int nfsv4_debug_auth		= 0;
int nfsv4_debug_secinfo		= 0;

int nfsv4_debug_check		= 0;
#endif  /* NFS4_DEBUG */

static struct {
	short	nfserr;
	short	syserr;
} nfs_errtbl[] = {
	{ NFS4_OK, 0 },
	{ NFS4ERR_PERM, -EPERM },
	{ NFS4ERR_NOENT, -ENOENT },
	{ NFS4ERR_IO, -EIO },
	{ NFS4ERR_NXIO, -ENXIO },
	{ NFS4ERR_ACCESS, -EACCES },
	{ NFS4ERR_EXIST, -EEXIST },
	{ NFS4ERR_XDEV, -EXDEV },
	{ NFS4ERR_MLINK, -EMLINK },
	{ NFS4ERR_NODEV, -ENODEV },
	{ NFS4ERR_NOTDIR, -ENOTDIR },
	{ NFS4ERR_ISDIR, -EISDIR },
	{ NFS4ERR_INVAL, -EINVAL },
	{ NFS4ERR_FBIG, -EFBIG },
	{ NFS4ERR_NOSPC, -ENOSPC },
	{ NFS4ERR_ROFS, -EROFS },
	{ NFS4ERR_MLINK, -EMLINK },
	{ NFS4ERR_NAMETOOLONG, -ENAMETOOLONG },
	{ NFS4ERR_NOTEMPTY, -ENOTEMPTY },
	{ NFS4ERR_NOTSUPP, -EOPNOTSUPP },
#ifdef EDQUOT
	{ NFS4ERR_DQUOT, -EDQUOT },
#endif
	{ NFS4ERR_STALE, -ESTALE },
	{ NFS4ERR_DENIED, -EAGAIN },
	{ NFS4ERR_SYMLINK, -ELOOP },
	{ NFS4ERR_BADXDR, -EBADRPC },
	{ NFS4ERR_WRONGSEC, -EPERM },
	{ -1, -EIO }
};

static struct {
	short	nfserr;
	char	*name;
} nfs_errnames[] = {
	{NFS4ERR_ACCESS,		"ACCESS"},
	{NFS4ERR_ATTRNOTSUPP,	"ATTRNOTSUPP"},
	{NFS4ERR_BADHANDLE,	"BADHANDLE"},
	{NFS4ERR_BADOWNER,	"BADOWNER"},
	{NFS4ERR_BADTYPE,	"BADTYPE"},
	{NFS4ERR_BAD_COOKIE,	"BAD_COOKIE"},
	{NFS4ERR_BAD_SEQID,	"BAD_SEQID"},
	{NFS4ERR_BAD_STATEID,	"BAD_STATEID"},
	{NFS4ERR_BADXDR,		"BADXDR"},
	{NFS4ERR_CLID_INUSE,	"CLID_INUSE"},
	{NFS4ERR_DELAY,		"DELAY"},
	{NFS4ERR_DENIED,		"DENIED"},
	{NFS4ERR_DQUOT,		"DQUOT"},
	{NFS4ERR_EXIST,		"EXIST"},
	{NFS4ERR_EXPIRED,	"EXPIRED"},
	{NFS4ERR_FBIG,		"FBIG"},
	{NFS4ERR_FHEXPIRED,	"FHEXPIRED"},
	{NFS4ERR_GRACE,		"GRACE"},
	{NFS4ERR_INVAL,		"INVAL"},
	{NFS4ERR_IO,		"IO"},
	{NFS4ERR_ISDIR,		"ISDIR"},
	{NFS4ERR_LEASE_MOVED,	"LEASE_MOVED"},
	{NFS4ERR_LOCKED,		"LOCKED"},
	{NFS4ERR_LOCK_RANGE,	"LOCK_RANGE"},
	{NFS4ERR_MINOR_VERS_MISMATCH,	"MINOR_VERS_MISMATCH"},
	{NFS4ERR_MLINK,		"MLINK"},
	{NFS4ERR_MOVED,		"MOVED"},
	{NFS4ERR_NAMETOOLONG,	"NAMETOOLONG"},
	{NFS4ERR_NODEV,		"NODEV"},
	{NFS4ERR_NOENT,		"NOENT"},
	{NFS4ERR_NOFILEHANDLE,	"NOFILEHANDLE"},
	{NFS4ERR_NO_GRACE,	"NO_GRACE"},
	{NFS4ERR_NOSPC,		"NOSPC"},
	{NFS4ERR_NOTDIR,		"NOTDIR"},
	{NFS4ERR_NOTEMPTY,	"NOTEMPTY"},
	{NFS4ERR_NOTSUPP,	"NOTSUPP"},
	{NFS4ERR_NOT_SAME,	"NOT_SAME"},
	{NFS4ERR_NXIO,		"NXIO"},
	{NFS4ERR_OLD_STATEID,	"OLD_STATEID"},
	{NFS4ERR_OPENMODE,	"OPENMODE"},
	{NFS4ERR_PERM,		"PERM"},
	{NFS4ERR_READDIR_NOSPC,	"READDIR_NOSPC"},
	{NFS4ERR_RECLAIM_BAD,	"RECLAIM_BAD"},
	{NFS4ERR_RECLAIM_CONFLICT,	"RECLAIM_CONFLICT"},
	{NFS4ERR_RESOURCE,	"RESOURCE"},
	{NFS4ERR_ROFS,		"ROFS"},
	{NFS4ERR_SAME,		"SAME"},
	{NFS4ERR_SERVERFAULT,	"SERVERFAULT"},
	{NFS4ERR_SHARE_DENIED,	"SHARE_DENIED"},
	{NFS4ERR_STALE,		"STALE"},
	{NFS4ERR_STALE_CLIENTID,	"STALE_CLIENTID"},
	{NFS4ERR_STALE_STATEID,	"STALE_STATEID"},
	{NFS4ERR_SYMLINK,	"SYMLINK"},
	{NFS4ERR_TOOSMALL,	"TOOSMALL"},
	{NFS4ERR_WRONGSEC,	"WRONGSEC"},
	{NFS4ERR_XDEV,		"XDEV"},
	{-1,			"UNKNOWN"}
};

int
kernerrno(nfsstat4 err)
{
	int i;
	char *name;
	
	/* XXX : not the optimal algorithm, but will do for now! */
	for (i = 0; nfs_errtbl[i].nfserr != -1; i++) {
		if (nfs_errtbl[i].nfserr == err)
			return nfs_errtbl[i].syserr;
	}
	for (i = 0; nfs_errnames[i].nfserr != -1; i++) {
		if (nfs_errnames[i].nfserr == err)
			break;
	}
	name = nfs_errnames[i].name;
	nfsv4_printk(level2, "kernerrno: unrecognized nfs err: %d %s\n", err, name);
	/* if no entry exists for this error, use EIO by default */
	return -EIO;
}

int
nfserrno(int errno)
{
	int i;

	/* XXX : not the optimal algorithm, but will do for now! */
	for (i = 0; nfs_errtbl[i].nfserr != -1; i++) {
		if (nfs_errtbl[i].syserr == errno)
			return nfs_errtbl[i].nfserr;
	}
	nfsv4_printk(level2, "nfserrno: unrecongnized errno: %d\n", errno);
	/* if no entry exists for this error, use NFS4ERR_IO by default */
	return NFS4ERR_IO;
}

#if NFS4_DEBUG
void
hex_dump(void *ptr, int nbytes)
{
	int i;
	unsigned char *cptr = (unsigned char *) ptr;

	if (nbytes < 0) {
		printk("hex_dump: nbytes < 0!\n");
		return;
	}
	for (i = 0; i < nbytes; i++) {
		printk("%2x", *cptr++);
		if ((i % 24) == 23)
			printk("\n");
		else if ((i % 4) == 3)
			printk("  ");
		else
			printk(" ");
	}
	printk("\n");
}

#define HASH_SIZE  256

static u32
ptr_hval(void *ptr)
{
	return opaque_hashval(&ptr, sizeof(void *)) % HASH_SIZE;
}

struct mem_hashent {
	int nbytes;
	char *file;
	int line;
	void *ptr;
	struct mem_hashent *next;
};
static struct mem_hashent *mem_hashtbl[HASH_SIZE];

/* when a filename is entered into the history tables, it may exist in the static memory of
   a different module, which can be unloaded seperately from this one.  therefore, we must
   copy the string.  on the other hand, we want to avoid excessive copying.  we keep a hash
   table of filenames, so that if a duplicate filename is seen, we can return the old copy
   instead of making a new one. */

struct filename_hashent {
	int len;
	char *name;
	struct filename_hashent *next;
};
static struct filename_hashent *filename_hashtbl[HASH_SIZE];

static char *
dup_filename(char *name)
{
	char *retval;
	struct filename_hashent *p;
	int len = strlen(name);
	int hashval = opaque_hashval(name, len) % HASH_SIZE;

	down(&check_sema);
	for (p = filename_hashtbl[hashval]; p; p = p->next)
		if ((p->len == len) && !memcmp(p->name, name, len)) {
			up(&check_sema);
			return p->name;
		}
	if (!(p = kmalloc(sizeof(*p), GFP_KERNEL))) {
		printk("!!! couldn't kmalloc filename_hashent !!!\n");
		return "unknown";
	}
	if (!(retval = kmalloc(len+1, GFP_KERNEL))) {
		printk("!!! couldn't kmalloc filename !!!\n");
		kfree(p);
		return "unknown";
	}
	memcpy(retval, name, len+1);
	p->len = len;
	p->name = retval;
	p->next = filename_hashtbl[hashval];
	filename_hashtbl[hashval] = p;
	up(&check_sema);
	return retval;
}

void *
__NFS4_ALLOC(int nbytes, char *file, int line)
{
	struct mem_hashent *p;
	void *ptr;
	u32 hashval;

	if (nbytes == 0) {
		printk("ZERO SIZE REQUESTED in NFS4_ALLOC (source %s:%d)!\n",
		       file, line);
		return NULL;
	}
	if (nbytes < 0) {
		printk("NEGATIVE SIZE REQUESTED in NFS4_ALLOC (source %s:%d)!\n",
		       file, line);
		return NULL;
	}

	file = dup_filename(file);
	if (!(p = kmalloc(sizeof(*p), GFP_KERNEL))) {
		printk("HASH KMALLOC FAILED in NFS4_ALLOC!\n");
		return NULL;
	}
	if (!(ptr = kmalloc(nbytes, GFP_KERNEL))) {
		printk("MAIN KMALLOC FAILED in NFS4_ALLOC!\n");
		return NULL;
	}
	hashval = ptr_hval(ptr);
	p->nbytes = nbytes;
	p->file = file;
	p->line = line;
	p->ptr = ptr;
	down(&check_sema);
	p->next = mem_hashtbl[hashval];
	mem_hashtbl[hashval] = p;
	up(&check_sema);
	return ptr;
}

void
__NFS4_FREE(void *ptr, char *file, int line)
{
	u32 hashval = ptr_hval(ptr);
	struct mem_hashent *p, **pp;

	file = dup_filename(file);
	down(&check_sema);
	pp = &mem_hashtbl[hashval];
	for (p = mem_hashtbl[hashval]; p; p = p->next) {
		if (p->ptr == ptr)
			break;
		pp = &p->next;
	}
	if (!p) {
		up(&check_sema);
		printk("BAD KFREE of ptr %p! (source: %s:%d)\n", ptr, file, line);
	}
	else {
		*pp = p->next;
		up(&check_sema);
		kfree(p);
		kfree(ptr);
	}
}

#endif  /* NFS4_DEBUG */

void *
kmalloc(size_t s, int t)
{
    void *p;

    MALLOC(p, void *, s, M_NFSREQ, M_WAITOK);
    return p;
}

void kfree(const void *p)
{
    FREE(p, M_NFSREQ);
}
