/*	$Id: nfs4_gss.c,v 1.1 2002/08/20 21:49:20 rees Exp $	*/

/*
 * copyright (c) 2002
 * the regents of the university of michigan
 * all rights reserved
 * 
 * permission is granted to use, copy, create derivative works and redistribute
 * this software and such derivative works for any purpose, so long as the name
 * of the university of michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software without specific,
 * written prior authorization.  if the above copyright notice or any other
 * identification of the university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must also be included.
 * 
 * this software is provided as is, without representation from the university
 * of michigan as to its fitness for any purpose, and without warranty by the
 * university of michigan of any kind, either express or implied, including
 * without limitation the implied warranties of merchantability and fitness for
 * a particular purpose. the regents of the university of michigan shall not be
 * liable for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising out of or in
 * connection with the use of the software, even if it has been or is hereafter
 * advised of the possibility of such damages.
 */

#ifdef __linux__
#include <linux/nfs4/nfs4_prot.h>
#include <linux/nfs4/nfs4_debug.h>
#include <linux/nfs4/nfs4.h>
#include <linux/sunrpc/auth_gss.h>
#else
#include <nfs4/list.h>
#include <nfs4/nfs4_prot.h>
#include <nfs4/nfs4_debug.h>
#include <nfs4/nfs4.h>
#include <nfs4/auth_gss.h>
#endif

#define GSS_HASH_BITS	8
#define GSS_HASH_SIZE	(1 << GSS_HASH_BITS)
#define GSS_HASH_MASK	(GSS_HASH_SIZE - 1)

/* XXX : the locking here is based on the assumption that kmalloc() and kfree() can both
   block.. make sure that this is correct! */

struct gss_mapping {
	/* we have a separate lock for each mapping.  we could even have a seperate lock for
	   each hash chain, but this seems a little excessive! */
	rwlock_t		lock;
	struct list_head	id_hash[GSS_HASH_SIZE];
	struct list_head	name_hash[GSS_HASH_SIZE];
};

#if 0
static struct gss_mapping	owner_map;
static struct gss_mapping	group_map;

static struct gss_cacheent *gss_alloc(unsigned int namelen);
#endif

#ifndef REALGSS
void gss_init(void)
{
}

int gss_get_name(int type, uid_t id, struct gss_cacheent **cachep)
{
	static struct gss_cacheent c;
	static char username[] = "rees@citi.umich.edu";
	static char groupname[] = "staff@citi.umich.edu";

	c.name.name = (type == GSS_OWNER) ? username : groupname ;
	c.name.len = strlen(c.name.name);
	*cachep = &c;
	return 0;
}

int gss_get_num(int type, unsigned int len, const unsigned char *name, uid_t *idp)
{
	*idp = 0;
	return 0;
}

void gss_free(struct gss_cacheent *p)
{
}

void gss_shutdown(void)
{
}
#else

#if NFS4_DEBUG
static char *
gss_opcode(int type)
{
	switch (type) {
	case GSS_OWNER:
		return "GSS_OWNER";
	case GSS_GROUP:
		return "GSS_GROUP";
	default:
		return "GSS_BAD";
	}
}
#endif

static int
__gss_get_num(int type, unsigned int namelen, const unsigned char *name, uid_t *idp)
{
	int status;
	struct xdr_netobj *x;

	nfsv4_printk(gss, "__gss_get_num\n");

	if (!namelen) {
		nfsv4_printk(gss, "__gss_get_num: empty string!\n");
		status = -EINVAL;
		goto out;
	}

	nfsv4_printk(gss, "__gss_get_num. name %s, namelen %d\n",name,namelen);

	if (namelen > GSS_NAME_MAX) {
		nfsv4_printk(gss, "__gss_get_num: name too long!\n");
		status = -EINVAL;
		goto out;
	}
	/* copy string into rpc buffer */
	
	nfsv4_printk(gss, "__gss_get_num alloc x.data\n");

	if (!(x = NFS4_ALLOC(sizeof(struct xdr_netobj)))) {
    status = -ENOMEM;
    goto out;	
	}
	x->len = namelen;
	x->data = name;

	nfsv4_printk(gss, "__gss_get_num making upcall \n");
	if (type == GSS_OWNER) {
		nfsv4_printk(gss, "__gss_get_num: gss_name_to_uid\n");
    if (gssd_name_to_uid(x, idp) < 0){
			status = -EIO;
			goto out;
		}
	}
	else if (type == GSS_GROUP) {
		nfsv4_printk(gss, "__gss_get_num: gss_name_to_gid\n");
    if (gssd_name_to_gid(x, idp) < 0){
			status = -EIO;
			goto out;
		}
	} 
	nfsv4_printk(gss, "__gss_get_num: returned from gssd\n");
	/* Once the id has been determined, this function is responsible for setting *idp. */
	nfsv4_printk(gss, "__gss_get_num: set id=%d\n", *idp);
	status = 0;
	NFS4_FREE(x);

out:
	nfsv4_printk(gss, "__gss_get_num: returning status %d\n", status);
	return status;
}

static int
__gss_get_name(int type, uid_t id, struct gss_cacheent **pp)
{
  struct gss_cacheent *p = NULL;
	struct xdr_netobj x;
  int status = 0;

  nfsv4_printk(gss, "__gss_get_name\n");

  if (type == GSS_OWNER) {
  	nfsv4_printk(gss, "__gss_get_name: gssd_uid_to_name\n");
    if (gssd_uid_to_name(&id, &x) < 0){
      status = -EIO;
			goto out;
		}
  }
  else if (type == GSS_GROUP) {
  	nfsv4_printk(gss, "__gss_get_name: gssd_gid_to_name\n");
    if (gssd_gid_to_name(&id, &x) < 0){
      status = -EIO;
      goto out;
		}
  }
	if (!(p = NFS4_ALLOC(sizeof(struct gss_cacheent)))) {
		status = -ENOMEM;
		goto out;
	}
	p->name.len = x.len;
	p->name.name = (char *)x.data;
	*pp = p;
  nfsv4_printk(gss, "__gss_get_name: set name=\"%.*s\"\n", (int)p->name.len,
         p->name.name);
out:
  nfsv4_printk(gss, "__gss_get_name: returning status %d\n", status);
  return status;
}

void
gss_init(void)
{
	int i;

	rwlock_init(&owner_map.lock);
	rwlock_init(&group_map.lock);
	for (i = 0; i < GSS_HASH_SIZE; i++) {
		INIT_LIST_HEAD(&owner_map.id_hash[i]);
		INIT_LIST_HEAD(&owner_map.name_hash[i]);
		INIT_LIST_HEAD(&group_map.id_hash[i]);
		INIT_LIST_HEAD(&group_map.name_hash[i]);
	}
}

static struct gss_cacheent *
gss_alloc(unsigned int len)
{
	struct gss_cacheent *p;

	/* caller must not hold spinlock, since this routine can block */

#if NFS4_DEBUG
	if (len > GSS_NAME_MAX) {
		nfsv4_printk(gss, "gss_alloc: name length too long!\n");
		p = NULL;
		goto out;
	}
#endif
	if (!(p = NFS4_ALLOC(sizeof(struct gss_cacheent))))
		goto nomem;
	if (!(p->name.name = NFS4_ALLOC(len))) {
		NFS4_FREE(p);
		goto nomem;
	}
	nfsv4_printk(gss, "gss_alloc: returning cacheent %p\n", p);

out:
	return p;
nomem:
	nfsv4_printk(gss, "gss_alloc: out of memory!\n");
	p = NULL;
	goto out;
}

void
gss_free(struct gss_cacheent *p)
{
	nfsv4_printk(gss, "gss_free: called on cacheent %p\n", p);
	
	if (p->name.name != p->iname)
		NFS4_FREE((char *)p->name.name);  /* cast is to get rid of "const" */
	NFS4_FREE(p);
}

static inline void
gss_insert(struct gss_cacheent *p, struct gss_mapping *map)
{
	unsigned int id_hashval = (p->id & GSS_HASH_MASK);
	unsigned int name_hashval = (p->name.hash & GSS_HASH_MASK);

	/* caller must hold writelock */
	atomic_set(&p->refcount, 1);    /* refcount==1 indicates in use by the cache */
	list_add(&p->id_hash, &map->id_hash[id_hashval]);
	list_add(&p->name_hash, &map->name_hash[name_hashval]);
}

static void
gss_clear_map(struct gss_mapping *map)
{
	int i;
	struct list_head *l;
	struct gss_cacheent *p;

	for (i = 0; i < GSS_HASH_SIZE; i++) {
		l = &map->id_hash[i];
		while (!list_empty(l)) {
			write_lock(&map->lock);
			p = list_entry(l->next, struct gss_cacheent, id_hash);
			list_del(&p->id_hash);
			list_del(&p->name_hash);
			/* Since gss_put() can block, it can only be called safely if the
			   spinlock is not held. */
			write_unlock(&map->lock);
			gss_put(p);
		}
	}
}

void
gss_shutdown(void)
{
	nfsv4_printk(gss, "gss_shutdown: starting\n");
	gss_clear_map(&owner_map);
	gss_clear_map(&group_map);
	nfsv4_printk(gss, "gss_shutdown: done\n");
}

static inline struct gss_cacheent *
gss_searchbyid(struct gss_mapping *map, uid_t id)
{
	unsigned int hashval = id & GSS_HASH_MASK;
	struct list_head *l;
	struct gss_cacheent *p;

#ifndef __OpenBSD__
	/* caller must hold readlock */
	list_for_each(l, &map->id_hash[hashval]) {
		p = list_entry(l, struct gss_cacheent, id_hash);
		if (p->id == id)
			return p;
	}
#endif
	return NULL;
}

static inline struct gss_cacheent *
gss_searchbyname(struct gss_mapping *map, unsigned int len, const unsigned char *name,
		 unsigned int hash)
{
	unsigned int hashval = hash & GSS_HASH_MASK;
	struct list_head *l;
	struct gss_cacheent *p;

#ifndef __OpenBSD__
	/* caller must hold readlock */
	list_for_each(l, &map->name_hash[hashval]) {
		p = list_entry(l, struct gss_cacheent, name_hash);
		if ((p->name.hash == hash) && (p->name.len == len) &&
		    !memcmp(p->name.name, name, len))
			return p;
	}
#endif
	return NULL;
}

int
gss_get_name(int type, uid_t id, struct gss_cacheent **pp)
{
	struct gss_mapping *map;
	struct gss_cacheent *p, *q;
	int status;

	nfsv4_printk(gss, "gss_get_name: called with type %s, id=%d\n", gss_opcode(type),
		     (int)id);
	
#if NFS4_DEBUG
	if ((type != GSS_OWNER) && (type != GSS_GROUP)) {
		printk("gss_get_name: bad type!\n");
		return -EINVAL;
	}
#endif
	map = (type == GSS_OWNER) ? &owner_map : &group_map;
	read_lock(&map->lock);
	if ((p = gss_searchbyid(map, id))) {
		atomic_inc(&p->refcount);         /* return a reference */
		read_unlock(&map->lock);
		nfsv4_printk(gss, "gss_get_name: cache hit\n");
		*pp = p;
		goto done;
	}
	read_unlock(&map->lock);
	nfsv4_printk(gss, "gss_get_name: cache miss\n");
	if ((status = __gss_get_name(type, id, &p)))
		goto out;
	write_lock(&map->lock);
	if ((q = gss_searchbyid(map, id))) {      /* retry the search, since we blocked */
		atomic_inc(&q->refcount);         /* return a reference */
		write_unlock(&map->lock);
		nfsv4_printk(gss, "gss_get_name: secondary cache hit (race condition)\n");
		gss_free(p);
		*pp = q;
		goto done;
	}
	p->id = id;
	p->name.hash = opaque_hashval(p->name.name, p->name.len);
	gss_insert(p, map);
	atomic_inc(&p->refcount);		  /* return a reference */
	write_unlock(&map->lock);
	*pp = p;
	
done:
	status = 0;
out:
	nfsv4_printk(gss, "gss_get_name: returning status %d\n", status);
	if (!status)
		nfsv4_printk(gss, "gss_get_name: returning id=\"%.*s\"\n",
			     (int)((*pp)->name.len), (*pp)->name.name);
	return status;
}

int
gss_get_num(int type, unsigned int len, const unsigned char *name, uid_t *pp)
{
	struct gss_mapping *map;
	struct gss_cacheent *p, *q;
	unsigned int hash = opaque_hashval(name, len);
	int status;

	nfsv4_printk(gss, "gss_get_num: called with type %s, name=\"%.*s\"\n",
		     gss_opcode(type), (int)len, name);
	
#if NFS4_DEBUG
	if ((type != GSS_OWNER) && (type != GSS_GROUP)) {
		printk("gss_get_name: bad type!\n");
		return -EINVAL;
	}
#endif
	if (len > GSS_NAME_MAX) {
		nfsv4_printk(gss, "gss_get_num: name too long!\n");
		status = -ENAMETOOLONG;
		goto out;
	}
	map = (type == GSS_OWNER) ? &owner_map : &group_map;
	read_lock(&map->lock);
	if ((p = gss_searchbyname(map, len, name, hash))) {
		*pp = p->id;
		read_unlock(&map->lock);
		nfsv4_printk(gss, "gss_get_num: cache hit\n");
		goto done;
	}
	read_unlock(&map->lock);
	nfsv4_printk(gss, "gss_get_num: cache miss\n");
	if (!(p = gss_alloc(len))) {
		nfsv4_printk(gss, "gss_get_num: gss_alloc failed!\n");
		status = -ENOMEM;
		goto out;
	}
	if ((status = __gss_get_num(type, len, name, &p->id)))
		goto out;
	p->name.len = len;
	memcpy((char *)p->name.name, name, len);         /* cast is to get rid of "const" */
	p->name.hash = hash;
	write_lock(&map->lock);
	if ((q = gss_searchbyname(map, len, name, hash))) {   /* blocked, so retry search */
		*pp = q->id;
		write_unlock(&map->lock);
		nfsv4_printk(gss, "gss_get_num: secondary cache hit (race condition)\n");
		gss_free(p);
		goto done;
	}
	gss_insert(p, map);
	*pp = p->id;
	write_unlock(&map->lock);
	
done:
	status = 0;
out:
	nfsv4_printk(gss, "gss_get_num: returning status %d\n", status);
	if (!status)
		nfsv4_printk(gss, "gss_get_num: returning id=%d\n", (int)(*pp));
	return status;
}
#endif
