/*
 *  spkm3/verify.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include<openssl/hmac.h>
#include <openssl/md5.h>
#include "gssapi_spkm3.h"

OM_uint32
spkm3_gss_verify_mic (OM_uint32 * minor_status,
		      gss_ctx_id_t context_handle,
		      gss_buffer_t message_buffer,
		      gss_buffer_t token_buffer,
		      gss_qop_t * qop_state)
{
  spkm3_context_t *ctx;
  SPKM_MIC_t *mic;
  InitialContextToken_t *ict = NULL;
  asn_dec_rval_t rval;
  int datalen = 0, mdlen = 0;
  long tmp;
  unsigned char *dataptr = NULL, *md = NULL;
  OM_uint32 major_status = GSS_S_FAILURE;
  
  *minor_status = 0;
  
  SPKM3_DEBUG(1, ("spkm3_gss_verify_mic: The incoming encoded MIC token:\n"));
  SPKM3_DEBUG_HEX(1, (token_buffer->value, token_buffer->length, 0));
  
  /* validate the context handle */
  if (!kg_validate_ctx_id (context_handle)) {
    *minor_status = GSS_S_G_VALIDATE_FAILED;
    major_status = GSS_S_NO_CONTEXT;
    spkm3_log_status ("spkm3_gss_verify_mic: kg_validate_ctx_id:",
		      major_status, *minor_status);
    goto err;
  }
  ctx = (spkm3_context_t *) context_handle;
  
  if (!ctx->established) {
    *minor_status = GSS_SPKM_S_SG_CTX_INCOMPLETE;
    spkm3_log_status
      ("spkm3_gss_verify_mic: context not established:",
       major_status, *minor_status);
    goto err;
  }
  
  /* decode token */
  rval = ber_decode(0, &asn_DEF_InitialContextToken, (void **)&ict,
		    token_buffer->value, token_buffer->length);
  if (rval.code != RC_OK) {
    *minor_status = GSS_S_G_VALIDATE_FAILED;
    spkm3_log_status
      ("spkm3_gss_verify_mic: failed to decode token:",
       major_status, *minor_status);
    SPKM3_DEBUG_HEX(1, (token_buffer->value, 
			token_buffer->length, 0));
    goto err;
  }
  
  mic = &ict->innerContextToken.choice.mic;
  
  /* token sanity checks */
  
  if ((asn_INTEGER2long (&mic->mic_header.tok_id, &tmp))
      || tmp != SPKM_TOK_SIGN_MSG) {
    major_status = GSS_S_DEFECTIVE_TOKEN;
    spkm3_log_err("spkm3_gss_verify_mic: Incorrect tokid %ld\n",
		  tmp);
    *minor_status = GSS_SPKM_S_SG_INVALID_TOKEN_FORMAT;
    goto err;
  }
  
  /* Our local ctx_id.length should always be SPKM3_CONTEXTID_LEN,
   * but due to bit string encoding, the transferred contextid
   * length may be less because of trailing zeros being
   * truncated.  We need to do the MD5 checksum on the
   * actual transferred data.  Make sure the transferred
   * contextid is not longer than ours and then only compare
   * for the length of the tranferred data.
   * (To be really anal, we could assert that any bytes in our
   * ctx_id that aren't compared must be zero.)
   */
  if ((mic->mic_header.context_id.size < SPKM3_MIN_CONTEXTID_LEN) ||
      (mic->mic_header.context_id.size > ctx->ctx_id.length) ||
      (memcmp (mic->mic_header.context_id.buf, ctx->ctx_id.data,
	       mic->mic_header.context_id.size))) {
    *minor_status = GSS_S_G_BAD_MSG_CTX;
    spkm3_log_status ("spkm3_gss_verify_mic: bad context:",
		      major_status, *minor_status);
    spkm3_log_err ("mic context mismatch\n  mic context:\n");
    SPKM3_DEBUG_HEX(0, (mic->mic_header.context_id.buf,
			mic->mic_header.context_id.size, 0));
    spkm3_log_err("  our context:\n");
    SPKM3_DEBUG_HEX(0, (ctx->ctx_id.data, ctx->ctx_id.length, 0));
    goto err;
  }
  
  /* get the sign algorithm. RFC 2203 states that QOP of 0 
   * is to be used. according to RFC 2025, this means 
   * that the QOP default integrity algorithm is the first
   * one on the initiator's int_alg list, and the default 
   * int_alg list is
   *  [1] md5WithRSA
   *  [2] DES-MAC
   * RFC 2847 (spkm3/lipkey) states that for spkm3, since there
   * is potentially no initiator credentials, the md5WithRSA is
   * relaxed to md5. 
   * SO! pant pant, we will use what krb5 uses, md5 !!!
   */
  
  /* must use default algorithm */
#if 0
  if (!((mic->mic_header.int_alg == NULL) ||
	((nid = get_nid_from_asn1_alg(mic->mic_header.int_alg)) ==
	 NID_md5))) {
    if (!((mic->mic_header.int_alg == NULL) ||
	  ((nid = get_nid_from_asn1_alg(mic->mic_header.int_alg)) ==
	   NID_hmac_md5))) {
      
      spkm3_log_err("spkm3_gss_verify_mic: intalg not md5. nid: %d\n",
		    nid);
      *minor_status = GSS_SPKM_S_SG_BAD_INT_ALG_SET;
      spkm3_log_status ("spkm3_gss_verify_mic: intalg not md5:",
			major_status, *minor_status);
      goto err;
    }
  }
 #endif
   
  /* return an indication of the qop applied */
  *qop_state = 0;

  /* XXX need to check that ctx->intg_alg == mic->micheader->intgalg.. */
  if(g_OID_equal(ctx->intg_alg, &null_mac_oid)) {
    SPKM3_DEBUG(1, ("spkm3_gss_get_mic: null-mac. nothing to do.\n"));
  } else if(g_OID_equal(ctx->intg_alg, &hmac_md5_oid)) {
    /* compute the checksum of the message. */
    
    /* The checksum length may be smaller than expected because it
     * has trailing zeros which are truncated in the bit string
     * encoding.  Just make sure that it isn't longer than we
     * are expecting.
     */
    if ((mic->int_cksum.size < SPKM3_MIN_MD5_DIGEST_LENGTH) ||
	(mic->int_cksum.size > MD5_DIGEST_LENGTH)) {
      *minor_status = GSS_S_G_WRONG_SIZE;
      spkm3_log_status
	("spkm3_gss_verify_mic: Bad checksum length:",
	 major_status, *minor_status);
      spkm3_log_err("spkm3_gss_verify_mic: Bad checksum length:\n");
      SPKM3_DEBUG_HEX(0, (mic->int_cksum.buf, 
			  mic->int_cksum.size, 0));
      goto err;
    }
    
    if((*minor_status = encode(&asn_DEF_Mic_Header, &mic->mic_header, 
			       &dataptr, &datalen)) != 0)
      goto err;
    
    SPKM3_DEBUG(1, ("spkm3_gss_get_mic: About to do HMAC-MD5 on %d bytes\n",
		    datalen));
    
    if((md = (unsigned char *)malloc(MD5_DIGEST_LENGTH)) == NULL) {
      *minor_status = GSS_S_G_MEMORY_ALLOC;
      goto err;
    }
    
    HMAC(EVP_md5(), ctx->derived_integ_key.value, 
	 ctx->derived_integ_key.length, dataptr, datalen, md, &mdlen);
    
    SPKM3_DEBUG (1, ("md\n"));
    SPKM3_DEBUG_HEX(1, ((unsigned char *) md, mdlen, 0));
    
    
    /* (To be really anal, we could assert that any bytes in our
     * digest that aren't compared must be zero.)
     */
    if (memcmp (md, mic->int_cksum.buf, mic->int_cksum.size)) {
      spkm3_log_err("spkm3_gss_verify_mic: md5 checksum mismatch\n");
      SPKM3_DEBUG_HEX(1, ((unsigned char *) md, mdlen, 0));
      *minor_status = GSS_S_G_WRONG_SIZE;	/* XXX */
      goto err;
    }
  } else if(g_OID_equal(ctx->intg_alg, &md5_rsa_encryption_oid)) {
    *minor_status = verify(ctx->intg_alg, &asn_DEF_Mic_Header, 
			   &mic->mic_header, NULL, -1, ctx->usercert,
			   NULL, mic->int_cksum.buf, mic->int_cksum.size);
  } else {
    *minor_status = -1;
    goto err;
  }
  
  major_status = GSS_S_COMPLETE;
  
 err:
  if(dataptr) free (dataptr);
  if(md) free(md);
  
  SPKM3_DEBUG(1, ("<--- spkm3_gss_verify_mic: "
		  "message_buffer(value %p length %d) "
		  "token_buffer(value %p length %d)\n",
		  message_buffer->value, message_buffer->length,
		  token_buffer->value, token_buffer->length));
  
  if(major_status != GSS_S_COMPLETE)
    spkm3_log_status ("spkm3_gss_verify_mic: returning:",
		      major_status, *minor_status);
  
  asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					  ict, 0);
  
  return major_status;
}

OM_uint32
spkm3_gss_verify (
			 OM_uint32 * min_status,
			 gss_ctx_id_t ctx_hndl,
			 gss_buffer_t msg_buf,
			 gss_buffer_t token_buf, int *qop_st)
{
  return (spkm3_gss_verify_mic
	  (min_status, ctx_hndl, msg_buf, token_buf, qop_st));
}
