/*
 *  spkm3/util.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <syslog.h>
#include "gssapi_spkm3.h"
#include "spkm3_mech.h"

void
print_hex (unsigned char *data, int len)
{
        unsigned char *p = data;
        int i;
        fprintf (stderr, "len: %5d    ", len);
        for (i = 0; i < len; i++) {
                fprintf (stderr, "%02x", *p++);
        }
        fprintf (stderr, "\n");
}

void
print_rand (unsigned char *r, int len)
{
	SPKM3_DEBUG(1, ("RANDOM NUM:\n"));
	SPKM3_DEBUG_HEX(1, (r, len, 0));

#if 0
	int i;
	for (i=0; i < len; i++) 
		fprintf (stderr, "%02x", r[i]);
	fprintf(stderr, "\n");
#endif
}

void
print_dh_sharekey (gss_buffer_desc * key)
{
	SPKM3_DEBUG (1, ("DH_SHARE_KEY:   \n"));
	SPKM3_DEBUG_HEX (1, (key->value, key->length, 0));
}

void
print_name_desc (spkm3_name_desc_t * name)
{
	int i;
	char *p = name->string_name;

	for (i = 0; i < name->length; i++)
		printf ("%c", *p++);
	printf ("\n");
}

void
print_x509_name (X509_NAME * xn)
{
	int i, j;
	STACK *sk;
	X509_NAME_ENTRY *ent = NULL;
	char *p;

	sk = xn->entries;
	for (j = 0; j < sk_num (sk); j++) {
		ent = (X509_NAME_ENTRY *) sk_value (sk, j);
		p = (char *) ent->value->data;
		for (i = 0; i < ent->value->length; i++)
			fprintf (stderr, "%c", *p++);
		fprintf(stderr, "\n");
	}
}
int
get_random_bytes (unsigned char *buffer, int n, int critical)
{
	unsigned char *p = buffer;
	int fd;
	int i;
	char *filename;

	if (n < 0)
		return -1;

	if (critical)
		filename = "/dev/random";
	else
		filename = "/dev/urandom";

	fd = open (filename, O_RDONLY);
	if (fd < 0) {
		perror ("opening random device");
		return -1;
	}

	for (i = 0; i < n; i++) {
		if (read (fd, p++, 1) != 1) {
			close (fd);
			return -1;
		}
	}

	close (fd);

	return i;
}

void
spkm3_log_status (char *m, OM_uint32 maj_stat, OM_uint32 min_stat)
{
	OM_uint32 min;
	gss_buffer_desc msg;
	int msg_ctx = 0;

	spkm3_gss_display_status (&min, maj_stat, GSS_C_GSS_CODE,
				  GSS_C_NULL_OID, &msg_ctx, &msg);
	spkm3_log_err("%*s - ", msg.length, (char *) msg.value);
	generic_gss_release_buffer (&min, &msg);

	spkm3_gss_display_status (&min, min_stat, GSS_C_MECH_CODE,
				  GSS_C_NULL_OID, &msg_ctx, &msg);
	spkm3_log_err("%*s\n", msg.length, (char *) msg.value);
	generic_gss_release_buffer (&min, &msg);
}


/*
 * XXX This needs to be be done better, but for now it'll work XXX
 */
int
spkm3_get_filename(spkm3_filetype_t filetype, spkm3_usertype_t usertype,
		   char *name, int namelen)
{
	if (name == NULL || namelen <= 0)
		return EINVAL;

	if (usertype == SPKM3_USERTYPE_SYSTEM) {
		if (filetype == SPKM3_FILETYPE_PRIV_KEY) {
			strncpy(name, KeyFile, namelen);
		} else if (filetype == SPKM3_FILETYPE_CERT) {
			strncpy(name, CertFile, namelen);
		} else {
			return EINVAL;
		}
	} else if (usertype == SPKM3_USERTYPE_USER) {
		if (filetype == SPKM3_FILETYPE_PRIV_KEY ||
		    filetype == SPKM3_FILETYPE_CERT) {
			snprintf(name, namelen, "/tmp/x509up_u%d", geteuid());
		} else
			return EINVAL;
	} else {
		return EINVAL;
	}
	return 0;
}

/*
 * This routine "repairs" a bit string that has been truncated because
 * of its ASN.1 encoding.  The BIT_STRING type is encoded such
 * that trailing zeros are not sent.  This is only an issue if an
 * entire byte at the end of the string is zero.  Then the entire byte
 * is not sent.
 * Returns 1 on success and 0 on failure.
 */
int
spkm3_bit_string_repair(BIT_STRING_t *bs, int newlen)
{
	int retval = 0;		/* assume failure */
	uint8_t *newdata;

	if (bs == NULL || newlen <= 0)
		goto out;
	SPKM3_DEBUG (1, ("spkm3_bit_string_repair: (before)\n"));
	SPKM3_DEBUG_HEX(1, (bs->buf, bs->size, 0));

	newdata = (uint8_t *) calloc(1, newlen);
	if (newdata == NULL)
		goto out;
	memcpy(newdata, bs->buf, bs->size);
	free(bs->buf);
	bs->buf = newdata;
	bs->size = newlen;

	SPKM3_DEBUG (1, ("spkm3_bit_string_repair: (after)\n"));
	SPKM3_DEBUG_HEX(1, (bs->buf, bs->size, 0));

	retval = 1;
out:
	return retval;
}

/*
 * Given an ascii string and it's length, parse it into an X509_NAME
 * structure. Returns a pointer to the X509_NAME structure, or NULL
 * if the string could not be parsed.
 *
 * XXX Currnetly handles escaping of commas, do we need to handle
 * escaping of other characters?
 */

X509_NAME *
a2i_X509_NAME (const char *ascii_string, int length)
{
	X509_NAME *nm = NULL;
	char *n, *e;		/* Index, end */
	char *copy = NULL;	/* Copy of input string we can change */
	int lastone = 0;


	if (length <= 0 || length > 4096)	/* XXX arbitrary range */
		goto err;

	copy = (char *) malloc (length + 1);
	if (copy == NULL)
		goto err;

	memcpy (copy, ascii_string, length);
	copy[length] = '\0';	/* Assure string is null-terminated */

	nm = X509_NAME_new ();
	if (!nm)
		goto err;

	n = copy;
	if ((e = strchr (n, '\n')))
		*e = '\0';
	else
		e = n + strlen (n);

	do {
		char *t;	/* type */
		char *v;	/* value */
		char *c;	/* comma */
		char *s;	/* space */
		char *p;	/* escape */
		int escaping;

		t = n;
		while (isspace (*t))
			t++;	/* skip leading white space */
		v = strchr (t, '=');
		if (v) {
			s = v - 1;	/* keep track of end */
			*v++ = '\0';
			while (isspace (*s))
				*s-- = '\0';	/* trim trailing spaces */
			while (isspace (*v))
				v++;	/* skip leading white space */
			c = strchr (v, ',');
			do {
				escaping = 0;
				if (c) {
					p = c - 1;
					if (*p == '\\') {
						escaping = 1;
						c = strchr (c + 1, ',');
					}
				}
			} while (escaping);
			if (c) {
				*c = '\0';
				s = c - 1;
			}
			else {
				lastone = 1;
				s = e - 1;
			}
			while (isspace (*s))
				*s-- = '\0';	/* trim trailing spaces */
			if (!X509_NAME_add_entry_by_txt (nm, t, MBSTRING_ASC,
							 v, -1, -1, 0))
				goto err;
		}
		else {
			/*
			 * No '=' in the string, assume it is a simple name and
			 * add it as a "CN" value
			 */
			if (!X509_NAME_add_entry_by_txt
			    (nm, "CN", MBSTRING_ASC, t, -1, -1, 0))
				goto err;
			lastone = 1;
		}
		n = ++c;
	} while (!lastone && n < e);

	free (copy);

	return (nm);

      err:
	if (copy)
		free (copy);
	if (nm)
		X509_NAME_free (nm);
	return (NULL);
}


/*
 * Return a string format of an X509_NAME in buf where
 * size is an in/out parameter.  On input it is the size
 * of the buffer, and on output it is the actual length
 * of the name.
 * If buf is NULL, returns the length req'd to hold name
 */
char *
X509_NAME_oneline_ex (X509_NAME * a, char *buf, int *size, unsigned long flag)
{
	BIO *out = NULL;

	out = BIO_new (BIO_s_mem ());
	if (X509_NAME_print_ex (out, a, 0, flag) > 0) {
		if (buf != NULL && *size > (int) BIO_number_written (out)) {
			memset (buf, 0, *size);
			BIO_read (out, buf, BIO_number_written (out));
		}
		else {
			*size = BIO_number_written (out);
		}
	}
	BIO_free (out);
	return (buf);
}

/*
 * Returns a character format of an X509_NAME in the "old" format
 */
char *
X509_NAME_oneline_readable (X509_NAME * a, char *buf, int *size)
{
	return X509_NAME_oneline_ex (a, buf, size, XN_FLAG_ONELINE);
}

/*
 * Returns a character format of an X509_NAME in rfc2253 format
 */
char *
X509_NAME_oneline_rfc2253 (X509_NAME * a, char *buf, int *size)
{
	return X509_NAME_oneline_ex (a, buf, size, XN_FLAG_SEP_COMMA_PLUS);
}

/*
 * Returns a character format of an X509_NAME in rfc2253 format
 * (with the names in reverse order)
 */
char *
X509_NAME_oneline_rfc2253_rev (X509_NAME * a, char *buf, int *size)
{
	return X509_NAME_oneline_ex (a, buf, size,
				     XN_FLAG_SEP_COMMA_PLUS | XN_FLAG_DN_REV);
}

/*
 * Utility function to get CN entry from an X509_NAME (DN)
 */
X509_NAME_ENTRY *
X509_NAME_get_cn (X509_NAME * dn)
{
	int cn_pos = -1;
	X509_NAME_ENTRY *cn = NULL;

	if (dn == NULL)
		return NULL;

	cn_pos = X509_NAME_get_index_by_NID (dn, OBJ_sn2nid ("CN"), -1);
	if (cn_pos == -1)
		return NULL;

	return X509_NAME_get_entry (dn, cn_pos);
}

/*
 * Given a target name CN entry, see if it matches
 * the CN entry in the given DN
 */
int
commonNamesMatch (X509_NAME_ENTRY * targCN, X509_NAME * certDN)
{
	X509_NAME_ENTRY *certCN;

	certCN = X509_NAME_get_cn (certDN);
	if (certCN == NULL)
		return 0;

	SPKM3_DEBUG (1, ("commonNamesMatch: targCN '%*s' certCN '%*s'\n",
		     targCN->value->length, targCN->value->data,
		     certCN->value->length, certCN->value->data));

	if ((targCN->value->length == certCN->value->length) &&
	    (strncmp (targCN->value->data, certCN->value->data,
		      targCN->value->length) == 0))
		return 1;

	return 0;
}

/*
 * Given a hostname and length, and an ASN1_IA5STRING, see if they match
 */
int
hostNamesMatch (char *hostname, int len, ASN1_IA5STRING * dnsName)
{
	char *dnsstring = NULL;
	int retval = 0;

	if (!dnsName || !dnsName->length)
		return 0;

	SPKM3_DEBUG (1, ("hostNamesMatch: hostname '%*s' dnsName '%*s'\n",
		     len, hostname, dnsName->length, dnsName->data));

	if ((len == dnsName->length) &&
	    (strncmp (hostname, dnsName->data, len) == 0)) {
		return 1;
	}

	return 0;
}

/*
 * Verify target name in request matches certificate name.
 * The rules agreed upon with IBM and others at the Austin 
 * October 2003 Bake-athon are:
 *
 * a) A match occurs if the target name is the same as the
 *    subject name or a DN entry for the subject alternate
 *    name (subjectAltName) in the target certificate.
 * b) A match occurs if the target name is just a common name
 *    and the common name matches the CN portion of the subject
 *    name or the CN portion of a DN entry for the subject
 *    alternate name (subjectAltNAme) in the target certificate.
 * c) A match occurs if the target name is a service name
 *    ("CN=service/host") and the host part matches the CN
 *    portion of the subject name or a DNS entry for the subject
 *    alternate name (subjectAltName) in the target certificate.
 *
 */
#if 0 
int
spkm3int_targetname_matches_cert (Name_t *target, Certificate_t *cert)
{
	int target_just_cn = 0;
	int target_cn_pos = 0;
	int target_is_service = 0;
	int subject_alt_pos = -1;
	int retval = 0;
	int i;
	int namesize;
	Name_t *cert_subject = &cert->certContents.subject;
	X509_NAME_ENTRY *targCN = NULL;
	STACK_OF (X509_EXTENSION) * subjAltNames;
	X509_EXTENSION *subjAltName;
	GENERAL_NAMES *gns;
	GENERAL_NAME *gn;
	char namebuff[2048];
	char *targHN = NULL;
	int targHNlen = 0;

	SPKM3_DEBUG (1, ("spkm3int_targetname_matches_cert entered\n"));

	/* First see if the target simply matches the subject name of the cert */
	if (X509_NAME_cmp (target, cert_subject) == 0) {
		/* That was too easy! */
#ifdef DEBUG
		fprintf (stderr,
			 "spkm3int_targetname_matches_cert: "
			 "target DN matches cert subject DN\n");
#endif
		retval = 1;
		goto exit;
	}

	/* Determine if target consists of only a commonName */
	if ((X509_NAME_entry_count (target) == 1) &&
	    ((target_cn_pos = X509_NAME_get_index_by_NID (target,
							  OBJ_sn2nid ("CN"),
							  -1)) != -1)) {
		targCN = X509_NAME_get_entry (target, target_cn_pos);
		target_just_cn = 1;
	}

	/* If target is only a CN and the common names match, we're done */
	if ((target_just_cn) && (commonNamesMatch (targCN, cert_subject))) {
#ifdef DEBUG
		fprintf (stderr,
			 "spkm3int_targetname_matches_cert: "
			 "target is just CN and matches cert subject CN\n");
#endif
		retval = 1;
		goto exit;
	}

	/*
	 * If target is a service name ("CN=service/fq.host.name") and
	 * the 'fq.host.name' matches the subject's CN (which must be 
	 * only a hostname) then we'll call it a match
	 */
	for (i = 0; i < targCN->value->length; i++) {
		if (targCN->value->data[i] == '/') {
			i++;
			targHN = &targCN->value->data[i];
			targHNlen = targCN->value->length - i;
			if (targHNlen > 0)
				target_is_service = 1;
			break;
		}
	}
	if (target_is_service) {
		X509_NAME_ENTRY *subjCN;

		if ((subjCN = X509_NAME_get_cn (cert_subject)) != NULL) {
			if ((targHNlen == subjCN->value->length) &&
			    (strncmp (targHN, subjCN->value->data, targHNlen)
			     == 0)) {
#ifdef DEBUG
				fprintf (stderr,
					 "spkm3int_targetname_matches_cert: "
					 "target is service name and hostname matches "
					 "cert subject CN\n");
#endif
				retval = 1;
				goto exit;
			}
		}
	}

	/* Otherwise, look through all subjectAltName entries for a match */
	while ((subject_alt_pos = X509_get_ext_by_NID (cert,
						       OBJ_sn2nid
						       ("subjectAltName"),
						       subject_alt_pos)) >=
	       0) {
#ifdef DEBUG
		fprintf (stderr,
			 "spkm3int_targetname_matches_cert: checking subjAltName %d\n",
			 subject_alt_pos);
#endif
		subjAltName = X509_get_ext (cert, subject_alt_pos);
		gns = (GENERAL_NAMES *) X509V3_EXT_d2i (subjAltName);

		for (i = 0; i < sk_GENERAL_NAME_num (gns); i++) {
			gn = sk_GENERAL_NAME_value (gns, i);

			/*
			 * If subjectAltName is a DN, then check if it matches target
			 * exactly.  Otherwise, if target is only a CN, see if it
			 * matches the CN part of the subjectAltName DN
			 */
			if (gn->type == GEN_DIRNAME) {
#ifdef DEBUG
				namesize = sizeof (namebuff);
				X509_NAME_oneline_rfc2253 (gn->d.
							   directoryName,
							   namebuff,
							   &namesize);
				fprintf (stderr,
					 "spkm3int_targetname_matches_cert: subjAltName %d: '%s'\n",
					 i, namebuff);
#endif
				if (X509_NAME_cmp
				    (target, gn->d.directoryName) == 0) {
#ifdef DEBUG
					fprintf (stderr,
						 "spkm3int_targetname_matches_cert: "
						 "target DN matches subjectAltName DN\n");
#endif
					retval = 1;
					goto exit;
				}
				else if ((target_just_cn) &&
					 (commonNamesMatch
					  (targCN, gn->d.directoryName))) {
#ifdef DEBUG
					fprintf (stderr,
						 "spkm3int_targetname_matches_cert: "
						 "target (only CN) matches subjectAltName CN\n");
#endif
					retval = 1;
					goto exit;
				}
			}
			/*
			 * If subjectAltName is a DNS name, and target is a service
			 * name, and the hostname part of the target's service name
			 * matches the DNS name, then we'll call it a match.
			 */
			else if ((gn->type == GEN_DNS) &&
				 (target_is_service) &&
				 (hostNamesMatch
				  (targHN, targHNlen, gn->d.dNSName))) {
#ifdef DEBUG
				fprintf (stderr,
					 "spkm3int_targetname_matches_cert: "
					 "target (service name hostname) matches "
					 "subjectAltName DNSname\n");
#endif
				retval = 1;
				goto exit;
			}
		}
	}
      exit:
#ifdef DEBUG
	fprintf (stderr, "spkm3int_targetname_matches_cert: returning %d\n",
		 retval);
#endif
	return (retval);
}
#else
int
targetname_matches_cert (Name_t *target, Certificate_t *cert)
{
	int target_just_cn = 0;
	int target_cn_pos = 0;
	int target_is_service = 0;
	int subject_alt_pos = -1;
	int retval = 0;
	int i;
	int namesize;
	Name_t *cert_subject = &cert->tbsCertificate.subject;
	X509_NAME_ENTRY *targCN = NULL;
	STACK_OF (X509_EXTENSION) * subjAltNames;
	X509_EXTENSION *subjAltName;
	GENERAL_NAMES *gns;
	GENERAL_NAME *gn;
	char namebuff[2048];
	char *targHN = NULL;
	int targHNlen = 0;

#ifdef DEBUG
	fprintf (stderr, "targetname_matches_cert entered\n");
#endif

	/* First see if the target simply matches the subject name of the cert */
	if (X509_NAME_cmp (target, cert_subject) == 0) {
		/* That was too easy! */
#ifdef DEBUG
		fprintf (stderr,
			 "targetname_matches_cert: "
			 "target DN matches cert subject DN\n");
#endif
		retval = 1;
		goto exit;
	}

	/* Determine if target consists of only a commonName */
	if ((X509_NAME_entry_count (target) == 1) &&
	    ((target_cn_pos = X509_NAME_get_index_by_NID (target,
							  OBJ_sn2nid ("CN"),
							  -1)) != -1)) {
		targCN = X509_NAME_get_entry (target, target_cn_pos);
		target_just_cn = 1;
	}

	/* If target is only a CN and the common names match, we're done */
	if ((target_just_cn) && (commonNamesMatch (targCN, cert_subject))) {
#ifdef DEBUG
		fprintf (stderr,
			 "targetname_matches_cert: "
			 "target is just CN and matches cert subject CN\n");
#endif
		retval = 1;
		goto exit;
	}

	/*
	 * If target is a service name ("CN=service/fq.host.name") and
	 * the 'fq.host.name' matches the subject's CN (which must be 
	 * only a hostname) then we'll call it a match
	 */
	for (i = 0; i < targCN->value->length; i++) {
		if (targCN->value->data[i] == '/') {
			i++;
			targHN = &targCN->value->data[i];
			targHNlen = targCN->value->length - i;
			if (targHNlen > 0)
				target_is_service = 1;
			break;
		}
	}
	if (target_is_service) {
		X509_NAME_ENTRY *subjCN;

		if ((subjCN = X509_NAME_get_cn (cert_subject)) != NULL) {
			if ((targHNlen == subjCN->value->length) &&
			    (strncmp (targHN, subjCN->value->data, targHNlen)
			     == 0)) {
#ifdef DEBUG
				fprintf (stderr,
					 "targetname_matches_cert: "
					 "target is service name and hostname matches "
					 "cert subject CN\n");
#endif
				retval = 1;
				goto exit;
			}
		}
	}

	/* Otherwise, look through all subjectAltName entries for a match */
	while ((subject_alt_pos = X509_get_ext_by_NID (cert,
						       OBJ_sn2nid
						       ("subjectAltName"),
						       subject_alt_pos)) >=
	       0) {
#ifdef DEBUG
		fprintf (stderr,
			 "targetname_matches_cert: checking subjAltName %d\n",
			 subject_alt_pos);
#endif
		subjAltName = X509_get_ext (cert, subject_alt_pos);
		gns = (GENERAL_NAMES *) X509V3_EXT_d2i (subjAltName);

		for (i = 0; i < sk_GENERAL_NAME_num (gns); i++) {
			gn = sk_GENERAL_NAME_value (gns, i);

			/*
			 * If subjectAltName is a DN, then check if it matches target
			 * exactly.  Otherwise, if target is only a CN, see if it
			 * matches the CN part of the subjectAltName DN
			 */
			if (gn->type == GEN_DIRNAME) {
#ifdef DEBUG
				namesize = sizeof (namebuff);
				X509_NAME_oneline_rfc2253 (gn->d.
							   directoryName,
							   namebuff,
							   &namesize);
				fprintf (stderr,
					 "targetname_matches_cert: subjAltName %d: '%s'\n",
					 i, namebuff);
#endif
				if (X509_NAME_cmp
				    (target, gn->d.directoryName) == 0) {
#ifdef DEBUG
					fprintf (stderr,
						 "targetname_matches_cert: "
						 "target DN matches subjectAltName DN\n");
#endif
					retval = 1;
					goto exit;
				}
				else if ((target_just_cn) &&
					 (commonNamesMatch
					  (targCN, gn->d.directoryName))) {
#ifdef DEBUG
					fprintf (stderr,
						 "targetname_matches_cert: "
						 "target (only CN) matches subjectAltName CN\n");
#endif
					retval = 1;
					goto exit;
				}
			}
			/*
			 * If subjectAltName is a DNS name, and target is a service
			 * name, and the hostname part of the target's service name
			 * matches the DNS name, then we'll call it a match.
			 */
			else if ((gn->type == GEN_DNS) &&
				 (target_is_service) &&
				 (hostNamesMatch
				  (targHN, targHNlen, gn->d.dNSName))) {
#ifdef DEBUG
				fprintf (stderr,
					 "targetname_matches_cert: "
					 "target (service name hostname) matches "
					 "subjectAltName DNSname\n");
#endif
				retval = 1;
				goto exit;
			}
		}
	}
      exit:
#ifdef DEBUG
	fprintf (stderr, "targetname_matches_cert: returning %d\n",
		 retval);
#endif
	return (retval);
}
#endif /* 0 */


/*
 * spkm3_check_anon_srcname()
 *
 * the SPKM3 anonymous user is indicated either by a NULL in_name, 
 * or an in_name with one entry containing the SPKM3_ANONYMOUS_STRING.
 *
 */
OM_uint32
spkm3_check_anon_srcname(Name_t *in_name)
{
	RelativeDistinguishedName_t *rdn;
	AttributeTypeAndValue_t *attr;
	AttributeType_t *atype;
	asn_set *as_rdn, *as_attr;

	if (!in_name)
		return 0;
	as_rdn = (asn_set *)&in_name->choice.rdnSequence.list;
	if (as_rdn->count != 1)
		return GSS_S_BAD_NAME;

	rdn = (RelativeDistinguishedName_t *)as_rdn->array[0];  
	as_attr = (asn_set *)&rdn->list;
	attr = (AttributeTypeAndValue_t *)as_attr->array[0];
	atype = &attr->type;

	if ((atype->size != strlen (SPKM3_ANONYMOUS_STRING)) || 
	    (memcmp ((char *) atype->buf, SPKM3_ANONYMOUS_STRING, atype->size)))
		return GSS_S_BAD_NAME;

	return 0;
}
/* 
 * spkm3_set_srcname()
 * 
 * given a Req_contents->srcname (RDNSequence) , create an internal name 
 * representation  suitable for spkm3_gss_display_name. 
 *
 * called by spkm3_accecpt_sec_context() to set the returned src_name.
 *
 */
OM_uint32
spkm3_set_srcname (gss_name_t * src_name, Name_t *in_name, int req_flags)
{
	OM_uint32 min_status = GSS_S_BAD_NAME;
	RelativeDistinguishedName_t *rdn;
	AttributeTypeAndValue_t *attr;
	AttributeType_t *atype;
	asn_set *as_rdn;
	asn_set *as_attr;
	X509_NAME_ENTRY *ent = NULL;
	spkm3_name_desc_t *cn;
	int i;
	gss_buffer_desc name;
	gss_OID oid;

	if (in_name && in_name->present == Name_PR_NOTHING)
			goto out;

	if (req_flags & GSS_C_ANON_FLAG) {

		if (spkm3_check_anon_srcname(in_name) == 0) {
			min_status = spkm3_gss_import_name (&min_status, NULL, 
						GSS_C_NT_ANONYMOUS, 
						src_name);
		}

	} else if (req_flags & GSS_C_MUTUAL_FLAG) {

		if ((cn = calloc (1, sizeof(spkm3_name_desc_t))) == NULL)
			goto out;
		min_status = asn1_name_to_string(in_name, cn);
		if (min_status)
			goto out_free_cn;
		if ((cn->asn1_name = asn1_name_dup(in_name)) == NULL) {
			min_status = GSS_S_G_MEMORY_ALLOC;
			goto out_free_cn_string;
		}
		cn->type = SPKM3_NT_USER_NAME;
		cn->oid = GSS_C_NT_USER_NAME;
		*src_name = cn;
	}
out:
	return min_status;

out_free_cn_string:
	free (cn->string_name);
out_free_cn:
	free (cn);
	goto out;
}

OM_uint32 encode(struct asn_TYPE_descriptor_s *asn_type,
                 void *req_contents,
                 unsigned char **der_contents,
                 size_t *der_len) {
                                                                                
        OM_uint32 minor_status = 0;
        asn_enc_rval_t ec;
        struct spkm3_ebuf seb;
                                                                                
        SPKM3_DEBUG(1, ("encode: START\n"));
        *der_contents = NULL;
                                                                                
        /* DER-encode the contents of the token */
        ec = der_encode (asn_type, req_contents, NULL, NULL);
        if(ec.encoded <= 0) {
                spkm3_log_err("encode: failed to encode reqcontents\n");
                minor_status = EINVAL;
                goto err;
        }
                                                                                
        if ((seb.buf = malloc(ec.encoded)) == NULL) {
                minor_status = GSS_S_G_MEMORY_ALLOC;
                goto err;
        }
                                                                                
        seb.cur = seb.buf;
        ec = der_encode(asn_type, req_contents, spkm3_encode_to_outbuf, &seb);
        if(ec.encoded <= 0) {
                spkm3_log_err("encode: failed to encode reqcontents\n");
                minor_status = EINVAL;
                goto err_buf;
        }
          
        SPKM3_DEBUG(1, ("encode: token of length = %d\n", ec.encoded));
#ifdef DEBUG_ENCODE
        SPKM3_DEBUG_HEX(1, (seb.buf, ec.encoded, 0));
#endif                                                                                
        if((*der_contents = (unsigned char *)malloc(ec.encoded)) == NULL) {
                minor_status = GSS_S_G_MEMORY_ALLOC;
                goto err_buf;
        }
        memcpy(*der_contents, seb.buf, ec.encoded);
        *der_len = ec.encoded;

#ifdef ASNDEBUG
	asn_fprint(stderr, asn_type, req_contents);
#endif
	
 err_buf:
        if(seb.buf) free(seb.buf);
 err:                                                                               
        SPKM3_DEBUG(1, ("encode: END\n"));
        return minor_status;
}

/* Logging */
void spkm3_log_err(char *format, ... ) {
        va_list vap;

        va_start(vap, format);
#ifdef DEBUG
	vfprintf(stderr, format, vap); 
#else
        vsyslog(LOG_ERR, format, vap);
#endif
        va_end(vap);
}

void spkm3_log_dbg(char *format, ... ) {
        va_list vap;

        va_start(vap, format);
#ifdef DEBUG
        vfprintf(stderr, format, vap);

#else
        vsyslog(LOG_INFO, format, vap);
#endif
        va_end(vap);
}

void spkm3_log_hexdump(const u_char *buf, int len, int offset)
{
        u_int i, j, jm;
        int c;
        char msgbuff[256];
        char *m = msgbuff;
        int written;

	if(buf == NULL || len == 0) return;

        for (i = 0; i < len; i += 0x10) {
                written = sprintf(m, "\t%04x: ", (u_int)(i + offset));
                m += written;
                jm = len - i;
                jm = jm > 16 ? 16 : jm;

                for (j = 0; j < jm; j++) {
                        if ((j % 2) == 1)
                                written = sprintf(m, "%02x ", (u_int) buf[i+j]);                        else
                                written = sprintf(m, "%02x", (u_int) buf[i+j]);
                        m += written;
                }
                for (; j < 16; j++) {
                        if ((j % 2) == 1)
                                written = sprintf(m, "\t ");
                        else
                                written = sprintf(m, "\t");
                        m += written;
                }
                sprintf(m, " ");
                m++;

		if(spkm3_debug_level > 2) {
                	for (j = 0; j < jm; j++) {
                        	c = buf[i+j];
                        	c = isprint(c) ? c : '.';
                        	sprintf(m, "%c", c);
                        	m++;
                	}
		}
                sprintf(m, "\n");

                spkm3_log_dbg("%s", msgbuff);
                m = msgbuff;
                memset(msgbuff, '\0', sizeof(msgbuff));
        }
}
