/*
 *  spkm3/spkm3_util_crypt.c
 *
 *  Copyright (c) 2002,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <openssl/sha.h>
#include <openssl/md5.h>
#include "gssapiP_generic.h"
#include "gssapi_spkm3.h"

/* Function: spkm3_derive_subkey()
 * 
 * Description: called upon context establishment. use the strongest
 * agreed upon conf_alg and integ_alg's, derive the sub keys as 
 * described, storing the results in the spkm3_gss_ctx_id_t
 * 
 * derive the subkey from the context_key as follows
 * (from rfc 2025, and replicated in rfc 2827)
 * rightmost_k_bits (OWF(context_key || x || n || s || context_key))
 * 
 * x: ASCII char "C" (0x43) if subkey for Conf alg, "I" (0x49) for integ alg
 * n: ASCII char  for position in algorithm list 0,1,2 .. (0x30,0x31,0x32...)
 * s: ASCII char for stage  0,1,2 .. (0x30,0x31,0x32...)
 *   where stage is always 0 unless k > OWF output size, in which case
 *   the OWF is computed repeated times with increasing stages, 
 *   concatenating the result to the result of the previous OWF.
 */

OM_uint32 spkm3_derive_subkey (int type,	/* 0 = conf, 1 = integ */
			       spkm3_context_t * ctx)
{
  OM_uint32 minor_status = 0;
  int x, n, s, bufsize = 0;
  unsigned char *buf = NULL, *p = NULL;
  gss_buffer_desc digest;
  
  ERR_load_crypto_strings ();

  SPKM3_DEBUG(1, ("spkm3_derive_subkey: START\n"));

  assert(ctx->share_key.value != NULL);

  bufsize = 2*ctx->share_key.length + 3*sizeof(int);
  if((p = buf = malloc(bufsize)) == NULL) {
    minor_status =  GSS_S_G_MEMORY_ALLOC;
    goto err;
  }
  memcpy(p, ctx->share_key.value, ctx->share_key.length);
  p += ctx->share_key.length;

  /* set x */
  if (type == 0)
    x = 0x43;
  else
    x = 0x49;
  memcpy(p, &x, sizeof(int));
  p += sizeof(int);
  
  /* check the qop to set n */
  /* set n */
  n = 0x30;
  memcpy(p, &n, sizeof(int));
  p += sizeof(int);
  
  /* set s */
  s =  0x30;
  memcpy(p, &s, sizeof(int));
  p += sizeof(int);
  
  memcpy(p, ctx->share_key.value, ctx->share_key.length);
  
  if(g_OID_equal(ctx->owf_alg, &sha1_oid)) {
    SHA_CTX sha1ctx;
    
    if((digest.value = (unsigned char *)malloc(SHA_DIGEST_LENGTH)) == NULL) {
      minor_status =  GSS_S_G_MEMORY_ALLOC;
      goto err;
    }
    digest.length = SHA_DIGEST_LENGTH;
    
    SHA1_Init(&sha1ctx);
    SHA1_Update(&sha1ctx, buf, bufsize);
    SHA1_Final(&(((unsigned char *)digest.value)[0]), &sha1ctx);
    
    SPKM3_DEBUG (1, ("digest\n"));
    SPKM3_DEBUG_HEX(1, ((unsigned char *) digest.value, 
			SHA_DIGEST_LENGTH, 0));
  } else {
    minor_status = -1;
    goto err;
  }
  
  if(type == 0) {
    int i = 0;

    if(g_OID_equal(ctx->conf_alg, &cast5_cbc_oid)) {
      /* need to get 128bits */
      if((ctx->derived_conf_key.value = 
	  malloc(MD5_DIGEST_LENGTH)) == NULL) {
	minor_status =  GSS_S_G_MEMORY_ALLOC;
	goto err;
      }
      ctx->derived_conf_key.length = MD5_DIGEST_LENGTH;
      memcpy(ctx->derived_conf_key.value, 
	     &((unsigned char *)digest.value)[digest.length-MD5_DIGEST_LENGTH], 
	     MD5_DIGEST_LENGTH);

      SPKM3_DEBUG (1, ("conf key:\n"));
      SPKM3_DEBUG_HEX(1, ((unsigned char *) ctx->derived_conf_key.value, 
			   ctx->derived_conf_key.length, 0));
    } else {
      minor_status = -1;
      goto err;
    }
  }
  else {
    if(g_OID_equal(ctx->intg_alg, &hmac_md5_oid)) {
      /* need to get MD5_DIGEST_LENGTH bytes */
      if((ctx->derived_integ_key.value = 
	  malloc(MD5_DIGEST_LENGTH)) == NULL) {
	minor_status =  GSS_S_G_MEMORY_ALLOC;
	goto err;
      }
      ctx->derived_integ_key.length = MD5_DIGEST_LENGTH;
      memcpy(ctx->derived_integ_key.value, 
	     &((unsigned char *)digest.value)[digest.length-MD5_DIGEST_LENGTH], 
	     MD5_DIGEST_LENGTH);

      SPKM3_DEBUG (1, ("integrity key:\n"));
      SPKM3_DEBUG_HEX(1, ((unsigned char *) ctx->derived_integ_key.value, 
			  ctx->derived_integ_key.length, 0));
      
    } else if(!g_OID_equal(ctx->intg_alg, &md5_rsa_encryption_oid)) {
      minor_status = -1;
      goto err;
    }
  }
  
 err:
  if(digest.value) free(digest.value);
  if(buf) free(buf);

  SPKM3_DEBUG(1, ("spkm3_derive_subkey: END\n"));

  return minor_status;
}
