/*
 *  spkm3/spkm3_parse_token.c
 *
 *  Copyright (c) 2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Kevin Coffman <kwc@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include "gssapi_spkm3.h"
#include "spkm3_mech.h"

OM_uint32 asn1_cmp_integers(INTEGER_t *asn1_i, OM_uint32 i) {

  long l = 0;
  asn_INTEGER2long(asn1_i, &l);
  if(l == i) return 1;
  else return 0;
}

OM_uint32
spkm3_parse_inner_context_token 
(InitialContextToken_t *ict,	/* decoded token */
 OM_uint32 * gss_token_type,	/* non-null => return gss_token_type */
 spkm3_context_t ** ctxPP)	/* non-null => return spkm3 ctx ptr */
{
  OM_uint32 retcode = -1;
  int returned_ict = 0;
  OM_uint32 gss_type = -1;
  OM_uint32 spkm_type = -1;
  BIT_STRING_t *token_ctx_id = NULL;
  INTEGER_t *token_tok_id = NULL;
  
  
  /* check that the token is 1 out of 6 valid types */
  /* for each case, check for correct valid of the token_id */
  retcode = GSS_S_DEFECTIVE_TOKEN;
  SPKM3_DEBUG(1, ("spkm3_parse_inner_context_token: "));
  switch (ict->innerContextToken.present) {
  case SPKMInnerContextToken_PR_req:            /* SPKM_REQ */
    SPKM3_DEBUG (1, ("SPKM_REQ\n"));
    gss_type = GSS_INIT_TOKEN;
    spkm_type = SPKM_TOK_CTX_REQ;
    token_tok_id =
      &ict->innerContextToken.choice.req.requestToken.req_contents.tok_id;
    if(!asn1_cmp_integers(token_tok_id, SPKM_TOK_CTX_REQ))
      goto err;
    break;

  case SPKMInnerContextToken_PR_rep_ti:         /* SPKM_REP_TI */
    SPKM3_DEBUG (1, ("SPKM_REP_TI\n"));
    gss_type = GSS_ACCEPT_TOKEN;
    spkm_type = SPKM_TOK_CTX_REPTI;
    token_tok_id =
    &ict->innerContextToken.choice.rep_ti.responseToken.rep_ti_contents.tok_id;
    if(!asn1_cmp_integers(token_tok_id, SPKM_TOK_CTX_REPTI))
      goto err;
    break;
  case SPKMInnerContextToken_PR_rep_it:         /* SPKM_REP_IT */
    SPKM3_DEBUG (1, ("SPKM_REP_IT\n"));
    gss_type = GSS_INIT_TOKEN;
    spkm_type = SPKM_TOK_CTX_REPIT;
    token_ctx_id =
      &ict->innerContextToken.choice.rep_it.responseToken.context_id;
    token_tok_id = &ict->innerContextToken.choice.rep_it.responseToken.tok_id;
    if(!asn1_cmp_integers(token_tok_id, SPKM_TOK_CTX_REPIT))
      goto err;
    break;
  case SPKMInnerContextToken_PR_error:          /* SPKM_ERROR */
    SPKM3_DEBUG (1, ("SPKM_ERROR\n"));
    gss_type = GSS_ERROR_TOKEN;
    spkm_type = SPKM_TOK_CTX_ERROR;
    token_ctx_id =
      &ict->innerContextToken.choice.error.errorToken.context_id;
   break;
  case SPKMInnerContextToken_PR_mic:            /* SPKM_MIC */
    SPKM3_DEBUG (1, ("SPKM_MIC\n"));
    gss_type = GSS_GETMIC_TOKEN;
    spkm_type = SPKM_TOK_MIC_MSG;
    token_ctx_id =
      &ict->innerContextToken.choice.mic.mic_header.context_id;
    break;
  case SPKMInnerContextToken_PR_wrap:           /* SPKM_WRAP */
    SPKM3_DEBUG (1, ("SPKM_WRAP\n"));
    gss_type = GSS_WRAP_TOKEN;
    spkm_type = SPKM_TOK_WRAP_MSG;
    token_ctx_id =
      &ict->innerContextToken.choice.wrap.wrap_header.context_id;
    break;
  case SPKMInnerContextToken_PR_del:            /* SPKM_DEL */
    SPKM3_DEBUG (1, ("SPKM_DEL\n"));
    gss_type = GSS_DELETE_TOKEN;
    spkm_type = SPKM_TOK_DEL_CTX;
    token_ctx_id =
      &ict->innerContextToken.choice.del.del_header.context_id;
    break;
  case SPKMInnerContextToken_PR_NOTHING:
    spkm3_log_err("spkm3_parse_inner_context_token: NO TOKEN\n");
    gss_type = -1;
    spkm_type = -1;
    break;
  default:
    spkm3_log_err("spkm3_parse_inner_context_token: *UNKNOWN* TOKEN\n");
    gss_type = -1;
    spkm_type = -1;
    break;
  }


  if (gss_token_type)
    *gss_token_type = gss_type;
  /*
   * If they supplied a context ptr ptr, but the pointer is NULL,
   * then locate the context for the given token.  Otherwise, if they
   * already have the context pointer, return the token type and ptr
   * to the ICT in the given context.
   */
  SPKM3_DEBUG(1, ("spkm3_parse_inner_context_token: "
	      "ctxPP %p *ctxPP %p\n", ctxPP, *ctxPP));
  if (ctxPP) {
    if (*ctxPP == NULL) {
      OM_uint32 minor_status;
      if((retcode = spkm3_context_create (&minor_status, ctxPP)) != 0) {
	spkm3_log_err("spkm3_parse_inner_context_token: failed to "
		      "create new context. minor_status = %d\n", minor_status);
	goto err;
      }
    } else {
      if (!(kg_validate_ctx_id ((gss_ctx_id_t) *ctxPP))) {
	retcode = GSS_S_G_VALIDATE_FAILED;
	spkm3_log_err("spkm3_parse_inner_context_token: "
		      "failed to verify supplied context\n");
	goto err;
      }

      /* verify that context_id in each token after the TI token contains
       * the right (agreed upon) identifier
       */
      if(ict->innerContextToken.present > SPKMInnerContextToken_PR_rep_ti) {
	if(token_ctx_id->size != (*ctxPP)->ctx_id.length ||
	   memcmp(token_ctx_id->buf, (*ctxPP)->ctx_id.data,
		  token_ctx_id->size)) {
	  spkm3_log_err("spkm3_parse_inner_context_token: failed to "
		        "validate context_id\n");
	  SPKM3_DEBUG(1, ("saved context_id:\n"));
	  SPKM3_DEBUG_HEX(1, ((*ctxPP)->ctx_id.data, 
                          (*ctxPP)->ctx_id.length, 0));
	  SPKM3_DEBUG(1, ("received context_id:\n"));
	  SPKM3_DEBUG_HEX(1, (token_ctx_id->buf, token_ctx_id->size, 0));
	  retcode = GSS_S_DEFECTIVE_TOKEN;
	  goto err;
	}
      }
    }
    (*ctxPP)->spkm_tok_type = spkm_type;
    (*ctxPP)->in_ict = ict;
  }
  

  
  retcode = 0;
  
 err:
  return (retcode);
}

OM_uint32
spkm3_parse_token_internal (spkm3_context_t ** ctxPP, gss_buffer_t input_token)
{
	gss_int32 err_code = 0;
	InitialContextToken_t *ict = NULL;
	asn_dec_rval_t rval;
	unsigned char *ptr = NULL;
	size_t len = 0;

	if (input_token == GSS_C_NO_BUFFER) {
		err_code = EINVAL;
		spkm3_log_err("spkm_parse_token_internal: no input token!\n");
		goto err;
	}

	rval = ber_decode(0, &asn_DEF_InitialContextToken, (void **) &ict,
			  input_token->value, input_token->length);
	if (rval.code != RC_OK) {
		err_code = GSS_S_DEFECTIVE_TOKEN;
		spkm3_log_err("spkm_parse_token_internal: failed to parse!\n");
		goto err;
	}

#ifdef ASNDEBUG
	asn_fprint(stderr, &asn_DEF_InitialContextToken, ict);
#endif

	/* We only get here if it is an SPKM3 token */
	err_code = spkm3_parse_inner_context_token (ict, NULL, ctxPP);
	if (err_code) {
		goto err;
	}

err:
	SPKM3_DEBUG(1, ("spkm3_parse_token_internal: returning with %d\n", 
		err_code));

	return (err_code);
}

/*
 * Take the wire-format token and verify its header contents.
 * If it is an spkm token, then call spkm3_parse_inner_context_token()
 * to internalize the SPKMInnerContextToken, locate the context,
 * and hang the internalized token off the context structure.
 */
OM_uint32
spkm3_parse_token (OM_uint32 * minor_status,
		   gss_buffer_t input_token,
		   gss_OID * mech_type,
		   OM_uint32 * token_type, 
		   gss_ctx_id_t * context_handle)
{
	OM_uint32 major_status;
	gss_int32 err_code;
	unsigned char *ptr;
	unsigned int len;
	asn_dec_rval_t rval;
	InitialContextToken_t *ict = NULL;
	spkm3_context_t *spkm3_ctx;

	rval = ber_decode(0, &asn_DEF_InitialContextToken, (void **)&ict,
			  input_token->value, input_token->length);

	if (rval.code != RC_OK) {
		*minor_status = GSS_S_G_VALIDATE_FAILED;
		major_status = GSS_S_DEFECTIVE_TOKEN;
		spkm3_log_err("spkm_parse_token: failed to ber_decode!\n");
		goto err;
	}

#ifdef ASNDEBUG
	asn_fprint(stderr, &asn_DEF_InitialContextToken, ict);
#endif

	/* We only get here if it is an SPKM3 token */

	err_code = spkm3_parse_inner_context_token(ict, token_type, &spkm3_ctx);
	if (err_code) {
		*minor_status = GSS_S_G_VALIDATE_FAILED;
		major_status = GSS_S_DEFECTIVE_TOKEN;
		goto err;
	}

	major_status = GSS_S_COMPLETE;

      err:
	asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
                                                ict, 0);

	return (major_status);
}
