/*
 * spkm3/spkm3_asn1_util.c
 *
 *  Copyright (c) 2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
*
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <sys/types.h>
#include <sys/errno.h>
#include "gssapi_spkm3.h"

/*
 * Function: spkm3_encode_to_outbuf()
 *
 * Description: buffer function for der_encode.
 *
 * ec = der_encode(&asn_DEF_Rectangle, rec, NULL, NULL);
 * returns the encode size in ec.encode, used to malloc seb->buf.
 *
 * caller mallocs seb->buf and initializes seb->cur = seb->buf
 */
int
spkm3_encode_to_outbuf(const void *buffer, size_t size, void *app_key)
{
	struct spkm3_ebuf *seb = app_key;

        memcpy(seb->cur, buffer, size);
	seb->cur += size;
        return 0;
}

/*
 *	
 * NOTE: caller needs to free namep and outp.
*/
OM_uint32
der_encode_asn1_name (Name_t **namep, gss_buffer_t *outp)
{
	Name_t *name = *namep;
	asn_enc_rval_t ec;
	struct spkm3_ebuf seb;
	gss_buffer_t out = *outp;
	int err = -1;

	ec = der_encode (&asn_DEF_Name, name, NULL, NULL);
	if (ec.encoded == -1)
                goto out; 
	if ((ec.encoded == 0) || ((seb.buf = malloc (ec.encoded)) == NULL))
                goto out;

        seb.cur = seb.buf;
	ec = der_encode (&asn_DEF_Name, name, spkm3_encode_to_outbuf, &seb);
	if (ec.encoded == -1)
		goto out;
#ifdef ASNDEBUG
        asn_fprint(stderr, &asn_DEF_Name, name);
#endif

	out->length = ec.encoded;
	out->value = seb.buf;
	err = 0;
out:
	return err;
}


/*
 * Function: asn1_copy_oid()
 *
 * Description: copy static oid into MechType struct.
 * returns a minor status
 */
OM_uint32
asn1_copy_oid (gss_OID oid,
		MechType_t *mech)
{
	unsigned char *p;

	mech->size = oid->length;
	p = malloc (mech->size);
	if (!p) {
		return GSS_S_G_MEMORY_ALLOC;
	}
	memcpy (p, oid->elements, mech->size);
	mech->buf = p;
	return 0;
}

OBJECT_IDENTIFIER_t *
txt_2_asn1_obj(char *s)
{
	ASN1_OBJECT *tmpoid;
	OBJECT_IDENTIFIER_t *oid;

	if ((oid = calloc (1, sizeof (OBJECT_IDENTIFIER_t))) == NULL)
		return NULL;
	tmpoid = OBJ_txt2obj("CN", 0);
	oid->size = tmpoid->length;

	if ((oid->buf = malloc (oid->size)) == NULL) {
		free (oid);
		ASN1_OBJECT_free (tmpoid);
		return NULL;
	}
	memcpy (oid->buf, tmpoid->data, oid->size);
	ASN1_OBJECT_free (tmpoid);
	return oid;	
}

OM_uint32
asn1_cmp_oid (gss_OID o,
		MechType_t *m)
{
	return ((o->length == m->size) && !memcmp(o->elements, m->buf, m->size));
}

AttributeValue_t *
asn1_get_entry_by_oid(Name_t *name, OBJECT_IDENTIFIER_t *oid)
{
	RelativeDistinguishedName_t *rn;
	AttributeTypeAndValue_t *av;
	AttributeType_t *at;
	AttributeValue_t *entry = NULL;
	asn_set *as_rdn, *as_rn;
	int i;

	if (!name || !oid)
		return entry;
 	as_rdn = (asn_set *)&name->choice.rdnSequence.list;
	for (i = 0; i < as_rdn->count; i++) {
		rn = (RelativeDistinguishedName_t *)as_rdn->array[i];
		as_rn = (asn_set *)&rn->list;
		av = (AttributeTypeAndValue_t *)as_rn->array[0];	
		at = &av->type;
		if ((oid->size == at->size) && 
				!memcmp (oid->buf, at->buf, at->size)) {
			entry = &av->value;
			break;
		}
	}
	return entry;
}

OM_uint32 oid_2_asn1alg(void *asn_set_of_x, const gss_OID_desc *oid) {

  OM_uint32 minor_status = 0;
  AlgorithmIdentifier_t *algid = NULL;

  if((algid = calloc(1, sizeof(AlgorithmIdentifier_t))) == NULL) {
    minor_status = GSS_S_G_MEMORY_ALLOC;
    goto out;
  }

  algid->algorithm.size = oid->length;
  if ((algid->algorithm.buf = malloc (algid->algorithm.size)) == NULL) {
    free(algid);
    minor_status = GSS_S_G_MEMORY_ALLOC;
    goto out;
  }

  memcpy (algid->algorithm.buf, oid->elements, algid->algorithm.size);
  algid->parameters = NULL;

  if (asn_set_of_x) {
    if((minor_status = asn_set_add(asn_set_of_x, (void *) algid))) {
      free(algid);
      free(algid->algorithm.buf);
      goto out;
    }
  }

out:
  return minor_status;
}

OM_uint32
asn1_algid_set_oid (AlgorithmIdentifier_t *x, const gss_OID_desc *oid)
{
        x->algorithm.size = oid->length;
        x->algorithm.buf = calloc (1, x->algorithm.size);
        if (!x->algorithm.buf) {
                return GSS_S_G_MEMORY_ALLOC;
        }
        memcpy (x->algorithm.buf, oid->elements, x->algorithm.size);
        x->parameters = NULL;

	return 0;
}

int
get_oids_from_token (OM_uint32 *minor_status, gss_OID_desc oids[ALGLISTSIZE], 
		     void *asn_set_of_x)
{
  asn_set *as = (asn_set *)asn_set_of_x;
  AlgorithmIdentifier_t *alg = NULL;
  int i = 0, limit = 0;

  limit = (as->count > ALGLISTSIZE) ? ALGLISTSIZE: as->count;
  for (i = 0; i < limit; i++) {
    alg = (AlgorithmIdentifier_t *)as->array[i];

    SPKM3_DEBUG(1, ("\tAlgorithm: "));
    SPKM3_DEBUG_HEX(1, (alg->algorithm.buf, alg->algorithm.size, 0));
    
    oids[i].length = alg->algorithm.size;
    if((oids[i].elements = 
         calloc(oids[i].length, sizeof(oids[i].length))) == NULL) {      
      *minor_status = GSS_S_G_MEMORY_ALLOC;
      return -1;
    }
    memcpy(oids[i].elements, alg->algorithm.buf, oids[i].length);
  }

  return i;
}

int
cmp_alg_oids(gss_OID_desc oid, const gss_OID_desc *oids) {
 
  int i = 0;

  for(i = 0; !g_OID_equal((oids+i), GSS_C_NULL_OID); i++) {
    if(g_OID_equal(&oid, (oids+i))) return 1;
  }
  return 0;

}

OM_uint32
spkm3_add_algs_oid (spkm3_context_t * ctx, void *in_asn_set_of_x, 
		    void *out_asn_set_of_x, const gss_OID_desc *alglist, 
		    gss_OID *ctxalg)
{
  OM_uint32 minor_status = 0;
  gss_OID_desc oids[ALGLISTSIZE];  
  int num = 0, i = 0;
  

  num = get_oids_from_token (&minor_status, oids, in_asn_set_of_x);

  for (i = 0; i < num; i++) {
    int j;
    for(j = 0; alglist[j].length != 0; j++) {
      if(g_OID_equal(&oids[i], &alglist[j])) { 

        if (out_asn_set_of_x) {
          AlgorithmIdentifier_t *algid;
      
          if((algid = calloc(1, sizeof(AlgorithmIdentifier_t))) == NULL) {
	    minor_status = GSS_S_G_MEMORY_ALLOC;
	    goto out;
          }
          algid->algorithm.size = oids[i].length;
          if ((algid->algorithm.buf = malloc (algid->algorithm.size)) == NULL) {
	    free(algid);
	    minor_status = GSS_S_G_MEMORY_ALLOC;
	    goto out;
          }
          memcpy (algid->algorithm.buf, oids[i].elements, 
                  algid->algorithm.size);
          algid->parameters = NULL;
          if((minor_status = asn_set_add(out_asn_set_of_x, (void *) algid))) {
	    free(algid);
	    free(algid->algorithm.buf);
	    goto out;
          }
        }
        spkm3_gss_copy_oid(&minor_status, &oids[i], ctxalg);
        goto out_loop;
      }
    }
  }

out_loop:
  for(i = 0; i < num; i++) 
    free(oids[i].elements);

out:
      
  return minor_status;
}

/* BIT_STRING stuff */

int
asn1_set_bit (BIT_STRING_t *bstr, int bitnum, int value)
{
	ASN1_BIT_STRING bitstr = {
		.length = 0,
	};

	if (bstr && ((bstr)->buf != NULL)) {
		bitstr.length = bstr->size;
		bitstr.data = bstr->buf;
	}
	/* XXX
	 * This lets OpenSSL realloc the space if
	 * it already allocated but not big enough.
	 * Is that a problem?
	 */
	if(!ASN1_BIT_STRING_set_bit(&bitstr, bitnum, value))
		return 0;

	if (bstr->buf == NULL) {
		/* Allocate and fill the memory */
		bstr->buf = malloc(bitstr.length + 1);
		if(bstr->buf == NULL) 
			return 0;
		bstr->size = bitstr.length;
		memcpy (bstr->buf, bitstr.data, bitstr.length);
		bstr->buf[bstr->size] = '\0';
		free(bitstr.data);
	 }

	 return 1;
}

int
asn1_get_bit (BIT_STRING_t *bstr, int bitnum)
{
	ASN1_BIT_STRING bitstr = {
		.length = 0,
	};

	if (!bstr)
		return 0;

	bitstr.length = (bstr)->size;
	bitstr.data = (bstr)->buf;
	return (ASN1_BIT_STRING_get_bit(&bitstr, bitnum));
}

int
asn1_bitstring_set(BIT_STRING_t *str, const void *data, size_t len)
{
	uint8_t *buf;

	if (str == NULL || (data == NULL & len != 0))
		return -1;	

	if (len < 0)
		return -1;

        /* Allocate and fill the memory */
        buf = (uint8_t *) malloc(len + 1);
        if (buf == NULL)
                return -1;

        memcpy(buf, data, len);
        buf[len] = '\0';

	str->buf = buf;
	str->size = len;
	str->bits_unused = 0;

        return 0;
}

BIT_STRING_t * asn1_bitstring_dup(BIT_STRING_t *in) {
  BIT_STRING_t *out = NULL;

  if(in == NULL || in->size <=0 || in->buf == NULL) return 0;

  if ((out = calloc (1, sizeof(BIT_STRING_t))) == NULL)
      goto err;

  if((out->buf = calloc(in->size, sizeof(uint8_t))) == NULL)
    goto err;

  memcpy(out->buf, in->buf, in->size);
  out->size = in->size;
  out->bits_unused = 0;

err:
 
  return out;
}

/* Names */

OM_uint32
asn1_name_compare(Name_t *name1, Name_t *name2)
{

	SPKM3_DEBUG(1, ("asn1_name_compare FUNCTION STUB\n"));
#ifdef ASNDEBUG
	asn_fprint(stderr, &asn_DEF_Name, name1);
	asn_fprint(stderr, &asn_DEF_Name, name2);
#endif
	return 0;
}

OM_uint32
asn1_name_to_string(Name_t *name, spkm3_name_desc_t *out)
{
	OM_uint32 min_stat = ENOMEM;
	ASN1_OBJECT *a;
	asn_set *as_attr, *as_rdn;
	RelativeDistinguishedName_t *rdn;
	AttributeTypeAndValue_t *attr;
	AttributeType_t *atype;
	AttributeValue_t *avalue;
	int i, nid;
	char nbuf[2048], *ptr, *valptr;
	const char *s;

	out->length = 0;
	memset(nbuf, 0, 2048);
	ptr = nbuf;
	a = ASN1_OBJECT_new();
	as_rdn = (asn_set *)&name->choice.rdnSequence.list;
	for (i = 0; i < as_rdn->count; i++) {
		rdn = (RelativeDistinguishedName_t *)as_rdn->array[i];	
		as_attr = (asn_set *)&rdn->list;	
		attr = (AttributeTypeAndValue_t *)as_attr->array[0];
		atype = &attr->type;
		avalue = &attr->value;
		
		a->length = atype->size;
		a->data = atype->buf;
		if ((nid=OBJ_obj2nid(a)) == NID_undef)
			goto out;
		s = OBJ_nid2sn(nid);
		if (strlen(s) + avalue->size + out->length > 247)
			goto out;
		strncat(ptr, s, strlen(s));
		ptr+= strlen(s);
		strncat(ptr, "=", 1);
		ptr++;
		/* move past the asn1 tag, plus string length */
		strncat(ptr, avalue->buf+2, avalue->size - 2);
		ptr += avalue->size - 2;
		strncat(ptr, ",", 1);
		ptr++;
		out->length += (strlen(s) + avalue->size);
	}
	/* constructed string includes a trailing comma */
	out->string_name = (char *) malloc (out->length);
	if (out->string_name == NULL)
		goto out;
	nbuf[out->length - 1] = 0; /* overwrite the trailing comma */
	memcpy (out->string_name, nbuf, out->length);

	min_stat = 0;
out:
	ASN1_OBJECT_free(a);
	return min_stat;
}

OM_uint32
string_to_asn1_name(spkm3_name_desc_t *cn)
{
	X509_NAME *xname;
	OM_uint32 min_stat = 0;

	xname = a2i_X509_NAME (cn->string_name, cn->length);
	if (xname == NULL)
		return EINVAL;
	if ((cn->asn1_name = calloc (1, sizeof(Name_t))) == NULL)
		return ENOMEM;

	min_stat = x509_to_asn1_name (xname, cn->asn1_name);
#ifdef ASNDEBUG
	if (!min_stat)
		asn_fprint(stderr, &asn_DEF_Name, cn->asn1_name);
#endif
	
	X509_NAME_free(xname);
	return min_stat;
}

OM_uint32
x509_2_asn1_name_entry(X509_NAME_ENTRY *xent, void *asn_set_of_x)
{
	AttributeTypeAndValue_t *at;
	OM_uint32 min_status = GSS_S_G_MEMORY_ALLOC;
	int err;
	OCTET_STRING_t *tmp;

	at = calloc(1, sizeof(AttributeTypeAndValue_t));
	if (!at)
		return min_status;
	tmp = calloc(1, sizeof(OCTET_STRING_t));
	if (!tmp)
		goto out_free_at;

	/* Allocate buffer for 'type' */
	at->type.size = xent->object->length;
	at->type.buf = calloc (1, at->type.size);
	if (!at->type.buf)
		goto out_free_at;
	memcpy(at->type.buf, xent->object->data, at->type.size);
		
	/* XXX This ends up taking the PrintableString (19) and
	 * converting it into just an OCTET_STRING ??? */
	if ((OCTET_STRING_fromBuf(tmp, xent->value->data, 
					xent->value->length)) != 0)
		goto out_free_tmp;
	if ((ANY_fromType(&at->value, &asn_DEF_OCTET_STRING, tmp)) != 0) 
		goto out_free_tmp;
		
	/* XXX free tmp !*/	
	//asn_DEF_OCTET_STRING.free_struct (&asn_DEF_OCTET_STRING, tmp, 1);


	SPKM3_DEBUG (1, ("x509_2_asn1_name_entry NAME: %.*s\n",
		     at->value.size, at->value.buf));

	min_status = asn_set_add(asn_set_of_x, (void *) at);

	asn_DEF_OCTET_STRING.free_struct (&asn_DEF_OCTET_STRING, tmp, 0);
out:
	return min_status;

out_free_tmp:
	asn_DEF_OCTET_STRING.free_struct (&asn_DEF_OCTET_STRING, tmp, 1);
out_free_at:
	asn_DEF_AttributeTypeAndValue.free_struct(&asn_DEF_AttributeTypeAndValue,
			at, 1);
	goto out;
}

OM_uint32
x509_to_asn1_name(X509_NAME *xname, Name_t *name)
{
	X509_NAME_ENTRY *entry;
	int i, count, ret = 0;
	RelativeDistinguishedName_t *rn;
	OM_uint32 min_status;

	count = X509_NAME_entry_count(xname);
	for (i = 0; i < count;i ++) {
		rn = calloc (1, sizeof(RelativeDistinguishedName_t));
		if (!rn)
			return GSS_S_G_MEMORY_ALLOC;
		entry =  X509_NAME_get_entry(xname, i);
		min_status = x509_2_asn1_name_entry(entry, &rn->list);
		if (min_status)
			goto out_free;
		min_status = asn_set_add(&name->choice.rdnSequence.list, (void *) rn);
		if (min_status)
			goto out_free;
	}
	name->present = Name_PR_rdnSequence;
out:
	return min_status;
out_free:
	asn_DEF_RelativeDistinguishedName.free_struct(
		&asn_DEF_RelativeDistinguishedName,
		rn, 0);
	goto out;
}

OM_uint32
asn1_attr_type_copy(AttributeType_t *out, AttributeType_t *in)
{
	if ((out->buf = malloc(in->size)) == NULL) {
		return GSS_S_G_MEMORY_ALLOC;
	}
	out->size = in->size;
	memcpy(out->buf, in->buf, in->size);
	return 0;
}

OM_uint32
asn1_attr_value_copy(AttributeValue_t *out, AttributeValue_t *in)
{
	if ((out->buf = malloc(in->size)) == NULL) {
		return GSS_S_G_MEMORY_ALLOC;
	}
	out->size = in->size;
	memcpy(out->buf, in->buf, in->size);
	return 0;
}

OM_uint32
asn1_name_entry_2_entry(AttributeTypeAndValue_t *at_in, void *asn_set_of_x)
{
	AttributeTypeAndValue_t *at_out;
	OM_uint32 min_status = GSS_S_G_MEMORY_ALLOC;

	at_out = calloc(1, sizeof(AttributeTypeAndValue_t));
	if (!at_out)
		return min_status;
	
	if ((min_status = asn1_attr_type_copy(&at_out->type, &at_in->type))) {
		free(at_out);
		return min_status;
	}
	if ((min_status = asn1_attr_value_copy(&at_out->value, &at_in->value))) {
		free(at_out->type.buf);
		free(at_out);
		return min_status;
	}
/*	This does a shallow copy instead of duplicating the buffers

	at_out->type.size = at_in->type.size;
	at_out->type.buf =  at_in->type.buf;

	at_out->value.size = at_in->value.size;
	at_out->value.buf = at_in->value.buf;
*/

	min_status = asn_set_add(asn_set_of_x, (void *) at_out);
	return min_status;
}

OM_uint32
asn1_name_entry_copy(RelativeDistinguishedName_t *rdn_dst,
		    RelativeDistinguishedName_t *rdn_src)
{
	OM_uint32 min_status = 0;
	AttributeTypeAndValue_t *at_in;
	asn_set *as_in;
	int i;

	as_in = (asn_set *)&rdn_src->list;
	for (i = 0; i < as_in->count; i++) {
		at_in =(AttributeTypeAndValue_t *)as_in->array[i];
		min_status = asn1_name_entry_2_entry(at_in, &rdn_dst->list);
		if (min_status)
			break;
	}
	return min_status;
}

OM_uint32
asn1_name_copy(Name_t *dst, const Name_t *src)
{
	OM_uint32 min_status = GSS_S_G_MEMORY_ALLOC;
	RelativeDistinguishedName_t *rdn_src;
	RelativeDistinguishedName_t *rdn_dst;
	asn_set *src_set;
	int i;

	if (src == NULL || dst == NULL)
		return GSS_S_G_MEMORY_ALLOC;	/* XXX */

	src_set =  (asn_set *)&src->choice.rdnSequence.list;

	/* unpack the SEQUENCE of RelativeDistinguishedName */
	for (i = 0; i< src_set->count; i++) {
		rdn_src = (RelativeDistinguishedName_t *)src_set->array[i];
		rdn_dst = calloc (1, sizeof(RelativeDistinguishedName_t));
		if (rdn_dst == NULL)
			goto out;
		min_status = asn1_name_entry_copy(rdn_dst, rdn_src);
		if (min_status)
			goto out_free;
		min_status = asn_set_add(&dst->choice.rdnSequence.list,
						(void *) rdn_dst);
		if (min_status)
			goto out_free;
	}
	dst->present = src->present;
	min_status = 0;
out:
	return min_status;
out_free:
	asn_DEF_RelativeDistinguishedName.free_struct(
			&asn_DEF_RelativeDistinguishedName,
			rdn_dst, 0);
	goto out;
}

Name_t * asn1_name_dup(const Name_t *src)
{
	Name_t *dst;

	if ((dst = calloc(1, sizeof(Name_t))) == NULL)
		return NULL;
	if ((asn1_name_copy(dst, src)) != 0) {
		free(dst);
		return NULL;
	}
	return dst;
}

/* Integer */

/*
 * Caller frees ASN1_INTEGER
 */
void
asn1_2_ssl_int(ASN1_INTEGER ** sslint, INTEGER_t *aint)
{
	ASN1_INTEGER *ptr;

	ptr = ASN1_INTEGER_new ();
	ptr->length = aint->size;
	ptr->data = malloc (ptr->length);
	memcpy (ptr->data, aint->buf, ptr->length);
	*sslint = ptr;
}

void
ssl_2_asn1_int(INTEGER_t ** aint, ASN1_INTEGER *sslint)
{
	INTEGER_t *ptr = *aint;

	ptr->size = sslint->length;
	ptr->buf = malloc (ptr->size);
	memcpy (ptr->buf, sslint->data, ptr->size);
}


/* Certificate */

Certificate_t *
asn1_cert_dup(Certificate_t *cert)
{
	Certificate_t *ret = NULL;
	asn_enc_rval_t ec;
	struct spkm3_ebuf seb;
	asn_dec_rval_t rval;

	if (cert == NULL) return NULL;

#if 0
	if ((ret = calloc (1, sizeof (Certificate_t))) == NULL)
		return NULL;
#endif
	ec = der_encode(&asn_DEF_Certificate, cert, NULL, NULL);
	if (ec.encoded == -1)
		goto out_free;
	if ((ec.encoded == 0) || ((seb.buf = malloc (ec.encoded)) == NULL))	
		goto out_free;

	seb.cur = seb.buf;
	ec = der_encode(&asn_DEF_Certificate, cert, spkm3_encode_to_outbuf, 
								&seb);
	if (ec.encoded == -1)
		goto out_free;
	rval = ber_decode(0, &asn_DEF_Certificate, (void **)&ret, 
				seb.buf, 
				ec.encoded);

	free (seb.buf);
	return ret;
out_free:
	free (ret);
	return NULL;
}

/*
 * Function:  asn1_algor_dh_set
 *
 * Description:  stuff the Key_Estb_Algs_t with diffie hellmen
 * initiator half of key.
 *
 * Returns: 0 on success
 * Caller responsible for freeing X509_ALGOR
 */

OM_uint32
asn1_algor_dh_set (void *asn_set_of_x, gss_OID oid, DH * dh)
{
	ASN1_INTEGER *aint;
	AlgorithmIdentifier_t *a;
	OM_uint32 minor_status = GSS_S_G_MEMORY_ALLOC;
	int ret;
	size_t len;
	unsigned char *output_buffer = NULL;
	DHParameter_t *dhp;
	INTEGER_t *iptr;


	if ((dhp = calloc(1, sizeof(DHParameter_t))) == NULL)
                goto out;

	a = calloc (1,sizeof(AlgorithmIdentifier_t));
	if (!a)
		goto out_free_p;
	
	minor_status = GSS_SPKM_S_SG_OPENSSL_CONVERT_ERROR;
	if ((aint = BN_to_ASN1_INTEGER (dh->p, NULL)) == NULL)
		goto out_free_p;
	iptr = &dhp->prime;
	ssl_2_asn1_int(&iptr, aint);
	M_ASN1_INTEGER_free (aint);

	if ((aint = BN_to_ASN1_INTEGER (dh->g, NULL)) == NULL)
		goto out_free_p;
	iptr = &dhp->base;
	ssl_2_asn1_int(&iptr, aint);
	M_ASN1_INTEGER_free (aint);

	if((minor_status = encode (&asn_DEF_DHParameter, dhp, &output_buffer,
				   &len)) != 0)
	  goto out;

	/* set the algorithm id */
	minor_status = GSS_S_G_MEMORY_ALLOC;
	a->algorithm.size = oid->length;
	if((a->algorithm.buf = malloc(a->algorithm.size)) == NULL)
		goto out_free_p;
	memcpy(a->algorithm.buf, oid->elements, a->algorithm.size);

	if ((a->parameters = calloc (1, sizeof(OCTET_STRING_t))) == NULL)
		goto out_free_p;
	a->parameters->size = len;
	a->parameters->buf = output_buffer;

	ret = asn_set_add(asn_set_of_x, (void *) a);
	asn_DEF_DHParameter.free_struct(&asn_DEF_DHParameter, dhp, 0);

	minor_status = 0;
out:
	return minor_status;

out_free_p:
	free(output_buffer);
out_free_dh:
	free (dhp);
	goto out;
}

OM_uint32 
asn1_algor_dh_get (unsigned char *p, long len, DH **dh) {

  OM_uint32 minor_status = 0;
  DHParameter_t *dhp = NULL;
  ASN1_INTEGER *prime, *base;
  asn_dec_rval_t rval;
  
  minor_status = GSS_S_G_MEMORY_ALLOC;
  if((*dh = DH_new ()) == NULL)
    goto err;
  
  if(((*dh)->p = BN_new ()) == NULL)
    goto err;
  
  if(((*dh)->g = BN_new ()) == NULL)
    goto err;
  
  /* Get the DHParameter structure that should have been passed */
  rval = ber_decode(0, &asn_DEF_DHParameter, (void **)&dhp, p, len);
  if (rval.code != RC_OK) 
    goto err;

      
  /* Now set the fields in the DH structure */
  asn1_2_ssl_int (&prime, &dhp->prime);
  asn1_2_ssl_int (&base, &dhp->base);
  ASN1_INTEGER_to_BN (prime, (*dh)->p);
  ASN1_INTEGER_to_BN (base, (*dh)->g);

  if (dhp->privateValueLength)
    asn_INTEGER2long(dhp->privateValueLength, &(*dh)->length);

  
  ASN1_INTEGER_free(prime);
  ASN1_INTEGER_free(base);
  minor_status = 0;

  asn_DEF_DHParameter.free_struct(&asn_DEF_DHParameter, dhp, 0);

 err:
  return minor_status;
}
