/*
 *  spkm3/sign.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <openssl/md5.h>
#include "gssapi_spkm3.h"


OM_uint32
spkm3_gss_get_mic (OM_uint32 * minor_status,
		   gss_ctx_id_t context_handle,
		   gss_qop_t qop_req,
		   gss_buffer_t message_buffer,
		   gss_buffer_t token_buffer)
{
  spkm3_context_t *ctx;
  InitialContextToken_t *ict = NULL;
  SPKM_MIC_t *mic;
  Mic_Header_t *mhdr;
  int datalen = 0, mdlen = 0;
  unsigned char *dataptr = NULL, *md = NULL;
  
  SPKM3_DEBUG(1, ("---> spkm3_gss_get_mic: "
		  "message_buffer(value %p length %d)\n",
		  message_buffer->value, message_buffer->length));
  
  /* validate the context handle */
  if (!kg_validate_ctx_id (context_handle)) {
    *minor_status = (OM_uint32) GSS_S_G_VALIDATE_FAILED;
    goto err;
  }
  ctx = (spkm3_context_t *) context_handle;
  
  if (!ctx->established) {
    *minor_status = GSS_SPKM_S_SG_CTX_INCOMPLETE;
    spkm3_log_status("spkm3_gss_get_mic: context not established:",
		     GSS_S_FAILURE, *minor_status);
    goto err;
  }
  
  token_buffer->length = 0;
  token_buffer->value = NULL;
  
  /* only default qop for now. */
  
  switch (qop_req) {
  case 0:
    /* check the sign algorithm. RFC 2203 states that QOP of 0
     * is to be used. according to RFC 2025, this means
     * that the QOP default integrity algorithm is the first
     * one on the initiator's int_alg list, and the default
     * int_alg list is
     *  [1] md5WithRSA
     *  [2] DES-MAC
     * RFC 2847 (spkm3/lipkey) states that for spkm3, since there
     * is potentially no initiator credentials, the md5WithRSA is
     * relaxed to md5.
     * SO! pant pant, we will use what krb5 uses, md5 !!!
     * XXX should use HMAC_MD5 - this is easy....
     * call HMAC() (openssl/crypto/hmac/hmac.c) instead of the MD5_Init/../Final
     * see seal.c
     */
/*
    if (ctx->intg_alg != NID_md5 && ctx->intg_alg != NID_hmac_md5 && 
	ctx->intg_alg != NID_null_mac && 
	ctx->intg_alg != NID_md5WithRSAEncryption) {
      spkm3_log_err("spkm3_gss_get_mic: unknown I-ALG (%s)\n",
		    (ctx->intg_alg == NID_undef) ? "UNKNOWN" : 
		    OBJ_nid2ln (ctx->intg_alg));
      *minor_status = GSS_SPKM_S_SG_BAD_INT_ALG_SET;
      goto err;
    }
*/
    
    break;
  default:
    spkm3_log_err("spkm3_gss_get_mic: unavailable qop (%d) "
		  "requested.\n", qop_req);
    *minor_status = GSS_S_G_UNAVAIL_QOP;
    goto err;
    break;
  }

  *minor_status = create_initial_ct (&ict);
  if (*minor_status)
    goto err;
  
  /* XXX set minor status for any ANS1 error */
  
  *minor_status = GSS_S_G_MEMORY_ALLOC;
  
  mic = &ict->innerContextToken.choice.mic;
  mhdr = &mic->mic_header;
  ict->innerContextToken.present = SPKMInnerContextToken_PR_mic;
  
  
  /* set tokenid */
  if ((asn_long2INTEGER(&mhdr->tok_id, SPKM_TOK_MIC_MSG)) == -1)
    goto err;
  
  /* set context id */
  if ((asn1_bitstring_set(&mhdr->context_id, 
			  ctx->ctx_id.data, 
			  ctx->ctx_id.length)) == -1) 
    goto err;
  
  /* mhdr->intgalg and sndseq left NULL */
  *minor_status = 0;
  
  SPKM3_DEBUG (1, ("Acceptor ctx->ctx_id.data\n"));
  SPKM3_DEBUG_HEX(1, ((unsigned char *) ctx->ctx_id.data,
		      ctx->ctx_id.length, 0));
  SPKM3_DEBUG (1, ("mhdr->context_id\n"));
  SPKM3_DEBUG_HEX(1, ((unsigned char *) mhdr->context_id.buf,
		      mhdr->context_id.size, 0));
  
  if(g_OID_equal(ctx->intg_alg, &null_mac_oid)) {
    SPKM3_DEBUG(1, ("spkm3_gss_get_mic: null-mac. nothing to do.\n"));
    mic->int_cksum.buf = NULL;
    mic->int_cksum.size = 0;
  } else if(g_OID_equal(ctx->intg_alg, &hmac_md5_oid)) {
    /* compute the checksum of the message. this assumes md5 */
    if((*minor_status = 
	encode(&asn_DEF_Mic_Header, mhdr, &dataptr, &datalen)) != 0)
      goto err;
    
    SPKM3_DEBUG(1, ("spkm3_gss_get_mic: About to do HMAC-MD5 on %d bytes\n",
		    datalen));
    
    if((md = (unsigned char *)malloc(MD5_DIGEST_LENGTH)) == NULL) {
      *minor_status = GSS_S_G_MEMORY_ALLOC;
      goto err;
    }
    
    HMAC(EVP_md5(), ctx->derived_integ_key.value, 
	 ctx->derived_integ_key.length, dataptr, datalen, md, &mdlen);
    
    SPKM3_DEBUG (1, ("md\n"));
    SPKM3_DEBUG_HEX(1, ((unsigned char *) md, mdlen, 0));
    
    /* place checksum in token */
    if ((asn1_bitstring_set(&mic->int_cksum, md, mdlen)) == -1) {
      *minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out_free_buf;
    }
  } else  if(g_OID_equal(ctx->intg_alg, &md5_rsa_encryption_oid)) {
    sign(ctx->intg_alg, &asn_DEF_Mic_Header, mhdr, NULL, NULL,
	 &mic->int_cksum, ctx->spkm3_cred);
  } else {
    spkm3_log_err("spkm3_gss_get_mic: unknown I-ALG:");
    SPKM3_DEBUG_HEX(1, (ctx->intg_alg->elements, ctx->intg_alg->length, 0));
    minor_status = -1;
    goto out_free_buf;
  }
  
  SPKM3_DEBUG (1, ("mic->int_cksum\n"));
  SPKM3_DEBUG_HEX(1, ((unsigned char *) mic->int_cksum.buf,
		      mic->int_cksum.size, 0));
  
  if ((*minor_status = encode(&asn_DEF_InitialContextToken, ict, 
			      &token_buffer->value,
			      &token_buffer->length)))
    goto out_free_buf;
  
  SPKM3_DEBUG(1, ("<--- spkm3_gss_get_mic: "
		  "message_buffer(value %p length %d) "
		  "token_buffer(value %p length %d)\n",
		    message_buffer->value, message_buffer->length,
		  token_buffer->value, token_buffer->length));
  
  if(md) free(md);
  if(dataptr) free(dataptr);
  asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					  ict, 0);
  
  return GSS_S_COMPLETE;
  
 err:
  SPKM3_DEBUG(1, ("<--- spkm3_gss_get_mic: failed\n"));

  return GSS_S_FAILURE;
  
 out_free_buf:
  if(md) free(md);
  if(dataptr) free (dataptr);
 out_free:
  /* We're done with the internal formats now */
  asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					  ict, 0);
  SPKM3_DEBUG(1, ("spkm3_gss_get_mic: returning with error: %d\n",
		  *minor_status));
  goto err;
}

  
  OM_uint32
    spkm3_gss_sign (
		    OM_uint32 * min_status,
		    gss_ctx_id_t ctx_hndl,
		    int qop_req,
		       gss_buffer_t msg_buf, gss_buffer_t token_buf)
{
	return (spkm3_gss_get_mic(min_status, ctx_hndl,
				  qop_req, msg_buf, token_buf));
}
