/*
 *  spkm3/lucid_context.c
 *
 *  Copyright (c) 2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Kevin Coffman <kwc@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "gssapi_spkm3.h"
#include "spkm3.h"

static void
buf_2_buf(gss_buffer_desc *out, gss_buffer_desc *in)
{
	/* XXX bound lengths? */
	out->value = malloc(in->length);
	out->length = in->length;
	memcpy(out->value, in->value, in->length);
}

static void
oid_2_buf(gss_buffer_desc *out, gss_OID in)
{
	/* XXX bound lengths? */
	out->value = malloc(in->length);
	out->length = in->length;
	memcpy(out->value, in->elements, in->length);
}

/*
 * Function: gss_spkm3_export_lucid_sec_context()
 *
 * Description: returns spkm3 context fields for transport into the kernel 
 */

OM_uint32
spkm3_gss_export_lucid_sec_context (
			  OM_uint32 * min_status,
			  gss_ctx_id_t *context_handle,
			  OM_uint32 version,
			  void ** vcdx)
{			
	OM_uint32 		retval, ret = 0;
	spkm3_context_t 	*ctx;
	gss_spkm3_lucid_ctx_t	*lctx = NULL;

	SPKM3_DEBUG (1, ("spkm3_gss_export_lucid_sec_context:\n"));

	retval = GSS_S_FAILURE;
	*min_status = 0;
	if (vcdx)
		*vcdx = NULL;
	else {
		ret = EINVAL;
		goto out_err;
	}
		
	/* XXX Validate context_handle */
	ctx = (spkm3_context_t *) *context_handle;

	/* Allocate the structure */
	if ((lctx = calloc(1, sizeof(gss_spkm3_lucid_ctx_t))) == NULL) {
		ret = ENOMEM;
		goto out_err;
	}

	lctx->version = 1;
	buf_2_buf(&lctx->ctx_id, (gss_buffer_desc *)&ctx->ctx_id);
	lctx->qop = ctx->qop;
	buf_2_buf(&lctx->mech_used, (gss_buffer_desc *)ctx->mech_used);
	lctx->ret_flags = ctx->ret_flags;
	lctx->req_flags = ctx->req_flags;
	buf_2_buf(&lctx->share_key, &ctx->share_key);
	buf_2_buf(&lctx->derived_conf_key, &ctx->derived_conf_key);
	buf_2_buf(&lctx->derived_integ_key, &ctx->derived_integ_key);
	oid_2_buf(&lctx->keyestb_alg, ctx->keyestb_alg);
	oid_2_buf(&lctx->owf_alg, ctx->owf_alg);
	oid_2_buf(&lctx->intg_alg, ctx->intg_alg);
	oid_2_buf(&lctx->conf_alg, ctx->conf_alg);
	
	*vcdx = lctx;
	*min_status = 0;
	retval = GSS_S_COMPLETE;

	/* Clean up the context state (it is an error for
	* someone to attempt to use this context again)
	*/
	(void)spkm3_gss_delete_sec_context(min_status, context_handle, NULL);
	*context_handle = GSS_C_NO_CONTEXT;
	goto out;

out_err:
	if (*min_status == 0)
		*min_status = ret;
out:
	if(retval != GSS_S_COMPLETE)
	  spkm3_log_status ("spkm3_gss_export_lucid_sec_context:", retval,
			    *min_status);

	return (retval);
}

OM_uint32
spkm3_gss_free_lucid_sec_context(
			OM_uint32 * min_status,
			void *lctx)
{
	gss_spkm3_lucid_ctx_t 	*ctx;

	SPKM3_DEBUG(1, ("spkm3_gss_free_lucid_sec_context:\n"));

	ctx =  (gss_spkm3_lucid_ctx_t *)lctx;
	if (ctx->ctx_id.value) free (ctx->ctx_id.value);
	if (ctx->mech_used.value) free(ctx->mech_used.value);
	if (ctx->share_key.value) free (ctx->share_key.value);
	if (ctx->derived_conf_key.value) free (ctx->derived_conf_key.value);
	if (ctx->derived_integ_key.value) free (ctx->derived_integ_key.value);
	if (ctx->keyestb_alg.value) free (ctx->keyestb_alg.value);
	if (ctx->owf_alg.value) free (ctx->owf_alg.value);
	if (ctx->intg_alg.value) free (ctx->intg_alg.value);
	if (ctx->conf_alg.value) free (ctx->conf_alg.value);

	free(ctx);

	*min_status = 0;
	return GSS_S_COMPLETE;
}
		
