/*
 *  spkm3/init_sec_context.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Kevin Coffman <kwc@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "gssapi_spkm3.h"
#include "gssapiP_generic.h"

/* Define here globally  XXX KWC Should be elsewhere!! XXX */
//void * kg_vdb = NULL;

/*
 * Function: spkm3_gss_init_sec_context()
 *
 * Description: 
 * called by initiator in a loop, sending output_token to target,
 * and processing returned input_token until the protocol 
 * is done.
 * 
 * NOTE: according to rfc 2847, spkm3_gss_init_sec_context will be called
 * from the LIPKEY mechanism's gss_init_sec_context(). 
 *
 * NOTE: spkm3's most basic form allows for no initiator or target credentials
 *
 * XXX: who is responsible for freeing target_name, 
 * XXX minor_status and major_status codes need improvement
 */

OM_uint32
spkm3_gss_init_sec_context (
				   OM_uint32 * minor_status,
				   const gss_cred_id_t initiator_cred_handle,
				   gss_ctx_id_t * context_handle_P,
				   gss_name_t target_name,
				   gss_OID mech_type,
				   OM_uint32 req_flags,
				   OM_uint32 time_req,
				   gss_channel_bindings_t input_chan_bindings,
				   gss_buffer_t input_token,
				   gss_OID * actual_mech_type,
				   gss_buffer_t output_token,
				   OM_uint32 * ret_flags,
				   OM_uint32 * time_rec)
{

	spkm3_context_t *ctx = *context_handle_P;
	OM_uint32 major_status;

	SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: "
		    "entered with context_handle %p input_token %p (len %d)\n",
                    *context_handle_P, input_token,
                    input_token ? input_token->length : -1));

	/* initialize variables */
	major_status = GSS_S_FAILURE;

	if (minor_status == NULL || output_token == NULL ||
	    context_handle_P == NULL)
		goto out;

	*minor_status = GSS_S_G_VALIDATE_FAILED;

	output_token->length = 0;
	output_token->value = NULL;

	/* validate mechanism */
	if (!g_OID_equal (mech_type, gss_mech_oid_spkm3)) {
		major_status = GSS_S_BAD_MECH;
		goto out;
	}


	/*
	 * If this is the first time in, create a new context,
	 * otherwise verify the context handle that was supplied.
	 */
	if ((*context_handle_P == (gss_ctx_id_t) GSS_C_NO_CONTEXT)) {
		if (input_token != GSS_C_NO_BUFFER &&
		    input_token->length != 0 && input_token->value != NULL) {
			major_status = GSS_S_DEFECTIVE_TOKEN;
			goto out;
		}

		/* first time for this connection */
		SPKM3_DEBUG (1, ("spkm3_gss_init_sec_context: 1st TIME IN\n"));
		if (spkm3_context_create (minor_status, &ctx)) {
			goto out;
		}
		ctx->req_flags = req_flags;	/* XXX is this right??? */

	} else {

		/* parse and validate input token (if supplied) */
		if ((spkm3_parse_token_internal(&ctx, input_token)) != 0) {
			spkm3_log_err("spkm3_gss_init_sec_context: "
				      "spkm3_parse_token_internal failed!\n");
			major_status = GSS_S_DEFECTIVE_TOKEN;
			goto out;
		}
	}

	/*
	 * If they supplied a credential handle, verify it.
	 * If they didn't supply a cred handle, but
	 * requested MUTUAL_AUTH, get default creds.
	 * XXX This should be done elsewhere ???
	 */

	if (initiator_cred_handle != GSS_C_NO_CREDENTIAL) {
		/* XXX Validate the cred handle */
		ctx->spkm3_cred = (spkm3_cred_int_t *) initiator_cred_handle;
	}

	switch (ctx->spkm_tok_type) {
	case SPKM_TOK_CTX_INITIAL:
		SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: REQ\n"));
		/* create initial request token */	        
		if (ctx->state != SPKM3_CTX_STATE_NEW) {
			spkm3_log_err("spkm3_gss_init_sec_context: "
				      "incorrect context state (%d) for "
                                      "initial request!\n", ctx->state);
			*minor_status = GSS_S_G_MEMORY_ALLOC; /* XXX */
			major_status = GSS_S_FAILURE;
			goto out;
		}

		*minor_status = REQ_create(ctx, output_token, 
                                           (spkm3_name_desc_t *) target_name);

		if (*minor_status) {
			spkm3_context_free(ctx);
			ctx = NULL;
			*context_handle_P == (gss_ctx_id_t) GSS_C_NO_CONTEXT;
			goto out;
		}

		/* we should now expect a REPTI */
		SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: "
			    "done with initial request, expecting "
			    "a REPTI token while in state %d\n", ctx->state));
		ctx->state = SPKM3_CTX_STATE_REPTI;
		major_status = GSS_S_CONTINUE_NEEDED;

		/*
		 * Save context so we will recognize it when we
		 * see the next token for this round of context
		 * establishment.
		 */
		SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: "
			    "saving context located at %p\n", ctx));
		if (!(kg_save_ctx_id ((gss_ctx_id_t) ctx))) {
			spkm3_context_free (ctx);
			ctx = 0;
			major_status = GSS_S_FAILURE;
			*minor_status = 0;	/* XXX */
			goto out;
		}

		break;

	case SPKM_TOK_CTX_ERROR:
		SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: ERROR\n"));
		/* handle error token */
		break;

	case SPKM_TOK_DEL_CTX:
		SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: DELETE\n"));
		/* handle delete token */
		break;

	case SPKM_TOK_CTX_REPTI:
		SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: REPTI\n"));
		/* handle reply from target */
		if (ctx->state != SPKM3_CTX_STATE_REPTI) {
			spkm3_log_err("spkm3_gss_init_sec_context: "
				      "not expecting a REPTI token while "
				      "in state %d\n", ctx->state);
			*minor_status = 0;	/* XXX */
			major_status = GSS_S_FAILURE;
			goto out;
		}

		if(*minor_status = REPTI_process(ctx))
			goto out;
	
		/* Create SPKM_REP_IT token if mutual auth requested */
		if (ctx->req_flags & GSS_C_MUTUAL_FLAG) {
			if(*minor_status =
	           		REPIT_create (ctx, output_token))
				goto out;
		}
		/* we're done! */
		ctx->state = SPKM3_CTX_STATE_AUTH;
		major_status = GSS_S_COMPLETE;
		ctx->established = 1;
		break;

	case SPKM_TOK_CTX_REQ:
	case SPKM_TOK_CTX_REPIT:
	default:
		spkm3_log_err("spkm3_gss_init_sec_context: BAD STATE\n");
		/* Any of these cases is an error, we should not get these! */
		major_status = GSS_S_FAILURE;
		*minor_status = GSS_S_DEFECTIVE_TOKEN;
		goto out;
		break;
	}


	/* Set the other output values */
	*context_handle_P = ctx;

	if (actual_mech_type != NULL)
		*actual_mech_type = (gss_OID) gss_mech_oid_spkm3;

	if (ret_flags != NULL)
		*ret_flags = 0;

	if (time_rec != NULL)
		*time_rec = GSS_C_INDEFINITE;

out:
	 if (ctx && ctx->in_ict) {
                SPKM3_DEBUG(1, ("spkm3_gss_init_sec_context: "
                            "freeing in_ict at %p\n", ctx->in_ict));
                asn_DEF_InitialContextToken.free_struct(
                                                &asn_DEF_InitialContextToken,
                                                ctx->in_ict, 0);
                ctx->in_ict = NULL;
        }

	if(major_status != GSS_S_COMPLETE && 
           major_status != GSS_S_CONTINUE_NEEDED)
	  spkm3_log_status ("spkm3_gss_init_sec_context returning:",
			    major_status, *minor_status);

	return(major_status);
}

