/*
 *  spkm3/import_name.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "gssapi_spkm3.h"

/*
 * Function: spkm3_set_name()
 * 
 * helper furnction for import name.
 */
OM_uint32
spkm3_set_name(spkm3_name_desc_t *cn, gss_buffer_t name_in)
{

	if (name_in->length <= 0)
		return EINVAL;	/* XXX */

	cn->string_name = (char *) malloc (name_in->length + 1);
	if (cn->string_name == NULL)
		return ENOMEM;

	memcpy (cn->string_name, name_in->value, name_in->length);
	cn->string_name[name_in->length] = 0;
	cn->length = name_in->length;
	return 0;
}

/*
 * Function: spkm3_gss_import_name()
 *
 * Description: creates an internal representation of a name.
 *
 * XXX: spkm3_name_desc_t saves both the oid and a type. do we need both?
 */
OM_uint32
spkm3_gss_import_name (
			      OM_uint32 * min_status,	/* minor_status */
			      gss_buffer_t name_in,	/* input_name_buffer */
			      gss_OID name_oid_in,	/* input_name_type */
			      gss_name_t * name_out)
{				/* output_name */
	spkm3_name_desc_t *cn;
	char *ptr;
	OM_uint32 maj_status = GSS_S_COMPLETE;
	gss_buffer_desc out = {
		.length = 0,
	};
	gss_buffer_t outp = &out;
	asn_set *as = NULL;
	int count;

	*name_out = NULL;
	*min_status = 0;

	SPKM3_DEBUG (1, ("spkm3_gss_import_name: START\n"));

	cn = (spkm3_name_desc_t *) calloc (1, sizeof (spkm3_name_desc_t));
	if (cn == NULL) {
		*min_status = ENOMEM;
		maj_status = GSS_S_FAILURE;
		goto exit;
	}
	cn->type = SPKM3_NT_INVALID;

	if ((name_oid_in == GSS_C_NO_OID) ||
	    g_OID_equal (name_oid_in, GSS_C_NT_ANONYMOUS)) {
		cn->string_name =
		   (char *) malloc (strlen (SPKM3_ANONYMOUS_STRING) + 1);
		if (cn->string_name == NULL) {
			*min_status = ENOMEM;
			maj_status = GSS_S_FAILURE;
			goto free_cn;
		}
		cn->length = strlen (SPKM3_ANONYMOUS_STRING);
		memcpy (cn->string_name, SPKM3_ANONYMOUS_STRING, cn->length);
		cn->string_name[strlen (SPKM3_ANONYMOUS_STRING)] = 0;
		cn->type = SPKM3_NT_ANONYMOUS_NAME;
		cn->oid = GSS_C_NT_ANONYMOUS;
	}
	else if (g_OID_equal (name_oid_in, GSS_C_NT_USER_NAME)) {

		SPKM3_DEBUG(1, ("spkm3_gss_import_name:  GSS_C_NT_USER_NAME\n"));

		*min_status = spkm3_set_name(cn, name_in);
		if (*min_status != 0) {
			maj_status = GSS_S_BAD_NAME;	
			goto free_cn;
		}	

		SPKM3_DEBUG(1, ("spkm3_gss_import_name: parsing name '%*s'\n",
			    cn->length, cn->string_name));

		if ((*min_status = string_to_asn1_name(cn)) != 0) {
			maj_status = GSS_S_BAD_NAME;
			goto free_asn1_name;
		}

		/* XXX check as->count to see if a name was actually encoded? */
		as = (asn_set *)&cn->asn1_name->choice.rdnSequence.list;
		if (as->count == 0) {
			*min_status = EINVAL;
			maj_status = GSS_S_BAD_NAME;
			goto free_asn1_name;
		}
		cn->type = SPKM3_NT_USER_NAME;
		cn->oid = GSS_C_NT_USER_NAME;
	}
	else if (g_OID_equal (name_oid_in, GSS_C_NT_EXPORT_NAME)) {
		/* XXX */
		SPKM3_DEBUG (1, ("spkm3_gss_import_name: need to support "
			     "GSS_C_NT_EXPORT_NAME?\n"));
		maj_status = GSS_S_BAD_NAMETYPE;
		goto free_cn;

	}
	else if ((g_OID_equal (name_oid_in, GSS_C_NT_HOSTBASED_SERVICE)) ||
		 (name_oid_in == GSS_C_NULL_OID)) {

		*min_status = spkm3_set_name(cn, name_in);
		if (*min_status != 0) {
			maj_status = GSS_S_BAD_NAME;	
			goto free_cn;
		}	
		if ((ptr = strchr (cn->string_name, '@')) != NULL) {
			*ptr = '/';
		}

		SPKM3_DEBUG (1, ("spkm3_gss_import_name: parsing name '%*s'\n",
			     cn->length, cn->string_name));

		if ((*min_status = string_to_asn1_name(cn)) != 0) {
			maj_status = GSS_S_BAD_NAME;
			goto free_asn1_name;
		}

		/* XXX check as->count to see if a name was actually encoded? */
		as = (asn_set *)&cn->asn1_name->choice.rdnSequence.list;
		if (as->count == 0) {
			*min_status = EINVAL;
			maj_status = GSS_S_BAD_NAME;
			goto free_asn1_name;
		}
		cn->oid = GSS_C_NT_HOSTBASED_SERVICE;
		if (ptr)
			cn->type = SPKM3_NT_SERVICE_NAME;
		else if (as->count == 1)
			cn->type = SPKM3_NT_SIMPLE_NAME;
		else
			cn->type = SPKM3_NT_DISTINGUISHED_NAME;
#if 0
	}
	if ((g_OID_equal (name_oid_in, GSS_C_NT_HOSTBASED_SERVICE)) ||
	    ((name_oid_in == GSS_C_NULL_OID) &&
	     (strchr (cn->string_name, '@')))) {
		/*
		 * They specified GSS_C_NT_HOSTBASED_SERVICE, or they
		 * didn't specify anything and provided what looks like
		 * a host-based service name
		 */
		cn->string_name = (char *) malloc (name_in->length + 1);
		if (cn->string_name == NULL) {
			*min_status = ENOMEM;
			maj_status = GSS_S_FAILURE;
			goto free_cn;
		}
		memcpy (cn->string_name, name_in->value, name_in->length);
		cn->string_name[name_in->length] = 0;

		if ((ptr = strchr (cn->string_name, '@'))) {
			*ptr = '/';
		}
		else {
			/* They specified GSS_C_NT_HOSTBASED_SERVICE,
			 * but there is no service name?
			 */
			*min_status = EINVAL;
			maj_status = GSS_S_BAD_NAME;
			goto free_cn_name;
		}
		cn->length = strlen (cn->string_name);
		cn->type = SPKM3_NT_SERVICE_NAME;

	}
	else if (name_oid_in == GSS_C_NULL_OID) {
		/*
		 * They didn't specify a name type.  See if it is parsable
		 * as an X509 name.
		 */
		cn->string_name = (char *) malloc (name_in->length + 1);
		if (cn->string_name == NULL) {
			*min_status = ENOMEM;
			maj_status = GSS_S_FAILURE;
			goto free_cn;
		}
		memcpy (cn->string_name, name_in->value, name_in->length);
		cn->string_name[name_in->length] = 0;

		if (( *min_status = string_to_asn1_name (cn)) != 0) {
			maj_status = GSS_S_BAD_NAME;
			goto free_asn1_name;
		}

		as = (asn_set *)&cn->asn1_name->choice.rdnSequence.list;
		if (as->count == 0) {
			*min_status = EINVAL;
			maj_status = GSS_S_BAD_NAME;
			goto free_asn1_name;
		}
		if (as->count == 1)
			cn->type = SPKM3_NT_SIMPLE_NAME;
		else
			cn->type = SPKM3_NT_DISTINGUISHED_NAME;
#endif /* 0 */
	}
	else {
		*min_status = EINVAL;
		maj_status = GSS_S_BAD_NAMETYPE;
		goto free_cn;
	}

	*name_out = cn;
	goto exit;

free_asn1_name:
	if (cn->asn1_name)
		asn_DEF_Name.free_struct (&asn_DEF_Name, cn->asn1_name, 0 /*1*/);
free_cn_name:
	if (cn->string_name)
		free (cn->string_name);
free_cn:
	if (cn)
		free (cn);

exit:
	if(maj_status != GSS_S_COMPLETE)
	  spkm3_log_status ("spkm3_gss_import_name:", maj_status, *min_status);

	return (maj_status);
}
