/*
 *  spkm3/gssapi_spkm3.h 
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/* Pre-req openssl headers */
#include <openssl/asn1.h>
#include <openssl/asn1_mac.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/err.h>
#include <openssl/x509_vfy.h>
/* DH libraries **/
#include <openssl/dh.h>
#include <openssl/bio.h>
#include <openssl/des.h>

#include "spkm3_asn1_all.h"
#include <gssapi/gssapi.h>
#include "gssapiP_generic.h"

#include <limits.h>

typedef A_SET_OF(void) asn_set;

/* krb5 routines - XXX these will be replaced */
void g_make_token_header(gss_OID mech, int body_size,
                          unsigned char **buf, int tok_type);

gss_int32 g_verify_token_header(gss_OID mech, int *body_size,
                          unsigned char **buf, int tok_type, int toksize);

/* certificate locations */
#define CertFile "/etc/spkm/spkm.cert"
#define KeyFile "/etc/spkm/spkm.key"
#define CAFile NULL
#define CADir "/etc/grid-security/certificates"
#define CRLFile NULL
#define CRLDir NULL

#if 0
/* krb5 declared static oid's */
extern gss_OID gss_nt_user_name;
extern gss_OID gss_nt_machine_uid_name;
extern gss_OID gss_nt_string_uid_name;
extern gss_OID gss_nt_service_name;
extern gss_OID gss_nt_exported_name;
#endif

/* spkm3 static oid's */
extern gss_OID gss_mech_oid_spkm3;
extern gss_OID gss_mech_oid_ialg_nullmac;
extern gss_OID_set spkm3_oid_set;

/* constants */
#define SPKM3_RAND_LEN              8
#define SPKM3_CONTEXTID_LEN         (2 * SPKM3_RAND_LEN)
#define SPKM3_MIN_CONTEXTID_LEN     (SPKM3_CONTEXTID_LEN - 2)
#define SPKM3_MIN_MD5_DIGEST_LENGTH (MD5_DIGEST_LENGTH -2)
#define SPKM3_SIGNATURE_LEN         128
#define SPKM3_MIN_SIGNATURE_LEN     (SPKM3_SIGNATURE_LEN - 2)

/* token types */
#define SPKM_TOK_CTX_INITIAL	0x0
#define SPKM_TOK_CTX_REQ    0x0100
#define SPKM_TOK_CTX_REPTI  0x0200
#define SPKM_TOK_CTX_REPIT  0x0300
#define SPKM_TOK_CTX_ERROR  0x0400
#define SPKM_TOK_SIGN_MSG   0x0101	/* V1 */
#define SPKM_TOK_MIC_MSG    0x0101
#define SPKM_TOK_SEAL_MSG   0x0201	/* V1 */
#define SPKM_TOK_WRAP_MSG   0x0201
#define SPKM_TOK_DEL_CTX    0x0301

/* Bit settings for the Options bit string within Context_Data */
typedef enum {
	SPKM_OPTIONS_DELEGATION_STATE = 0,
	SPKM_OPTIONS_MUTUAL_STATE = 1,
	SPKM_OPTIONS_REPLAY_DET_STATE = 2,
	SPKM_OPTIONS_SEQUENCE_STATE = 3,
	SPKM_OPTIONS_CONF_AVAIL = 4,
	SPKM_OPTIONS_INTEG_AVAIL = 5,
	SPKM_OPTIONS_TARGET_CERTIF_DATA_REQUIRED = 6
} spkm_options_t;

/* Context states */
typedef enum {
	SPKM3_CTX_STATE_INVALID = 0,	/* context is invalid */
	SPKM3_CTX_STATE_NEW = 1,	/* context has been initialized */
	SPKM3_CTX_STATE_REPTI = 2,	/* expecting REPTI message */
	SPKM3_CTX_STATE_REPIT = 3,	/* expecting REPIT message */
	SPKM3_CTX_STATE_AUTH = 4	/* context negotiation is complete */
} spkm3_ctx_state_type;

/* XXX This belongs somewhere else! */
typedef enum {
	GSS_INIT_TOKEN = 1,
	GSS_ACCEPT_TOKEN = 2,
	GSS_ERROR_TOKEN = 3,
	GSS_SIGN_TOKEN = 4,		/* V1 */
	GSS_GETMIC_TOKEN = 4,
	GSS_SEAL_TOKEN = 5,		/* V1 */
	GSS_WRAP_TOKEN = 5,
	GSS_DELETE_TOKEN = 6
} gss_token_type;

typedef enum {
	SPKM3_USERTYPE_NONE = 0,
	SPKM3_USERTYPE_SYSTEM = 1,
	SPKM3_USERTYPE_USER = 2
} spkm3_usertype_t;

typedef enum {
	SPKM3_FILETYPE_PRIV_KEY = 1,
	SPKM3_FILETYPE_CERT = 2
} spkm3_filetype_t;

/* minor status values (implementation specific...)*/
/* XXX - RFC 2025 section 5.1.1 has value constraints that i'm
 * ignoring....
 */
#define GSS_S_G_VALIDATE_FAILED                   0x0001
#define GSS_S_G_BUFFER_ALLOC                      0x0002
#define GSS_S_G_BAD_MSG_CTX                       0x0003
#define GSS_S_G_WRONG_SIZE                        0x0004
#define GSS_S_G_BAD_USAGE                         0x0005
#define GSS_S_G_UNAVAIL_QOP                       0x0006
#define GSS_S_G_MEMORY_ALLOC                      0x0007
#define GSS_SPKM_S_SG_CONTEXT_ESTABLISHED         0x0008
#define GSS_SPKM_S_SG_BAD_INT_ALG_TYPE            0x0009
#define GSS_SPKM_S_SG_BAD_CONF_ALG_TYPE           0x000A
#define GSS_SPKM_S_SG_BAD_KEY_ESTB_ALG_TYPE       0x000B
#define GSS_SPKM_S_SG_CTX_INCOMPLETE              0x000C
#define GSS_SPKM_S_SG_BAD_INT_ALG_SET             0x000D
#define GSS_SPKM_S_SG_BAD_CONF_ALG_SET            0x000E
#define GSS_SPKM_S_SG_BAD_KEY_ESTB_ALG_SET        0x000F
#define GSS_SPKM_S_SG_NO_PVNO_IN_COMMON           0x0010
#define GSS_SPKM_S_SG_INVALID_TOKEN_DATA          0x0011
#define GSS_SPKM_S_SG_INVALID_TOKEN_FORMAT        0x0012
#define GSS_SPKM_S_SG_CONTEXT_DELETED             0x0013
#define GSS_SPKM_S_SG_BAD_DELETE_TOKEN_RECD       0x0014
#define GSS_SPKM_S_SG_CONTEXT_ESTB_ABORT          0x0015
#define GSS_SPKM_S_SG_OPENSSL_CONVERT_ERROR	  0x0016


/*
 * the SPKM mechanism demands machine credentials.
 * here are the default locations.
 */

#define RPCSEC_GSS_SPKM_CERT  "/etc/spkm/spkm.cert"
#define RPCSEC_GSS_SPKM_KEY  "/etc/spkm/spkm.key"


/* dh stuff - first pass */
#define DES_KEY_SIZE 8

/* structure typedefs */

/* 
 * bookeeping struct for spkm3_encode_to_outbuf ()
 */
struct spkm3_ebuf {
        unsigned char * cur;
        unsigned char * buf;
};

/*
 * Per-session context-id established between
 * initiator and target
 */
typedef struct spkm3_context_id_struct {
    int length;
    unsigned char * data;
} spkm3_context_id_t;

/*
 * store full X509 DN as null terminated string 
 * in data. i currently concatenate the on the wire
 * entries into one string which may change...
 */
typedef struct spkm3_name_desc_struct {
    gss_OID oid;		/* name OID */
    unsigned int type;		/* name type */
    int length;			/* length of string */
    char * string_name;		/* string representation */
    Name_t * asn1_name;		/* wire ready name */
} spkm3_name_desc_t;

#define SPKM3_ANONYMOUS_STRING "<anonymous>"
#define SPKM3_NT_INVALID                0x00000000
#define SPKM3_NT_UNKNOWN                0xffffffff
#define SPKM3_NT_SERVICE_NAME           0x00000001
#define SPKM3_NT_SIMPLE_NAME            0x00000002
#define SPKM3_NT_DISTINGUISHED_NAME     0x00000003
#define SPKM3_NT_ANONYMOUS_NAME         0x00000004
#define SPKM3_NT_USER_NAME              0x00000005

/* used to hold requestor alg oid's */
#define ALGLISTSIZE  9

typedef struct spkm3_alg {
	AlgorithmIdentifier_t alg;
	int nid;
} spkm3_alg_t;

/*
 * spkm3 internal context structure
 */

typedef struct spkm3_context_struct {
    spkm3_context_id_t ctx_id;	/* per spkm token contextid */
    spkm3_ctx_state_type state;
    int established;
    int qop;			/* negotiated qop */
    gss_OID mech_used;
    OM_uint32 ret_flags;
    OM_uint32 req_flags;
    /* DH should be abstracted to an EVP_ struct able to hold
     * various kalg results */
    DH *dh;
    gss_buffer_desc share_key;
    /* derived keys are result from applying the owf_alg to the 
     * shared key - see spkm3_derive_supkey */
    gss_buffer_desc derived_conf_key;
    gss_buffer_desc derived_integ_key;
 
   /* OID's of the negotiated algorithms */
    gss_OID keyestb_alg;	/* key establishment */
    gss_OID owf_alg;		/* one way function */
    gss_OID intg_alg;		/* integrity */
    gss_OID conf_alg;		/* privacy */
 
   /* der encoded REQ_TOKEN reqcontets and length */
    unsigned char * der_reqcontents;
    int der_req_len;
    InitialContextToken_t * in_ict;
    gss_cred_id_t spkm3_cred;	/* credentials handle */
    OM_uint32 spkm_tok_type;
    X509 *usercert;
    STACK_OF(X509 *) *user_certchain;
    Random_Integer_t *randSrc;
    Random_Integer_t *randTarg;
    Name_t *src_name;
    Name_t *targ_name;
} spkm3_context_t;


/* first pass at spkm3 internal credentials structure. */
typedef struct spkm3_cred_int_struct {
    Certificate_t 	* wire_cert;
    Name_t 		* wire_name;
    CertificatePair_t 	* wire_certchain;
    int 		  certchain_len;
    X509		* cert;	
    EVP_PKEY		* priv_key;
    int cert_type;
} spkm3_cred_int_t;

#define SPKM3_STANDART_CERT 		0
#define SPKM3_GLOBUS_PROXY_CERT    	1

/* function prototypes */

EVP_PKEY *
spkm3_load_key(
    char * file);

X509  *spkm3_load_cert(
    char * file,
    int * cert_type);

Certificate_t *
spkm3_load_asn_cert(
    char * file,
    int cert_type,
    Name_t **wire_name);

CertificatePair_t *
spkm3_load_asn_certchain(
	X509 *cert, 
	char *file, 
	char *cafile, 
	char *cadir, 
	int *certchain_len,
	OM_uint32 *minor_status);

OM_uint32 spkm3_set_srcname(
    gss_name_t * src_name,
    Name_t * in_name,
    int req_flags);

int get_random_bytes(
    unsigned char * buffer,
    int n,
    int critical);

void spkm3_log_status(
    char * m,
    OM_uint32 maj_stat,
    OM_uint32 min_stat);

void print_rand(
    unsigned char * r,
    int len);

void print_hex(
    unsigned char * data,
    int len);
    
void print_dh_sharekey(
    gss_buffer_desc * key);

void print_name_desc(
    spkm3_name_desc_t * name);

void print_req_token(
    REQ_TOKEN_t * req);

void print_rep_ti_token(
    REP_TI_TOKEN_t * rep);

void print_context_data(
    Context_Data_t * cd);

void print_x509_name(
    X509_NAME * xn);

char * X509_NAME_oneline_readable(
    X509_NAME * a,
    char * buf,
    int * size);

char * X509_NAME_oneline_rfc2253(
    X509_NAME * a,
    char * buf,
    int * size);

char * X509_NAME_oneline_rfc2253_rev(
    X509_NAME * a,
    char * buf,
    int * size);

X509_NAME * a2i_X509_NAME(
    const char * ascii_string,
    int length);

OM_uint32 spkm3_parse_token_internal(
    spkm3_context_t ** ctx,
    gss_buffer_t input_token);

OM_uint32 spkm3_add_srcname(
    spkm3_context_t * ctx,
    STACK * sk);

OM_uint32 spkm3_context_create(
    OM_uint32 * minor_status,
    spkm3_context_t ** context_handle_P);

void spkm3_context_free(
    spkm3_context_t * ctx);

OM_uint32 spkm3_context_get(
    OM_uint32 * min_status,
    spkm3_context_id_t * ctx,
    spkm3_context_t ** context);

OM_uint32 spkm3_gss_context_data(
    Context_Data_t * reqdata,
    OM_uint32 req_flags);

OM_uint32 x509_algid_set(
    X509_ALGOR * x,
    int nid);

OM_uint32 x509_algor_dh_set(
    X509_ALGOR ** x,
    int nid,
    DH * dh);

OM_uint32 REQ_create(
    spkm3_context_t * ctx,
    gss_buffer_t output_token,
    spkm3_name_desc_t * target_name);

OM_uint32 REQ_process(
    spkm3_context_t * ctx);

OM_uint32 REQ_integrity_sign(
    spkm3_context_t * ctx,
    SPKM_REQ_t *spkm_req);

OM_uint32 REQ_integrity_verify(
    spkm3_context_t * ctx,
    SPKM_REQ_t *spkmreq);

OM_uint32 REPIT_create(
    spkm3_context_t * ctx,
    gss_buffer_t output_token);

OM_uint32 REPIT_process(
    spkm3_context_t * ctx);

OM_uint32 REPIT_integrity_sign(
    spkm3_context_t * ctx,
    SPKM_REP_IT_t * repit);

OM_uint32 REPIT_integrity_verify(
    spkm3_context_t * ctx,
    SPKM_REP_IT_t *req);

OM_uint32 REPTI_create(
    spkm3_context_t * ctx,
    gss_buffer_t output_token);

OM_uint32 REPTI_process(
    spkm3_context_t * ctx);

OM_uint32 REPTI_integrity_sign(
    spkm3_context_t * ctx,
    REP_TI_TOKEN_t * rep,
    REQ_TOKEN_t * req);

OM_uint32 REPTI_integrity_verify(
    spkm3_context_t * ctx,
    SPKM_REP_TI_t *);

int verify_callback(int ok, X509_STORE_CTX *ctx);
int get_cert_type(X509 *x);

#if 0
int spkm3int_targetname_matches_cert(
    X509_NAME * target,
    X509 * cert);
#else
int targetname_matches_cert(
    Name_t * target,
    Certificate_t * cert);
#endif

int spkm3_bit_string_repair(
    BIT_STRING_t *bs,
    int newlen);

OM_uint32 create_initial_ct(
    InitialContextToken_t **ict);

/* ASN1 asn1c helper routines */
OM_uint32 asn1_copy_oid(
    gss_OID oid,
    MechType_t * mech);

OM_uint32 asn1_cmp_oid(
    gss_OID o,
    MechType_t * m);

int spkm3_encode_to_outbuf(
    const void * buffer,
    size_t size,
    void * outbuf);

OM_uint32 oid_2_asn1alg(void *asn_set_of_x, const gss_OID_desc *oid);

BIT_STRING_t * asn1_bitstring_dup(
    BIT_STRING_t *in);

int asn1_bitstring_set(
    BIT_STRING_t * str,
    const void * data,
    size_t len);

int asn1_set_bit(
    BIT_STRING_t * bstr,
    int bitnum,
    int value);

int asn1_get_bit(
    BIT_STRING_t * bstr,
    int bitnum);

OM_uint32 x509_to_asn1_name(
    X509_NAME * xname,
    Name_t * name);

OM_uint32 asn1_algor_dh_set(
    void * asn_set_of_x,
    gss_OID oid,
    DH * dh);
OM_uint32 asn1_algor_dh_get (
    unsigned char *p, 
    long len, 
    DH **dh);
void asn1_2_ssl_int(
    ASN1_INTEGER ** sslint,
    INTEGER_t * aint);

OM_uint32 asn1_algor_stack_get_list(
    spkm3_context_t * ctx,
    void * asn_set_of_x);

Certificate_t * asn1_cert_dup(
    Certificate_t * cert);

AttributeValue_t * asn1_get_entry_by_oid(
    Name_t * name,
    OBJECT_IDENTIFIER_t * oid);

OBJECT_IDENTIFIER_t * txt_2_asn1_obj(
    char * s);

OM_uint32 der_encode_asn1_name(
    Name_t ** namep,
    gss_buffer_t * outp);

OM_uint32 asn1_name_to_string(
    Name_t * name,
    spkm3_name_desc_t * out);

OM_uint32 asn1_name_copy(
    Name_t * dst,
    const Name_t * src);

Name_t * asn1_name_dup(
    const Name_t * src);

int spkm3_get_filename(
    spkm3_filetype_t filetype,
    spkm3_usertype_t usertype,
    char * name,
    int namelen);

OM_uint32 spkm3_derive_subkey (int type, spkm3_context_t * ctx);
OM_uint32 encode(struct asn_TYPE_descriptor_s *asn_type,
                 void *req_contents,
                 unsigned char **der_contents,
                 size_t *der_len);

OM_uint32 sign(AlgorithmIdentifier_t *,
	       struct asn_TYPE_descriptor_s *req_asn_type,
	       void *req_contents,
	       struct asn_TYPE_descriptor_s *rep_asn_type,
	       void *rep_contents,
	       BIT_STRING_t *integrity,	       
	       spkm3_cred_int_t *creds);

OM_uint32 verify (AlgorithmIdentifier_t *,
                  struct asn_TYPE_descriptor_s *req_asn_type,
                  void *req_contents,
                  unsigned char *saved_content,
                  int saved_len,
                  X509 * x509,
		  STACK_OF(X509 *) *certchain,
                  unsigned char *sig_buf, int sig_len);

/* context cache helper macros */
extern void * kg_vdb;

#define kg_save_ctx_id(ctx)   g_save_ctx_id(&kg_vdb,ctx)
#define kg_validate_ctx_id(ctx)   g_validate_ctx_id(&kg_vdb,ctx)
#define kg_delete_ctx_id(ctx)   g_delete_ctx_id(&kg_vdb,ctx)

/* Debugging */
int spkm3_set_debug_level (int new_level);
void spkm3_log_err (char *format, ...);
void spkm3_log_dbg (char *format, ...);
void spkm3_log_hexdump (const u_char *buf, int len, int offset);

#define spkm3_debug_level 2

#define BEGINMAC do {
#define ENDMAC   } while (0)

#ifdef DEBUG
#define SPKM3_DEBUG(lvl, msg) \
        BEGINMAC \
        if ((lvl) <= spkm3_debug_level) { \
        	spkm3_log_dbg("(%s:%d) ", __FILE__, __LINE__); \
                spkm3_log_dbg msg; } \
        ENDMAC
#else
#define SPKM3_DEBUG(lvl, msg) \
        BEGINMAC \
        if ((lvl) <= spkm3_debug_level) { \
                spkm3_log_dbg msg; } \
        ENDMAC
#endif
#define SPKM3_DEBUG_HEX(lvl, msg) \
        BEGINMAC \
        if ((lvl) <= spkm3_debug_level) \
               	spkm3_log_hexdump msg; \
        ENDMAC

#define SPKM3_DEBUG_X509_NAME(lvl, msg) \
        BEGINMAC \
        if ((lvl) <= spkm3_debug_level) \
                print_x509_name msg; \
        ENDMAC

/* OIDs declarations for K-ALG, I-ALG, C-ALG, and OWF-ALG */
static const gss_OID_desc dh_oid = {9, "\x2A\x86\x48\x86\xF7\x0D\x01\x03\x01"};
static const gss_OID_desc hmac_md5_oid = { 8, "\x2B\x06\x01\x05\x05\x08\x01\x01"};
static const gss_OID_desc null_mac_oid = { 6, "\x2B\x06\x01\x05\x03\x03"};
static const gss_OID_desc md5_rsa_encryption_oid = {9, "\x2A\x86\x48\x86\xF7\x0D\x01\x01\x04"};

static const gss_OID_desc cast5_cbc_oid = {9, "\x2A\x86\x48\x86\xF6\x7D\x07\x42\x0A"};
static const gss_OID_desc des_cbc_oid = {5, "\x2B\x0E\x03\x02\x07"};

static const gss_OID_desc sha1_oid = {5, "\x2B\x0E\x03\x02\x1A"};

static const gss_OID_desc globus_proxy_oid_1 = {10, "\x2B\x06\x01\x04\x01\x9b\x50\x01\x81\x5e"};


#define GLOBUS_GSI_PROXY_GENERIC_POLICY_OID "1.3.6.1.4.1.3536.1.1.1.8"
#define GLOBUS_GSI_PROXY_GENERIC_POLICY_OID_2 "1.3.6.1.4.1.3536.1.222"
#define GLOBUS_GSI_PROXY_GENERIC_POLICY_OID_3 "1.3.6.1.5.5.7"

