/*
 *  spkm3/cred.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <sys/types.h>
#include <strings.h>
#include <unistd.h>
#include "gssapi_spkm3.h"


/*
* Function: spkm3_load_key()
*
* Description: read a private key from the filesystem.
*
*/

EVP_PKEY *
spkm3_load_key (char *file)
{
	EVP_PKEY *pkey = NULL;
	BIO *key;

	if ((key = BIO_new (BIO_s_file ()))
	    && (BIO_read_filename (key, file) > 0))
		pkey = PEM_read_bio_PrivateKey (key, NULL, NULL, NULL);
	if (key != NULL)
		BIO_free (key);
	return (pkey);
}

/*
* Function:  spkm3_load_cert()
*
* Description: read a certificate from the filesystem.
*
*/

X509 *
spkm3_load_cert (char *file, int *cert_type)
{
	X509 *x = NULL;
	BIO *cert = NULL;

	*cert_type = SPKM3_STANDART_CERT;

	if ((cert = BIO_new (BIO_s_file ()))
	    && (BIO_read_filename (cert, file) > 0))
		x = PEM_read_bio_X509 (cert, NULL, NULL, NULL);
	if (cert != NULL)
		BIO_free (cert);

	/* test for the type of the cert */
#if OPENSSL_LIBRARY_VERSION > 0x00908000
      	if (xs->ex_flags & EXFLAG_PROXY) {	
		SPKM3_DEBUG(1, ("spkm3_load_cert: this is a proxy cert\n"));
		*cert_type = SPKM3_GLOBUS_PROXY_CERT;
		goto out;
      	} 
#endif
	if(x) { 
		SPKM3_DEBUG(1, ("spkm3_load_cert: this is a proxy cert\n"));
		*cert_type = get_cert_type(x);
	}
out:
	return (x);
}

Certificate_t *
spkm3_load_asn_cert(char *file, int cert_type, Name_t **wire_name)
{
	Certificate_t *cert = NULL;
	OM_uint32 minor_status;
	asn_dec_rval_t rval;
	BIO *bc;
	unsigned char *pp = NULL, *data = NULL;
	X509 *ret;
	long len;

	if ((bc = BIO_new (BIO_s_file ()))
			&& (BIO_read_filename (bc, file) > 0)) {

		if (!PEM_bytes_read_bio(&data, &len, NULL, PEM_STRING_X509, bc, NULL, NULL))
			goto out_free;
	} else
		goto out;

        rval = ber_decode(0, &asn_DEF_Certificate,
                        (void **)&cert, data, (size_t) len);

        if (rval.code != RC_OK) {
		spkm3_log_err("decode failed\n");
		goto out_free;
        }
#ifdef ASNDEBUG
        asn_fprint(stderr, &asn_DEF_Certificate, cert);
#endif

	/* if this is a proxy certificate, save the subject name of the
         * entity certificate.
         */
	if(cert_type == SPKM3_GLOBUS_PROXY_CERT) {
		Certificate_t *ee_cert = NULL;
		free(data); data = NULL;
		len = 0;

		/* read the 2nd certificate in the file.
		 * it should be the EE cert.
                 */
		if (!PEM_bytes_read_bio(&data, &len, NULL, PEM_STRING_X509, 
					bc, NULL, NULL))
                        goto out_free;

		rval = ber_decode(0, &asn_DEF_Certificate,
                        (void **)&ee_cert, data, (size_t) len);

		if (rval.code != RC_OK) {
                	spkm3_log_err("decode failed\n");
                	goto out_free;
		}

		*wire_name = asn1_name_dup(&ee_cert->tbsCertificate.subject);

		asn_DEF_Certificate.free_struct(&asn_DEF_Certificate, 
						ee_cert, 0);
	}
	else if(cert_type == SPKM3_STANDART_CERT) {
		*wire_name = asn1_name_dup(&cert->tbsCertificate.subject);
	}


out:
	if(data) free(data);
	if(bc) BIO_free(bc);
	return cert;
out_free:
	asn_DEF_Certificate.free_struct (&asn_DEF_Certificate, cert, 0);
	cert = NULL;
	goto out;
}

CertificatePair_t *spkm3_load_asn_certchain(X509 *cert, char *filename, 
                                            char *cafile, char *cadir,
					    int *certchain_len,
					    OM_uint32 *minor_status) {

	char buf[256];
	int i = 0;
	X509_STORE *certstore = NULL;
	X509_STORE_CTX certstorectx;
	STACK_OF(X509) *chn = NULL;
	CertificatePair_t *asnchn;
	
	if(cert == NULL) return NULL;

	SPKM3_DEBUG(1, ("spkm3_load_asn_certchain: loading cert chain for %s\n",
			X509_NAME_oneline(X509_get_subject_name(cert), buf, 
			sizeof(buf))));


	/* build a list of all the certs */
	CRYPTO_malloc_init();
  	OpenSSL_add_all_algorithms();

  	if((certstore = X509_STORE_new()) == NULL) {
    		ERR_print_errors_fp (stderr);
    		spkm3_log_err("verify_X509_cert: X509_STORE_new failed\n");
    		return NULL; 
  	}

	X509_STORE_set_verify_cb_func(certstore, verify_callback);

	if(!X509_STORE_load_locations(certstore, filename, NULL)) {
                ERR_print_errors_fp (stderr);
                spkm3_log_err("verify_X509_cert: failed reading %s\n",
                              filename);
                *minor_status = -1; goto err;
        }


	if(!X509_STORE_load_locations(certstore, CAFile, CADir)) {
    		ERR_print_errors_fp (stderr);
    		spkm3_log_err("verify_X509_cert: failed reading %s or %s\n",
                  	      CADir, CAFile);
    		*minor_status = -1; goto err;
  	}

  	X509_STORE_CTX_init(&certstorectx, certstore, cert, NULL);
	if(!X509_verify_cert(&certstorectx)) {
		ERR_print_errors_fp (stderr);
		spkm3_log_err("verify_X509_cert: failed to verify cert\n");
		*minor_status = -1; goto err;
	}

	chn =  X509_STORE_CTX_get1_chain(&certstorectx);
	if((*certchain_len = sk_X509_num(chn)) > 0) {

  		SPKM3_DEBUG(1, ("spkm3_load_asn_certchain: cert chain length "
				"%d\n", *certchain_len));

		if((asnchn = (CertificatePair_t *)
                	calloc(*certchain_len, 
			sizeof(CertificatePair_t))) == NULL) {
			
			*minor_status = GSS_S_G_MEMORY_ALLOC;
			goto err;
		}
		for(i = 0; i < *certchain_len; i++) {
			X509 *x = NULL;
			unsigned char *data = NULL, *p = NULL;
			int len = 0;
			asn_dec_rval_t rval;

			if((x = sk_X509_value(chn, i)) != NULL) {
        			SPKM3_DEBUG(1, 
				("spkm3_load_asn_certchain: #%d cert in "
				"chain %s\n", i, 
				X509_NAME_oneline(X509_get_subject_name(x), 
						  buf, sizeof(buf))));

				if((asnchn[i].issuedByThisCA = 
					(Certificate_t *) calloc(1, 
                                        sizeof(Certificate_t))) == NULL) {
				
					*minor_status = GSS_S_G_MEMORY_ALLOC;
					goto err;
				}	

				asnchn[i].issuedToThisCA = NULL;

				len = i2d_X509(x, NULL);
				if(len <= 0 || len >= INT_MAX || 
			   	(data = (unsigned char *)
                             	calloc(len, sizeof(unsigned char))) == NULL) {
					*minor_status = GSS_S_G_MEMORY_ALLOC;
					goto err;
				}
				p = data;
				if((len = i2d_X509(x, &p)) <= 0) {
					spkm3_log_err("i2d_X509() failed\n");
					*minor_status = -1;
					free(data);
					goto err;
				}

				rval = ber_decode(0, &asn_DEF_Certificate,
                        		(void **)&asnchn[i].issuedByThisCA, 
					data, (size_t) len);

        			if (rval.code != RC_OK) {
                			spkm3_log_err("ber_decode failed\n");
					*minor_status = -1;
					free(data);
                			goto err;
        			}
				free(data);
			}

		}
	}

err:

	if(chn) {
		for(i = 0; i < sk_X509_num(chn); i++) {
			X509_free(sk_X509_value(chn, i));
		}
		sk_X509_free(chn);
	}
	if(certstore) X509_STORE_free(certstore);
  	X509_STORE_CTX_cleanup(&certstorectx);

  	SPKM3_DEBUG(1, ("spkm3_load_asn_certchain: END %d\n", *minor_status));

  	return asnchn;

}
