/*
 *  spkm3/context.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <signal.h>
#include "gssapi_spkm3.h"
#include "spkm3_mech.h"
#include "gssapiP_generic.h"

void * kg_vdb = NULL;

/* ordered list of supported confidentiality algorithms */
static gss_OID_desc sup_conf_algs[] = { 
	{9, "\x2A\x86\x48\x86\xF6\x7D\x07\x42\x0A"}, /* cast5 oid */
	{0,0}
};

/* ordered list of supported integrity algorithms */
static const gss_OID_desc sup_intg_algs[] = { 
	{8, "\x2B\x06\x01\x05\x05\x08\x01\x01"},     /* hmac-md5 */
	{9, "\x2A\x86\x48\x86\xF7\x0D\x01\x01\x04"}, /* md5WithRSAEncryption */
	{6, "\x2B\x06\x01\x05\x03\x03"},	     /* null-mac */
	{0,0}
};

/* ordered list of supported oneway functions */
static const gss_OID_desc sup_owf_algs[] = { 
	{5, "\x2B\x0E\x03\x02\x1A"}, 		     /* sha1_oid */
	{0,0} 
};

/* ordered list of supported key establishment algorithms */
static const gss_OID_desc keyest_algs[] = { 
	{9, "\x2A\x86\x48\x86\xF7\x0D\x01\x03\x01"}, /* dh_oid */
	{0,0} 
};

/*********************************************************/
/***            Initiator side DH functions            ***/
/*********************************************************/

/*
 * The following prime was generated via DH_generate_parameters
 * Use it instead of constantly getting new primes.
 */
static int spkm3_dh_generator = DH_GENERATOR_5;
static unsigned char spkm3_dh_prime[] = {
  0xa3, 0xe7, 0x5f, 0xcd, 0x25, 0x15, 0x42, 0x2d,
  0x43, 0x29, 0xe2, 0xec, 0xd9, 0xd0, 0x67, 0x4c,
  0xab, 0xd8, 0xb0, 0x7a, 0x39, 0x2c, 0xb5, 0x1d,
  0xdb, 0x03, 0xde, 0xef, 0xb0, 0x60, 0x95, 0x48,
  0x55, 0xc7, 0xea, 0x35, 0xe3, 0xb6, 0xc5, 0x46,
  0x75, 0x59, 0xd3, 0x3b, 0x7a, 0x5e, 0x99, 0xe2,
  0x3b, 0xda, 0x64, 0x27, 0xb5, 0xf6, 0xe5, 0xfe,
  0xf7, 0x86, 0xea, 0xad, 0x68, 0x8e, 0x40, 0x3a,
  0x9f, 0x02, 0x90, 0x28, 0xd0, 0x37, 0x19, 0xdc,
  0x1d, 0xaa, 0x44, 0xf8, 0x9a, 0xa6, 0xd4, 0xda,
  0xde, 0xa1, 0x07, 0xa0, 0xb9, 0x2b, 0x33, 0x7e,
  0xc6, 0x08, 0x3e, 0xd3, 0xb0, 0xf7, 0x28, 0x18,
  0x1c, 0x1d, 0x6f, 0xff, 0xe1, 0xac, 0x55, 0x13,
  0x58, 0x7e, 0x50, 0x00, 0x0b, 0x07, 0x62, 0xad,
  0xe2, 0xd8, 0x30, 0x5b, 0x73, 0xba, 0xbb, 0xbd,
  0xf2, 0xe6, 0x4f, 0x9b, 0xf5, 0x46, 0x32, 0xb3
};

static unsigned char dh1024_g[]={
                0x02,
                };
static unsigned char dh1024_p[]={
  0xF8,0x81,0x89,0x7D,0x14,0x24,0xC5,0xD1,0xE6,0xF7,0xBF,0x3A,
  0xE4,0x90,0xF4,0xFC,0x73,0xFB,0x34,0xB5,0xFA,0x4C,0x56,0xA2,
  0xEA,0xA7,0xE9,0xC0,0xC0,0xCE,0x89,0xE1,0xFA,0x63,0x3F,0xB0,
  0x6B,0x32,0x66,0xF1,0xD1,0x7B,0xB0,0x00,0x8F,0xCA,0x87,0xC2,
  0xAE,0x98,0x89,0x26,0x17,0xC2,0x05,0xD2,0xEC,0x08,0xD0,0x8C,
  0xFF,0x17,0x52,0x8C,0xC5,0x07,0x93,0x03,0xB1,0xF6,0x2F,0xB8,
  0x1C,0x52,0x47,0x27,0x1B,0xDB,0xD1,0x8D,0x9D,0x69,0x1D,0x52,
  0x4B,0x32,0x81,0xAA,0x7F,0x00,0xC8,0xDC,0xE6,0xD9,0xCC,0xC1,
  0x11,0x2D,0x37,0x34,0x6C,0xEA,0x02,0x97,0x4B,0x0E,0xBB,0xB1,
  0x71,0x33,0x09,0x15,0xFD,0xDD,0x23,0x87,0x07,0x5E,0x89,0xAB,
  0x6B,0x7C,0x5F,0xEC,0xA6,0x24,0xDC,0x53,
};

void spkm3_init_gss_buffer(gss_buffer_desc *buf);

OM_uint32 spkm3_certifdata_create(spkm3_cred_int_t *cred,
				  CertificationData_t ** certif_data);
OM_uint32 spkm3_keyestab_create(BIT_STRING_t **key_est, DH *dh);
OM_uint32 spkm3_dh_decode(spkm3_context_t * ctx, void *asn_set_of_x);
OM_uint32 spkm3_dh_compute_key(DH *dh, gss_buffer_desc *share_key, 
			       BIT_STRING_t *keyest);

OM_uint32 spkm3_contextid_create(unsigned char **out_buf,
				 int *out_len,
				 unsigned char *in_buf,
				 int in_len);
OM_uint32 spkm3_verify_randsrc(BIT_STRING_t *randSrcREQ,
                               BIT_STRING_t *randSrcTI);

OM_uint32 REQ_contextdata_create(Context_Data_t * reqdata, OM_uint32 flags);
OM_uint32 REQ_keyestab_create(spkm3_context_t *ctx, Req_contents_t *reqcnt);
OM_uint32 REQ_create_mutual (spkm3_context_t * ctx, SPKM_REQ_t *spkm_req);


OM_uint32 REPTI_contextdata_create(spkm3_context_t * ctx, 
				   Context_Data_t * req_data, 
				   Context_Data_t * rep_data);
OM_uint32 REPTI_keyestab_create (spkm3_context_t * ctx, 
				 REP_TI_TOKEN_t * repti);
OM_uint32 REPTI_certifdata_create (spkm3_cred_int_t *ctx, 
				   SPKM_REP_TI_t *spkm_repti, 
				   Name_t *targname);

#define DH_KEYSIZE 1024

/****************************************************************************/
/* spkm3 context create/delete functions */
/****************************************************************************/
/*
 * Function:  spkm3_context_create
 *
 * Description:
 * called by spkm3_gss_init_sec_context and accept_sec_context
 * to create and initalize the context.
 *
 * NOTE:  - no creds used.
 * ctx->req_flags must not be -1 (why not?). determine what are the right
 * initialization values.
 */

OM_uint32
spkm3_context_create (OM_uint32 * minor_status,
		      spkm3_context_t ** context_handle_P)
{
  spkm3_context_t *ctx = NULL;
  
  SPKM3_DEBUG(1, ("spkm3_context_create: START\n"));
 
  *minor_status = GSS_S_G_MEMORY_ALLOC;
  
  ctx = (spkm3_context_t *) calloc (1, sizeof(spkm3_context_t));
  if (ctx == NULL) {
    *minor_status = GSS_S_G_MEMORY_ALLOC;
    return GSS_S_FAILURE;
  }
  
  ctx->state = SPKM3_CTX_STATE_NEW;
  ctx->established = -1;
  ctx->qop = 0;
  ctx->mech_used = gss_mech_oid_spkm3;
  ctx->ret_flags = 0;
  ctx->req_flags = 0; /*XXXX must not be -1 */
  ctx->dh = NULL;

  spkm3_init_gss_buffer(&ctx->share_key);
  spkm3_init_gss_buffer(&ctx->derived_conf_key);
  spkm3_init_gss_buffer(&ctx->derived_integ_key);

  ctx->keyestb_alg = NULL;
  ctx->owf_alg = NULL;
  ctx->intg_alg = NULL;
  ctx->conf_alg = NULL;

  ctx->der_reqcontents = NULL;
  ctx->der_req_len = -1;
  ctx->in_ict = NULL;
  ctx->spkm_tok_type = SPKM_TOK_CTX_INITIAL;
  ctx->usercert = NULL;
  ctx->user_certchain = NULL;
  ctx->src_name = NULL;
  ctx->targ_name = NULL;
  ctx->randSrc = NULL;
  ctx->randTarg = NULL;
  
  *context_handle_P = ctx;
  *minor_status = 0;
  
  return GSS_S_COMPLETE;
}

void spkm3_init_gss_buffer(gss_buffer_desc *buf) {
  buf->length = 0;
  buf->value = NULL;
}

/*
 * Function:  spkm3_context_free
 *
 * Description:
 * called to reap context
 */
void
spkm3_context_free (spkm3_context_t * ctx)
{
  
  if(ctx == NULL) return;
  
  SPKM3_DEBUG(3, ("spkm3_context_free\n"));
  
  if(ctx->ctx_id.data && ctx->ctx_id.length > 0) {
    SPKM3_DEBUG(3, ("freeing ctx_id\n"));
    free(ctx->ctx_id.data);
  }
  if(ctx->der_reqcontents && ctx->der_req_len > 0) {
    SPKM3_DEBUG(3, ("freeing saved reqcontents\n"));
    free(ctx->der_reqcontents);
  }
  if(ctx->usercert) {
    SPKM3_DEBUG(3, ("freeing saved usercert\n"));
    X509_free(ctx->usercert);
  }
  if(ctx->user_certchain) {
    int i = 0;
    SPKM3_DEBUG(3, ("freeing saved user certchain\n"));
    for(i = 0; i < sk_X509_num(ctx->user_certchain); i++) {
      X509_free(sk_X509_value(ctx->user_certchain, i));
    }
    
    sk_X509_free(ctx->user_certchain);
  }
  if(ctx->spkm3_cred) {
    spkm3_cred_int_t *cred = ctx->spkm3_cred;
    if(cred->cert) {
      SPKM3_DEBUG(3, ("freeing X509 cert\n"));
      X509_free(cred->cert);
    }
    if(cred->priv_key) {
      SPKM3_DEBUG(3, ("freeing X509 priv key\n"));
      EVP_PKEY_free(cred->priv_key);
    }
    if(cred->wire_cert) {
      SPKM3_DEBUG(3, ("freeing ASN1 cert\n"));
      asn_DEF_Certificate.free_struct(&asn_DEF_Certificate,
				      cred->wire_cert, 0);
    }
    /* need to free wire_certchain */
    if(cred->wire_certchain && cred->certchain_len > 0) {
      int i =0;
      for(i = 0; i < cred->certchain_len; i++) {
	SPKM3_DEBUG(3, ("freeing ASN1 cert chain pair\n"));
	asn_DEF_Certificate.free_struct(&asn_DEF_CertificatePair,
					&cred->wire_certchain[i], 1);
      }
      free(cred->wire_certchain);
    }

    if(cred->wire_name && cred->wire_name->present != Name_PR_NOTHING) {
      SPKM3_DEBUG(3, ("freeing wire_name\n"));
      asn_DEF_Name.free_struct(&asn_DEF_Name, cred->wire_name, 0);
    }
    free(cred);
  }

  if(ctx->dh) {
    SPKM3_DEBUG(3, ("freeing DH keys\n"));
    DH_free(ctx->dh);
  }

  if(ctx->share_key.value && ctx->share_key.length > 0) {
    SPKM3_DEBUG(3, ("freeing share key\n"));
    free(ctx->share_key.value);
  }
  if(ctx->derived_conf_key.value && ctx->derived_conf_key.length > 0) {
    SPKM3_DEBUG(3, ("freeing derived conf key\n"));
    free(ctx->derived_conf_key.value);
  }
  if(ctx->derived_integ_key.value && ctx->derived_integ_key.length > 0) {
    SPKM3_DEBUG(3, ("freeing derived integ key\n"));
    free(ctx->derived_integ_key.value);
  }
  
  if(ctx->in_ict) {
    SPKM3_DEBUG(3, ("freeing InitialContextToken\n"));
    asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken, 
					    ctx->in_ict, 0);
  }

  if(ctx->src_name && ctx->src_name->present != Name_PR_NOTHING) {
    SPKM3_DEBUG(3, ("freeing src_name\n"));
    asn_DEF_Name.free_struct(&asn_DEF_Name, ctx->src_name, 0);
  }
  if(ctx->targ_name && ctx->targ_name->present != Name_PR_NOTHING) {
    SPKM3_DEBUG(3, ("freeing targ_name\n"));
    asn_DEF_Name.free_struct(&asn_DEF_Name, ctx->targ_name, 0);
  }

  if(ctx->randSrc && ctx->randSrc->size >= 0 && ctx->randSrc->buf != NULL) {
    SPKM3_DEBUG(3, ("freeing randSrc\n"));
    free(ctx->randSrc->buf);
    free(ctx->randSrc);
  }

  if(ctx->randTarg && ctx->randTarg->size >= 0 && ctx->randTarg->buf != NULL) {
    SPKM3_DEBUG(3, ("freeing randTarg\n"));
    free(ctx->randTarg->buf);
    free(ctx->randTarg);
  }

  if(ctx->keyestb_alg) {
    SPKM3_DEBUG(3, ("freeing K-ALGs\n"));
    if(ctx->keyestb_alg->length > 0 && ctx->keyestb_alg->elements != NULL)
      free(ctx->keyestb_alg->elements);
    free(ctx->keyestb_alg);
  }

  if(ctx->owf_alg) {
    SPKM3_DEBUG(3, ("freeing OWF-ALGs\n"));
    if(ctx->owf_alg->length > 0 && ctx->owf_alg->elements != NULL)
      free(ctx->owf_alg->elements);
    free(ctx->owf_alg);
  }

  if(ctx->intg_alg) {
    SPKM3_DEBUG(3, ("freeing I-ALGs\n"));    
    if(ctx->intg_alg->length > 0 && ctx->intg_alg->elements != NULL)
      free(ctx->intg_alg->elements);
    free(ctx->intg_alg);
  }

  if(ctx->conf_alg) {
    SPKM3_DEBUG(3, ("freeing I-ALGs\n"));    
    if(ctx->conf_alg->length > 0 && ctx->conf_alg->elements != NULL)
      free(ctx->conf_alg->elements);
    free(ctx->conf_alg);
  }


  kg_delete_ctx_id((gss_ctx_id_t)ctx);

  free(ctx);
  ctx = NULL;
}

/***************************************************************************/
/* REQ_create */
/* REPTI_create */
/* REPIT_create */
/***************************************************************************/

/*
 *
 * XXX This is DH-specific.  We should be doing something more generic? XXX
 *
 * Function: REQ_create()
 *
 * Description: creates the initiator side of the diffie-hellman
 * request. constructs a wire-ready spkm3 inner context
 * token to be passed to the receiver gss_accept_sec_context(). 
 *
 * no initiator nor target certificate 
 *
 * dh inner context token format:
 *
 * SPKM-REQ (0)
 *    requestToken      REQ-TOKEN
 *    certif-data [0]   CertificationData OPTIONAL (not used when anonymous)
 *    auth-data [1]     AuthorizationData OPTIONAL (not used)
 *
 * REQ-TOKEN
 *    req-contents      Req-contents
 *    algId             AlgorithmIdentifier
 *    req-integrity     Integrity
 *
 * Req-contents
 *    tok-id            0x0100
 *    context-id        Random-Integer (fresh) passed to target
 *                         which concatenates another random-#
 *                         to form the context-id which is
 *                         mapped to the context-handle
 *    pvno              BIT STRING protocol version # (1)
 *    timestamp         UTCTime -- OPTIONAL not used.
 *    randSrc           Random-Integer
 *    targ-name         Name --  of target machine.
 *    src-name          Name OPTIONAL -- spkm3 always anonymous
 *    rep-data          context-data
 *    validity [2]      Validity - OPTIONAL not used.
 *    key-estb-set      Key-Estb-Algs - initiator half of the dh key
 *    key-estb-req      BIT STRING  - OPTIONAL not used.
 *    key-src-bind      OCTET STRING - OPTIONAL not used.
 *    
 */

OM_uint32
REQ_create (spkm3_context_t * ctx,
	    gss_buffer_t output_token,
	    spkm3_name_desc_t * target_name)
{
  OM_uint32 minor_status;
  unsigned char randnum[SPKM3_RAND_LEN];
  unsigned char *p;
  SPKM_REQ_t*spkm_req;
  Name_t *cred_subject = NULL;
  Req_contents_t *reqcnt;
  asn_enc_rval_t ec;
  struct spkm3_ebuf seb;
  InitialContextToken_t * ict = NULL;
  
  SPKM3_DEBUG(1, ("REQ_create: START\n"));
  
  CRYPTO_malloc_init();
  OpenSSL_add_all_algorithms();
  
  if ((minor_status = create_initial_ct (&ict)))
    goto out;
  
  ict->innerContextToken.present = SPKMInnerContextToken_PR_req;
  spkm_req = &ict->innerContextToken.choice.req;
  reqcnt = &spkm_req->requestToken.req_contents;
  
  minor_status = GSS_S_G_MEMORY_ALLOC;  /* XXX used for generic error */
  
  /* set token id */
  if ((asn_long2INTEGER(&reqcnt->tok_id, SPKM_TOK_CTX_REQ)) == -1)
    goto out;
  
  /* context_id
   * Get a "fresh" random number, rfc2025 says number needs to be
   * fresh, but need not be overly strong.
   */
  if (get_random_bytes (randnum, SPKM3_RAND_LEN, 0) != SPKM3_RAND_LEN) {
    goto out;
  }
  if ((asn1_bitstring_set(&reqcnt->context_id, randnum, 
			  SPKM3_RAND_LEN)) == -1)
    goto out;
  
  /* set protocol version, set bit 0 */
  reqcnt->pvno.buf = NULL;
  if (!asn1_set_bit (&reqcnt->pvno, 0, 1))
    goto out;
  
  /* skip timestamp */
  
  /* Now get another random number */
  if (get_random_bytes (randnum, SPKM3_RAND_LEN, 0) != SPKM3_RAND_LEN)
    goto out;
  
  if ((asn1_bitstring_set (&reqcnt->randSrc, randnum, 
			   SPKM3_RAND_LEN)) == -1)
    goto out;
  if((ctx->randSrc = asn1_bitstring_dup(&reqcnt->randSrc)) == NULL)
    goto out;
  
  /* targ-name         Name --  of target machine. */
  /* copy the target name into the req_contents, also save it in our context */
  if ((minor_status = asn1_name_copy (&reqcnt->targ_name,
				      target_name->asn1_name)) != 0) 
    goto out;

  if ((ctx->targ_name = asn1_name_dup (target_name->asn1_name)) == NULL)  
    goto out;
#ifdef ASNDEBUG
  asn_fprint(stderr, &asn_DEF_Name, target_name->asn1_name);
#endif
  
  /* src_name (OPTIONAL) is set in REQ_create_mutual */
  
  /* Context Data */
  if((minor_status = REQ_contextdata_create (&reqcnt->req_data, 
					     ctx->req_flags)) != 0)
    goto out;
  
  /* validity - OPTIONAL */
  
  /* key-estb-set */
  if((minor_status = REQ_keyestab_create(ctx, reqcnt)) != 0)
    goto out;
  
  /* key-src-bind OPTIONAL */
  
  /* AlgId and requestToken->reqintegrity */
  if (ctx->req_flags & GSS_C_MUTUAL_FLAG) {
    /*
     * If they requested mutual auth, but didn't supply
     * credentials, get default credentials.
     */
    if (ctx->spkm3_cred == NULL) {
      OM_uint32 maj;
      if (( maj = 
	    spkm3_gss_acquire_cred (&minor_status, (gss_name_t) NULL,
				    GSS_C_INDEFINITE, GSS_C_NULL_OID_SET,
				    GSS_C_INITIATE, &ctx->spkm3_cred, NULL,
				    NULL)) != GSS_S_COMPLETE)
	goto out; /* key_estb_req->buf free'd 
		   * with the rest of the 
		   * InitialContextToken */
    }
    minor_status = REQ_create_mutual(ctx, spkm_req);
  } else {  
    /* Anonymous SPKM3: Null MAC algID */ 
    minor_status = asn1_algid_set_oid (&spkm_req->requestToken.algId, 
				       &null_mac_oid);
  }
  if (minor_status)
    goto out;
  
  /* save the der encoded reqcontents in the gss_context - this
   * is used to validate the targets response.
   */
  if((minor_status = encode(&asn_DEF_Req_contents, reqcnt, 
			    &ctx->der_reqcontents, 
			    (size_t *)&ctx->der_req_len)) != 0)
    goto out;
  
  
  minor_status = encode(&asn_DEF_InitialContextToken,
			ict, (unsigned char **)&output_token->value,
			&output_token->length);
  
#ifdef ASNDEBUG
  asn_fprint(stderr, &asn_DEF_SPKM_REQ, spkm_req);	
#endif
 out:
  asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					  ict, 0);
  SPKM3_DEBUG(1, ("REQ_create: END %d\n", minor_status));
  
  return minor_status;
}

OM_uint32
REQ_contextdata_create (Context_Data_t * reqdata, OM_uint32 req_flags)
{
  OM_uint32 ret = 0;
  OM_uint32 minor_status = GSS_SPKM_S_SG_OPENSSL_CONVERT_ERROR;
  BIT_STRING_t *bstr = NULL;
  
  SPKM3_DEBUG (1, ("REQ_contextdata_create: START\n"));
  
  /* channelId - OPTIONAL - not used in first pass coding */
  
  /* seq-number  - OPTIONAL - not used in first pass coding */
  
  /* options - the set of features that will be active on the context.
   * see rfc 1508
   * (0) delegation-state - delegation_req_flag set  
   * (1) mutual-state
   * (2) replay-det-state -  replay_det_flag set
   * (3) sequence-state - 
   * (4) conf-avail - per-message confidentialiy available
   * (5) integ-avail - per-message integrity services available
   * (6) target-certif-data-required - MUST be set for spkm3 
   *
   * Note: see Context_Data_free() which frees all the 
   * x509_algor_list_set() alloc's.
   */
  
  reqdata->options.buf = NULL;
  if (req_flags & GSS_C_MUTUAL_FLAG) {
    if (!asn1_set_bit (&reqdata->options,
		       SPKM_OPTIONS_MUTUAL_STATE, 1))
      goto out;
  }
  
  if (!asn1_set_bit (&reqdata->options,
		     SPKM_OPTIONS_TARGET_CERTIF_DATA_REQUIRED, 1))
    goto out;
  
  /* Conf-Algs 
   * add cast5-cbc as per rfc 2847
   *
   * this is used to encode the initialization vector once the context
   * is set up
   * ret=EVP_CIPHER_set_asn1_iv(cipher,a->parameter);
   */
  reqdata->conf_alg.present = Conf_Alg_PR_algId;
  if ((minor_status = oid_2_asn1alg(&reqdata->conf_alg.choice.algs.list,
				    &cast5_cbc_oid)) != 0)
	goto out;
  
  /* Intg-Algs
   * add md5WithRSAEncryption as RECOMMENDED in rfc2847
   */
  if(req_flags & GSS_C_MUTUAL_FLAG) {
    if ((minor_status = oid_2_asn1alg(&reqdata->intg_alg.list, 
                                      &hmac_md5_oid)) != 0)
	goto out;
    if ((minor_status = oid_2_asn1alg(&reqdata->intg_alg.list, 
                                      &md5_rsa_encryption_oid)) != 0) 
	goto out;
    if ((minor_status = oid_2_asn1alg(&reqdata->intg_alg.list, 
				      &null_mac_oid)) != 0)
	goto out;
  } else {
    if ((minor_status = oid_2_asn1alg(&reqdata->intg_alg.list, 
                                      &hmac_md5_oid)) != 0)
        goto out;
    if ((minor_status = oid_2_asn1alg(&reqdata->intg_alg.list,
                                      &null_mac_oid)) != 0)
        goto out;
  }

  /* OWF-Algs 
   * add id-sha1 as per rfc2847
   */
 
  if((minor_status = oid_2_asn1alg(&reqdata->owf_alg.list, &sha1_oid)) != 0)
   goto out;
  
  minor_status = 0;
  
 out:
 SPKM3_DEBUG (1, ("REQ_contextdata_create: END %d\n", minor_status));
  
  return minor_status;
}

OM_uint32
REQ_keyestab_create(spkm3_context_t *ctx, Req_contents_t *reqcnt) {
  
  OM_uint32 minor_status = 0;
  int len = 0;
  BIO *out = NULL;
  
  SPKM3_DEBUG (1, ("REQ_keyestab_create: START\n"));
  
  /* The initiator half of the dh key info is split between the
   * keyestbset field which holds the parameters p & g 
   * and the keyestbreq which holds the initiators pubkey.
   */
  
  /*
   * Diffie-Hellman initiator.
   * 
   * Use 1024-bit key instead of 128-bit key.  This is apparently
   * what IBM is using. Also various postings indicate that a
   * 128-bit DH key is not secure.
   * In doing that, found that the DH_generate_parameters() can
   * take a very long to find a good 1024-bit prime.  Instead of
   * calling DH_generate_parameters each time, we use a pre-
   * generated value (generated with DH_generate_parameters).
   * (This is what IBM is doing as well.)
   */
  
  minor_status = GSS_S_G_MEMORY_ALLOC;
  spkm3_gss_copy_oid(&minor_status, &dh_oid, &ctx->keyestb_alg);

#if defined(DSA_DH_GENERATE) 
  {
    DSA *dsa = DSA_new();
    if(!dsa || !DSA_generate_parameters_ex(dsa, DH_KEYSIZE, NULL, 0, NULL, 
					   NULL, NULL)) {
      if(dsa) DSA_free(dsa);
      goto out;
    }
    
    ctx->dh = DSA_dup_DH(dsa);
    DSA_free(dsa);
  }
#else
  if ((ctx->dh = DH_new ()) == NULL)
    goto out;

#if defined (DH_GENERATE)
  if(!DH_generate_parameters_ex(dh, DH_KEYSIZE, 2, NULL))
    goto out;
#else

  if ((ctx->dh->g = BN_new ()) == NULL)
    goto out;
  
  ctx->dh->p =
    BN_bin2bn (spkm3_dh_prime, sizeof (spkm3_dh_prime), NULL);
  BN_set_word (ctx->dh->g, spkm3_dh_generator);
#endif /* DH_GENERATE */
#endif /* DSA_DH_GENERAGE */
  
  /* Generate pub_key */
  DH_generate_key (ctx->dh);
  
#ifdef DEBUG
  if(spkm3_debug_level >= 1) {
    out = BIO_new (BIO_s_file ());
    if (out == NULL)
      exit (1);
    
    BIO_set_fp (out, stderr, BIO_NOCLOSE);
    minor_status = DHparams_print_fp (stderr, ctx->dh);
    BIO_puts (out, "\n\nINITIATOR pub_key =");
    BN_print (out, ctx->dh->pub_key);
    BIO_puts (out, "\n\n\n");
    BIO_free (out);
  }
#endif
  
  if((minor_status = asn1_algor_dh_set (&reqcnt->key_estb_set.list, 
					ctx->keyestb_alg, 
					ctx->dh)) != 0)
    goto out;
  
  
  /*
   * keyestbreq holds the diffie-hellman initiator public key,
   * send it as an ASN1_BIT_STRING
   */
  if((minor_status = 
      spkm3_keyestab_create(&reqcnt->key_estb_req, ctx->dh)) != 0)
    goto out;
  
 out:
  SPKM3_DEBUG(1, ("REQ_keyestab_create: END %d\n", minor_status));
  
  return minor_status;
}

/*
 * Function: REQ_create_mutual()
 *
 * Description: creates the initiator side of the diffie-hellman 
 * SPKM-REQ CertificationData and computes the checksum over the 
 * RequestToken stored in the reqintegrity field
 * required for mutual authentication.
 */

OM_uint32
REQ_create_mutual (spkm3_context_t * ctx, SPKM_REQ_t *spkm_req)
{
  
  OM_uint32 minor_status = 0;
  REQ_TOKEN_t *req = &spkm_req->requestToken;
  Req_contents_t *rcnt = &req->req_contents;
  spkm3_cred_int_t * spkm3_cred = ctx->spkm3_cred;
  Name_t *src_subject = NULL;
  CertificationPath_t *cpath = NULL;
  
  SPKM3_DEBUG(1, ("REQ_create_mutual: START\n"));
  
  /* 
   * AlgId: One of id-dsa-with-sha1, md5WithRSAEncryption, 
   * or sha1WithRSAEncryption 
   * 
   * XXX only coding NID_md5WithRSAEncryption
   */
  if((minor_status = 
      asn1_algid_set_oid (&req->algId, &md5_rsa_encryption_oid)) != 0)
    return minor_status;
  
  if((minor_status = 
      spkm3_certifdata_create(spkm3_cred, &spkm_req->certif_data)) != 0)
    goto out;
  
  /* srcname */
  minor_status = GSS_S_G_MEMORY_ALLOC;
#if 0
  src_subject = &spkm3_cred->wire_cert->tbsCertificate.subject;
#else
  src_subject = spkm3_cred->wire_name;
#endif
  if (src_subject->present) {
    spkm3_name_desc_t spkm3_name;
    asn1_name_to_string(src_subject, &spkm3_name);
    SPKM3_DEBUG(1, ("ASN src_name = %s\n", spkm3_name.string_name));
    free(spkm3_name.string_name);
    asn_DEF_Name.free_struct(&asn_DEF_Name, &spkm3_name.asn1_name, 1);

    if ((rcnt->src_name = asn1_name_dup(src_subject)) == NULL)
      goto out;
    if ((ctx->src_name = asn1_name_dup (src_subject)) == NULL)
      goto out;
  }
    
  minor_status = REQ_integrity_sign(ctx, spkm_req);
  
 out:  
  SPKM3_DEBUG (1, ("REQ_create_mutual: END %d\n", minor_status));
  
  return minor_status; 

}


/*
 * Function: REPIT_create()
 *
 * Description: Create an SPKM_REP_IT token 
 * 
 * For dhKeyAgreement mutual authentication the initiator must construct 
 * a REP-IT token.
 * 
 * The REP-IT-TOKEN randTarg field is the same value as the 
 * REP-TI-TOKEN randTarg field.
 * 
 * The REP-IT-TOKEN randSrc field is the same value as the 
 * REQ-TOKEN randSrc field which is the same vaule as the 
 * REP-TI-TOKEN randSrc field.
 *
 * REP_IT_TOKEN srcname and keyestbrep are NULL after REP_IT_TOKEN_new.
 */

OM_uint32
REPIT_create (spkm3_context_t *ctx, gss_buffer_t output_token)
{
  OM_uint32 minor_status = 0;
  REP_TI_TOKEN_t *repti = 
    &ctx->in_ict->innerContextToken.choice.rep_ti.responseToken;
  Rep_ti_contents_t *repticnt = &repti->rep_ti_contents;	
  REP_IT_TOKEN_t *repit_tok = NULL;
  AlgorithmIdentifier_t *algp = NULL;
  InitialContextToken_t * ict = NULL;
  
  SPKM3_DEBUG(1, ("REPIT_create: START\n"));
  
  minor_status = GSS_S_G_MEMORY_ALLOC;  /* XXX - used for everything */
  
  if ((minor_status = create_initial_ct(&ict)))
    goto out;
  
  ict->innerContextToken.present = SPKMInnerContextToken_PR_rep_it;
  repit_tok = &ict->innerContextToken.choice.rep_it.responseToken;
  
  /* tokid */
  if ((asn_long2INTEGER(&repit_tok->tok_id, SPKM_TOK_CTX_REPIT)) == -1)
    goto out_free_ict;
  
  /* contextid */
  if ((asn1_bitstring_set(&repit_tok->context_id, 
			  repticnt->context_id.buf,
			  repticnt->context_id.size)) == -1)
    goto out_free_ict;
  
  /* randSrc */
  if ((asn1_bitstring_set(&repit_tok->randSrc, 
			  repticnt->randSrc.buf,
			  repticnt->randSrc.size)) == -1)
    goto out_free_ict;
  
  /* randTarg */
  if ((asn1_bitstring_set(&repit_tok->randTarg, 
			  repticnt->randTarg.buf,
			  repticnt->randTarg.size)) == -1)
    goto out_free_ict;
  
  /* targname */
  if ((asn1_name_copy(&repit_tok->targ_name, &repticnt->targ_name)) != 0)
    goto out_free_ict;
  
  /* srcname */
  if (repticnt->src_name) {
    if ((repit_tok->src_name =
	 asn1_name_dup(repticnt->src_name)) == NULL)
      goto out_free_ict;
  }
  
  /*keyestbrep     BIT STRING OPTIONAL
   *
   * XXX - do we need this?
   * contains the response to targets key-estb-str,
   * (if target selected a 2-pass K-ALG)
   */
  
  /* Set the Algorthm.  XXX where should this come from? */
  algp = &ict->innerContextToken.choice.rep_it.algId;
  if((minor_status = 
      asn1_algid_set_oid(algp, &md5_rsa_encryption_oid)) != 0)
    goto out;
  algp->parameters == NULL;
  
  if((minor_status = 
      REPIT_integrity_sign(ctx, 
			   &ict->innerContextToken.choice.rep_it)) != 0)
    goto out;
  
  minor_status = encode(&asn_DEF_InitialContextToken,
			ict, (unsigned char **)&output_token->value,
			&output_token->length);
  
  asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					  ict, 0);
  
#ifdef ASNDEBUG
  asn_fprint(stderr, &asn_DEF_SPKM_REP_IT, repit_tok);	
#endif
 out:
  SPKM3_DEBUG(1, ("REPIT_create: END %d\n", minor_status));
  return minor_status;
  
 out_free_ict:
  /* We're done with the internal formats now */
  asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					  ict, 0);
  goto out;
  
}

/*
 * Function: REPTI_create()
 *
 * Description: constructs a wire-ready token to be passed back to 
 * gss_init_sec_context() caller.
 */

OM_uint32
REPTI_create (spkm3_context_t * ctx, gss_buffer_t output_token)
{
  OM_uint32 minor_status = 0;
  int pvno = 0;
  int i, done = 0, start_index = 0, num = 0;
  unsigned char randnum[SPKM3_RAND_LEN];
  unsigned char sharednum[SPKM3_CONTEXTID_LEN];
  unsigned char *init_rand;
  int init_rand_len;
  SPKM_REQ_t * spkm_req = &ctx->in_ict->innerContextToken.choice.req;
  REQ_TOKEN_t *req = &spkm_req->requestToken;
  Rep_ti_contents_t *repti_cont;
  SPKM_REP_TI_t *spkm_repti = NULL;
  REP_TI_TOKEN_t *repptr = NULL;
  InitialContextToken_t * ict = NULL;
  
  SPKM3_DEBUG (1, ("REPTI_create: START\n"));
  
  if ((minor_status = create_initial_ct (&ict)) != 0)
    return minor_status;
  
  ict->innerContextToken.present = SPKMInnerContextToken_PR_rep_ti;
  spkm_repti = &ict->innerContextToken.choice.rep_ti;
  repptr = &spkm_repti->responseToken;
  
  repti_cont = &repptr->rep_ti_contents;
  
  /* set token id */
  if ((asn_long2INTEGER(&repti_cont->tok_id, SPKM_TOK_CTX_REPTI)) == -1)
    goto out;
  
  /* context-id: "fresh" random-integer passed from initiator
   * to target.  This is concatenated with our own random number
   * to form the context-id.
   */
  
  /* Get our own random number */
  if (get_random_bytes (randnum, SPKM3_RAND_LEN, 0) != SPKM3_RAND_LEN) {
    minor_status = GSS_S_G_MEMORY_ALLOC;	/* XXX */
    goto out;
  }
  
  if (req->req_contents.context_id.buf) {
    init_rand = req->req_contents.context_id.buf;
    init_rand_len = req->req_contents.context_id.size;
  }
  else {
    init_rand = NULL;
    init_rand_len = SPKM3_RAND_LEN;
  }
  
  /* Print out the initiator's random number and our random number */
  if (init_rand) {
    SPKM3_DEBUG(1, ("REPTI_create: initiator context_id rand\n"));
    print_rand (init_rand, init_rand_len);
  }
  else {
    /* XXX Is this a protocol error??? */
    SPKM3_DEBUG(1, ("REPTI_create: initiator context_id is NULL\n"));
  }
  SPKM3_DEBUG (1, ("REPTI_create: target context_id rand\n"));
  print_rand (randnum, sizeof (randnum));
  
  /* Use no more than SPKM3_RAND_LEN bytes from initiator */
  if (init_rand_len > SPKM3_RAND_LEN) {
    SPKM3_DEBUG(1, ("REPTI_create: init_rand_len is %d "
	        "SPKM3_RAND_LEN is %d -- truncating!\n",
	        init_rand_len, SPKM3_RAND_LEN));
    init_rand_len = SPKM3_RAND_LEN;
  }
  
  /* Must use ASN1_BIT_STRINGs here rather than integers */
  memset (sharednum, 0, sizeof (sharednum));
  if (init_rand)
    memcpy (sharednum, init_rand, init_rand_len);
  memcpy (&sharednum[SPKM3_RAND_LEN], randnum, SPKM3_RAND_LEN);
  
  SPKM3_DEBUG (1, ("REPTI_create: sharednum contextid rand\n"));
  print_rand (sharednum, sizeof (sharednum));
  
  if ((asn1_bitstring_set(&repti_cont->context_id, 
			  sharednum, sizeof (sharednum))) == -1){
    minor_status = GSS_S_G_MEMORY_ALLOC;	/* XXX */
    free (sharednum);
    goto out;
  }
  
  /* save contextid */
  if((minor_status = 
      spkm3_contextid_create(&ctx->ctx_id.data, &ctx->ctx_id.length,
			     repti_cont->context_id.buf, 
			     repti_cont->context_id.size)) != 0)
    goto out;
  
  /* protocol version 
   * if req->reqcontents->pvno has the correct bit (bit 0) set to 1
   * then return no repti_cont->pvno.
   * else if req->reqcontents->pvno is not supported 
   *      return DELET TOKEN (well, not quite yet..)
   */
  
  /* NOTE: ASN1_BIT_STRING_get_bit is destructive */
  pvno = asn1_get_bit (&req->req_contents.pvno, 0);
  if (pvno != 1) {
    minor_status = GSS_SPKM_S_SG_NO_PVNO_IN_COMMON;
    goto out;
  }
  /* skip timestamp */
  
  /* set randTarg */
  if (get_random_bytes (randnum, SPKM3_RAND_LEN, 0) != SPKM3_RAND_LEN) {
    minor_status = GSS_S_G_MEMORY_ALLOC;	/* XXX */
    goto out;
  }
  if ((asn1_bitstring_set(&repti_cont->randTarg, 
			  randnum, SPKM3_RAND_LEN)) == -1) {
    minor_status = GSS_S_G_MEMORY_ALLOC;	/* XXX */
    goto out;
  }
  /* save randTag into our context */
  if((ctx->randTarg = asn1_bitstring_dup(&repti_cont->randTarg)) == NULL) {
    minor_status = GSS_S_G_MEMORY_ALLOC;	/* XXX */
    goto out;
  }
  
  minor_status = 0;
  
  /* source name - copy from the REQ_TOKEN (XXX need to validate...) */
  if (req->req_contents.src_name && 
      req->req_contents.src_name->present != Name_PR_NOTHING) {
    if ((repptr->rep_ti_contents.src_name =
	 asn1_name_dup (req->req_contents.src_name)) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
    if((ctx->src_name = asn1_name_dup(req->req_contents.src_name)) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
#ifdef ASNDEBUG
    fprintf (stderr, "REPTI_create: source name:\n");
    asn_fprint(stderr , &asn_DEF_Name,
	       &repptr->rep_ti_contents.src_name);
#endif
  }
  
  /* target name - copy from the REQ_TOKEN (XXX need to validate...) */
  /* copy target name into the ti_token and also save it in our context */
  if((minor_status = asn1_name_copy (&repptr->rep_ti_contents.targ_name,
				     &req->req_contents.targ_name)) != 0)
    goto out;
  if((ctx->targ_name = asn1_name_dup(&req->req_contents.targ_name)) == NULL) {
    minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
  }

#ifdef ASNDEBUG
  fprintf (stderr, "REPTI_create: target name:\n");
  asn_fprint(stderr , &asn_DEF_Name, &repptr->rep_ti_contents.targ_name);
#endif
  /* randSrc - copy from the REQ_TOKEN */
  /* copy randSrc into the ti_token and also save it in our context */  
  if ((asn1_bitstring_set( &repptr->rep_ti_contents.randSrc,
			   req->req_contents.randSrc.buf,
			   req->req_contents.randSrc.size)) == -1)
    goto out;
  if((ctx->randSrc = 
      asn1_bitstring_dup(&repptr->rep_ti_contents.randSrc)) == NULL)
    goto out;
  
  /* Context Data */
  if((minor_status = 
      REPTI_contextdata_create (ctx, &req->req_contents.req_data,
				&repptr->rep_ti_contents.rep_data)) != 0)
    goto out;
  
  /* validity  - OPTIONAL - not used in first pass coding */
  
  /* keyestbid - OPTIONAL
   * used if target is changing key estb. algorithm (must be
   * a member of initiators key-estb-set)
   */
  
  /*
   * add the first reqdata->keysetbset algorithm to the context. 
   */
  
  /* set the first key alg encounterd in ordered list */
  if((minor_status = 
      spkm3_add_algs_oid (ctx, &req->req_contents.key_estb_set, 
		          (STACK *)NULL, &keyest_algs, 
		          &ctx->keyestb_alg)) != 0)
    goto out;
  
  /* finish any key establishment work */
  if((minor_status = REPTI_keyestab_create (ctx, repptr)) != 0)
    goto out;
  
  
  /* integrity of repti_token*/
  if((minor_status = REPTI_integrity_sign (ctx, repptr, req)) != 0)
    goto out;
 
 
  /* certif-data */
  if((minor_status =
      REPTI_certifdata_create (ctx->spkm3_cred, spkm_repti,
			       &req->req_contents.targ_name)) != 0)
    goto free_context;
  
  
  minor_status = encode(&asn_DEF_InitialContextToken,
			ict, (unsigned char **)&output_token->value,
			&output_token->length);
  
#ifdef ASNDEBUG
  asn_fprint(stderr, &asn_DEF_SPKM_REP_TI, repptr);
#endif
  
 out:
 free_context:
  asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					  ict, 0);
  
  SPKM3_DEBUG (1, ("REPTI_create: END %d\n", minor_status));
  
  return minor_status;
}

/*
 * Function: REPTI_contextdata_create()
 *
 * Description: Sets the Context_Data field of the REP_TI_TOKEN.
 *     and sets the first matching alg nid in the spkm3_gss_context. 
 *
 *     Choose the set of C_alg, I_alg, K_alg, and O_alg from 
 *     the choices presented in the reqdata parameter.
 *
 * XXX need to have a configurable list of algorithms supported
 * by the target.
 *
 * XXX error_returns!
 */

OM_uint32
REPTI_contextdata_create (spkm3_context_t * ctx,
			  Context_Data_t * req_data, Context_Data_t * rep_data)
{
  OM_uint32 minor_status = 0;
  
  SPKM3_DEBUG(1, ("REPTI_contextdata_create: START\n"));
  
  /* channelId - OPTIONAL - not used in first pass coding
   *
   * seq-number  - OPTIONAL - not used in first pass coding 
   *
   * options - the set of features that will be active on the context.
   * see rfc 1508
   * (0) delegation-state - delegation_req_flag set
   * (1) mutual-state
   * (2) replay-det-state -  replay_det_flag set
   * (3) sequence-state -
   * (4) conf-avail - per-message confidentialiy available
   * (5) integ-avail - per-message integrity services available
   * (6) target-certif-data-required - MUST be set for spkm3
   */
  
  if (ctx->req_flags & GSS_C_MUTUAL_FLAG) {
    if (!asn1_set_bit (&rep_data->options,
		       SPKM_OPTIONS_MUTUAL_STATE, 1))
      return -1;  /* XXX */
  }
  
  /* Owf_Algs: needs to be set first because after we choose Conf and Integ 
   * algorithms, we would generate the subkeys and for that we need to have
   * chosen an owf algorithm.
   *
   * XXX need error for 'no common owf alg found' ???
   */
  SPKM3_DEBUG(1, ("adding owf algs\n"));
  if ((minor_status = spkm3_add_algs_oid (ctx, &req_data->owf_alg.list,
				          &rep_data->owf_alg.list,
				          &sup_owf_algs, 
				          &ctx->owf_alg)))
    return minor_status;


  /* Conf_Algs */
  if (req_data->conf_alg.present == Conf_Algs_PR_algs) { 
    SPKM3_DEBUG(1, ("adding conf algs\n"));
    if ((minor_status = 
         spkm3_add_algs_oid (ctx, &req_data->conf_alg.choice.algs.list,
			     &rep_data->conf_alg.choice.algs.list, 
                             &sup_conf_algs, &ctx->conf_alg)))
	return minor_status;

    if(ctx->conf_alg == NULL) {
      /* no common conf alg found */
      SPKM3_DEBUG(1, ("no matching conf algs found\n"));
      return GSS_SPKM_S_SG_BAD_CONF_ALG_SET;
    } else {
      rep_data->conf_alg.present = Conf_Algs_PR_algs;

      /* Generate conf subkeys */
      if((minor_status = spkm3_derive_subkey(0, ctx)) != 0)
        return minor_status;
    }
  } else {
    SPKM3_DEBUG(1, ("no conf algs\n"));
    /* No conf alg in request */
    rep_data->conf_alg.present = Conf_Algs_PR_NOTHING;
  }
  
  /* Intg_Algs */
  SPKM3_DEBUG(1, ("adding integ algs\n"));
  if ((minor_status = spkm3_add_algs_oid (ctx, &req_data->intg_alg.list, 
				          &rep_data->intg_alg.list, 
				          &sup_intg_algs, 
				          &ctx->intg_alg)))
    return minor_status;
  
  if (ctx->intg_alg == NULL)            
    /* no common integrity alg found */
    return GSS_SPKM_S_SG_BAD_INT_ALG_SET;
  else if(!g_OID_equal(ctx->intg_alg, &null_mac_oid)) 
    /* Generate integ subkeys */    
    if((minor_status = spkm3_derive_subkey(1, ctx)) != 0)
      return minor_status;

  SPKM3_DEBUG(1, ("REPTI_contextdata_create: END\n"));
  
  return 0;
}


/*
 * Function: REPTI_keyestab_create()
 *
 * Description: stuffs the keyestab field of the REP_TI_TOKEN 
 */

OM_uint32
REPTI_keyestab_create (spkm3_context_t * ctx, REP_TI_TOKEN_t * repti)
{
  OM_uint32 minor_status = 0;
  Rep_ti_contents_t *repti_cont = &repti->rep_ti_contents;
  
  SPKM3_DEBUG (1, ("REPTI_keyestab_create: START\n"));
  
  if(g_OID_equal(ctx->keyestb_alg, &dh_oid)) {  
    /* diffie-hellman place the receivers pub_key into the keyestbstr */
    if((minor_status = 
	spkm3_keyestab_create(&repti_cont->key_estb_str, ctx->dh)) != 0)
      goto out;
    
    if((minor_status = 
	asn1_algid_set_oid(&repti->algId, &md5_rsa_encryption_oid)) != 0)
      goto out;
    
    repti->algId.parameters == NULL;
  } else {  
    SPKM3_DEBUG (1, ("Unknown K-ALG OID: "));
    SPKM3_DEBUG_HEX(1, (ctx->keyestb_alg->elements, ctx->keyestb_alg->length, 0));
    minor_status = GSS_SPKM_S_SG_BAD_KEY_ESTB_ALG_TYPE;
  }

 out:
  
  SPKM3_DEBUG (1, ("REPTI_keyestab_create: END %d\n", minor_status));
  
  return minor_status;
}


/*
 * Function: REPTI_certifdata_create(*)
 *
 * Description: stuffs the certifdata field of the SPKM_REP_TI
 * token with the certificate of the target_name.
 */

OM_uint32
REPTI_certifdata_create (spkm3_cred_int_t *cred, SPKM_REP_TI_t *spkm_repti, 
			 Name_t *targname)
{
  OM_uint32 minor_status = 0;
  Certificate_t *cert = NULL;
  
  SPKM3_DEBUG (1, ("REPTI_certifdata_create: START\n"));
  
  if((minor_status = 
      spkm3_certifdata_create(cred, &spkm_repti->certif_data)) != 0)
    goto out;
  
#if 0
  SPKM3_DEBUG(1, ("ASN target name\n"));
  //asn_fprint(stderr, &asn_DEF_Name, targname);	
  { spkm3_name_desc_t spkm3_name;
    asn1_name_to_string(targname, &spkm3_name); 
  SPKM3_DEBUG(1, ("ASN string = %s\n", spkm3_name.string_name));
  SPKM3_DEBUG(1, ("ASN cert subject name\n"));
  //asn_fprint(stderr, &asn_DEF_Name, &spkm_repti->certif_data->certificationPath->userCertif->tbsCertificate.subject);
  asn1_name_to_string(&spkm_repti->certif_data->certificationPath->userCertif->tbsCertificate.subject, &spkm3_name);
  SPKM3_DEBUG(1, ("ASN string = %s\n", spkm3_name.string_name));
  
  }
#endif

#if 0	/* This doesn't work yet... */
	/* Check if targname is compatible with certificate */
  cert = spkm_repti->certif_data->certificationPath->userCertif;
  if (targetname_matches_cert (targname, cert) == 0) {
    fprintf (stderr,
	     "REPTI_certifdata_create targname does not match cert\n");
    return GSS_S_BAD_NAME;
    
  }
#endif
  
 out:
  SPKM3_DEBUG (1, ("REPTI_certifdata_create: END %d\n", minor_status));
  
  return minor_status;
}

/***************************************************************************/
/* REQ_process */
/* REPTI_process */
/* REPIT_process */
/***************************************************************************/


/*
 * Function: REQ_process()
 *
 * Description: processes a request token
 */

OM_uint32
REQ_process (spkm3_context_t * ctx)
{
  OM_uint32 minor_status = 0;
  int key_len = 0;
  SPKM_REQ_t * spkm_req = &ctx->in_ict->innerContextToken.choice.req;
  REQ_TOKEN_t * req = &spkm_req->requestToken ;
  BIO *out = NULL;
  
  SPKM3_DEBUG(1, ("REQ_process: START\n"));
#ifdef ASNDEBUG
  asn_fprint(stderr, &asn_DEF_SPKM_REQ, req);	
#endif

  /* acquire creds */
  if (ctx->spkm3_cred == NULL) {
    OM_uint32 maj;
    if((maj = 
	spkm3_gss_acquire_cred (&minor_status, (gss_name_t) NULL,
				GSS_C_INDEFINITE, GSS_C_NULL_OID_SET,
				GSS_C_ACCEPT, &ctx->spkm3_cred, NULL,
				NULL)) != GSS_S_COMPLETE)
      goto out; 
  }

  /* if mutual authentication, verify initiator */
  if (ctx->req_flags & GSS_C_MUTUAL_FLAG) {
    if((minor_status = REQ_integrity_verify(ctx, spkm_req)) != 0)
      goto out;
  }
  
  /* decode DH params into context DH struct */
  if((minor_status = 
      spkm3_dh_decode(ctx, &req->req_contents.key_estb_set)) != 0)
    goto out;
    
  /* Generate a shared key */
  if((minor_status = 
      spkm3_dh_compute_key(ctx->dh, &ctx->share_key,
			   req->req_contents.key_estb_req)) != 0)
    goto out;

 out:
  SPKM3_DEBUG(1, ("REQ_process: END %d\n", minor_status));
  
  return (minor_status);
}

OM_uint32
REPTI_process (spkm3_context_t *ctx)
{
  OM_uint32 minor_status;
  SPKMInnerContextToken_t *spkm_ict = &ctx->in_ict->innerContextToken;
  SPKM_REP_TI_t *spkm_repti = &spkm_ict->choice.rep_ti;
  REP_TI_TOKEN_t *repti = &spkm_repti->responseToken;
  int j = 0;
  X509_ALGOR *ent;
  Rep_ti_contents_t *repptr =
    &spkm_ict->choice.rep_ti.responseToken.rep_ti_contents;
  
  SPKM3_DEBUG (1, ("REPTI_process: START\n"));
#ifdef ASNDEBUG
  asn_fprint(stderr, &asn_DEF_SPKM_REP_TI, repti);	
  fprintf (stderr, "subject_name on target Cert:\n");
  asn_fprint(stdout, &asn_DEF_Name, &spkm_ict->choice.rep_ti.certif_data->certificationPath->userCertif->tbsCertificate.subject);
#endif
    
  /* verify that randSrc in TI token is the same as in REQ token */
  if((minor_status = 
    spkm3_verify_randsrc(ctx->randSrc, &repti->rep_ti_contents.randSrc)) != 0)
    goto out;
  /*
   * verify target x509 Certificate signature. 
   */
  if((minor_status = REPTI_integrity_verify (ctx, spkm_repti)) != 0)
    goto out;
  
  /* dh compute the shared key */
  if((minor_status = 
      spkm3_dh_compute_key(ctx->dh, &ctx->share_key,
			   repti->rep_ti_contents.key_estb_str)) != 0)
    goto out;
    
  /* calculated per token contexid */
  if((minor_status = 
      spkm3_contextid_create(&ctx->ctx_id.data, &ctx->ctx_id.length,
			     repptr->context_id.buf,
			     SPKM3_CONTEXTID_LEN)) != 0)
      goto out;

  /*
   * save first negotiated intg_alg, owf_alg, and conf_alg
   */
  
  /* owf alg: need to this one first because when we chosen the conf and integ
   * algorithm we would also generate the subkeys and need to have owf value
    */
  j = repptr->rep_data.owf_alg.list.count;
  if (j > 0) {
     AlgorithmIdentifier_t *alg = (AlgorithmIdentifier_t *)
       repptr->rep_data.owf_alg.list.array[0];

    if((ctx->owf_alg = malloc(sizeof(gss_OID))) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
    ctx->owf_alg->length = alg->algorithm.size;
    if((ctx->owf_alg->elements = malloc(ctx->owf_alg->length)) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
    memcpy(ctx->owf_alg->elements, alg->algorithm.buf, ctx->owf_alg->length);

    SPKM3_DEBUG(1, ("\tOWF-ALG"));
    SPKM3_DEBUG_HEX(1, (ctx->owf_alg->elements, ctx->owf_alg->length, 0));
  }

  /* conf alg */
  if (repptr->rep_data.conf_alg.present == Conf_Alg_PR_algId) {
     AlgorithmIdentifier_t *alg = (AlgorithmIdentifier_t *)
       repptr->rep_data.conf_alg.choice.algs.list.array[0];

    if((ctx->conf_alg = malloc(sizeof(gss_OID))) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
    ctx->conf_alg->length = alg->algorithm.size;
    if((ctx->conf_alg->elements = malloc(ctx->conf_alg->length)) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
    memcpy(ctx->conf_alg->elements, alg->algorithm.buf, ctx->conf_alg->length);

    SPKM3_DEBUG(1, ("\tCONF-ALG"));
    SPKM3_DEBUG_HEX(1, (ctx->conf_alg->elements, ctx->conf_alg->length, 0));

    /* Generate conf subkey */
    if((minor_status = spkm3_derive_subkey(0, ctx)) != 0)
      goto out;

  }

  
  /* intg alg */
  j = repptr->rep_data.intg_alg.list.count;
  if (j > 0) {
    AlgorithmIdentifier_t *alg = (AlgorithmIdentifier_t *)
      repptr->rep_data.intg_alg.list.array[0];

    if((ctx->intg_alg = malloc(sizeof(gss_OID))) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
    ctx->intg_alg->length = alg->algorithm.size;
    if((ctx->intg_alg->elements = malloc(ctx->intg_alg->length)) == NULL) {
      minor_status = GSS_S_G_MEMORY_ALLOC;
      goto out;
    }
    memcpy(ctx->intg_alg->elements, alg->algorithm.buf, ctx->intg_alg->length);

    SPKM3_DEBUG(1, ("\tI-ALG"));
    SPKM3_DEBUG_HEX(1, (ctx->intg_alg->elements, ctx->intg_alg->length, 0));
 
    /* Generate integ subkeys */
    if(!g_OID_equal(ctx->intg_alg, &null_mac_oid))
      if((minor_status = spkm3_derive_subkey(1, ctx)) != 0)
	goto out;

  }

  ctx->mech_used = gss_mech_oid_spkm3;
  ctx->established = 1;
  minor_status = 0;



  SPKM3_DEBUG(1, ("REPTI_process: END %d\n", minor_status));
 out:
  
  return minor_status;
}

OM_uint32
REPIT_process (spkm3_context_t * ctx)
{
  OM_uint32 minor_status = 0;
  SPKMInnerContextToken_t * spkm_ict = &ctx->in_ict->innerContextToken;
  SPKM_REP_IT_t *rep = &spkm_ict->choice.rep_it;
  
  SPKM3_DEBUG(1, ("REPIT_process: START\n"));
#ifdef ASNDEBUG
  asn_fprint(stderr, &asn_DEF_SPKM_REP_IT, rep);	
#endif
 
  /* verify that randTarg in IT token is the same as randTarg in TI token */
  /* verify that randSrc in TI token is the same as in REQ token */
  if((minor_status = 
    spkm3_verify_randsrc(ctx->randTarg, &rep->responseToken.randTarg)) != 0)
    goto out;

  if((minor_status = 
    spkm3_verify_randsrc(ctx->randSrc, &rep->responseToken.randSrc)) != 0)
    goto out;

  /* check token integrity */
  minor_status = REPIT_integrity_verify(ctx, rep);

 out:  
  SPKM3_DEBUG(1, ("REPIT_process: END %d\n", minor_status));
  return (minor_status);
}

/****************************************************************************/
/* Helper functions for create/process functions */
/****************************************************************************/
OM_uint32 spkm3_certifdata_create(spkm3_cred_int_t *cred,
				  CertificationData_t ** certif_data) {

  OM_uint32 minor_status = GSS_S_G_MEMORY_ALLOC;
  int i = 0;
  unsigned char *output_buffer = NULL;
  size_t len = 0;

  if ((*certif_data = 
       calloc (1, sizeof (CertificationData_t))) == NULL)
    goto out;	
  
  if (((*certif_data)->certificationPath = 
       calloc (1, sizeof (CertificationPath_t))) == NULL)
    goto out;
  
  if (((*certif_data)->certificationPath->userCertif = 
       asn1_cert_dup(cred->wire_cert)) == NULL)
    goto out;
 
  if(cred->wire_certchain) {
    if(((*certif_data)->certificationPath->theCACertificates =
	calloc(1, 
	       sizeof(struct CertificationPath__theCACertificates))) == NULL)
      goto out;
    
    for(i = 0; i < cred->certchain_len; i++) {
      CertificatePair_t *pair = NULL;

      if((pair = calloc(1, sizeof(CertificatePair_t))) == NULL)
	goto out;

      if((pair->issuedByThisCA = 
	  asn1_cert_dup(cred->wire_certchain[i].issuedByThisCA)) == NULL)
	goto out;

      pair->issuedToThisCA = NULL;

      asn_set_add(&(*certif_data)->certificationPath->theCACertificates->list,
		  (void *)pair);
    }
  }

  minor_status = 0;
 out:
  
  return minor_status;
  
}

OM_uint32 spkm3_keyestab_create(BIT_STRING_t **key_estb_req, DH *dh) {
  
  OM_uint32 minor_status = GSS_S_G_MEMORY_ALLOC;
  int len = 0;
  
  if (*key_estb_req == NULL) {
    if ((*key_estb_req = 
	 calloc (1, sizeof(BIT_STRING_t))) == NULL)
      goto out;
  }
  len = BN_num_bytes (dh->pub_key);
  if (((*key_estb_req)->buf = malloc (len)) == NULL)
    goto out; /* key_estb_req free'd with the rest of the
	       * InitialContextToken */
  
  (*key_estb_req)->size =
    BN_bn2bin (dh->pub_key, (*key_estb_req)->buf);
  
  minor_status = 0;
  
 out:
  return minor_status;
  
}

/*
 * Function: spkm3_dh_decode()
 *
 * Description:  adds the X509_ALGOR key establishment algorithms
 * to the context.
 * NOTE: currently expect ONLY DH with initiator 1/2 key, with no initiator
 * nor target credentials....
 */

OM_uint32
spkm3_dh_decode(spkm3_context_t * ctx, void *asn_set_of_x)
{
  asn_set *as = (asn_set *)asn_set_of_x;
  AlgorithmIdentifier_t *alg;
  asn_dec_rval_t rval;
  int i, nid;
  OM_uint32 minor_status = 0;
  
  SPKM3_DEBUG (1, ("spkm3_dh_decode: START\n"));
  
  for (i = 0; i < as->count; i++) {
    alg = (AlgorithmIdentifier_t *)as->array[i];

    if(alg->algorithm.size == dh_oid.length &&
	!memcmp(alg->algorithm.buf, dh_oid.elements, dh_oid.length)) { 
    
      SPKM3_DEBUG(1, ("matched to NID_dhKeyAgreement\n"));
      if((minor_status = 
	  asn1_algor_dh_get(alg->parameters->buf, alg->parameters->size, 
			    &ctx->dh)))
        goto out;
  
      /* Generate the receiver public key */
      DH_generate_key (ctx->dh);
      break;
    } else { 
      SPKM3_DEBUG(1, ("unsupported key establishment algorithm "));
      SPKM3_DEBUG_HEX(1, (alg->algorithm.buf, alg->algorithm.size, 0));
      minor_status = GSS_SPKM_S_SG_BAD_KEY_ESTB_ALG_TYPE;
    }
  }

 out:
  SPKM3_DEBUG (1, ("spkm3_dh_decode: END %d\n", minor_status));

  return minor_status;
}

OM_uint32 spkm3_dh_compute_key(DH *dh, gss_buffer_desc *share_key, 
			       BIT_STRING_t *key_estb_req) {
  
  OM_uint32 minor_status = 0;
  BIO *out = NULL;
  BIGNUM *pkey = NULL;

  /* grab the initiator's pub_key from the wire */
  if ((pkey = 
       BN_bin2bn (key_estb_req->buf, key_estb_req->size, NULL)) == NULL) {
    minor_status = GSS_S_G_MEMORY_ALLOC;	/* XXX */
    goto out;
  }
  
  /* dh shared key */
  share_key->length = DH_size (dh);
  if ((share_key->value = (unsigned char *) 
       malloc (share_key->length * sizeof (unsigned char))) == NULL) {
    minor_status = GSS_S_G_MEMORY_ALLOC;
    goto out;
  }
  
  share_key->length =
    DH_compute_key (share_key->value, pkey, dh);
 
#ifdef DEBUG 
  if(spkm3_debug_level >= 1) {
    out = BIO_new (BIO_s_file ());
    if (out == NULL) exit (1);
    BIO_set_fp (out, stderr, BIO_NOCLOSE);
    fprintf (stderr, "initiator key from token\n");
    DHparams_print_fp (stderr, dh);
    
    BIO_puts (out, "\nINITIATOR pub_key = ");
    BN_print (out, pkey);
    BIO_puts (out, "\n");
    BIO_puts (out, "RECEIVOR pub_key = ");
    BN_print (out, dh->pub_key);
    BIO_puts (out, "\n");
    print_dh_sharekey (share_key);
    fprintf (stderr, "\n\n");
    
    BIO_free(out);
  }
#endif
  
 out:
  if(pkey) BN_free(pkey);
  return minor_status;
}

/*
 * Function: spkm3_cmp_alg()
 *
 * Description: Boolean. searches supported algorithm list starting at
 *     the start_index (to maintain algorithm ordering) for input 
 *     algorithm. returns 1 iff found, 0 on failure.
 */

OM_uint32
spkm3_cmp_alg (int nid, int alglist[], int *start_index)
{
  int i = *start_index;
  for (;; i++) {
    if (alglist[i] == nid) {
      *start_index = i;
      return 1;
    }
    if (alglist[i+1] == 0)
      break;
  }
  return 0;
}

/*
 * Function: spkm3_set_req_flags()
 *
 * Description: sets ctx->req_flags.
 * 
 * XXX currently only sets MUTUAL/ANON
 */

void
spkm3_set_req_flags(spkm3_context_t *ctx, Req_contents_t *reqcnt)
{
  ASN1_BIT_STRING *tmp_bstr;
  
  if (asn1_get_bit (&reqcnt->req_data.options, 1)) {
    ctx->req_flags |= GSS_C_MUTUAL_FLAG;
    SPKM3_DEBUG(1, ("spkm3_set_req_flags:\t\tGSS_C_MUTUAL_FLAG\n"));
  } else {
    ctx->req_flags |= GSS_C_ANON_FLAG;
    SPKM3_DEBUG(1, ("spkm3_set_req_flags:\t\tGSS_C_ANON_FLAG\n"));
  }
}

OM_uint32 spkm3_contextid_create(unsigned char **out_buf,
				 int *out_len,
				 unsigned char *in_buf,
				 int in_len) {
  OM_uint32 minor_status = GSS_S_G_MEMORY_ALLOC;

  *out_len = in_len;
  if((*out_buf = (unsigned char *) 
      calloc (in_len, sizeof(unsigned char))) == NULL)
    goto out;
  memcpy (*out_buf, in_buf, in_len);
    
  SPKM3_DEBUG (1, ("ctx->ctx_id.data\n"));
  SPKM3_DEBUG_HEX(1, ((unsigned char *) *out_buf, in_len, 0));

  minor_status = 0;
 out:

  return minor_status;
}

/*
 * Function: create_initial_ct()
 * 
 * Description: create ICT and initialize mechanism.
 */
OM_uint32
create_initial_ct (InitialContextToken_t **ict)
{
  InitialContextToken_t *local;
  OM_uint32 minor_status;
  
  if ((local = calloc (1, sizeof(InitialContextToken_t))) == NULL)
    return GSS_S_G_MEMORY_ALLOC;
  
  if((minor_status = 
      asn1_copy_oid (gss_mech_oid_spkm3, &local->thisMech)) != 0)
    goto out;

  *ict = local;
 out:
  
  SPKM3_DEBUG(1, ("create_initial_ct: returning %d (ict = %p)\n",
	      minor_status, local));
  
  return minor_status;
}

OM_uint32 spkm3_verify_randsrc(BIT_STRING_t *randSrcREQ,
                               BIT_STRING_t *randSrcTI) {
  OM_uint32 minor_status = 0;

  if(randSrcTI->size != randSrcREQ->size ||
     memcmp(randSrcTI->buf, randSrcREQ->buf, randSrcTI->size)) {
    SPKM3_DEBUG(1, ("spkm3_parse_inner_context_token: failed to "
	        "validate randSrc in TI token\n"));
    SPKM3_DEBUG(1, ("randSrc in REQ token (%d):\n", randSrcREQ->size));
    SPKM3_DEBUG_HEX(1, (randSrcREQ->buf, randSrcREQ->size, 0));
    SPKM3_DEBUG(1, ("randSrc in TI token (%d):\n", randSrcTI->size));
    SPKM3_DEBUG_HEX(1, (randSrcTI->buf, randSrcTI->size, 0));
    minor_status = -1;
  }

  return minor_status;
}

int get_cert_type(X509 *x) {

  int i = 0;
  int cert_type = SPKM3_STANDART_CERT;

#if OPENSSL_LIBRARY_VERSION > 0x00908000
  if (xs->ex_flags & EXFLAG_PROXY) {	
    SPKM3_DEBUG(1, ("get_cert_type: this is a proxy cert\n"));
    cert_type = SPKM3_GLOBUS_PROXY_CERT;
  } 
#endif

  for(i = 0; i < X509_get_ext_count(x); i++) {
    X509_EXTENSION *ex = X509_get_ext(x, i);
    ASN1_OBJECT *obj = X509_EXTENSION_get_object(ex);;

    if(X509_EXTENSION_get_critical(ex)) {

      SPKM3_DEBUG(1, ("get_cert_type: found a critical extension\n"));
      
      if(globus_proxy_oid_1.length == obj->length &&
	 !memcmp(globus_proxy_oid_1.elements, obj->data, obj->length)) {
	SPKM3_DEBUG(1, ("get_cert_type: got globus proxy oid\n"));
	cert_type = SPKM3_GLOBUS_PROXY_CERT;
	break;
      }            
    }
  }
  
  return cert_type;
}
/****************************************************************************/
/* Token Integrity Functions                                                */
/****************************************************************************/
int verify_callback(int ok, X509_STORE_CTX *ctx) {
  char buf[256];
  static int v_verbose=0;
  
  if (!ok) {
    spkm3_log_err("verify_callback: WARNING: need extra cert verification\n");
    X509_NAME_oneline(X509_get_subject_name(ctx->current_cert),
		      buf,256);
    spkm3_log_err("verify_callback: %s: error %d at %d depth lookup: %s\n",
	          buf, ctx->error, ctx->error_depth,
	          X509_verify_cert_error_string(ctx->error));

    if(ctx->error == X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION) {
      X509 *xs = ctx->current_cert;

      if(get_cert_type(xs) == SPKM3_GLOBUS_PROXY_CERT)
	ok = 1;
    }

    /*
     * If a cert within the verification path is expired, then
     * fail the request.  Log the problem, and return the
     * current value of "ok".
     */
    if (ctx->error == X509_V_ERR_CERT_HAS_EXPIRED) {
      spkm3_log_err("verify_callback: Cert expired\n");
      return ok;
    }
    
    /*
     * Since we are just checking the certificates, it is
     * ok if they are self signed. But we should still warn
     * the user.
     */
    if (ctx->error == X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) {
      SPKM3_DEBUG(1, ("verify_callback: ignoring: self signed cert\n"));
      ok=1;
    }
    
    /*
     * Continue after extension errors too
     */
    if (ctx->error == X509_V_ERR_INVALID_CA) {
      SPKM3_DEBUG(1, ("verify_callback: ignoring: invalid ca\n"));
      ok=1;
    }
    if (ctx->error == X509_V_ERR_PATH_LENGTH_EXCEEDED) {
      SPKM3_DEBUG(1, ("verify_callback: ignoring: path length exceeded\n"));
      ok=1;
    }
    if (ctx->error == X509_V_ERR_INVALID_PURPOSE) {
      SPKM3_DEBUG(1, ("verify_callback: ignoring: invalid purpose\n"));
      ok=1;
    }
  }
  if (!v_verbose) ERR_clear_error();
  return(ok);
  
}

int check_revocation(X509_STORE_CTX *ctx)
{
  X509_STORE *certstore = NULL;
  X509_STORE_CTX certstorectx;
  X509_OBJECT obj;
  X509_NAME *subject = NULL ;
  X509_NAME *issuer = NULL;
  X509 *xs = NULL;
  X509_CRL *crl = NULL;
  X509_REVOKED *revoked = NULL;
  EVP_PKEY *pubkey = NULL;
  long serial;
  BIO *bio = NULL ;
  int i, n, rc;
  char *cp = NULL ;
  char *cp2 = NULL;
  ASN1_TIME *t = NULL;
  int ok = 0;
  
  SPKM3_DEBUG(1, ("check_revocation: START\n"));
  /* If no CRL file or directory specified, just return with success */
  if(CRLDir == NULL && CRLFile == NULL) return 0;
  
  if((certstore = X509_STORE_new()) == NULL) {
    ERR_print_errors_fp (stderr);
    spkm3_log_err("revocation_check: X509_STORE_new failed\n");
    return -1;
  }
  
  if(!X509_STORE_load_locations(certstore, CRLFile, CRLDir)) {
    ERR_print_errors_fp (stderr);
    spkm3_log_err("check_revocation: failed reading %s or %s\n", 
                  CRLDir, CRLFile);
    ok = -1; goto free_certstore;
  }
  
  X509_STORE_CTX_init(&certstorectx, certstore, NULL, NULL);
  
  /*
   * Determine certificate ingredients in advance
   */
  xs      = X509_STORE_CTX_get_current_cert(ctx);
  subject = X509_get_subject_name(xs);
  issuer  = X509_get_issuer_name(xs);
  
  /*
   * OpenSSL provides the general mechanism to deal with CRLs but does not
   * use them automatically when verifying certificates, so we do it
   * explicitly here. We will check the CRL for the currently checked
   * certificate, if there is such a CRL in the store.
   *
   * We come through this procedure for each cert in the certificate
   * chain, starting with the root-CA's certificate. At each step we've to
   * both verify the signature on the CRL (to make sure it's a valid CRL)
   * and it's revocation list (to make sure the current certificate isn't
   * revoked).  But because to check the signature on the CRL we need the
   * public key of the issuing CA certificate (which was already processed
   * one round before), we've a little problem. But we can both solve it
   * and at the same time optimize the processing by using the following
   * verification scheme (idea and code snippets borrowed from the GLOBUS
   * project):
   *
   * 1. We'll check the signature of a CRL in each step when we find a CRL
   *    through the _subject_ name of the current certificate. This CRL
   *    itself will be needed the first time in the next round, of course.
   *    But we do the signature processing one round before this where the
   *    public key of the CA is available.
   *
   * 2. We'll check the revocation list of a CRL in each step when
   *    we find a CRL through the _issuer_ name of the current certe.
   *    This CRLs signature was then already verified one round before.
   *
   * This verification scheme allows a CA to revoke its own certificate as
   * well, of course.
   */
  
  /*
   * Try to retrieve a CRL corresponding to the _subject_ of
   * the current certificate in order to verify it's integrity.
   */
  memset((char *)&obj, 0, sizeof(obj));
  rc = X509_STORE_get_by_subject(&certstorectx, X509_LU_CRL, 
				 subject, &obj);
  crl = obj.data.crl;
  if (rc > 0 && crl != NULL) {
    /*
     * Log information about CRL
     * (A little bit complicated because of ASN.1 and BIOs...)
     */
    bio = BIO_new(BIO_s_file());
    if(bio == NULL) { 
      ERR_print_errors_fp (stderr);
      spkm3_log_err("failed to initialized BIO\n");
      ok = -1; goto free_certstore;
    }
    BIO_set_fp(bio, stderr, BIO_NOCLOSE);
    
    BIO_printf(bio, "lastUpdate: ");
    ASN1_UTCTIME_print(bio, X509_CRL_get_lastUpdate(crl));
    BIO_printf(bio, ", nextUpdate: ");
    ASN1_UTCTIME_print(bio, X509_CRL_get_nextUpdate(crl));
    n = BIO_pending(bio);
    if((cp = malloc(n+1)) == NULL) {
      ok = -1; goto free_certstore;
    }
    n = BIO_read(bio, cp, n);
    cp[n] = '\0';
    BIO_free(bio);
    cp2 = X509_NAME_oneline(subject, NULL, 0);
    fprintf(stderr, "CA CRL: Issuer: %s, %s", cp2, cp);
    OPENSSL_free(cp2);
    free(cp);
    BIO_free(bio);
    
    /*
     * Verify the signature on this CRL
     */
    pubkey = X509_get_pubkey(xs);
    if (X509_CRL_verify(crl, pubkey) <= 0) {
      spkm3_log_err("Invalid signature on CRL");
      X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_SIGNATURE_FAILURE);
      ERR_print_errors_fp (stderr);
      ok = 1; goto free_certstorectx;
    }
    
    /*
     * Check date of CRL to make sure it's not expired
     */
    if ((t = X509_CRL_get_nextUpdate(crl)) == NULL) {
      spkm3_log_err("Found CRL has invalid nextUpdate field");
      X509_STORE_CTX_set_error(ctx, X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD);
      ERR_print_errors_fp (stderr);
      ok = 1; goto free_certstorectx;
    }
    
    if (X509_cmp_current_time(t) < 0) {
      spkm3_log_err("Found CRL is expired - revoking all certificates until "
                    "you get updated CRL");
      X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_HAS_EXPIRED);
      ERR_print_errors_fp (stderr);
      ok = 1; goto free_certstorectx;
    }
    X509_OBJECT_free_contents(&obj);
  }
  
  /*
   * Try to retrieve a CRL corresponding to the _issuer_ of
   * the current certificate in order to check for revocation.
   */
  memset((char *)&obj, 0, sizeof(obj));
  rc = X509_STORE_get_by_subject(&certstorectx, X509_LU_CRL, 
				 issuer, &obj);
  crl = obj.data.crl;
  if (rc > 0 && crl != NULL) {
    /*
     * Check if the current certificate is revoked by this CRL
     */
#if OPENSSL_LIBRARY_VERSION < 0x00904000
    n = sk_num(X509_CRL_get_REVOKED(crl));
#else
    n = sk_X509_REVOKED_num(X509_CRL_get_REVOKED(crl));
#endif
    for (i = 0; i < n; i++) {
#if OPENSSL_LIBRARY_VERSION < 0x00904000
      revoked = (X509_REVOKED *)sk_value(X509_CRL_get_REVOKED(crl), i);
#else
      revoked = sk_X509_REVOKED_value(X509_CRL_get_REVOKED(crl), i);
#endif
      if (ASN1_INTEGER_cmp(revoked->serialNumber, 
			   X509_get_serialNumber(xs)) == 0) {
	
	serial = ASN1_INTEGER_get(revoked->serialNumber);
	cp = X509_NAME_oneline(issuer, NULL, 0);
	spkm3_log_err("Certificate with serial %ld (0x%lX) revoked per CRL "
                      "from issuer %s", serial, serial, cp);
	OPENSSL_free(cp);
	
	X509_STORE_CTX_set_error(ctx, X509_V_ERR_CERT_REVOKED);
	ERR_print_errors_fp (stderr);
	ok = 1; goto free_certstorectx;
      }
    }
  }
  
  SPKM3_DEBUG(1, ("check_revocation: Passed\n"));

 free_certstorectx:
  if (pubkey != NULL) EVP_PKEY_free(pubkey);
  X509_STORE_CTX_cleanup(&certstorectx);
  X509_OBJECT_free_contents(&obj);

 free_certstore:
  X509_STORE_free(certstore);
  SPKM3_DEBUG(1, ("check_revocation: END %d\n", ok));

  return ok;
}

OM_uint32 verify_X509_cert(X509 *cert, STACK_OF(X509 *) *certchain) {
  
  OM_uint32 minor_status = 0;
  X509_STORE *certstore = NULL;
  X509_STORE_CTX certstorectx;
  int i = 0;
  
  SPKM3_DEBUG(1, ("verify_X509_cert: START\n"));
  
  signal(SIGPIPE,SIG_IGN);
  CRYPTO_malloc_init();
  OpenSSL_add_all_algorithms();
  
  if((certstore = X509_STORE_new()) == NULL) {
    ERR_print_errors_fp (stderr);
    spkm3_log_err("verify_X509_cert: X509_STORE_new failed\n");
    return -1;
  }
  
  X509_STORE_set_verify_cb_func(certstore, verify_callback);
  
  if(!X509_STORE_load_locations(certstore, CAFile, CADir)) {
    ERR_print_errors_fp (stderr);
    spkm3_log_err("verify_X509_cert: failed reading %s or %s\n",
	          CADir, CAFile);
    minor_status = -1; goto err;
  }

  
  X509_STORE_CTX_init(&certstorectx, certstore, cert, NULL);	
  if(certchain) {
    X509_STORE_CTX_trusted_stack(&certstorectx, certchain);
  }
#if OPENSSL_LIBRARY_VERSION > 0x00908000
  X509_STORE_CTX_set_flags(&certstorectx, X509_V_FLAG_ALLOW_PROXY_CERTS);
#endif

  i = X509_verify_cert(&certstorectx);
  if(!i) {
    ERR_print_errors_fp (stderr);
    spkm3_log_err("verify_X509_cert: failed to verify certificate\n");
    minor_status = -1; goto err;
  }
  i = check_revocation(&certstorectx);
  if (i) {
    spkm3_log_err("verify_X509_cert: revocation check returned error\n");
    minor_status = -1; goto err;
  }
  
  SPKM3_DEBUG (1, ("verify_X509_cert: Certificate Verified OK\n"));
  
 err:
  X509_STORE_free(certstore);
  X509_STORE_CTX_cleanup(&certstorectx);
  
  SPKM3_DEBUG(1, ("verify_X509_cert: END %d\n", minor_status));
  return minor_status;
  
}

/* XXX This routine should be returning minor_status values 
   (at least in the manor it is currently called) */
OM_uint32 verify (AlgorithmIdentifier_t *alg, 
		  struct asn_TYPE_descriptor_s *req_asn_type, 
		  void *req_contents,
		  unsigned char *saved_content, 
		  int saved_len,
		  X509 * x509,
		  STACK_OF(X509 *) *certchain,
		  unsigned char *sig_buf, int sig_len)
{
  OM_uint32 minor_status = 0;
  int status = 0;
  EVP_MD_CTX md_ctx;
  EVP_PKEY *pkey = NULL;
  unsigned char *req_encoded_buf = NULL;
  size_t req_encoded_len = 0;
  
  SPKM3_DEBUG(1, ("verify: START\n"));
  ERR_load_crypto_strings ();
  
  /* verify the certificate */
  minor_status = verify_X509_cert(x509, certchain);
  if(minor_status) goto err_verify;
  
  minor_status = encode(req_asn_type, req_contents, 
			&req_encoded_buf, &req_encoded_len);
  if(minor_status) goto err;
  
  SPKM3_DEBUG (1, ("verify: Received Signature sig_len=%d\n", sig_len));
  SPKM3_DEBUG_HEX(1, (sig_buf, sig_len, 0));
  
  /* Read public key */
  if(alg->algorithm.size == md5_rsa_encryption_oid.length &&
     !memcmp(alg->algorithm.buf, md5_rsa_encryption_oid.elements, 
             alg->algorithm.size)) {	
    pkey = X509_get_pubkey (x509);
    if (pkey == NULL) {
      ERR_print_errors_fp (stderr);
      spkm3_log_err("verify: error getting X509 public key\n");
      minor_status = -1;	/* XXX need error code */
      goto err;
    }
  } else {
    spkm3_log_err("verify: unsupported C-ALG\n\n");
    SPKM3_DEBUG_HEX(1, (alg->algorithm.buf, alg->algorithm.size, 0));
    SPKM3_DEBUG (1, ("expecting OID\n"));
    SPKM3_DEBUG_HEX(1, (md5_rsa_encryption_oid.elements, 
			md5_rsa_encryption_oid.length, 0));
    minor_status = EINVAL; /*XXX unsupported signing algorith */
    goto err;
  }
  
  /* Initialize with appropriate hash function */
  if(alg->algorithm.size == md5_rsa_encryption_oid.length &&
     !memcmp(alg->algorithm.buf, md5_rsa_encryption_oid.elements,              
             alg->algorithm.size)) { 
    EVP_VerifyInit (&md_ctx, EVP_md5 ());
  } 
  //case NID_sha1WithRSAEncryption:
  //  EVP_VerifyInit (&md_ctx, EVP_sha1 ());
  else {
    minor_status = EINVAL; /*XXX unsupported signing algorith */
    goto err;
  }	
  
  /* Verify the signature */
  EVP_VerifyUpdate (&md_ctx, req_encoded_buf, req_encoded_len);
  if(saved_content)
    EVP_VerifyUpdate (&md_ctx, saved_content, saved_len);
  
  status = EVP_VerifyFinal (&md_ctx, sig_buf, sig_len, pkey);
  
  if (status != 1) {
    ERR_print_errors_fp (stderr);
    spkm3_log_err ("verify: error verifying signature\n");
#ifdef DEBUG
    exit(1);
#else
    minor_status = GSS_S_G_VALIDATE_FAILED; 
#endif
  } else {
    SPKM3_DEBUG (1, ("verify: Signature Verified OK\n"));
  }

  EVP_MD_CTX_cleanup(&md_ctx);
 err:
  if(req_encoded_buf) free(req_encoded_buf);
  if(pkey) EVP_PKEY_free (pkey);
  
 err_verify:
  ERR_free_strings();
  
  SPKM3_DEBUG(1, ("verify: END %d\n", minor_status));
  return minor_status;
}

OM_uint32 sign (AlgorithmIdentifier_t *alg,
		struct asn_TYPE_descriptor_s *req_asn_type,
		void *req_contents,
		struct asn_TYPE_descriptor_s *rep_asn_type,
		void *rep_contents,
		BIT_STRING_t *integrity,	       
		spkm3_cred_int_t *creds) {

  OM_uint32 minor_status = 0;
  int status = 0;
  EVP_MD_CTX md_ctx;
  FILE *fp = NULL;
  char keyfile[1024];
  unsigned char *sig_buf = NULL;
  size_t sig_len = 0;
  unsigned char *req_encoded_buf = NULL, *rep_encoded_buf = NULL;
  size_t req_encoded_len = 0, rep_encoded_len = 0;
  
  SPKM3_DEBUG(1, ("sign: START\n"));
  ERR_load_crypto_strings ();
  
  if(req_contents)
    minor_status = encode(req_asn_type, req_contents, 
			  &req_encoded_buf, &req_encoded_len);
  if(minor_status) goto err;
  
  if(rep_contents)
    minor_status = encode(rep_asn_type, rep_contents, 
			  &rep_encoded_buf, &rep_encoded_len);
  if(minor_status) goto err;
  
  /* Do the signature */
  if(alg->algorithm.size == md5_rsa_encryption_oid.length &&
     !memcmp(alg->algorithm.buf, md5_rsa_encryption_oid.elements,
             alg->algorithm.size)) {
    EVP_VerifyInit (&md_ctx, EVP_md5 ());
  }
  //case NID_sha1WithRSAEncryption:
  //  EVP_VerifyInit (&md_ctx, EVP_sha1 ());
  else {
    minor_status = EINVAL; /*XXX unsupported signing algorith */
    goto err;
  }	
  
  if(req_encoded_buf)
    EVP_SignUpdate (&md_ctx, req_encoded_buf, req_encoded_len);
  if(rep_encoded_buf)
    EVP_SignUpdate (&md_ctx, rep_encoded_buf, rep_encoded_len);
  
  sig_len = EVP_PKEY_size(creds->priv_key);
  if((sig_buf = (unsigned char *)malloc(sig_len)) == NULL) {
    minor_status = GSS_S_G_MEMORY_ALLOC;
    goto err;
  }
  
  status = EVP_SignFinal (&md_ctx, sig_buf, &sig_len, creds->priv_key);
  
  if (status != 1) {
    ERR_print_errors_fp (stderr);
    spkm3_log_err("sign: error reading private key\n");
    minor_status = EINVAL;	/* XXX need error code */
    goto err;
  }
  
  SPKM3_DEBUG (1, ("sign: Signature Created OK sig_len=%d\n", sig_len));
  SPKM3_DEBUG_HEX(1, (sig_buf, sig_len, 0));
  
  if ((asn1_bitstring_set(integrity, sig_buf, sig_len)) == -1) {
    spkm3_log_err("sign: failed to ASN encode signature %d %s\n",
	          sig_len, sig_buf);
    minor_status = EINVAL;
    goto err;
  }
  
 err:
  if(sig_buf) free(sig_buf);
  if(req_encoded_buf) free(req_encoded_buf);
  if(rep_encoded_buf) free(rep_encoded_buf);
  EVP_MD_CTX_cleanup(&md_ctx);
  ERR_free_strings();
  
  SPKM3_DEBUG(1, ("sign: END %d\n", minor_status));
  return minor_status;
}

/*
 * Function: REQ_integrity_sign()
 *
 * Description:  Computes the checksum over the RequestToken
 * stored in the reqintegrity field required for mutual authentication.
 */
OM_uint32
REQ_integrity_sign(spkm3_context_t * ctx,
		   SPKM_REQ_t *req)
{
  
  OM_uint32 minor_status = 0;
  SPKM3_DEBUG(1, ("REQ_integrity_sign: START\n"));
  minor_status = sign (&req->requestToken.algId, 
		       &asn_DEF_Req_contents, 
		       &req->requestToken.req_contents, NULL, NULL,
		       &req->requestToken.req_integrity, ctx->spkm3_cred);
  
  SPKM3_DEBUG(1, ("REQ_integrity_sign: END %d\n", minor_status));
  return minor_status;
}

/*
 * Function: REQTI_integrity_sign()
 *
 * Description:  Computes the checksum over the RequestToken
 * stored in the reqintegrity field required for mutual authentication.
 */
OM_uint32
REPTI_integrity_sign (spkm3_context_t * ctx, REP_TI_TOKEN_t * rep, 
		      REQ_TOKEN_t* req)
{

  OM_uint32 minor_status = 0;
  
  SPKM3_DEBUG(1, ("REPTI_integrity_sign: START\n"));
  
  /* Initialize with appropriate hash function */
  if(req->algId.algorithm.size == md5_rsa_encryption_oid.length &&
     !memcmp(req->algId.algorithm.buf, md5_rsa_encryption_oid.elements,
	     req->algId.algorithm.size)) {
    SPKM3_DEBUG(1, ("REPTI_integrity_sign: RSA-based signature\n"));
    minor_status = sign (&rep->algId, 
			 &asn_DEF_Rep_ti_contents, 
			 &rep->rep_ti_contents, NULL, NULL,
			 &rep->rep_ti_integ, ctx->spkm3_cred);
  } else if(req->algId.algorithm.size == null_mac_oid.length &&
	    !memcmp(req->algId.algorithm.buf, null_mac_oid.elements,
                    req->algId.algorithm.size)) {
    SPKM3_DEBUG(1, ("REPTI_integrity_sign: NULL-MAC request\n"));
    minor_status = sign (&rep->algId, 
			 &asn_DEF_Rep_ti_contents, 
			 &rep->rep_ti_contents, 
			 &asn_DEF_Req_contents, 
			 &req->req_contents, &rep->rep_ti_integ,  
			 ctx->spkm3_cred);
  } else {
    minor_status = EINVAL; /*XXX unsupported signing algorith */
  }	
  
  
  SPKM3_DEBUG(1, ("REPTI_integrity_sign: END %d\n", minor_status));
  
  return minor_status;
}

/*
 * Function: REQIT_integrity_sign()
 *
 * Description:  Computes the checksum over the RequestToken
 * stored in the reqintegrity field required for mutual authentication.
 */
OM_uint32
REPIT_integrity_sign (spkm3_context_t * ctx, SPKM_REP_IT_t * repit)
{
  
  OM_uint32 minor_status = 0;
  
  SPKM3_DEBUG(1, ("REPIT_integrity_sign: START\n"));
  minor_status = sign (&repit->algId, 
		       &asn_DEF_REP_IT_TOKEN, 
		       &repit->responseToken, NULL, NULL,
		       &repit->rep_it_integ, ctx->spkm3_cred);
  SPKM3_DEBUG(1, ("REPIT_integrity_sign: END %d\n", minor_status));
  
  return minor_status;
}

/*
 * Function: REQ_integrity_verify()
 *
 * Description: grab the srcname and verify that the
 *	req-integrity field was computed by the owner
 *	of the user-certif field
 */

OM_uint32
REQ_integrity_verify (spkm3_context_t * ctx, SPKM_REQ_t *token)
{
  
  OM_uint32 minor_status = 0;
  REQ_TOKEN_t*req = &token->requestToken;
  Certificate_t *cert = 
    token->certif_data->certificationPath->userCertif;
  unsigned char *p = NULL, *encoded_buf = NULL;
  size_t encoded_len = 0;
  int i = 0;

  asn_set *asn_certchain = (asn_set *) 
    token->certif_data->certificationPath->theCACertificates;
  CertificatePair_t *pair = NULL;
  unsigned char *buf = NULL;
  size_t len = 0;
  
  SPKM3_DEBUG(1, ("REQ_integrity_verify: START\n"));
  minor_status = encode(&asn_DEF_Certificate, cert, 
			&encoded_buf, &encoded_len);
  if(minor_status) goto err;
  
  /* at this point, we have DER-encoded cert in seb.buf */
  p = encoded_buf;
  ctx->usercert = d2i_X509(NULL, (char **)&p, encoded_len);
  
  /* let's retrive certificate chain if present */
  if(asn_certchain) {

    SPKM3_DEBUG(1, ("REQ_integrity_verify: %d certs in CA chain\n", 
		    asn_certchain->count));
   
    for(i = 0; i < asn_certchain->count; i++) {

      if(!i) 
	if ((ctx->user_certchain = sk_X509_new_null()) == NULL)
	  goto err;
      
      pair = (CertificatePair_t *)asn_certchain->array[i];

      if(pair->issuedByThisCA) {
	X509 *x = NULL;
	X509_NAME *name = NULL;
	char tmp[256];

	minor_status = encode(&asn_DEF_Certificate, pair->issuedByThisCA, 
			      &buf, &len);
	if(minor_status) goto err;
	p = buf;
	x = d2i_X509(NULL, (char **)&p, len);
	sk_X509_push(ctx->user_certchain, x);

	name = X509_get_subject_name(x);
	SPKM3_DEBUG(1, ("REQ_integrity_verify: #%d cert in chain %s\n", i,
			X509_NAME_oneline(name, tmp, sizeof(tmp))));

	free(buf);
      }

      if(pair->issuedToThisCA) {
      }
    }

    SPKM3_DEBUG(1, ("REQ_integrity_verify: created X509 stack of size %d\n",
		    sk_X509_num(ctx->user_certchain)));
  }

  minor_status = verify (&req->algId, 
			 &asn_DEF_Req_contents, 
			 &req->req_contents, NULL, -1,
			 ctx->usercert, ctx->user_certchain, 
			 req->req_integrity.buf, 
			 req->req_integrity.size);
  
 err:
  if(encoded_buf) free(encoded_buf);
  SPKM3_DEBUG(1, ("REQ_integrity_verify: END %d\n", minor_status));
  
  return minor_status;
  
}

OM_uint32
REPTI_integrity_verify (spkm3_context_t *ctx, SPKM_REP_TI_t *token)
{
  OM_uint32 minor_status = 0;
  REP_TI_TOKEN_t *req = &token->responseToken;
  Certificate_t *cert = 
    token->certif_data->certificationPath->userCertif;
  unsigned char *p, *encoded_buf = NULL;
  size_t encoded_len = 0;
  
  SPKM3_DEBUG(1, ("REPTI_integrity_verify: START\n"));
  minor_status = encode(&asn_DEF_Certificate, cert, 
			&encoded_buf, &encoded_len);
  if(minor_status) goto err;
  
  /* at this point, we have DER-encoded cert in seb.buf */
  p = encoded_buf;
  ctx->usercert = d2i_X509(NULL, (char **)&p, encoded_len);

  if(ctx->req_flags & GSS_C_MUTUAL_FLAG) {
    SPKM3_DEBUG(1, ("REPTI_integrity_verify: RSA-based signature\n"));
    minor_status = verify (&req->algId, 
			   &asn_DEF_Rep_ti_contents, 
			   &req->rep_ti_contents, NULL, -1, 
			   ctx->usercert, NULL, req->rep_ti_integ.buf,  
			   req->rep_ti_integ.size);
  } else {
    SPKM3_DEBUG(1, ("REPTI_integrity_verify: NULL-MAC request\n"));
    minor_status = verify (&req->algId, 
			   &asn_DEF_Rep_ti_contents, 
			   &req->rep_ti_contents, 
			   ctx->der_reqcontents, ctx->der_req_len, 
			   ctx->usercert, NULL, req->rep_ti_integ.buf,
			   req->rep_ti_integ.size);
  }
 err:
  if(encoded_buf) free(encoded_buf);

  SPKM3_DEBUG(1, ("REPIT_integrity_verify: END %d\n", minor_status));
  return minor_status;
  
}

OM_uint32
REPIT_integrity_verify(spkm3_context_t * ctx,
		       SPKM_REP_IT_t *token)
{
  OM_uint32 minor_status = 0;
  
#if 0	/* XXX XXX SPKM3_SIGNATURE_LEN is not the right length here!?!?!?! */
	/* Because of trailing zeros and bit string encoding, the signature
	 * length may not be correct.  The OpenSSL routine called in vrfyit
	 * insists on the correct signature length, so we need to restore
	 * the trailing zeros.
	 */
#ifdef DEBUG
  SPKM3_DEBUG(1, ("REPIT_integrity_verify: calling vrfyit with sig_len %d\n",
	      token->rep_it_integ.size));
#endif
  
  if (token->rep_it_integ.size != SPKM3_SIGNATURE_LEN) {
    if (!spkm3_bit_string_repair (&token->rep_it_integ,
				  SPKM3_SIGNATURE_LEN)) {
      SPKM3_DEBUG(1, ("REPIT_integrity_verify: Failed to repair signature\n"));
      goto err;
    }
  }
#endif
  SPKM3_DEBUG(1, ("REPIT_integrity_verify: START\n"));
  minor_status = verify (&token->algId, 
			 &asn_DEF_REP_IT_TOKEN, 
			 &token->responseToken, NULL, -1, ctx->usercert, 
			 ctx->user_certchain, token->rep_it_integ.buf, 
			 token->rep_it_integ.size);
  
  SPKM3_DEBUG(1, ("REPIT_integrity_verify: END %d\n", minor_status));
  return minor_status;
}

