/*
 *  spkm3/compare_name.c
 *
 *  Copyright (c) 2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "gssapi_spkm3.h"

/*
 * Function: spkm3_gss_compare_name()
 *
 * Description: compares two internal-form names.
 *
 * Sets *name_equal to non-zero if the names refer to the same entity,
 * zero if they cannot be verified to be the same entity.
 *
 */
OM_uint32
spkm3_gss_compare_name (
			OM_uint32 * min_status,	/* minor_status */
			const gss_name_t name1,	/* input_name_buffer */
			const gss_name_t name2,	/* input_name_buffer */
			int *name_equal)
				/* non-zero => names refer to same entity,
				   zero => names are not known to refer to
				   same entity */
{
	AttributeValue_t *entry1, *entry2;
	OBJECT_IDENTIFIER_t *oid = NULL;
	OM_uint32 maj_status = GSS_S_COMPLETE;
	spkm3_name_desc_t *n1, *n2;
	int ndx;

	*min_status = 0;
	*name_equal = 0;	/* Assume names are not the same */

	SPKM3_DEBUG(1, ("spkm3_gss_compare_name:\n"));

	/* XXX Need to verify these are valid names XXX */
	n1 = (spkm3_name_desc_t *) name1;
	n2 = (spkm3_name_desc_t *) name2;

	/* If they're both NULL or both GSS_C_NO_NAME, then they are equal */
	if ((n1 == NULL && n2 == NULL) ||
	    (n1 == GSS_C_NO_NAME && n2 == GSS_C_NO_NAME)) {
		*name_equal = 1;
		goto exit;
	}

	/* However, if only one is NULL or GSS_C_NO_NAME, they can't be equal */
	if ((n1 == NULL || n2 == NULL) ||
	    (n1 == GSS_C_NO_NAME || n2 == GSS_C_NO_NAME)) {
		goto exit;
	}

	/*
	 * rfc2744 says if either name "denotes an anonymous principal, the
	 * routines should indicate that the two names do not refer to the
	 * same identity."
	 */
	if (g_OID_equal (n1->oid, GSS_C_NT_ANONYMOUS) ||
	    g_OID_equal (n2->oid, GSS_C_NT_ANONYMOUS)) {
		goto exit;
	}

	/* If either is missing an X509 name, then they can't be equal */
	if (n1->asn1_name == NULL || n2->asn1_name == NULL)
		goto exit;


	/*
	 * XXX The Globus GSI code has a bunch of different checks that we
	 * may need to follow here.  But for now, just do a straight
	 * compare of the _first_ common name. XXX
	 */

	oid =txt_2_asn1_obj("CN");
	entry1 = asn1_get_entry_by_oid (n1->asn1_name, oid);
	entry2 = asn1_get_entry_by_oid (n2->asn1_name, oid);
	
	asn_DEF_OBJECT_IDENTIFIER.free_struct( &asn_DEF_OBJECT_IDENTIFIER,
			oid, 0);

	/* If we can't find CN, then they don't match */
	if (entry1 == NULL || entry2 == NULL) {
		goto exit;
	}

	/* Compare the values of the two CNs */
#if 0
	SPKM3_DEBUG(1, ("spkm3_gss_compare_name: comparing '%*s' and '%*s'\n",
		    entry1->size, entry1->buf,
		    entry2->size, entry2->buf));
#else
	SPKM3_DEBUG(1, ("spkm3_gss_compare_name: comparing '%s' and '%s'\n",
		    entry1->buf, entry2->buf));
#endif

	if ((entry1->size == entry2->size) &&
	    strncasecmp (entry1->buf, entry2->buf,
			 entry1->size)) {
		*name_equal = 1;
		goto exit;
	}

exit:
	if(maj_status != GSS_S_COMPLETE)
	  spkm3_log_status ("spkm3_gss_compare_name:", maj_status, *min_status);

	if (entry1)
		asn_DEF_AttributeValue.free_struct (&asn_DEF_AttributeValue,
			entry1, 0);
	if (entry2)
		asn_DEF_AttributeValue.free_struct (&asn_DEF_AttributeValue,
			entry2, 0);
	
	return (maj_status);
}
