/*
 * spkm3/acquire_cred.c 
 *
 *  Copyright (c) 2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Kevin Coffman <kwc@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include "gssapi_spkm3.h"


/*
 * Locate certificate with name matching x509name and the
 * corresponding private key
	kinit, kx509, then
	kxlist -p to get PEM kx509 key into file /tmp/x509up_uUID
 */
OM_uint32
spkm3_get_cred (OM_uint32 * min_status,
		gss_cred_usage_t cred_usage,
		spkm3_cred_int_t ** cred_handle,   /* output_cred_handle */
		X509_NAME * x509name)
{
	OM_uint32 maj_status = GSS_S_FAILURE;
	spkm3_cred_int_t *spkm3_cred;
	char filename[1024];
	spkm3_usertype_t utype;
	int cert_type;

	*cred_handle = NULL;

	spkm3_cred = calloc (1, sizeof (spkm3_cred_int_t));
	if (spkm3_cred == NULL) {
		*min_status = ENOMEM;
		goto out;
	}

	/* XXX
	 * Need better defaults for cert files.
	 * Need a way to specify them?
	 */
	if (cred_usage & GSS_C_ACCEPT) {
		SPKM3_DEBUG(1, ("get_cred: reading system creds\n"));
		utype = SPKM3_USERTYPE_SYSTEM;
	}
	else if (cred_usage & GSS_C_INITIATE) {
		SPKM3_DEBUG(1, ("get_cred: reading user creds "
				"uid=%d euid=%d\n", getuid(), geteuid()));
		utype = SPKM3_USERTYPE_USER;
	}
	else {
		spkm3_log_err("get_cred: unknown cred usage\n");
		*min_status = EINVAL;
		free(spkm3_cred);
		maj_status = GSS_S_FAILURE;
		goto out;
	}
	
	if (spkm3_get_filename(SPKM3_FILETYPE_CERT, utype,
				filename, sizeof(filename))) {
		spkm3_log_err("get_cred: spkm3_get_filename failed to create "
			      "filename %s\n", filename);
		*min_status = EPERM;
		maj_status = GSS_S_FAILURE;
		free(spkm3_cred);
		goto out;
	}

	if((spkm3_cred->cert = 
	    spkm3_load_cert(filename, &cert_type)) == NULL) {
		spkm3_log_err("get_cred: spkm3_get_filename failed to get "
			      "x509 cert from %s\n", filename);
		asn_DEF_Certificate.free_struct(&asn_DEF_Certificate,
						spkm3_cred->wire_cert, 0);
		*min_status = -1;
		maj_status = GSS_S_FAILURE;
		asn_DEF_Certificate.free_struct(&asn_DEF_Certificate,
						spkm3_cred->wire_cert, 0);
		free(spkm3_cred);		
		goto out;

	}
	
	if((spkm3_cred->wire_cert = 
	    spkm3_load_asn_cert(filename, cert_type, 
			        &spkm3_cred->wire_name)) == NULL) {
		spkm3_log_err("get_cred: spkm3_get_filename failed to get " 
			      "asn1 encoded cert from %s\n", filename);
		*min_status = EPERM;
		maj_status = GSS_S_FAILURE;
		free(spkm3_cred);
		goto out;
	}
#if 1
	spkm3_cred->wire_certchain = 
		spkm3_load_asn_certchain(spkm3_cred->cert, filename, 
                                         CAFile, CADir, 
					 &spkm3_cred->certchain_len,
					 min_status);
	if(*min_status) {
        	spkm3_log_err("get_cred: spkm3_load_asn_certchain failed to "
			      "get asn1 encoded certchaing from %s %s %s\n",
			      filename, CAFile, CADir);
		*min_status = -1;
		maj_status = GSS_S_FAILURE;
		asn_DEF_Certificate.free_struct(&asn_DEF_Certificate,
						spkm3_cred->wire_cert, 0);
		X509_free(spkm3_cred->cert);
		free(spkm3_cred);		
		goto out;
	}
#endif
	if (spkm3_get_filename(SPKM3_FILETYPE_PRIV_KEY, utype,
				filename, sizeof(filename))) {
		spkm3_log_err("get_cred: spkm3_get_filename failed to create "
			      "filename %s\n", filename);
		*min_status = EPERM;
		maj_status = GSS_S_FAILURE;
		asn_DEF_Certificate.free_struct(&asn_DEF_Certificate,
						spkm3_cred->wire_cert, 0);
		X509_free(spkm3_cred->cert);
		if(spkm3_cred->wire_certchain && 
		   spkm3_cred->certchain_len > 0) {
		  int i =0;
		  for(i = 0; i < spkm3_cred->certchain_len; i++) {
		    asn_DEF_Certificate.free_struct(&asn_DEF_CertificatePair,
		      &spkm3_cred->wire_certchain[i], 1);       
		  }
		  free(spkm3_cred->wire_certchain);
		}
		free(spkm3_cred);		

		goto out;
	}

	if((spkm3_cred->priv_key = spkm3_load_key(filename)) == NULL) {
		spkm3_log_err("get_cred: spkm3_get_filename failed to get "
			      "x509 key frp, %s\n", filename);
		*min_status = -1;
		maj_status = GSS_S_FAILURE;
		asn_DEF_Certificate.free_struct(&asn_DEF_Certificate,
						spkm3_cred->wire_cert, 0);
		X509_free(spkm3_cred->cert);

		if(spkm3_cred->wire_certchain && 
		   spkm3_cred->certchain_len > 0) {
		  int i =0;
		  for(i = 0; i < spkm3_cred->certchain_len; i++) {
		    asn_DEF_Certificate.free_struct(&asn_DEF_CertificatePair,
		      &spkm3_cred->wire_certchain[i], 1);       
		  }
		  free(spkm3_cred->wire_certchain);
		}
		free(spkm3_cred);		
		
		goto out;

	}

	*cred_handle = spkm3_cred;
	maj_status = GSS_S_COMPLETE;

      out:
	return maj_status;

}

/* XXX XXX XXX XXX XXX XXX XXX XXX XXX */
OM_uint32
spkm3_get_cred_lifetime (OM_uint32 min_status,
			 spkm3_cred_int_t * cred, OM_uint32 * time)
{
	*time = 555;		/* XXX */
	return GSS_S_COMPLETE;
}


OM_uint32
spkm3_gss_acquire_cred (
			OM_uint32 * min_status,
			gss_name_t desired_name,
			OM_uint32 time_req,
			gss_OID_set desired_mechs,
			gss_cred_usage_t cred_usage,
			gss_cred_id_t * output_cred_handle,
			gss_OID_set * actual_mechs,
			OM_uint32 * time_rec)
{
	OM_uint32 maj_status = GSS_S_FAILURE;
	OM_uint32 tmp_min_status;
	Name_t *asn1name = NULL;
	spkm3_cred_int_t *spkm3_cred = NULL;


	*min_status = 0;
	*output_cred_handle = NULL;

	if (actual_mechs != NULL) {
		maj_status = spkm3_gss_indicate_mechs (&tmp_min_status,
						       actual_mechs);
		if (GSS_ERROR (maj_status)) {
			goto out;
		}
	}

	if (desired_name) {
		/* XXX Assume this is a handle to one of our name structures */
		asn1name = ((spkm3_name_desc_t *) desired_name)->asn1_name;
	}

	maj_status = spkm3_get_cred (min_status,
				     cred_usage, &spkm3_cred, asn1name);
	if (GSS_ERROR (maj_status)) {
		goto out;
	}

	*output_cred_handle = spkm3_cred;

	if (time_rec != NULL) {
		/* XXX Need to fill in the time (use certificate's expiration?) */
		maj_status =
		   spkm3_get_cred_lifetime (*min_status, spkm3_cred, 
					    time_rec);
		if (GSS_ERROR (maj_status)) {
			goto out;
		}
	}

	maj_status = GSS_S_COMPLETE;

      out:
        if(maj_status != GSS_S_COMPLETE)
	  spkm3_log_status ("spkm3_gss_acquire_cred:", maj_status, *min_status);

	return maj_status;
}
