/*
 *  spkm3/accept_sec_context.c
 *
 *  Copyright (c) 2001,2004 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *  Olga Kornievskaia <aglo@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "gssapi_spkm3.h"
#include "spkm3_mech.h"


/*
 * Function: spkm3_gss_accept_sec_context()
 *
 * Description: 
 * responds to init_sec_context queries, processes input_token
 * returns output token
 * 
 * NOTE: spkm3's most basic form allows for no initiator or target credentials
 */

OM_uint32
spkm3_gss_accept_sec_context (
			      OM_uint32 * minor_status,
			      gss_ctx_id_t * context_handle_P,
			      gss_cred_id_t verifier_cred_handle,
			      gss_buffer_t input_token,
			      gss_channel_bindings_t input_chan_bindings,
			      gss_name_t * src_name,
			      gss_OID * mech_type,
			      gss_buffer_t output_token,
			      OM_uint32 * ret_flags,
			      OM_uint32 * time_rec,
			      gss_cred_id_t * delegated_cred_handle)
{
  OM_uint32 major_status = 0;
  spkm3_context_t *ctx = *context_handle_P;
  SPKMInnerContextToken_t *spkm_ict = NULL;
  REQ_TOKEN_t *reqptr = NULL;
  OM_uint32 err_code;
  
  SPKM3_DEBUG(1, ("spkm3_gss_accept_sec_context: START\n"
		  "context_handle %p input_token %p (len %d) "
		  "src_name %p\n",
		  *context_handle_P, input_token,
		  input_token ? input_token->length : -1,
		  src_name));
  
  /* initialize variables */
  major_status = GSS_S_FAILURE;
  *minor_status = GSS_S_G_VALIDATE_FAILED;
  
  if (output_token == NULL || context_handle_P == NULL)
    goto err;
  
  output_token->length = 0;
  output_token->value = NULL;
  
  if(src_name) 
    *src_name = (gss_name_t) NULL;

  /* spkm3: no credentials. XXX This may need to be changed...XXX */
  if (verifier_cred_handle != GSS_C_NO_CREDENTIAL)
    verifier_cred_handle = GSS_C_NO_CREDENTIAL;
  
  /*
   * Parse the input token.  This routine always returns a
   * context.  It either creates a new one, or locates the
   * appropriate existing context.
   */
  if ((*minor_status = spkm3_parse_token_internal (&ctx, input_token)) != 0) {
    spkm3_log_err("spkm3_gss_accept_sec_context: failed to "
		  "parse internal token\n");
    major_status = GSS_S_DEFECTIVE_TOKEN;
    goto err;
  }
  spkm_ict = &ctx->in_ict->innerContextToken;
  
  /*
   * Verify this token is for us, or process appropriately ...
   */
  if (ctx->spkm_tok_type != SPKM_TOK_CTX_REQ &&
      ctx->spkm_tok_type != SPKM_TOK_CTX_REPIT) {
    switch (ctx->spkm_tok_type) {
    case SPKM_TOK_CTX_ERROR:
      /* XXX handle error token */
      SPKM3_DEBUG(1, ("accept_sec_context received ERROR token\n"));
      goto err;
      break;
    case SPKM_TOK_DEL_CTX:
      /* XXX handle delete token */
      SPKM3_DEBUG(1, ("accept_sec_context received DELETE token\n"));
      goto err;
      break;
    default:
      spkm3_log_err("accept_sec_context received unknown token type %d\n", 
		    ctx->spkm_tok_type);
      major_status = GSS_S_FAILURE;
      goto err;
    }
  }
  
  if ((*context_handle_P == (gss_ctx_id_t) GSS_C_NO_CONTEXT)) {
    /* We should be dealing with a fresh context */
    if (ctx->state != SPKM3_CTX_STATE_NEW) {
      spkm3_log_err("accept_sec_context context state is %d with no input "
		    "context handle!\n", ctx->state);
      major_status = GSS_S_FAILURE;
      goto err;
    }
    if (ctx->spkm_tok_type != SPKM_TOK_CTX_REQ) {
      spkm3_log_err("accept_sec_context token type is %d, expecting "
		    "SPKM_TOK_CTX_REQ!\n", ctx->spkm_tok_type);
      major_status = GSS_S_FAILURE;
      goto err;
    }
    
    reqptr = &spkm_ict->choice.req.requestToken;
#ifdef ASNDEBUG
    asn_fprint(stderr, &asn_DEF_REQ_TOKEN, reqptr);
#endif
    
    /* XXX
     * Verify the key establishment algorithm requested
     * and call the appropriate routine
     */
    
    spkm3_set_req_flags (ctx, &reqptr->req_contents);
    
    /* receiver diffie-hellman */
    if((*minor_status = REQ_process(ctx)) != 0) {
      spkm3_log_err("Failed processing REQ token\n");
      spkm3_context_free(ctx);
      ctx = NULL;
      *context_handle_P == (gss_ctx_id_t) GSS_C_NO_CONTEXT;

      goto err;
    }
    
    /*
     * OK, prepare the reply token
     */
    major_status = GSS_S_FAILURE;
    
    /* generate a REP_TI_TOKEN - add contextid to ctx */
    if((*minor_status = REPTI_create(ctx, output_token)) != 0) {
      spkm3_log_err("Failed creating REP-TI token\n");
      spkm3_context_free(ctx);
      ctx = NULL;
      *context_handle_P == (gss_ctx_id_t) GSS_C_NO_CONTEXT;
      goto err;
    }
  } else {
    if (ctx->state != SPKM3_CTX_STATE_REPIT) {
      spkm3_log_err("accept_sec_context context state is %d, expecting "
		    "SPKM3_CTX_STATE_REPIT!\n", ctx->state);
      major_status = GSS_S_FAILURE;
      goto err;
    }
    if (ctx->spkm_tok_type != SPKM_TOK_CTX_REPIT) {
      /* Note that if the peer's gss_accept_sec_context() receives 
       * SPKM-REQ token when it is expecting a SPKM-REP-IT token, 
       * it will ignore SPKM-REQ and return a zero-length output 
       * token with a major status of GSS_S_CONTINUE_NEEDED
       */
      if(ctx->spkm_tok_type == SPKM_TOK_CTX_REQ) {
	SPKM3_DEBUG(1, ("WARNING accept_sec_context expecting "
			"SPKM_TOK_CTX_REPIT but received SPKM3_CTX_REQ\n"));
	major_status = GSS_S_CONTINUE_NEEDED;
      } else {
	spkm3_log_err("accept_sec_context token type is %d, expecting "
		      "SPKM_TOK_CTX_REPIT!\n", ctx->spkm_tok_type);
	major_status = GSS_S_FAILURE;
      }
      goto err;
    }
    
    /* Process the REPIT token */
    if((*minor_status = REPIT_process(ctx)) != 0) {
      spkm3_log_err("Failed processing REPIT token\n");
      spkm3_context_free(ctx);
      ctx = NULL;
      *context_handle_P == (gss_ctx_id_t) GSS_C_NO_CONTEXT;
      goto err;
    }
  }
    
  if (mech_type)
    *mech_type = gss_mech_oid_spkm3;
  
  SPKM3_DEBUG(1, ("BEFORE state %x req_flags %x\n", ctx->state, ctx->req_flags));

  switch (ctx->state) {
  case SPKM3_CTX_STATE_NEW:
    if (ctx->req_flags & GSS_C_MUTUAL_FLAG) {
      ctx->state = SPKM3_CTX_STATE_REPIT;
      major_status = GSS_S_CONTINUE_NEEDED;
      
    } else {
      ctx->state = SPKM3_CTX_STATE_AUTH;
      major_status = GSS_S_COMPLETE;
      ctx->established = 1;
    }
    
    SPKM3_DEBUG(1, ("spkm3_gss_accept_sec_context: Saving context %p\n", ctx));
    
    if (!(kg_save_ctx_id ((gss_ctx_id_t) ctx))) {
      spkm3_context_free (ctx);
      ctx = 0;
      *minor_status = GSS_S_G_VALIDATE_FAILED;
      major_status = GSS_S_FAILURE;
      goto err;
    }
    
    break;
  case SPKM3_CTX_STATE_REPIT:
    ctx->state = SPKM3_CTX_STATE_AUTH;
    major_status = GSS_S_COMPLETE;
    ctx->established = 1;
    break;
  }
  
  if(major_status == GSS_S_COMPLETE && src_name) {
    Name_t * src;
    if (ctx->spkm_tok_type == SPKM_TOK_CTX_REQ)
      src = reqptr->req_contents.src_name;
    else if (ctx->spkm_tok_type == SPKM_TOK_CTX_REPIT)
      src = spkm_ict->choice.rep_it.responseToken.src_name;
    else {
      goto err;
    }
    
    if(*minor_status = spkm3_set_srcname (src_name, src, ctx->req_flags)) {
      major_status = GSS_S_FAILURE;
      goto err;
    }
  }

  *context_handle_P = ctx;
  SPKM3_DEBUG(1, ("AFTER  state %x req_flags %x ctx %p context_handle_P %p\n", 
		  ctx->state, ctx->req_flags, ctx, *context_handle_P));
  
 err:
  if (ctx && ctx->in_ict) {
    SPKM3_DEBUG(1, ("spkm3_gss_accept_sec_context: freeing in_ict at %p\n", ctx->in_ict));
    asn_DEF_InitialContextToken.free_struct(&asn_DEF_InitialContextToken,
					    ctx->in_ict, 0);
    ctx->in_ict = NULL;
  }
  
  
  if(major_status != GSS_S_COMPLETE && 
     major_status != GSS_S_CONTINUE_NEEDED)
    spkm3_log_status ("spkm3_gss_accept_sec_context:", major_status, *minor_status);
  
  SPKM3_DEBUG(1, ("spkm3_gss_accept_sec_context: END\n"));
  
  return major_status;
}
