/*
  gssd.c

  Copyright (c) 2000 The Regents of the University of Michigan.
  All rights reserved.

  Copyright (c) 2000 Dug Song <dugsong@UMICH.EDU>.
  Copyright (c) 2002 Andy Adamson <andros@UMICH.EDU>.
  All rights reserved, all wrongs reversed.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of the University nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  $Id: gssd.c,v 1.13 2002/02/12 22:38:36 andros Exp $
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <rpc/rpc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include "gss.h"

#define GSS_SERVER_NAME		"nfs"

/* gss_proc.c */
extern bool_t gssd_acquire_cred(char *server_name);

/* gss_svc.c */
extern void gssprog_1(struct svc_req *rqstp, SVCXPRT *xprt);

/* XXX - for gss_svc.c */
int _rpcpmstart, _rpcfdtype, _rpcsvcdirty;

void
sig_die(int signal)
{
	pmap_unset(GSSPROG, GSSVERS);
  /* destroy krb5 machine creds */
	gss_kt_destroy_tickets();
	syslog(LOG_INFO, "exiting on signal %d", signal);
	exit(1);
}

static void
usage(char *progname)
{
	fprintf(stderr, "usage: %s [-n|--no-krb5]\n", progname);
	exit(1);
}

int
main(int argc, char *argv[])
{
	SVCXPRT *xprt;
	int c, sock;
	struct sockaddr_in sin;
	int one = 1;
	int i;
	int get_creds = 1;

	for (i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-n"))
			get_creds = 0;
		else if (!strcmp(argv[i], "--no-krb5"))
			get_creds = 0;
		else
			usage(argv[0]);
	}

#ifndef DEBUG
	if (daemon(0, 0) < 0) {
		perror("fork");
		exit(1);
	}
#endif
	openlog("gssd", LOG_PID, LOG_DAEMON);

	signal(SIGINT, sig_die);
	signal(SIGTERM, sig_die);
	signal(SIGHUP, sig_die);
	
	pmap_unset(GSSPROG, GSSVERS);
	
	/* Register UDP transport. */
	if ((sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
		syslog(LOG_ERR, "cannot create udp socket: %m");
		exit(1);
	}
	setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one));
	
	memset(&sin, 0, sizeof(sin));
#ifndef __linux__
	sin.sin_len = sizeof(sin);
#endif
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	
	if ((xprt = svcudp_create(sock)) == (SVCXPRT *)NULL) {
		syslog(LOG_ERR, "couldn't do udp_create");
		exit(1);
	}
	if (!svc_register(xprt, GSSPROG, GSSVERS, gssprog_1, IPPROTO_UDP)) {
		syslog(LOG_ERR, "unable to register (GSSPROG, GSSVERS, udp).");
		exit(1);
	}
	
	/* Register TCP transport. */
	if ((sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
		syslog(LOG_ERR, "cannot create tcp socket: %m");
		exit(1);
	}
	setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one));
	
	memset(&sin, 0, sizeof(sin));
#ifndef __linux__
	sin.sin_len = sizeof(sin);
#endif
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	
	if ((xprt = svctcp_create(sock, 0, 0)) == (SVCXPRT *)NULL) {
		syslog(LOG_ERR, "couldn't do tcp_create");
		exit(1);
	}
	if (!svc_register(xprt, GSSPROG, GSSVERS, gssprog_1, IPPROTO_TCP)) {
		syslog(LOG_ERR, "unable to register (GSSPROG, GSSVERS, tcp).");
		exit(1);
	}

	/* Initialize server. */
	if (!gssd_init_defaults()) {
		syslog(LOG_ERR, "couldn't initialize defaults");
		exit(1);
	}
	if (get_creds && !gssd_acquire_cred(GSS_SERVER_NAME)) {
		syslog(LOG_ERR, "couldn't get credentials");
		exit(1);
	}
	if (get_creds && !krb5_kt_get_tickets()) {
		syslog(LOG_ERR, "couldn't get krb5 machine tickets");
		exit(1);
	}

	/* Run server. */
	svc_run();
	
	syslog(LOG_ERR, "svc_run returned!");
	
	abort();
	
	/* NOTREACHED */
}
