/*
  gss_proc.c
  
  Copyright (c) 2000 The Regents of the University of Michigan.
  All rights reserved.

  Copyright (c) 2000 Dug Song <dugsong@UMICH.EDU>.
  Copyright (c) 2001 Andy Adamson<andros@UMICH.EDU>.
  All rights reserved, all wrongs reversed.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of the University nor the names of its
     contributors may be used to endorse or promote products derived
     from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  $Id: gss_proc.c,v 1.19 2002/06/03 17:18:55 andros Exp $
*/

#include <sys/types.h>
#include <rpc/rpc.h>
#include <syslog.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#ifdef HAVE_KRB5
#include <gssapi/gssapi.h>
#include <gssapi/gssapi_generic.h>
#define CA_RUN_AS_MACHINE  0x00000200
#elif HAVE_HEIMDAL
#include <gssapi.h>
#define gss_nt_service_name GSS_C_NT_HOSTBASED_SERVICE
#endif
#include "gss.h"

#ifdef DEBUG
void
print_hexl(u_int *p, u_int length, u_int offset)
{
  u_int i, j, jm;
  unsigned char c, *cp;
 
  printf("\n");
  cp = (unsigned char *) p;
 
  for (i = 0; i < length; i += 0x10) {
    printf("  %04x: ", (u_int)(i + offset));
    jm = length - i;
    jm = jm > 16 ? 16 : jm;

    for (j = 0; j < jm; j++) {
      if ((j % 2) == 1)
        printf("%02x ", (u_int)cp[i+j]);
      else
        printf("%02x", (u_int)cp[i+j]);
    }
    for (; j < 16; j++) {
      if ((j % 2) == 1)
        printf("   ");
      else
        printf("  ");
    }
    printf(" ");

    for (j = 0; j < jm; j++) {
      c = cp[i+j];
      c = isprint(c) ? c : '.';
      printf("%c", c);
    }
    printf("\n");
  }
}
#endif /* DEBUG */



/* Default values. */
u_int	nobody_uid;
u_int	nobody_gid;
char   *nobody_user = "nobody";
char   *nobody_group = "nobody";

/* adapedt from mit kerberos 5 ../lib/gssapi/mechglue/mglueP.h 
 * this is what gets passed around when the mechglue code is
 * enabled */
typedef struct gss_union_ctx_id_t {
  GSS_OID     mech_type;
  GSS_CTX_ID_T    internal_ctx_id;
} gss_union_ctx_id_desc, *gss_union_ctx_id_t;


/* GSS server credentials. */
static gss_cred_id_t	gssd_creds;

static bool_t
hosts_access(struct svc_req *rqstp)
{
	struct sockaddr_in *sinp;
	SVCXPRT *xprt = rqstp->rq_xprt;
	struct authunix_parms *aup;

	sinp = svc_getcaller(xprt);
	
	/* Require loopback source address, reserved source port. */
	if (ntohl(sinp->sin_addr.s_addr) != INADDR_LOOPBACK ||
	    ntohs(sinp->sin_port) >= IPPORT_RESERVED) {
		goto bad_auth;
	}
	/* Require root AUTH_UNIX credentials. */
	aup = (struct authunix_parms *)rqstp->rq_clntcred;
	if (rqstp->rq_cred.oa_flavor != AUTH_UNIX || aup->aup_uid != 0) {
		syslog(LOG_ERR,"RPC_AUTHA_TOOWEAK: flavor %d uid %d\n",rqstp->rq_cred.oa_flavor,aup->aup_uid);
		goto bad_auth;
	}
	return (TRUE);
	
 bad_auth:
	syslog(LOG_ERR, "connection refused from %s port %d",
	       inet_ntoa(sinp->sin_addr), ntohs(sinp->sin_port));
	svcerr_weakauth(xprt);
	
	return (FALSE);
}

/* XXX put machine login here! */
bool_t
gssd_init_defaults(void)
{
	struct passwd *pw;
	struct group *gr;

	if ((pw = getpwnam(nobody_user)) == NULL ||
	    (gr = getgrnam(nobody_group)) == NULL)
		return (FALSE);

	nobody_uid = pw->pw_uid;
	nobody_gid = gr->gr_gid;

	return (TRUE);
}

bool_t
gssd_acquire_cred(char *server_name)
{
	gss_buffer_desc name;
	gss_name_t target_name;
	OM_uint32 maj_stat, min_stat;
	
	name.value = (void *)server_name;
	name.length = strlen(server_name);
	
	maj_stat = gss_import_name(&min_stat, &name, gss_nt_service_name,
				   &target_name);
	if (maj_stat != GSS_S_COMPLETE)
		return (FALSE);

	maj_stat = gss_acquire_cred(&min_stat, target_name, 0,
				    GSS_C_NULL_OID_SET, GSS_C_ACCEPT,
				    &gssd_creds, NULL, NULL);
	
	(void)gss_release_name(&min_stat, &target_name);
	
	return (maj_stat == GSS_S_COMPLETE);
}

GSS_BUFFER_T *
uid_to_name_1_svc(u_int *uidp, struct svc_req *rqstp)
{
	static GSS_BUFFER_T name;
	static char user[16];
	struct passwd *pw;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "uid_to_name_svc arg (%d)\n", *uidp);
#endif
	name.GSS_BUFFER_T_val = user;
	
	if ((pw = getpwuid(*uidp)) == NULL) {
		strncpy(user, nobody_user, sizeof(user) - 1);
		user[sizeof(user) - 1] = '\0';
	}
	else {
		strncpy(user, pw->pw_name, sizeof(user) - 1);
		user[sizeof(user) - 1] = '\0';
	}
	name.GSS_BUFFER_T_len = strlen(user);
#ifdef DEBUG
	fprintf(stderr, "uid_to_name_svc res (%.*s)\n",
		name.GSS_BUFFER_T_len, name.GSS_BUFFER_T_val);
#endif	
	return (&name);
}

u_int *
name_to_uid_1_svc(GSS_BUFFER_T *namep, struct svc_req *rqstp)
{
	static u_int uid;
	char user[256];
	struct passwd *pw;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "name_to_uid_svc arg (%.*s)\n",
		namep->GSS_BUFFER_T_len, namep->GSS_BUFFER_T_val);
#endif
	uid = nobody_uid;
	
	if (namep->GSS_BUFFER_T_len < sizeof(user) - 1) {
		memcpy(user, namep->GSS_BUFFER_T_val, namep->GSS_BUFFER_T_len);
		user[namep->GSS_BUFFER_T_len] = '\0';
		
		/* XXX - user@REALM handling */
		strtok(user, "@");
		if ((pw = getpwnam(user)) != NULL)
			uid = pw->pw_uid;
	}
#ifdef DEBUG
	fprintf(stderr, "name_to_uid_svc res (%d)\n", uid);
#endif
	return (&uid);
}

GSS_BUFFER_T *
gid_to_name_1_svc(u_int *gidp, struct svc_req *rqstp)
{
	static GSS_BUFFER_T name;
	static char group[16];
	struct group *gr;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "gid_to_name_svc arg (%d)\n", *gidp);
#endif
	name.GSS_BUFFER_T_val = group;
	
	if ((gr = getgrgid(*gidp)) == NULL) {
		strncpy(group, nobody_group, sizeof(group) - 1);
		group[sizeof(group) - 1] = '\0';
	}
	else {
		strncpy(group, gr->gr_name, sizeof(group) - 1);
		group[sizeof(group) - 1] = '\0';
	}
	name.GSS_BUFFER_T_len = strlen(group);
#ifdef DEBUG
	fprintf(stderr, "gid_to_name_svc res (%.*s)\n",
		name.GSS_BUFFER_T_len, name.GSS_BUFFER_T_val);
#endif
	return (&name);
}

u_int *
name_to_gid_1_svc(GSS_BUFFER_T *namep, struct svc_req *rqstp)
{
	static u_int gid;
	char group[256];
	struct group *gr;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "name_to_gid_svc arg (%.*s)\n",
		namep->GSS_BUFFER_T_len, namep->GSS_BUFFER_T_val);
#endif
	gid = nobody_gid;
	
	if (namep->GSS_BUFFER_T_len < sizeof(group) - 1) {
		memcpy(group, namep->GSS_BUFFER_T_val,
		       namep->GSS_BUFFER_T_len);
		group[namep->GSS_BUFFER_T_len] = '\0';
		
		if ((gr = getgrnam(group)) != NULL)
			gid = gr->gr_gid;
	}
#ifdef DEBUG
	fprintf(stderr, "name_to_gid_svc res (%d)\n", gid);
#endif
	return (&gid);
}

gss_init_sec_context_res *
gss_init_sec_context_1_svc(gss_init_sec_context_arg *argp,
			   struct svc_req *rqstp)
{
	static gss_init_sec_context_res res;
	static gss_ctx_id_t ctx_id;
	static char buf[MAX_NETOBJ_SZ];

	gss_name_t service_name;
	gss_OID_desc mech_desc;
	gss_OID mech = &mech_desc;
	gss_buffer_desc *input_tokenp, input_token, output_token, name;
	OM_uint32 i;
	
	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "init_sec_context_svc arg (uid %d, flags %d,ctx %p:%d, "
		"name %.*s, mech %p:%d, token %p:%d)\n", argp->uid, argp->flags,
		argp->ctx_id.GSS_CTX_ID_T_val, argp->ctx_id.GSS_CTX_ID_T_len,
		argp->target_name.GSS_BUFFER_T_len,
		argp->target_name.GSS_BUFFER_T_val,
		argp->mech.GSS_OID_val, argp->mech.GSS_OID_len,
		argp->input_token.GSS_BUFFER_T_val,
		argp->input_token.GSS_BUFFER_T_len);
#endif	
	ctx_id = (argp->ctx_id.GSS_CTX_ID_T_len == 0 ? GSS_C_NO_CONTEXT :
		  *(gss_ctx_id_t *)argp->ctx_id.GSS_CTX_ID_T_val);
	
	name.value = (void *)argp->target_name.GSS_BUFFER_T_val;
	name.length = argp->target_name.GSS_BUFFER_T_len;

	memset(&res, 0, sizeof(res));

	if (res.maj_stat = gss_import_name(&res.min_stat, &name,
					   gss_nt_service_name,
					   &service_name) != GSS_S_COMPLETE) {
		res.ctx_id = argp->ctx_id;
		res.output_token.GSS_BUFFER_T_len = 0;
		res.ret_flags = 0;
		return (&res);
	}
	if (argp->mech.GSS_OID_len != 0) {
		mech->length = (OM_uint32)argp->mech.GSS_OID_len;
		mech->elements = (void *)argp->mech.GSS_OID_val;
	}
	else mech = GSS_C_NULL_OID;
	
	if (argp->input_token.GSS_BUFFER_T_len != 0) {
		input_token.value = (void *)argp->input_token.GSS_BUFFER_T_val;
		input_token.length = argp->input_token.GSS_BUFFER_T_len;
		input_tokenp = &input_token;
	}
	else input_tokenp = GSS_C_NO_BUFFER;

	/* XXX - haaaack. how else to do this? */
#ifdef HAVE_KRB5
  if((argp->flags & CA_RUN_AS_MACHINE) || (argp->uid == 0))
		snprintf(buf, sizeof(buf), "FILE:/tmp/krb5cc_machine");
	else
		snprintf(buf, sizeof(buf), "FILE:/tmp/krb5cc_%u", argp->uid);
	gss_krb5_ccache_name(&res.min_stat, buf, NULL);

#elif HAVE_HEIMDAL
	snprintf(buf, sizeof(buf),
		 "KRB5CCNAME=FILE:/tmp/krb5cc_%u", argp->uid);
	putenv(buf);
#endif

	res.maj_stat = gss_init_sec_context(&res.min_stat,
					    GSS_C_NO_CREDENTIAL,
					    &ctx_id,
					    service_name,
					    mech,
					    GSS_C_MUTUAL_FLAG,
					    0,
					    NULL,
					    input_tokenp,
					    NULL,
					    &output_token,
					    &res.ret_flags,
					    NULL);

	(void)gss_release_name(&i, &service_name);
	
	/* Save our results. */
	if (ctx_id != GSS_C_NO_CONTEXT) {
		res.ctx_id.GSS_CTX_ID_T_val = (void *)&ctx_id;
		res.ctx_id.GSS_CTX_ID_T_len = sizeof(ctx_id);
	}
	else res.ctx_id.GSS_CTX_ID_T_len = 0;
	
	if (output_token.length != 0) {
		memcpy(buf, output_token.value, output_token.length);
		res.output_token.GSS_BUFFER_T_val = buf;
		res.output_token.GSS_BUFFER_T_len = (u_int)output_token.length;
		(void)gss_release_buffer(&i, &output_token);
	}
	else res.output_token.GSS_BUFFER_T_len = 0;

#ifdef DEBUG
	log_status("init_sec_context", res.maj_stat, res.min_stat);
	fprintf(stderr, "init_sec_context_svc res (ctx %p:%d, maj_stat %d, "
		"min_stat %d, token %p:%d, flags %d)\n",
		res.ctx_id.GSS_CTX_ID_T_val, res.ctx_id.GSS_CTX_ID_T_len,
		res.maj_stat, res.min_stat,
		res.output_token.GSS_BUFFER_T_val,
		res.output_token.GSS_BUFFER_T_len,
		res.ret_flags);
#endif	
	return (&res);
}

gss_accept_sec_context_res *
gss_accept_sec_context_1_svc(gss_accept_sec_context_arg *argp,
			     struct svc_req *rqstp)
{
	static struct gss_accept_sec_context_res res;
	static gss_ctx_id_t ctx_id;
	static char buf1[MAX_NETOBJ_SZ], buf2[MAX_NETOBJ_SZ];

	gss_buffer_desc input_token, output_token, name;
	gss_name_t client_name;
	gss_OID mech;
	OM_uint32 i, j;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "accept_sec_context_svc arg (ctx_id %p:%d, "
		"token %p:%d)\n",
	       argp->ctx_id.GSS_CTX_ID_T_val, argp->ctx_id.GSS_CTX_ID_T_len,
	       argp->input_token.GSS_BUFFER_T_val, argp->input_token.GSS_BUFFER_T_len);
#endif
	ctx_id = (argp->ctx_id.GSS_CTX_ID_T_len == 0 ? GSS_C_NO_CONTEXT :
		  *(gss_ctx_id_t *)argp->ctx_id.GSS_CTX_ID_T_val);
	
	input_token.value = (void *)argp->input_token.GSS_BUFFER_T_val;
	input_token.length = argp->input_token.GSS_BUFFER_T_len;

	memset(&res, 0, sizeof(res));

	res.maj_stat = gss_accept_sec_context(&res.min_stat,
					      &ctx_id,
					      gssd_creds,
					      &input_token,
					      GSS_C_NO_CHANNEL_BINDINGS,
					      &client_name,
					      &mech,
					      &output_token,
					      &res.ret_flags,
					      NULL,
					      NULL);
	printf("GSSD: gss_accecp_sec_context returns maj:min (%d:%d)\n",res.maj_stat,res.min_stat);

	if((res.maj_stat != GSS_S_COMPLETE) && 
           (res.maj_stat != GSS_S_CONTINUE_NEEDED)) {

		printf("GSSD: gss_accecp_sec_context FAILED. maj:min (%d:%d)\n",res.maj_stat,res.min_stat);
		return(NULL);
	}

	/* Save our results. */
	if (ctx_id != GSS_C_NO_CONTEXT) {
		res.ctx_id.GSS_CTX_ID_T_val = (void *)&ctx_id;
		res.ctx_id.GSS_CTX_ID_T_len = sizeof(gss_union_ctx_id_desc);
	}
	else res.ctx_id.GSS_CTX_ID_T_len = 0;
	
	i = gss_display_name(&j, client_name, &name, &mech);

	if (i == GSS_S_COMPLETE) {
		memcpy(buf1, name.value, name.length);
		res.src_name.GSS_BUFFER_T_val = buf1;
		res.src_name.GSS_BUFFER_T_len = name.length;
		(void)gss_release_buffer(&i, &name);
	}
	else res.src_name.GSS_BUFFER_T_len = 0;
	
	if (output_token.length != 0) {
		memcpy(buf2, output_token.value, output_token.length);
		res.output_token.GSS_BUFFER_T_val = buf2;
		res.output_token.GSS_BUFFER_T_len = (u_int)output_token.length;
		(void)gss_release_buffer(&i, &output_token);
	}
	else res.output_token.GSS_BUFFER_T_len = 0;
	
#ifdef DEBUG
	log_status("accept_sec_context", res.maj_stat, res.min_stat);
	fprintf(stderr, "accept_sec_context_svc res (ctx %p:%d, maj_stat %d, "
	       "min_stat %d, name %.*s, token %p:%d, flags %d)\n",
	       res.ctx_id.GSS_CTX_ID_T_val, res.ctx_id.GSS_CTX_ID_T_len,
	       res.maj_stat, res.min_stat,
	       res.src_name.GSS_BUFFER_T_len, res.src_name.GSS_BUFFER_T_val,
	       res.output_token.GSS_BUFFER_T_val,
	       res.output_token.GSS_BUFFER_T_len,
	       res.ret_flags);
#endif
	return (&res);
}

gss_import_sec_context_res *
gss_import_sec_context_1_svc(gss_import_sec_context_arg *argp,
			     struct svc_req *rqstp)
{
	static struct gss_import_sec_context_res res;
	static gss_ctx_id_t ctx_id;
	gss_buffer_desc  output_token;
	static char buf1[MAX_NETOBJ_SZ];
	OM_uint32  min = 0,maj = 0;
	OM_uint32  mins = 0,resu = 0;

	if (!hosts_access(rqstp))
		return (NULL);
#ifdef DEBUG
	fprintf(stderr, "import_sec_context_svc arg (ctx %p:%d\n", 
		argp->ctx_id.GSS_CTX_ID_T_val, argp->ctx_id.GSS_CTX_ID_T_len);
#endif	

	output_token.length = 0;
	output_token.value = NULL;

	ctx_id = (argp->ctx_id.GSS_CTX_ID_T_len == 0 ? GSS_C_NO_CONTEXT :
		  *(gss_ctx_id_t *)argp->ctx_id.GSS_CTX_ID_T_val);

	memset(&res, 0, sizeof(res));

	/* This call switches on the mechanism oid containted in the 
	 * ctx_it (a union type - see mechglue code)
	 */
	res.maj_stat = gss_export_sec_context(&res.min_stat,
					    &ctx_id, 
	            &output_token);

#ifdef DEBUG
	print_hexl((u_int *)output_token.value,8,0);
#endif /* DEBUG */

resu = gss_import_sec_context(&mins, &output_token, &ctx_id);

  /* Note: the krb5 context is returned in an 'externalized form'
	 * packed in an asn1-like fashion into the output_token.
	 * so krb5 userland code has been added to the kernel rpc 
	 * to transform the output_token back into an interalized
	 * form, e.g. a krb5_gss_ctx_rec_id structure.  
	 */

	if (output_token.length != 0) {
		memcpy(buf1, output_token.value, output_token.length);
		res.ctx_token.GSS_BUFFER_T_val = buf1;
		res.ctx_token.GSS_BUFFER_T_len = (u_int)output_token.length;
		maj = gss_release_buffer(&min, &output_token);
	}
	else res.ctx_token.GSS_BUFFER_T_len = 0;
#ifdef DEBUG
	log_status("import_sec_context: export_sec_context call", res.maj_stat, res.min_stat);
	fprintf(stderr, "import_sec_context_svc res (maj_stat %d, "
	       "min_stat %d ctx_token %p:%d)\n",
	       res.maj_stat, res.min_stat,
	       res.ctx_token.GSS_BUFFER_T_val,
	       res.ctx_token.GSS_BUFFER_T_len);
#endif
	return (&res);
}
