/*
 * util.c 
 *
 *  Copyright (c) 2000 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Jake Moilanen <jmoilane@umich.edu>
 *  Kendrick Smith <kmsmith@umich.edu>
 *  
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include "exportfs4.h"
#include "pseudodev.h"

struct clist {
  int ncomponents;
  char *components[MAX_COMPONENTS];
};

struct clist *
parse_path(char *path) {
  struct clist * new;
  char *p, *workpath;
  int len;
  int i;

  if (!(new = calloc(1,sizeof(*new)))) {
    fprintf(stderr,"exportfs4: out of memory in parse_path");
    destroy_server();
  }
  if (!(workpath = strdup(path))) {
    fprintf(stderr,"exportfs4: out of memory in parse_path");
    destroy_server();
  }
  len = strlen(workpath);
  p=workpath;
  while ( (p = strchr(p,'/')) ) {
    *p='\0';
    p++;
  }
  i=0;
  p=workpath;
  while (i<len) {
    if (*(p+i) == '\0') {
      i++;
    } else {
      if (++new->ncomponents>MAX_COMPONENTS) {
	fprintf(stderr,"exportfs4: pathname %s has too many components\n",path);
	destroy_server();
      }
      if (!(new->components[new->ncomponents-1] = strdup(p+i))) {
	fprintf(stderr,"exportfs4: out of memory in parse_path");
	destroy_server();
      }
      i+= strlen(new->components[new->ncomponents-1])+1;
    }
  }
  free(workpath);
  return(new);
}

void
free_clist(struct clist *cl) {
  int i;

  for (i=0; i<cl->ncomponents; i++) {
    free(cl->components[i]);
  }
  free(cl);
}

void
add_pseudo(int ncomponents, char ** components)
{
  int result;
  struct nfsctl_arg arg;
  union nfsctl_res res;

  memset(&arg, 0, sizeof(arg));
  arg.ca_version = NFSCTL_VERSION;
  arg.ca_addpseudo.namlen = strlen(components[ncomponents - 1]);
  arg.ca_addpseudo.name = components[ncomponents - 1];

  result = get_pseudodev(ncomponents - 1, components,
      &arg.ca_addpseudo.parent_major,
      &arg.ca_addpseudo.parent_minor, &arg.ca_addpseudo.parent_ino);
  if (result) {
    fprintf(stderr, "couldn't access pseudofs: %s\n", strerror(result));
    destroy_server();
  }

  result = get_pseudodev(ncomponents, components, &arg.ca_addpseudo.major,
      &arg.ca_addpseudo.minor, &arg.ca_addpseudo.ino);
  if (result) {
    fprintf(stderr, "couldn't access pseudofs: %s\n", strerror(result));
    destroy_server();
  }

  result = nfsservctl(NFS4CTL_ADDPSEUDO, &arg, &res);
  if (result){
    perror("addpseudo");
    destroy_server();
  }
}

void
start_server(int grace_period)
{
  struct nfsctl_arg arg;
  union nfsctl_res res;

  memset(&arg, 0, sizeof(arg));

  arg.ca_version = NFSCTL_VERSION;
  arg.ca_start.nthreads = NFSD4_NTHREADS;
  arg.ca_start.grace_period = grace_period;

  if (nfsservctl(NFS4CTL_START, &arg, &res)) {
    perror("export");
    destroy_server();
  }
}

int
unexport_all(void)
{
  struct nfsctl_arg arg;
  union nfsctl_res res;

  memset(&arg, 0, sizeof(arg));
  arg.ca_version = NFSCTL_VERSION;
  if (nfsservctl(NFS4CTL_REMOVEPSEUDO, &arg, &res))
    return errno;
  return 0;
}

void
destroy_server(void)
{
  struct nfsctl_arg arg;
  union nfsctl_res res;

  unexport_all();

  memset(&arg, 0, sizeof(arg));
  arg.ca_version = NFSCTL_VERSION;
  arg.ca_start.nthreads = 0;
  nfsservctl(NFS4CTL_START, &arg, &res);
  exit(1);
}

int
unexport(char *filename)
{
  struct nfsctl_arg arg;
  union nfsctl_res res;

  memset(&arg, 0, sizeof(arg));
  arg.ca_version = NFSCTL_VERSION;
  arg.ca_unexport.namelen = strlen(filename);
  arg.ca_unexport.name = filename;
  if (nfsservctl(NFS4CTL_UNEXPORT, &arg, &res))
    return errno;
  return 0;
}

static struct nfsctl_arg new_export_arg;
char * ppath = NULL;

void
start_new_export(void)
{
  if (new_export_arg.ca_export.path) {
    free(new_export_arg.ca_export.path);
    new_export_arg.ca_export.pathlen = 0;
    new_export_arg.ca_export.path = NULL;
  }
  if (ppath) {
    free(ppath);
    ppath = NULL;
  }
  memset(&new_export_arg, 0, sizeof(new_export_arg));
  new_export_arg.ca_export.read_only = 1; /* default to read-only */
}

void
set_pseudo_path(char * path)
{
  if (!(ppath = strdup(path))) {
    fprintf(stderr, "exportfs4: out of memory in set_pseudo_path\n");
    destroy_server();
  }
}

void
set_export_path(char * path)
{
  new_export_arg.ca_export.pathlen = strlen(path);
  if (!(new_export_arg.ca_export.path = strdup(path))) {
    fprintf(stderr, "exportfs4: out of memory in set_export_path\n");
    destroy_server();
  }
}

void
do_add_export(struct clist * path)
{
  struct stat s;
  int result;
  union nfsctl_res res;

  new_export_arg.ca_version = NFSCTL_VERSION;
  if (stat(new_export_arg.ca_export.path, &s)){
    perror("stat (in do_add_export)");
    exit(1);
  }

  result = get_pseudodev(path->ncomponents, path->components,
      &new_export_arg.ca_export.major,
      &new_export_arg.ca_export.minor, &new_export_arg.ca_export.ino);
  if (result) {
    fprintf(stderr, "couldn't access pseudofs: %s\n", strerror(result));
    destroy_server();
  }
  result = nfsservctl(NFS4CTL_EXPORT, &new_export_arg, &res);
  if (result){
    perror(new_export_arg.ca_export.path);
    destroy_server();
  }
}

void
lowerstr(char * str) {
  while (*str) {
    tolower(*str);
    str++;
  }
}

void
add_sec(char * sec) {
  lowerstr(sec);

  if (!strcmp(sec,"none")) {
    new_export_arg.ca_export.sec_none = 1;
  } else if (!strcmp(sec,"sys")) {
    new_export_arg.ca_export.sec_sys = 1;
  } else if (!strcmp(sec,"dh")) {
    new_export_arg.ca_export.sec_dh = 1;
  } else if (!strcmp(sec,"krb5")) {
    new_export_arg.ca_export.sec_krb5 = 1;
  } else if (!strcmp(sec,"krb5i")) {
    new_export_arg.ca_export.sec_krb5i = 1;
  } else if (!strcmp(sec,"krb5p")) {
    new_export_arg.ca_export.sec_krb5p = 1;
  } else if (!strcmp(sec,"spkm3")) {
    new_export_arg.ca_export.sec_spkm3 = 1;
  } else if (!strcmp(sec,"spkm3i")) {
    new_export_arg.ca_export.sec_spkm3i = 1;
  } else if (!strcmp(sec,"spkm3p")) {
    new_export_arg.ca_export.sec_spkm3p = 1;
  } else if (!strcmp(sec,"lkey")) {
    new_export_arg.ca_export.sec_lkey = 1;
  } else if (!strcmp(sec,"lkeyi")) {
    new_export_arg.ca_export.sec_lkeyi = 1;
  } else if (!strcmp(sec,"lkeyp")) {
    new_export_arg.ca_export.sec_lkeyp = 1;
  } else {
    fprintf(stderr,
	    "exportfs4: unknown security option \"%s\" in add_sec\n",sec);
    destroy_server();
  }
}

void
set_ro(int ro) {
  if (ro<0 || ro>1) {
    fprintf(stderr,"exportfs4: bad argument in set_ro\n");
    destroy_server();
  } else {
    new_export_arg.ca_export.read_only = ro;
  }
}

void add_export() {
  struct clist * pclist;
  int i;

  if (new_export_arg.ca_export.path == NULL) {
    fprintf(stderr,"exportfs4: missing export path in add_export\n");
    destroy_server();
  }
  if (ppath==NULL) {
    /* use default: */
    if (!(ppath = strdup(new_export_arg.ca_export.path))) {
      fprintf(stderr,"exportfs4: out of memory in add_export\n");
    }
  } 
  fprintf(stderr,"exporting local path \"%s\" as \"%s\"\n",
		  new_export_arg.ca_export.path,ppath);
  pclist = parse_path(ppath);
  for (i=1; i<=pclist->ncomponents; i++) {
    add_pseudo(i,pclist->components);
  }
  do_add_export(pclist);
  free_clist(pclist);
}
