/*
 *  pseudodev.c
 *
 *  Copyright (c) 2001 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Kendrick Smith <kmsmith@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <assert.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "pseudodev.h"

#define FILENAME  "/etc/nfs4-pseudofs"
#define BUFSIZE   8192

static int	fd = -1;
static int	parsed = 0;

static struct pseudodev {
	int		ncomponents;
	char **		components;
	__u16		major;
	__u16		minor;
	ino_t		ino;
	struct pseudodev *next;
} *devlist = NULL;

static char	file_buf[BUFSIZE];
static int	bufpos;

static int allocate_dev(unsigned int ncomponents, char **components,
			__u16 major, __u16 minor, ino_t ino);

static int
readline(char *buf)
{
	char *p;
	int nbytes;
	int start = 0;
	
	assert(fd >= 0);
	
	for (;;) {
		p = memchr(&file_buf[start], 0, bufpos - start);
		if (p)
			return EINVAL;
		
		p = memchr(&file_buf[start], '\n', bufpos - start);
		if (p) {
			nbytes = p - file_buf;
			assert(nbytes >= 0 && nbytes < bufpos);
			memcpy(buf, file_buf, nbytes);
			buf[nbytes] = 0;
			bufpos -= nbytes + 1;
			memmove(file_buf, &file_buf[nbytes + 1], bufpos);
			return 0;
		}

		if (bufpos == BUFSIZE)
			return EINVAL;
		
		nbytes = read(fd, &file_buf[bufpos], BUFSIZE - bufpos);
		if (nbytes < 0)
			return errno;
		if (!nbytes) {
			if (bufpos == 0)
				return -1;
			memcpy(buf, file_buf, bufpos);
			buf[bufpos] = 0;
			bufpos = 0;
			return 0;
		}
		bufpos += nbytes;
	}
}

static void
skip_whitespace(char **bufp)
{
	char *buf = *bufp;
	
	while (*buf && isspace(*buf))
		buf++;
	*bufp = buf;
}

static int
get_component(char **bufp)
{
	char *save = *bufp;
	char *buf = *bufp;
	int err;
	
	for (;;) {
		if (!*buf)
			return EINVAL;
		if (*buf == '/') {
			err = EAGAIN;
			break;
		}
		if (isspace(*buf)) {
			err = 0;
			break;
		}
		if (!isprint(*buf))
			return EINVAL;
		buf++;
	}

	*buf++ = 0;
	if (!strcmp(save, "") || !strcmp(save, ".") || !strcmp(save, ".."))
		return EINVAL;
	*bufp = buf;
	return err;
}

static int
get_long(char **bufp, unsigned long *longp)
{
	char *save;
	char *buf = *bufp;
	
	skip_whitespace(bufp);
	save = buf;
	while (isdigit(*buf))
		buf++;
	if (buf == save)
		return EINVAL;
	if (*buf) {
		if (!isspace(*buf))
			return EINVAL;
		*buf++ = 0;
	}
	*bufp = buf;
	*longp = strtoul(save, NULL, 10);
	return 0;
}

static int
parseline(char *buf)
{
	int slashcount, ncomponents;
	char *p;
	char **components;
	int err;
	unsigned long major, minor, ino;
	
	skip_whitespace(&buf);
	if (!*buf || *buf == '#')
		return 0;
	if (*buf++ != '/')
		return EINVAL;

	slashcount = 1;
	for (p = buf; *p; p++)
		if (*p == '/') slashcount++;
	if (!(components = malloc(slashcount * sizeof(char *))))
		return ENOMEM;

	ncomponents = 0;
	do {
		assert(ncomponents < slashcount);
		components[ncomponents++] = buf;
		err = get_component(&buf);
	} while (err == EAGAIN);

	if (!err)
		err = get_long(&buf, &major);
	if (!err)
		err = get_long(&buf, &minor);
	if (!err)
		err = get_long(&buf, &ino);
	if (!err)
		err = allocate_dev(ncomponents, components, major, minor, ino);
	
	free(components);
	return err;
}

static struct pseudodev *
dev_by_name(unsigned int ncomponents, char **components)
{
	struct pseudodev *p;
	unsigned int i;

	for (p = devlist; p; p = p->next) {
		if (p->ncomponents != ncomponents)
			continue;
		for (i = 0; i < ncomponents; i++)
			if (strcmp(components[i], p->components[i]))
				break;   /* out of _inner_ loop */
		if (i == ncomponents)
			return p;
	}
	return NULL;
}

static struct pseudodev *
dev_by_num(__u16 major, __u16 minor, ino_t ino)
{
	struct pseudodev *p;
	
	for (p = devlist; p; p = p->next) {
		if (p->major == major && p->minor == minor && p->ino == ino)
			return p;
	}
	return NULL;
}

static int
allocate_dev(unsigned int ncomponents, char **components,
	     __u16 major, __u16 minor, ino_t ino)
{
	struct pseudodev *p;
	unsigned int i;

	if (!ncomponents)
		return EINVAL;
	if (!major && !minor && !ino)
		return EINVAL;
	if (dev_by_name(ncomponents, components) || dev_by_num(major, minor, ino))
		return EINVAL;
	
	p = malloc(sizeof(*p));
	if (!p)
		return ENOMEM;
	p->components = malloc(ncomponents * sizeof(char *));
	if (!p->components) {
		free(p);
		return ENOMEM;
	}
	
	for (i = 0; i < ncomponents; i++) {
		p->components[i] = strdup(components[i]);
		if (p->components[i])
			continue;
		while (i--)
			free(p->components[i]);
		free(p->components);
		free(p);
		return ENOMEM;
	}
	p->ncomponents = ncomponents;
	p->major = major;
	p->minor = minor;
	p->ino = ino;
	p->next = devlist;
	devlist = p;
	return 0;
}

int
open_pseudodev(void)
{
	struct flock lock;
	
	assert(fd < 0);

	if (geteuid())
		return EPERM;
	
	fd = open(FILENAME, O_RDWR | O_CREAT | O_APPEND | O_SYNC, 0644);
	if (fd < 0)
		return errno;
	
	lock.l_type = F_WRLCK;
	lock.l_start = 0;
	lock.l_whence = SEEK_SET;
	lock.l_len = 0;
	
	if (fcntl(fd, F_SETLKW, &lock) < 0) {
		close(fd);
		fd = -1;
		return errno;
	}
	
	parsed = 0;
	return 0;
}

int
get_pseudodev(unsigned int ncomponents, char **components,
	      __u16 *majorp, __u16 *minorp, ino_t *inop)
{
	static ino_t trial_ino = 1;
	struct pseudodev *p;
	unsigned int i, j;
	int len, totlen;
	char buf[BUFSIZE + 1];
	char *cp;
	int err;

	assert(fd >= 0);

	/*
	 * Validate arguments.  We allow any printable character in
	 * a filename component except for whitespace and '/'.  The
	 * filenames "", ".", and ".." are disallowed.
	 */
	totlen = 0;
	for (i = 0; i < ncomponents; i++) {
		if (!strcmp(components[i], ""))
			return EINVAL;
		if (!strcmp(components[i], "."))
			return EINVAL;
		if (!strcmp(components[i], ".."))
			return EINVAL;
		len = strlen(components[i]);
		for (j = 0; j < len; j++)
			if (!isprint(components[i][j]) || isspace(components[i][j]))
				return EINVAL;
		totlen += (len + 1);
	}
	if (totlen > BUFSIZE - 100)
		return ENAMETOOLONG;

	/*
	 * The pseudofs root always has major and minor device numbers 0,
	 * and inode number 0.
	 */
	if (!ncomponents) {
		*majorp = 0;
		*minorp = 0;
		*inop = 0;
		return 0;
	}

	/*
	 * If /etc/pseudofs hasn't been parsed, parse it now.
	 */
	if (!parsed) {
		bufpos = 0;
		for (;;) {
			err = readline(buf);
			if (err == -1)
				break;
			if (err)
				return err;

			err = parseline(buf);
			if (err)
				return err;
		}
		parsed = 1;
	}

	/*
	 * Try to lookup the pathname among the existing entries.
	 */
	p = dev_by_name(ncomponents, components);
	if (p) {
		*majorp = p->major;
		*minorp = p->minor;
		*inop = p->ino;
		return 0;
	}

	/*
	 * The pathname doesn't exist, so allocate a new pseudodev entry.
	 */
	while (dev_by_num(0, 0, trial_ino))
		trial_ino++;
	err = allocate_dev(ncomponents, components, 0, 0, trial_ino);
	if (err)
		return err;

	/*
	 * Write the appropriate line into /etc/pseudofs, and return success.
	 */
	cp = buf;
	totlen = BUFSIZE - 100;
	for (i = 0; i < ncomponents; i++) {
		len = snprintf(cp, totlen, "/%s", components[i]);
		assert(len <= totlen);
		cp += len;
		totlen -= len;
	}
	len = snprintf(cp, 100, " %u %u %lu\n", 0, 0, trial_ino);
	cp += len;
	
	totlen = cp - buf;
	cp = buf;
	do {
		len = write(fd, cp, totlen);
		assert(len <= totlen);
		if (len < 0)
			return errno;
		if (len == 0)
			return EIO;        /* totally confused */
		totlen -= len;
	} while (totlen);

	*majorp = 0;
	*minorp = 0;
	*inop = trial_ino;
#if 0
	trial_ino++;
#endif
	return 0;
}

int
close_pseudodev(void)
{
	struct pseudodev *p;
	int i;
	
	assert(fd >= 0);
	if (close(fd) < 0)
		return errno;

	while (devlist) {
		p = devlist;
		for (i = 0; i < p->ncomponents; i++)
			free(p->components[i]);
		free(p->components);
		devlist = p->next;
		free(p);
	}
	
	fd = -1;
	return 0;
}
