%{
/*
 * exportfs4.y
 *
 *  Copyright (c) 2000 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Jake Moilanen <jmoilane@umich.edu>
 *  Kendrick Smith <kmsmith@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "exportfs4.h"
#include "pseudodev.h"

#define YYSTYPE char *
#define YYDEBUG 1
int yydebug = 0;
char *export_file_name = "/etc/exports";

void
freestr(char * str) {
  free(str);
  str=NULL;
}

static int grace_period;
%}

%token PATH PSEUDO ELEMENT END_PATH NL RO RW SEC SECTYPE

%%

finish: tree                           { start_server(grace_period); }

tree:   /* empty */
| entry
| entry NL tree
| NL tree
;

entry:
PATH	                               { set_export_path($1);
                                         add_export();
                                         freestr($1); 
					 start_new_export();}
| PATH options                         { set_export_path($1);
                                         add_export();
                                         freestr($1);
					 start_new_export();}
;

options:
'(' option_list ')'
;

option_list: option
| option ',' option_list
;

option: pseudo
| f_perm
| f_sec
;

pseudo: PSEUDO '=' PATH                { set_pseudo_path($3);
                                         freestr($3); }
;

f_perm: RO                             { set_ro(1); }
| RW                                   { set_ro(0); }
;

f_sec: SEC '=' f_seclist               { add_sec($3); freestr($3); }
;

f_seclist: SECTYPE
|    SECTYPE ':' f_seclist
;

%%

extern FILE *yyin;

void
usage()
{
  fprintf(stderr, "usage: exportfs4 [-g|-s|-u filename]\n");
  exit(1);
}

int
main (int argc, char **argv)
{
  struct stat s;
  char c;
  int err;
  extern char * optarg;
  extern int optind;

  /* init */
  grace_period = 0;

  err = open_pseudodev();
  if (err) {
    fprintf(stderr, "couldn't open /etc/nfs4-pseudofs: %s\n", strerror(err));
    exit(1);
  }

  /* command line checking */
  if (argc > 1){
    while ((c = getopt(argc, argv, "gsu:")) != EOF) {
      switch (c) {
      case 'u':
	err = unexport(optarg);
	if (err) {
	  fprintf(stderr, "unexport: %s\n", strerror(err));
	  exit(1);
	}
	exit(0);
      case 's':
	destroy_server();
	break;
      case 'g': 
	grace_period = 1;
	break;
      default:
        fprintf(stderr, "exportfs4: invalid option -- %c\n", c);
	usage();
      break;
      } // switch
    } // while
  } // if

  if (optind + 1 < argc) {
    usage();
  }

  if (optind + 1 == argc) {
    export_file_name = argv[optind];
  }
  
  if (stat(export_file_name, &s)){
    perror("stat: File probably doesn't exist\n");
    exit(1);
  }
  if (s.st_uid != 0){
    perror("Export file not owned by root\n");
    exit(1);
  }

  yyin = fopen(export_file_name, "r");
  if (!yyin) {
    fprintf(stderr,"could not open %s\n", export_file_name);
    exit(1);
  }

  err = unexport_all();
  if (err) {
    perror("when attempting to remove old exports");
    exit(1);
  }
  
  start_new_export();
  while(!feof(yyin)) {  
    yyparse();
  }

  err = close_pseudodev();
  if (err) {
    fprintf(stderr, "close_pseudodev failed\n");
    exit(1);
  }
  exit(0);
}

