/* $FreeBSD$ */
/* $Id: nfs4_vnops.c,v 1.5 2004/05/21 18:47:53 marius Exp $ */

/*
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Rick Macklem at The University of Guelph.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)nfs_vnops.c	8.16 (Berkeley) 5/27/95
 */

#include <sys/cdefs.h>

#ifdef __FreeBSD__
__FBSDID("$FreeBSD$");
#endif

/*
 * vnode op calls for Sun NFS version 2 and 3
 */

#ifdef __FreeBSD__
#include "opt_inet.h"
#endif

#include <sys/param.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/resourcevar.h>
#include <sys/proc.h>
#include <sys/mount.h>
#ifdef __FreeBSD__
#include <sys/bio.h>
#endif
#include <sys/buf.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/namei.h>
#include <sys/socket.h>
#include <sys/vnode.h>
#include <sys/dirent.h>
#include <sys/fcntl.h>
#include <sys/lockf.h>
#include <sys/stat.h>
#include <sys/sysctl.h>

#ifdef __APPLE__
#include <sys/kdebug.h>
#endif

#ifdef __FreeBSD__
#include <vm/vm.h>
#include <vm/vm_extern.h>

#include <fs/fifofs/fifo.h>
#endif /* __FreeBSD__ */

#ifdef __APPLE__
#include <vfs/vfs_support.h>
#endif

#include <nfs4client/nfs4_glue.h>

#include <rpcx/rpcclnt.h>
#include <nfsx/rpcv2.h>
#include <nfsx/nfsproto.h>
#include <nfsxclient/nfs.h>
#include <nfs4client/nfs4.h>
#include <nfsxclient/nfsnode.h>
#include <nfsxclient/nfsmount.h>
/* #include <nfsx/nfs_lock.h>	/\* XXX *\/ */
#include <nfsx/xdr_subs.h>
#include <nfsxclient/nfsm_subs.h>
#include <nfsx/nfs_socket.h>

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_var.h>

/* NFSv4 */
#include <nfs4client/nfs4m_subs.h>
#include <nfs4client/nfs4_vn.h>

#include <nfs4client/nfs4m_glue.h>

#include <nfsxclient/nfs_vnops.h>

#ifdef __APPLE__		/* XXX - OpenBSD */
/* XXXMARIUS: name collision */
extern  struct nfsstats	nfsstats_nfsx;
#define  nfsstats nfsstats_nfsx


#define PDIRUNLOCK 0

#define FSDBG(A, B, C, D, E) \
        KERNEL_DEBUG((FSDBG_CODE(DBG_FSRW, (A))) | DBG_FUNC_NONE, \
                (int)(B), (int)(C), (int)(D), (int)(E), 0)
#define FSDBG_TOP(A, B, C, D, E) \
        KERNEL_DEBUG((FSDBG_CODE(DBG_FSRW, (A))) | DBG_FUNC_START, \
                (int)(B), (int)(C), (int)(D), (int)(E), 0)
#define FSDBG_BOT(A, B, C, D, E) \
        KERNEL_DEBUG((FSDBG_CODE(DBG_FSRW, (A))) | DBG_FUNC_END, \
                (int)(B), (int)(C), (int)(D), (int)(E), 0)

#endif /* __APPLE__ */

/* Defs */
#define	TRUE	1
#define	FALSE	0

/*
 * Ifdef for FreeBSD-current merged buffer cache. It is unfortunate that these
 * calls are not in getblk() and brelse() so that they would not be necessary
 * here.
 */
#ifndef B_VMIO
#define vfs_busy_pages(bp, f)
#endif

static	int	nfs4_create(struct vop_create_args *);
static	int	nfs4_open(struct vop_open_args *);
static	int	nfs4_access(struct vop_access_args *);
static	int	nfs4_getattr __P((struct vop_getattr_args *));
static	int	nfs4_link(struct vop_link_args *);
static	int	nfs4_mkdir(struct vop_mkdir_args *);
static	int	nfs4_rmdir(struct vop_rmdir_args *);
static	int	nfs4_symlink(struct vop_symlink_args *);

/* XXX??? */
int nfs4_removeit(struct sillyrename *);

/* XXX these belong in nfsx.h.  and they need x appendages */
/* XXX nfsx_vnop.[ch] */

int nfs_setattr(struct vop_setattr_args *ap);
int nfs_close(struct vop_close_args *);
int nfs_remove(struct vop_remove_args *);
int nfs_rename(struct vop_rename_args *);
int nfs_advlock(struct vop_advlock_args *);
int nfs_fsync(struct vop_fsync_args *);
int nfs_print(struct vop_print_args *);
int nfs_sillyrename(struct vnode *, struct vnode *, struct componentname *);
int nfs_readlink(struct vop_readlink_args *);
int nfs_read(struct vop_read_args *);
int nfs_mknod(struct vop_mknod_args *ap);

int nfsspec_access(struct vop_access_args *);
int nfsspec_read(struct vop_read_args *);
int nfsspec_write(struct vop_write_args *);
int nfsspec_close(struct vop_close_args *);

int nfsfifo_read(struct vop_read_args *);
int nfsfifo_write(struct vop_write_args *);
int nfsfifo_close(struct vop_close_args *);

#ifndef __FreeBSD__
extern vop_t **spec_vnodeop_p;
extern vop_t **fifo_vnodeop_p;
#endif 

#ifdef __FreeBSD__
#define DEFAULTOP vop_defaultop
#else
#define DEFAULTOP vn_default_error
#endif

/*
 * Global vfs data structures for nfs
 */
vop_t **nfs4_vnodeop_p;
static struct vnodeopv_entry_desc nfs4_vnodeop_entries[] = {
	{ &vop_default_desc,		(vop_t *) DEFAULTOP },
	{ &vop_access_desc,		(vop_t *) nfs4_access },
	{ &vop_advlock_desc,		(vop_t *) DEFAULTOP },
	{ &vop_close_desc,		(vop_t *) nfsx_close },
	{ &vop_create_desc,		(vop_t *) nfs4_create },
	{ &vop_fsync_desc,		(vop_t *) nfsx_fsync },
	{ &vop_getattr_desc,		(vop_t *) nfs4_getattr },
	{ &vop_inactive_desc,		(vop_t *) nfsx_inactive },
	{ &vop_lease_desc,		(vop_t *) DEFAULTOP },
	{ &vop_link_desc,		(vop_t *) nfs4_link },
	{ &vop_lookup_desc,		(vop_t *) nfsx_lookup },
	{ &vop_mkdir_desc,		(vop_t *) nfs4_mkdir },
	{ &vop_mknod_desc,		(vop_t *) nfsx_mknod },
	{ &vop_open_desc,		(vop_t *) nfs4_open },
	{ &vop_print_desc,		(vop_t *) nfsx_print },
	{ &vop_read_desc,		(vop_t *) nfsx_read },
	{ &vop_readdir_desc,		(vop_t *) nfsx_readdir },
	{ &vop_readlink_desc,		(vop_t *) nfsx_readlink },
	{ &vop_reclaim_desc,		(vop_t *) nfsx_reclaim },
	{ &vop_remove_desc,		(vop_t *) nfsx_remove },
	{ &vop_rename_desc,		(vop_t *) nfsx_rename },
	{ &vop_rmdir_desc,		(vop_t *) nfs4_rmdir },
	{ &vop_setattr_desc,		(vop_t *) nfsx_setattr },
#ifdef __APPLE__
	{ &vop_strategy_desc,		(vop_t *) err_strategy },
#else
	{ &vop_strategy_desc,		(vop_t *) nfsx_strategy },
#endif /* __APPLE__ */
	{ &vop_symlink_desc,		(vop_t *) nfs4_symlink },
	{ &vop_write_desc,		(vop_t *) nfsx_write },
#ifdef __FreeBSD__
	{ &vop_getpages_desc,		(vop_t *) nfsx_getpages },
	{ &vop_putpages_desc,		(vop_t *) nfsx_putpages },
#endif /* __FreeBSD__ */
#ifndef __FreeBSD__
	{ &vop_lock_desc,               (vop_t *) nfsx_lock },
	{ &vop_unlock_desc,             (vop_t *) nfsx_unlock },
	{ &vop_islocked_desc,           (vop_t *) nfsx_islocked },
#endif /* ~__FreeBSD__ */
#ifdef __APPLE__
	{ &vop_bmap_desc,               (vop_t *) nfsx_bmap },
	{ &vop_blkatoff_desc,           (vop_t *) nfsx_blkatoff },
	{ &vop_valloc_desc,             (vop_t *) nfsx_valloc },
	{ &vop_reallocblks_desc,        (vop_t *) nfsx_reallocblks },
	{ &vop_vfree_desc,              (vop_t *) nfsx_vfree },
	{ &vop_truncate_desc,           (vop_t *) nfsx_truncate },
	{ &vop_update_desc,             (vop_t *) nfsx_update },
	{ &vop_bwrite_desc,             (vop_t *) err_bwrite },
	{ &vop_pagein_desc,             (vop_t *) nfsx_pagein },
	{ &vop_pageout_desc,            (vop_t *) nfsx_pageout },
	{ &vop_copyfile_desc,           (vop_t *) err_copyfile },
	{ &vop_copyfile_desc,           (vop_t *) err_copyfile },
	{ &vop_blktooff_desc,           (vop_t *) nfsx_blktooff },
	{ &vop_offtoblk_desc,           (vop_t *) nfsx_offtoblk },
	{ &vop_cmap_desc,               (vop_t *) nfsx_cmap },
        { &vop_abortop_desc,            (vop_t *) nop_abortop },
#endif /* __APPLE__ */ 
	{ NULL, NULL }
};
struct vnodeopv_desc nfs4_vnodeop_opv_desc =
	{ &nfs4_vnodeop_p, nfs4_vnodeop_entries };
#ifdef __FreeBSD__
VNODEOP_SET(nfs4_vnodeop_opv_desc);
#endif

/*
 * Special device vnode ops
 */
vop_t **spec_nfs4nodeop_p;
static struct vnodeopv_entry_desc nfs4_specop_entries[] = {
#ifdef __FreeBSD__
	{ &vop_default_desc,		(vop_t *) spec_vnoperate },
#else
	{ &vop_default_desc,		(vop_t *) vn_default_error },
#endif /* __FreeBSD__ */
	{ &vop_access_desc,		(vop_t *) nfsspec_access },
	{ &vop_close_desc,		(vop_t *) nfsspec_close },
	{ &vop_fsync_desc,		(vop_t *) nfsx_fsync },
	{ &vop_getattr_desc,		(vop_t *) nfs4_getattr },
	{ &vop_inactive_desc,		(vop_t *) nfsx_inactive },
	{ &vop_print_desc,		(vop_t *) nfsx_print },
	{ &vop_read_desc,		(vop_t *) nfsspec_read },
	{ &vop_reclaim_desc,		(vop_t *) nfsx_reclaim },
	{ &vop_setattr_desc,		(vop_t *) nfsx_setattr },
	{ &vop_write_desc,		(vop_t *) nfsspec_write },
#ifdef __APPLE__
	{ &vop_strategy_desc,		(vop_t *) err_strategy },
#endif
	{ NULL, NULL }
};
struct vnodeopv_desc spec_nfs4nodeop_opv_desc =
	{ &spec_nfs4nodeop_p, nfs4_specop_entries };
#ifdef __FreeBSD__
VNODEOP_SET(spec_nfs4nodeop_opv_desc);
#endif

vop_t **fifo_nfs4nodeop_p;
static struct vnodeopv_entry_desc nfs4_fifoop_entries[] = {
#ifdef __FreeBSD__
	{ &vop_default_desc,		(vop_t *) fifo_vnoperate },
#else
	{ &vop_default_desc,		(vop_t *) vn_default_error },	
#endif /* __FreeBSD__ */
	{ &vop_access_desc,		(vop_t *) nfsspec_access },
	{ &vop_close_desc,		(vop_t *) nfsfifo_close },
	{ &vop_fsync_desc,		(vop_t *) nfsx_fsync },
	{ &vop_getattr_desc,		(vop_t *) nfs4_getattr },
	{ &vop_inactive_desc,		(vop_t *) nfsx_inactive },
	{ &vop_print_desc,		(vop_t *) nfsx_print },
	{ &vop_read_desc,		(vop_t *) nfsfifo_read },
	{ &vop_reclaim_desc,		(vop_t *) nfsx_reclaim },
	{ &vop_setattr_desc,		(vop_t *) nfsx_setattr },
	{ &vop_write_desc,		(vop_t *) nfsfifo_write },
	{ NULL, NULL }
};
struct vnodeopv_desc fifo_nfs4nodeop_opv_desc =
	{ &fifo_nfs4nodeop_p, nfs4_fifoop_entries };
#ifdef __FreeBSD__
VNODEOP_SET(fifo_nfs4nodeop_opv_desc);
#endif

static int      nfs4_openrpc(struct vnode *, struct vnode **,
                            struct componentname *, int, struct vattr *);
static int	nfs4_open_confirm(struct vnode *vp, struct nfs4_compound *cpp,
				 struct nfs4_oparg_open *openap,
				 struct nfs4_oparg_getfh *gfh,
				 struct ucred *cred, cthread_t *td);
static int      nfs4_createrpc(struct vnode *, struct vnode **,
                              struct componentname *, nfstype,
                              struct vattr *, char *);

/*
 * Global variables
 */
struct nfs4_lowner nfs4_masterlowner;

#define	DIRHDSIZ	(sizeof (struct dirent) - (MAXNAMLEN + 1))

SYSCTL_DECL(_vfs_nfs4);

static int	nfsaccess_cache_timeout = NFS_MAXATTRTIMO;
SYSCTL_INT(_vfs_nfs4, OID_AUTO, access_cache_timeout, CTLFLAG_RW,
	   &nfsaccess_cache_timeout, 0, "NFS ACCESS cache timeout");

static int	nfsv3_commit_on_close = 0;
SYSCTL_INT(_vfs_nfs4, OID_AUTO, nfsv3_commit_on_close, CTLFLAG_RW,
	   &nfsv3_commit_on_close, 0, "write+commit on close, else only write");
#if 0
SYSCTL_INT(_vfs_nfs4, OID_AUTO, access_cache_hits, CTLFLAG_RD,
	   &nfsstats.accesscache_hits, 0, "NFS ACCESS cache hit count");

SYSCTL_INT(_vfs_nfs4, OID_AUTO, access_cache_misses, CTLFLAG_RD,
	   &nfsstats.accesscache_misses, 0, "NFS ACCESS cache miss count");
#endif

#define	NFSV3ACCESS_ALL (NFSV3ACCESS_READ | NFSV3ACCESS_MODIFY		\
			 | NFSV3ACCESS_EXTEND | NFSV3ACCESS_EXECUTE	\
			 | NFSV3ACCESS_DELETE | NFSV3ACCESS_LOOKUP)

/* XXX */
static int
nfs3_access_otw(struct vnode *vp, int wmode, cthread_t *td,
    struct ucred *cred)
{
	const int v3 = 1;
	u_int32_t *tl;
	int error = 0, attrflag;

	return (0);

	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	caddr_t bpos, dpos;
	u_int32_t rmode;
	struct nfsnode *np = VTONFS(vp);

	nfsxstats.rpccnt[NFSPROC_ACCESS]++;
	mreq = nfsm_reqhead(vp, NFSPROC_ACCESS, NFSX_FH(v3) + NFSX_UNSIGNED);
	mb = mreq;
	bpos = mtod(mb, caddr_t);
	nfsm_fhtom(vp, v3);
	tl = nfsm_build(u_int32_t *, NFSX_UNSIGNED);
	*tl = txdr_unsigned(wmode);
	nfsm_request(vp, NFSPROC_ACCESS, td, cthread_ucred(td));
	nfsm_postop_attr(vp, attrflag);
	if (!error) {
		tl = nfsm_dissect(u_int32_t *, NFSX_UNSIGNED);
		rmode = fxdr_unsigned(u_int32_t, *tl);
		np->n_mode = rmode;
		np->n_modeuid = cred->cr_uid;
		np->n_modestamp = time_second;
	}
	m_freem(mrep);
nfsmout:
	return error;
}

/*
 * nfs access vnode op.
 * For nfs version 2, just return ok. File accesses may fail later.
 * For nfs version 3, use the access rpc to check accessibility. If file modes
 * are changed on the server, accesses might still fail later.
 */
/* XXXMARIUS: nfsxify */
static int
nfs4_access(struct vop_access_args *ap)
{
	struct vnode *vp = ap->a_vp;
	int error = 0;
	u_int32_t mode, wmode;
	int v3 = NFS_ISV3(vp);	/* v3 \in v4 */
	struct nfsnode *np = VTONFS(vp);
	caddr_t bpos, dpos;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;
	struct nfs4_oparg_access acc;
	cthread_t *td = cthread__ap(ap);
	struct ucred *cred = ap->a_cred;

	/*
	 * Disallow write attempts on filesystems mounted read-only;
	 * unless the file is a socket, fifo, or a block or character
	 * device resident on the filesystem.
	 */
	if ((ap->a_mode & VWRITE) && (vp->v_mount->mnt_flag & MNT_RDONLY)) {
		switch (vp->v_type) {
		case VREG:
		case VDIR:
		case VLNK:
			return (EROFS);
		default:
			break;
		}
	}
	/*
	 * For nfs v3, check to see if we have done this recently, and if
	 * so return our cached result instead of making an ACCESS call.
	 * If not, do an access rpc, otherwise you are stuck emulating
	 * ufs_access() locally using the vattr. This may not be correct,
	 * since the server may apply other access criteria such as
	 * client uid-->server uid mapping that we do not know about.
	 */
	/* XXX Disable this for now; needs fixing of _access_otw() */
	if (0 && v3) {
		if (ap->a_mode & VREAD)
			mode = NFSV3ACCESS_READ;
		else
			mode = 0;
		if (vp->v_type != VDIR) {
			if (ap->a_mode & VWRITE)
				mode |= (NFSV3ACCESS_MODIFY | NFSV3ACCESS_EXTEND);
			if (ap->a_mode & VEXEC)
				mode |= NFSV3ACCESS_EXECUTE;
		} else {
			if (ap->a_mode & VWRITE)
				mode |= (NFSV3ACCESS_MODIFY | NFSV3ACCESS_EXTEND |
				    NFSV3ACCESS_DELETE);
			if (ap->a_mode & VEXEC)
				mode |= NFSV3ACCESS_LOOKUP;
		}
		/* XXX safety belt, only make blanket request if caching */
		if (nfsaccess_cache_timeout > 0) {
			wmode = NFSV3ACCESS_READ | NFSV3ACCESS_MODIFY |
			    NFSV3ACCESS_EXTEND | NFSV3ACCESS_EXECUTE |
			    NFSV3ACCESS_DELETE | NFSV3ACCESS_LOOKUP;
		} else {
			wmode = mode;
		}

		/*
		 * Does our cached result allow us to give a definite yes to
		 * this request?
		 */
		if ((time_second < (np->n_modestamp + nfsaccess_cache_timeout)) &&
		    (ap->a_cred->cr_uid == np->n_modeuid) &&
		    ((np->n_mode & mode) == mode)) {
#ifdef __FreeBSD__
			nfsxstats.accesscache_hits++
#endif /* __FreeBSD__ */
			    ;
		} else {
			/*
			 * Either a no, or a don't know.  Go to the wire.
			 */
#ifdef __FreeBSD__
			nfsxstats.accesscache_misses++;
#endif
		        error = nfs3_access_otw(vp, wmode,
			    cthread__ap(ap), ap->a_cred);
			if (!error) {
				if ((np->n_mode & mode) != mode) {
					error = EACCES;
				}
			}
		}
		return (error);
	}

	/* XXX use generic access code here? */
	mode = ap->a_mode & VREAD ? NFSV4ACCESS_READ : 0;
	if (vp->v_type == VDIR) {
		if (ap->a_mode & VWRITE)
			mode |= NFSV4ACCESS_MODIFY |
			    NFSV4ACCESS_EXTEND | NFSV4ACCESS_DELETE;
		if (ap->a_mode & VEXEC)
			mode |= NFSV4ACCESS_LOOKUP;
	} else {
		if (ap->a_mode & VWRITE)
			mode |= NFSV4ACCESS_MODIFY | NFSV4ACCESS_EXTEND;
		if (ap->a_mode & VEXEC)
			mode |= NFSV4ACCESS_EXECUTE;
	}

	nfs_v4initcompound(&cp);
	acc.mode = mode;

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	nfsm_v4build_compound(&cp, "nfs4_access()");
	nfsm_v4build_putfh(&cp, vp);
	nfsm_v4build_access(&cp, &acc);
	nfsm_v4build_finalize(&cp);

	nfsm_request(vp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_access(&cp, &acc);

	if ((acc.rmode & mode) != mode)
		error = EACCES;

 nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);

	return (error);
}

static int
nfs4_openrpc(struct vnode *dvp, struct vnode **vpp, struct componentname *cnp,
    int flags, struct vattr *vap)
{
	struct vnode *vp = *vpp;
	struct nfs4_oparg_getattr getattr;
	struct nfs4_oparg_getfh getfh;
	struct nfs4_oparg_open opena;
	struct nfs4_compound cp;
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct ucred *cred = cnp->cn_cred;
	cthread_t *td = cthread__cnp(cnp);
	struct nfs4_fctx xfc, *fcp;
	struct nfsnode *np;

	if (vp == NULL) {
		/* Create a new file */
		np = NULL;
		fcp = &xfc;
		bzero(fcp, sizeof(*fcp));
	} else {
		np = VTONFS(vp);
		fcp = flags & FWRITE ? &np->n_wfc : &np->n_rfc;
	}


	/*
	 * Since we are currently only one lockowner; we only open the
	 * file once each for reading and writing.
	 */
	if (fcp->refcnt++ != 0) {
		*vpp = vp;
		/*printf("not opening %s\n", np->n_name != NULL ? np->n_name : "");*/
		/* XXX truncate flag? */
		return (0);
	}

	fcp->lop = &nfs4_masterlowner;
#ifdef __FreeBSD__ 		/* XXX - ugly. */
	fcp->pid = cnp->cn_thread->td_proc->p_pid;
#else
	fcp->pid = td->p_pid;
#endif /* __FreeBSD__ */

	fcp->np = np;

	nfs_v4initcompound(&cp);
	cp.nmp = VFSTONFS(dvp->v_mount);

	opena.ctype = NCLNULL;
	opena.flags = flags;
	opena.vap = vap;
	opena.fcp = fcp;		/* For lockowner */
	opena.cnp = cnp;

	getattr.bm = &nfsv4_getattrbm;

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	nfsm_v4build_compound(&cp, "nfs4_openrpc()");
	nfsm_v4build_putfh(&cp, dvp);
	nfsm_v4build_open(&cp, &opena);
	nfsm_v4build_getattr(&cp, &getattr);
	nfsm_v4build_getfh(&cp, &getfh);
	nfsm_v4build_finalize(&cp);

	nfsm_request(vp != NULL ? vp : dvp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_open(&cp, &opena);
	nfsm_v4dissect_getattr(&cp, &getattr);
	nfsm_v4dissect_getfh(&cp, &getfh);

	error = nfs_v4postop(&cp, error);

	if (opena.rflags & NFSV4OPENRES_CONFIRM) {
		error = nfs4_open_confirm(vp ? vp : dvp,
		    &cp, &opena, &getfh, cred, td);
		if (error != 0)
			goto nfsmout;
	}

	if (vp == NULL) {
		/* New file */
		error = nfsx_nget(dvp->v_mount, &getfh.fh_val,
		    getfh.fh_len, &np);
		if (error != 0)
			goto nfsmout;

		vp = NFSTOV(np);
		np->n_dvp = dvp;
		np->n_namelen = cnp->cn_namelen; /* XXX memory leaks on these; track! */
		if (np->n_name != NULL)
			FREE(np->n_name, M_NFSREQ);
		MALLOC(np->n_name, u_char *,
		    np->n_namelen + 1, M_NFSREQ, M_WAITOK);
		bcopy(cnp->cn_nameptr, np->n_name, np->n_namelen);
		np->n_name[np->n_namelen] = '\0';
		if (flags & FWRITE)
			np->n_wfc = *fcp;
		else
			np->n_rfc = *fcp;

		/*printf("opened new file %s\n", np->n_name);*/

		error = nfs4_vnop_loadattrcache(vp, &getattr.fa, NULL);
		if (error != 0)
			goto nfsmout;
		*vpp = vp;
	} else {
		/*printf("openend \"old\" %s\n", np->n_name != NULL ? np->n_name : "");*/

		if (flags & O_TRUNC && np->n_size != 0) {
			struct vattr va;

			VATTR_NULL(&va);
			va.va_size = 0;
			error = nfs4_setattrrpc(vp, &va,
			    cnp->cn_cred, cthread__cnp(cnp));
		}
		np->n_attrstamp = 0;
	}

 nfsmout:
	if (mrep != NULL)
		m_freem(mrep);

	return (error);
}

static int
nfs4_open_confirm(struct vnode *vp, struct nfs4_compound *cpp,
    struct nfs4_oparg_open *openap, struct nfs4_oparg_getfh *gfh,
    struct ucred *cred, cthread_t *td)
{
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;

	nfs_v4initcompound(cpp);
	cpp->nmp = VFSTONFS(vp->v_mount);

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	nfsm_v4build_compound(cpp, "nfs4_open_confirm()");
	nfsm_v4build_putfh_nv(cpp, gfh);
	nfsm_v4build_open_confirm(cpp, openap);
	nfsm_v4build_finalize(cpp);

	nfsm_request(vp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(cpp);
	nfsm_v4dissect_putfh(cpp);
	nfsm_v4dissect_open_confirm(cpp, openap);

 nfsmout:
	error = nfs_v4postop(cpp, error);

	if (mrep != NULL)
		m_freem(mrep);

	return (error);
}


/*
 * nfs open vnode op
 * Check to see if the type is ok
 * and that deletion is not in progress.
 * For paged in text files, you will need to flush the page cache
 * if consistency is lost.
 */
/* ARGSUSED */
static int
nfs4_open(struct vop_open_args *ap)
{
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	enum vtype vtype = vp->v_type;
	int mode = ap->a_mode;
	struct componentname cn;

	if (vtype != VREG) {
		if (vtype != VDIR && vtype != VLNK) {
#ifdef DIAGNOSTIC
			printf("open eacces vtyp=%d\n", vp->v_type);
#endif
			return (EACCES);
		} else
			return (0);
	}

	if (np->n_flag & NCREATED) {
		np->n_flag &= ~NCREATED;
		return (0);
	}

	cn.cn_nameptr = np->n_name;
	cn.cn_namelen = np->n_namelen;
	cn.cn_cred = ap->a_cred;
	cthread__cnp(&cn) = cthread__ap(ap);

	printf("in nfs4open... going in\n");

	return (nfs4_openrpc(np->n_dvp, &vp, &cn, mode, NULL));
}

int
nfs4_closerpc(struct vnode *vp, struct ucred *cred, cthread_t *td, int flags)
{
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_fctx *fcp;
	struct nfs4_compound cp;
	struct nfsnode *np = VTONFS(vp);

	fcp = flags & FWRITE ? &np->n_wfc : &np->n_rfc;

	if (--fcp->refcnt != 0)
		return (0);

	nfs_v4initcompound(&cp);

	cp.fcp = fcp;

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	nfsm_v4build_compound(&cp, "nfs4_closerpc()");
	nfsm_v4build_putfh(&cp, vp);
	nfsm_v4build_close(&cp, fcp);
	nfsm_v4build_finalize(&cp);

	nfsm_request(vp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_close(&cp, fcp);

 nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);

	return (error);
}

/*
 * nfs getattr call from vfs.
 */
static int
nfs4_getattr(struct vop_getattr_args *ap)
{
	struct vnode *vp = ap->a_vp;
	struct nfsnode *np = VTONFS(vp);
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_oparg_getattr ga;
	struct nfs4_compound cp;

	printf("nfs4_getattr!\n");

	/*
	 * Update local times for special files.
	 */
	if (np->n_flag & (NACC | NUPD))
		np->n_flag |= NCHG;
	/*
	 * First look in the cache.
	 */
	if (nfsx_getattrcache(vp, ap->a_vap) == 0)
		return (0);

	nfsxstats.rpccnt[NFSPROC_GETATTR]++;

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, NFSX_FH(1));
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	ga.bm = &nfsv4_getattrbm;
	nfs_v4initcompound(&cp);

	nfsm_v4build_compound(&cp, "nfs4_getattr()");
	nfsm_v4build_putfh(&cp, vp);
	nfsm_v4build_getattr(&cp, &ga);
	nfsm_v4build_finalize(&cp);

	nfsm_request(vp, NFSV4PROC_COMPOUND, cthread__ap(ap), ap->a_cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_getattr(&cp, &ga);

	error = nfs4_vnop_loadattrcache(vp, &ga.fa, ap->a_vap);

nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);
	return (error);
}

/*
 * Do an nfs setattr rpc.
 */
int
nfs4_setattrrpc(struct vnode *vp, struct vattr *vap, struct ucred *cred,
    cthread_t *td)
{
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;
	struct nfs4_oparg_getattr ga;
	struct nfsnode *np = VTONFS(vp);
	struct nfs4_fctx *fcp;

	printf("SETATTR\n");

	nfsxstats.rpccnt[NFSPROC_SETATTR]++;
	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	ga.bm = &nfsv4_getattrbm;
	fcp = (vap->va_size != VNOVAL) ? &np->n_wfc : NULL;
	nfs_v4initcompound(&cp);

	nfsm_v4build_compound(&cp, "nfs4_setattrrpc");
	nfsm_v4build_putfh(&cp, vp);
	nfsm_v4build_setattr(&cp, vap, fcp);
	nfsm_v4build_getattr(&cp, &ga);
	nfsm_v4build_finalize(&cp);

	nfsm_request(vp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_setattr(&cp);
	nfsm_v4dissect_getattr(&cp, &ga);

	error = nfs4_vnop_loadattrcache(vp, &ga.fa, NULL);
	if (error != 0)
		goto nfsmout;

	/* XXX -- need to implement this in nfs4_setattr*/
	if (np->n_flag & NTRUNCATE) {
		printf("ntruncate, woulda closed;\n");

/* 		error = nfs4_closerpc(vp, cred, td, FWRITE); */
/* 		np->n_flag &= ~NTRUNCATE; */
	}

nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);
	
	return (error);
}

/*
 * nfs lookup call, one step at a time...
 * First look in cache
 * If not found, unlock the directory nfsnode and do the rpc
 */
int
nfs4_lookuprpc(struct vnode *dvp, struct componentname *cnp,
    int isdotdot, nfsfh_t **fhp, int *fhsize, void *stp)
{
	caddr_t bpos, dpos;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;
	struct nfs4_oparg_getattr *ga, dga;
	struct nfs4_oparg_lookup l;
	static struct nfs4_oparg_getfh gfh; /* XXX majorly. */
	int error = 0, isdot;

	printf("XXXMARIUS: nfs4_lookuprpc\n");

	ga = stp;

	printf("XXXMARIUS: nfs4_lookuprpc before stats\n");

	nfsxstats.rpccnt[NFSPROC_LOOKUP]++;

	printf("XXXMARIUS: nfs4_lookuprpc after stats\n");

	isdot = cnp->cn_namelen == 1 && cnp->cn_nameptr[0] == '.';

	mreq = nfsm_reqhead(NULL, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	printf("XXXMARIUS: nfs4_lookuprpc after head!\n");

	ga->bm = &nfsv4_getattrbm;
	dga.bm = &nfsv4_getattrbm;
	nfs_v4initcompound(&cp);

	printf("XXXMARIUS: nfs4_lookuprpc after initcmpd!\n");

	nfsm_v4build_compound(&cp, "nfs4_lookup()");
	nfsm_v4build_putfh(&cp, dvp);
	nfsm_v4build_getattr(&cp, &dga);
	if (isdotdot)
		nfsm_v4build_lookupp(&cp);
	else if (!isdot) {
		l.name = cnp->cn_nameptr;
		l.namelen = cnp->cn_namelen;
		nfsm_v4build_lookup(&cp, &l);
	}
	nfsm_v4build_getattr(&cp, ga);
	nfsm_v4build_getfh(&cp, &gfh);
	nfsm_v4build_finalize(&cp);

	printf("XXXMARIUS: build; about to send req\n");

	nfsm_request(dvp, NFSV4PROC_COMPOUND, cthread__cnp(cnp), cnp->cn_cred);
	if (error != 0)
		goto nfsmout;

	printf("XXXMARIUS: req out\n");

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_getattr(&cp, &dga);
	if (isdotdot)
		nfsm_v4dissect_lookupp(&cp);
	else if (!isdot)
		nfsm_v4dissect_lookup(&cp);
	nfsm_v4dissect_getattr(&cp, ga);
	nfsm_v4dissect_getfh(&cp, &gfh);

	printf("XXXMARIUS: dissect done, doing postops\n");

/* 	nfs4_vnop_loadattrcache(dvp, &dga.fa, NULL); */
	
	if (fhp != NULL && fhsize != NULL) {
		*fhp = &gfh.fh_val;
		*fhsize = gfh.fh_len;
	}

nfsmout:
	error = nfs_v4postop(&cp, error);

	if (m_freem != NULL)
		m_freem(mrep);

	return (error);
}

int
nfs4_lookuploadattr(struct vnode *newvp, struct vnode *dvp, void *stp)
{
	struct nfs4_oparg_getattr *gap = stp;

	return (nfs4_vnop_loadattrcache(newvp, &gap->fa, NULL));
}

int
nfs4_mknodrpc(struct vnode *dvp, struct vnode **newvp,
    struct componentname *cnp, struct vattr *vap)
{
	int error;

	error = nfs4_createrpc(dvp, newvp,
	    cnp, (nfstype)vap->va_type, vap, NULL);

#ifdef __APPLE__
	vput(dvp);
#endif

	return (error);
}

/*
 * Do a readlink rpc.
 * Called by nfs_doio() from below the buffer cache.
 */
int
nfs4_readlinkrpc(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;

	nfsxstats.rpccnt[NFSPROC_READLINK]++;

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	nfs_v4initcompound(&cp);

	nfsm_v4build_compound(&cp, "nfs4_readlinkrpc()");
	nfsm_v4build_putfh(&cp, vp);
	nfsm_v4build_readlink(&cp);
	nfsm_v4build_finalize(&cp);
 
	nfsm_request(vp, NFSV4PROC_COMPOUND, uio__td(uiop), cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_readlink(&cp, uiop);

nfsmout:
	error = nfs_v4postop(&cp, error);

	if (m_freem != NULL)
		m_freem(mrep);
	return (error);
}

/*
 * nfs read rpc call
 * Ditto above
 */
int
nfs4_readrpc(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
	caddr_t bpos, dpos;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfsmount *nmp;
	int error = 0, len, tsiz;
	struct nfs4_compound cp;
	struct nfs4_oparg_read read;
	struct nfsnode *np = VTONFS(vp);

	nmp = VFSTONFS(vp->v_mount);
	tsiz = uiop->uio_resid;
#ifdef __FreeBSD__		/* XXX - OpenBSD */
	if (uiop->uio_offset + tsiz > nmp->nm_maxfilesize)
		return (EFBIG);
#else
        if (0 && ((u_int64_t)uiop->uio_offset + (unsigned int)tsiz > 0xffffffff))
		return (EFBIG);
#endif /* __FreeBSD__ */

	if (tsiz == 0)
		return (0);

	read.uiop = uiop;
	read.fcp = np->n_rfc.refcnt > 0 ? &np->n_rfc : &np->n_wfc;

	while (tsiz > 0) {
		nfsxstats.rpccnt[NFSPROC_READ]++;
		len = (tsiz > nmp->nm_rsize) ? nmp->nm_rsize : tsiz;

		read.off = uiop->uio_offset;
		read.maxcnt = len;
		nfs_v4initcompound(&cp);

		mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
		mb = mreq;
		bpos = mtod(mb, caddr_t);

		nfsm_v4build_compound(&cp, "nfs4_readrpc()");
		nfsm_v4build_putfh(&cp, vp);
		nfsm_v4build_read(&cp, &read);
		nfsm_v4build_finalize(&cp);

		nfsm_request(vp, NFSV4PROC_COMPOUND, uio__td(uiop), cred);
		if (error != 0) {
			error = nfs_v4postop(&cp, error);
			goto nfsmout;
		}

		nfsm_v4dissect_compound(&cp);
		nfsm_v4dissect_putfh(&cp);
		nfsm_v4dissect_read(&cp, &read);

		if (read.eof || read.retlen == 0)
			tsiz = 0;
		else
			tsiz -= read.retlen;

		error = nfs_v4postop(&cp, error);

		m_freem(mrep);
		mrep = NULL;
	}
nfsmout:
	if (mrep != NULL)
		m_freem(mrep);
	
	return (error);
}

/*
 * nfs write call
 */
int
nfs4_writerpc(struct vnode *vp, struct uio *uiop, struct ucred *cred,
    int *iomode, int *must_commit)
{
	int32_t backup;
	caddr_t bpos, dpos;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfsmount *nmp = VFSTONFS(vp->v_mount);
	int error = 0, len, tsiz, wccflag = 1, rlen;
	struct nfs4_compound cp;
	struct nfs4_oparg_write write;
	nfsv4stablehow commit, committed = NSHFILESYNC;
	caddr_t verf;
	struct nfsnode *np = VTONFS(vp);

#ifndef DIAGNOSTIC
	if (uiop->uio_iovcnt != 1)
		panic("nfs: writerpc iovcnt > 1");
#endif
	*must_commit = 0;
	tsiz = uiop->uio_resid;

#ifdef __FreeBSD__		/* XXX - OpenBSD */
	if (uiop->uio_offset + tsiz > nmp->nm_maxfilesize)
		return (EFBIG);
#else
        if (0 && ((u_int64_t)uiop->uio_offset + (unsigned int)tsiz > 0xffffffff)) {
		FSDBG_BOT(536, vp, uiop->uio_offset, uiop->uio_resid, EFBIG);
		return (EFBIG);
	}
#endif /* __FreeBSD__ */

	if (tsiz == 0)
		return (0);

	write.stable = (nfsv4stablehow)*iomode;
	write.uiop = uiop;
	write.fcp = &np->n_wfc;

	while (tsiz > 0) {
		nfsxstats.rpccnt[NFSPROC_WRITE]++;
		len = (tsiz > nmp->nm_wsize) ? nmp->nm_wsize : tsiz;

		write.off = uiop->uio_offset;
		write.cnt = len;
		nfs_v4initcompound(&cp);

		mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
		mb = mreq;
		bpos = mtod(mb, caddr_t);

		nfsm_v4build_compound(&cp, "nfs4_writerpc()");
		nfsm_v4build_putfh(&cp, vp);
		nfsm_v4build_write(&cp, &write);
		nfsm_v4build_finalize(&cp);

		nfsm_request(vp, NFSV4PROC_COMPOUND, uio__td(uiop), cred);
		if (error != 0) {
			error = nfs_v4postop(&cp, error);
			goto nfsmout;
		}

		nfsm_v4dissect_compound(&cp);
		nfsm_v4dissect_putfh(&cp);
		nfsm_v4dissect_write(&cp, &write);

		rlen = write.retlen;
		if (rlen == 0) {
			error = NFSERR_IO;
			break;
		} else if (rlen < len) {
			backup = len - rlen;
			(char *)uiop->uio_iov->iov_base -= backup;
			uiop->uio_iov->iov_len += backup;
                        uiop->uio_offset -= backup;
                        uiop->uio_resid += backup;
                        len = rlen;
		}

		commit = write.committed;

		if (committed == NSHFILESYNC ||
		    (committed = NSHDATASYNC && commit == NSHUNSTABLE))
			committed = commit;

		verf = (caddr_t)write.wverf;

                if ((nmp->nm_flag & NFSSTA_HASWRITEVERF) == 0) {
                        bcopy(verf, nmp->nm_verf, NFSX_V4VERF);
                        nmp->nm_flag |= NFSMNT_HASWRITEVERF;
                } else if (bcmp(verf, nmp->nm_verf, NFSX_V4VERF)) {
                        *must_commit = 1;
                        bcopy(verf, nmp->nm_verf, NFSX_V4VERF);
                }

		/* XXX wccflag */
		if (wccflag)
			VTONFS(vp)->n_mtime =
			    VTONFS(vp)->n_vattr.va_mtime.tv_sec;

		error = nfs_v4postop(&cp, error);

		m_freem(mrep);
		mrep = NULL;
		if (error)
			break;
		tsiz -= len;
	}
nfsmout:
	if (mrep != NULL)
		m_freem(mrep);
	*iomode = committed;
	if (error)
		uiop->uio_resid = tsiz;
	return (error);
}

static int
nfs4_createrpc(struct vnode *dvp, struct vnode **vpp, struct componentname *cnp,
    nfstype ftype, struct vattr *vap, char *linktarget)
{
	struct nfsnode *dnp = VTONFS(dvp);
	struct nfsnode *np = NULL;
	struct vnode *newvp = NULL;
	struct nfs4_compound cp;
	struct nfs4_oparg_create c;
	struct nfs4_oparg_getattr ga;
	struct nfs4_oparg_getfh gfh;
	caddr_t bpos, dpos;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	int error = 0;

	nfsxstats.rpccnt[NFSPROC_CREATE]++;

	mreq = nfsm_reqhead(dvp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	bzero(&c, sizeof(c));
	bzero(&ga, sizeof(ga));

	c.type = ftype;
	c.vap = vap;
	c.linktext = linktarget;
	c.name = cnp->cn_nameptr;
	c.namelen = cnp->cn_namelen;

	ga.bm = &nfsv4_getattrbm;
	nfs_v4initcompound(&cp);

	nfsm_v4build_compound(&cp, "nfs4_createrpc()");
	nfsm_v4build_putfh(&cp, dvp);
	nfsm_v4build_create(&cp, &c);
	nfsm_v4build_getattr(&cp, &ga);
	nfsm_v4build_getfh(&cp, &gfh);	
	nfsm_v4build_finalize(&cp); 

	nfsm_request(dvp, NFSV4PROC_COMPOUND, cthread__cnp(cnp), cnp->cn_cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_create(&cp, &c);
	nfsm_v4dissect_getattr(&cp, &ga);
	nfsm_v4dissect_getfh(&cp, &gfh);	

	error = nfsx_nget(dvp->v_mount, &gfh.fh_val, gfh.fh_len, &np);
	if (error != 0)
		goto nfsmout;

	newvp = NFSTOV(np);
	error = nfs4_vnop_loadattrcache(newvp, &ga.fa, NULL);
	if (error != 0)
		goto nfsmout;

	if (cnp->cn_flags & MAKEENTRY)
		cache_enter(dvp, newvp, cnp);

	dnp->n_flag |= NMODIFIED;
	dnp->n_attrstamp = 0;

 nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);

	/* XXX */
	/*FREE(cnp->cn_pnbuf, M_NAMEI);*/
	if (error != 0 && newvp != NULL)
#ifdef __APPLE__		/* XXXMARIUS: fix this properly. */
		vput(newvp);
#else
		vrele(newvp);
#endif
	else if (error == 0)
		*vpp = newvp;

	return (error);
}

int
nfs4_renamerpc(struct vnode *fdvp, const char *fnameptr, int fnamelen,
    struct vnode *tdvp, const char *tnameptr, int tnamelen,
    struct ucred *cred, cthread_t *td)
{

	struct nfsnode *fnp = VTONFS(fdvp), *tnp = VTONFS(tdvp);
	caddr_t bpos, dpos;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;
	struct nfs4_oparg_rename r;
	int error = 0;

	nfsxstats.rpccnt[NFSPROC_RENAME]++;

	r.fname = fnameptr;
	r.fnamelen = fnamelen;
	r.tname = tnameptr;
	r.tnamelen = tnamelen;
	nfs_v4initcompound(&cp);

	mreq = nfsm_reqhead(fdvp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	nfsm_v4build_compound(&cp, "nfs4_renamerpc()");
	nfsm_v4build_putfh(&cp, fdvp);
	nfsm_v4build_savefh(&cp);
	nfsm_v4build_putfh(&cp, tdvp);
	nfsm_v4build_rename(&cp, &r);
	nfsm_v4build_finalize(&cp);

	nfsm_request(fdvp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_savefh(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_rename(&cp);

	/* XXX should this always be performed?  */
	fnp->n_flag |= NMODIFIED;
	tnp->n_flag |= NMODIFIED;
	fnp->n_attrstamp = tnp->n_attrstamp = 0;

 nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);

	return (error);
}

/*
 * nfs file create call
 */
/* XXX nfsxify */
static int
nfs4_create(struct vop_create_args *ap)
{
	struct vnode *dvp = ap->a_dvp;
	struct vattr *vap = ap->a_vap;
	struct nfsnode *dnp = VTONFS(dvp);
	struct componentname *cnp = ap->a_cnp;
	struct vnode *newvp = NULL;
	int error = 0, fmode = (O_CREAT | FREAD | FWRITE);
	struct vattr vattr;

	if ((error = VOP_GETATTR(dvp, &vattr,
		 cnp->cn_cred, cthread__cnp(cnp))) != 0) {
#ifdef __APPLE__
		vput(dvp);
#endif
		return (error);
	}

	if (vap->va_vaflags & VA_EXCLUSIVE)
		fmode |= O_EXCL;

	error = nfs4_openrpc(dvp, &newvp, cnp, fmode, vap);
	if (error != 0)
		goto out;

	VTONFS(newvp)->n_flag |= NCREATED;

	if (cnp->cn_flags & MAKEENTRY)
		cache_enter(dvp, newvp, cnp);

	*ap->a_vpp = newvp;

	dnp->n_flag |= NMODIFIED;
	dnp->n_attrstamp = 0;	/* XXX; wccflag */


 out:
	/* XXX might need to vput newvp on error. */
#ifdef __APPLE__
	vput(dvp);
#endif

	return (error);
}

/*
 * nfs file remove rpc called from nfs_inactive
 */
int
nfs4_removeit(struct sillyrename *sp)
{
	/*
	 * Make sure that the directory vnode is still valid.
	 * XXX we should lock sp->s_dvp here.
	 */
	if (sp->s_dvp->v_type == VBAD)
		return (0);
	return (nfs4_removerpc(sp->s_dvp, sp->s_name, sp->s_namlen, sp->s_cred,
		NULL));
}

/*
 * Nfs remove rpc, called from nfs4_remove() and nfs4_removeit().
 */
int
nfs4_removerpc(struct vnode *dvp, const char *name, int namelen,
    struct ucred *cred, cthread_t *td)
{
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;

	nfsxstats.rpccnt[NFSPROC_REMOVE]++;

	mreq = nfsm_reqhead(dvp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	printf("REMOVE(%.*s)\n", namelen, name);

	nfs_v4initcompound(&cp);

	nfsm_v4build_compound(&cp, "nfs4_removerpc()");
	nfsm_v4build_putfh(&cp, dvp);
	nfsm_v4build_remove(&cp, name, namelen);
	nfsm_v4build_finalize(&cp);

	nfsm_request(dvp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_remove(&cp);

 nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);

	VTONFS(dvp)->n_flag |= NMODIFIED;
	VTONFS(dvp)->n_attrstamp = 0; /* XXX wccflag */

	return (error);
}

/*
 * nfs hard link create call
 */
/* XXXMARIUS: nfsxify */
static int
nfs4_link(struct vop_link_args *ap)
{
	struct vnode *vp = ap->a_vp;
	struct vnode *tdvp = ap->a_tdvp;
	struct componentname *cnp = ap->a_cnp;
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;
	struct nfs4_oparg_link l;

	if (vp->v_mount != tdvp->v_mount) {
#ifdef __APPLE__
		vput(tdvp);
#endif
		return (EXDEV);
	}

	/*
	 * Push all writes to the server, so that the attribute cache
	 * doesn't get "out of sync" with the server.
	 * XXX There should be a better way!
	 */
	VOP_FSYNC(vp, cnp->cn_cred, MNT_WAIT, cthread__cnp(cnp));

	nfsxstats.rpccnt[NFSPROC_LINK]++;

	l.name = cnp->cn_nameptr;
	l.namelen = cnp->cn_namelen;
	nfs_v4initcompound(&cp);

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	nfsm_v4build_compound(&cp, "nfs4_link()");
	nfsm_v4build_putfh(&cp, vp);
	nfsm_v4build_savefh(&cp);
	nfsm_v4build_putfh(&cp, tdvp);
	nfsm_v4build_link(&cp, &l);
	nfsm_v4build_finalize(&cp);

	nfsm_request(vp, NFSV4PROC_COMPOUND, cthread__cnp(cnp), cnp->cn_cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_savefh(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_link(&cp);

	VTONFS(tdvp)->n_flag |= NMODIFIED;
	VTONFS(vp)->n_attrstamp = 0;
	VTONFS(tdvp)->n_attrstamp = 0;

nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep != NULL)
		m_freem(mrep);

#ifdef __APPLE__
	vput(tdvp);
#endif

	return (error);
}

/*
 * nfs symbolic link create call
 */
static int
nfs4_symlink(struct vop_symlink_args *ap)
{
	struct vnode *dvp = ap->a_dvp;
	int error = 0;
	struct vnode *newvp = NULL;

	nfsxstats.rpccnt[NFSPROC_SYMLINK]++;

	error = nfs4_createrpc(ap->a_dvp, &newvp, ap->a_cnp, NFLNK,
	    ap->a_vap, ap->a_target);

	if (newvp != NULL)
		vput(newvp);
	else if (error == 0)
		 *ap->a_vpp = newvp;

	VTONFS(dvp)->n_flag |= NMODIFIED;
	VTONFS(dvp)->n_attrstamp = 0; /* XXX wccflags */
#ifdef __APPLE__
	vput(dvp);
#endif

	return (error);
}

/*
 * nfs make dir call
 */
static int
nfs4_mkdir(struct vop_mkdir_args *ap)
{
	int error;

	error = nfs4_createrpc(ap->a_dvp, ap->a_vpp, ap->a_cnp, NFDIR,
	    ap->a_vap, NULL);

	printf("mkdir\n");
#ifdef __APPLE__
	printf("mkdir, unlocking\n");
	vput(ap->a_dvp);
#endif

	return (error);
}

/*
 * nfs remove directory call
 */
/* XXXMARIUS: nfsxify */
static int
nfs4_rmdir(struct vop_rmdir_args *ap)
{
	struct vnode *vp = ap->a_vp;
	struct vnode *dvp = ap->a_dvp;
	struct nfsnode *dnp = VTONFS(dvp);
	struct componentname *cnp = ap->a_cnp;
	int error = 0;

	if (dvp == vp) {
#ifdef __APPLE__
		vput(dvp);
		vput(vp);
#endif /* __APPLE__ */
		return (EINVAL);
	}

	error = (nfs4_removerpc(dvp, cnp->cn_nameptr,
		     cnp->cn_namelen, cnp->cn_cred, NULL));
	if (error)
		return (error);

	dnp->n_flag |= NMODIFIED;
	dnp->n_attrstamp = 0;
	cache_purge(dvp);
	cache_purge(vp);

#ifdef __APPLE__
	vput(dvp);
	vput(vp);
#endif /* __APPLE__ */

	return (error);
}

static u_char fty_to_dty[] = {
	DT_UNKNOWN,		/* NFNON */
	DT_REG,			/* NFREG */
	DT_DIR,			/* NFDIR */
	DT_BLK,			/* NFBLK */
	DT_CHR,			/* NFCHR */
	DT_LNK,			/* NFLNK */
	DT_SOCK,		/* NFSOCK */
	DT_FIFO,		/* NFFIFO */
        DT_UNKNOWN,		/* NFATTRDIT */
        DT_UNKNOWN,		/* NFNAMEDATTR */
        DT_UNKNOWN,		/* NFBAD */
};

/*
 * Readdir rpc call.
 * Called from below the buffer cache by nfs_doio().
 */
int
nfs4_readdirrpc(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
	int len, left;
	struct dirent *dp = NULL;
	u_int32_t *tl;
	caddr_t p;
	uint64_t *cookiep;
	caddr_t bpos, dpos;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	uint64_t cookie;
	struct nfsmount *nmp = VFSTONFS(vp->v_mount);
	struct nfsnode *dnp = VTONFS(vp);
	int error = 0, tlen, more_dirs = 1, blksiz = 0, bigenough = 1;
	struct nfs4_compound cp;
	struct nfs4_oparg_readdir readdir;
	struct nfsv4_fattr fattr;
	u_int fty;

#ifndef DIAGNOSTIC
	if (uiop->uio_iovcnt != 1 || (uiop->uio_offset & (DIRBLKSIZ - 1)) ||
		(uiop->uio_resid & (DIRBLKSIZ - 1)))
		panic("nfs readdirrpc bad uio");
#endif

	/*
	 * If there is no cookie, assume directory was stale.
	 */
	cookiep = nfs4_getcookie(dnp, uiop->uio_offset, 0);
	if (cookiep)
		cookie = *cookiep;
	else
		return (NFSERR_BAD_COOKIE);

	/* Generate fake entries for "." and ".." */
	while (cookie < 2 && bigenough) {
		cookie++;
		len = 4 + DIRHDSIZ;

		if (len > uiop->uio_resid) {
			bigenough = 0;
			break;
		}
		dp = (struct dirent *)uiop->uio_iov->iov_base;

		dp->d_namlen = cookie;
		dp->d_reclen = len;
		dp->d_type = DT_DIR;
		if (cookie == 1)
			dp->d_fileno = dnp->n_vattr.va_fileid; /* XXX has problems with pynfs virtualhandles */
		else
			dp->d_fileno = dnp->n_dvp != NULL ?
			    VTONFS(dnp->n_dvp)->n_vattr.va_fileid : cookie;

		p = dp->d_name;
		*p++ = '.';
		if (cookie == 2)
			*p++ = '.';
		*p = '\0';

		blksiz += len;
		if (blksiz == DIRBLKSIZ)
			blksiz = 0;
		uiop->uio_offset += len;
		uiop->uio_resid -= len;
		(char *)uiop->uio_iov->iov_base += len;
		uiop->uio_iov->iov_len -= len;
	}

	if (cookie == 2)
		cookie = 0;

	/* This is sort of ugly, to prevent v4postop() from acting weird */
	bzero(&cp, sizeof(cp));

	/*
	 * Loop around doing readdir rpc's of size nm_readdirsize
	 * truncated to a multiple of DIRBLKSIZ.
	 * The stopping criteria is EOF or buffer full.
	 */
	/*
	 * XXX this is sort of ugly for nfsv4; we don't maintain the
	 * strict abstraction, but do the decoding inline.  that's ok.
	 */
	while (more_dirs && bigenough) {
		nfsxstats.rpccnt[NFSPROC_READDIR]++;

		mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
		mb = mreq;
		bpos = mtod(mb, caddr_t);

		readdir.cnt = nmp->nm_readdirsize;
		readdir.cookie = cookie;
		readdir.bm = &nfsv4_readdirbm;
		if (cookie == 0)
			bzero(&readdir.verf, sizeof(readdir.verf));
		else
			bcopy(&dnp->n_cookieverf, &readdir.verf,
			    sizeof(readdir.verf));

		nfs_v4initcompound(&cp);

		nfsm_v4build_compound(&cp, "nfs4_readdirrpc()");
		nfsm_v4build_putfh(&cp, vp);
		nfsm_v4build_readdir(&cp, &readdir);
		nfsm_v4build_finalize(&cp);

		nfsm_request(vp, NFSV4PROC_COMPOUND, uio__td(uiop), cred);
		if (error != 0)
			goto nfsmout;

		nfsm_v4dissect_compound(&cp);
		nfsm_v4dissect_putfh(&cp);

		/*
		 * XXX - Readdir gets handled inline like in
		 * NFSv{2,3}.  This is a nasty inconsistency and
		 * should be fixed.
		 */

		tl = nfsm_dissect(uint32_t *, 5 * NFSX_UNSIGNED);
		if (fxdr_unsigned(uint32_t, *tl++) != NFSV4OP_READDIR) {
			error = EBADRPC;
			goto nfsmout;
		}
		if (fxdr_unsigned(uint32_t, *tl++) != 0) {
			error = EBADRPC;
			goto nfsmout;
		}

		bcopy(tl, &dnp->n_cookieverf, NFSX_V4VERF);
		tl += 2;
		more_dirs = fxdr_unsigned(int, *tl++);

		/* loop thru the dir entries, doctoring them to 4bsd form */
		while (more_dirs && bigenough) {
			tl = nfsm_dissect(uint32_t *, 3 * NFSX_UNSIGNED);
			cookie = fxdr_hyper(tl);
			tl += 2;
			/* XXX cookie sanity check */
			len = fxdr_unsigned(int, *tl++);
			if (len <= 0 || len > NFS_MAXNAMLEN) {
				error = EBADRPC;
				goto nfsmout;
			}
			tlen = nfsm_rndup(len);
			if (tlen == len)
				tlen += 4;	/* To ensure null termination */
			left = DIRBLKSIZ - blksiz;
			if ((tlen + DIRHDSIZ) > left) {
				dp->d_reclen += left;
				uiop->uio_iov->iov_base =
				    (char *)uiop->uio_iov->iov_base + left;
				uiop->uio_iov->iov_len -= left;
				uiop->uio_offset += left;
				uiop->uio_resid -= left;
				blksiz = 0;
			}
			if ((tlen + DIRHDSIZ) > uiop->uio_resid)
				bigenough = 0;
			if (bigenough) {
				dp = (struct dirent *)uiop->uio_iov->iov_base;

				dp->d_namlen = len;
				dp->d_reclen = tlen + DIRHDSIZ;

				blksiz += dp->d_reclen;
				if (blksiz == DIRBLKSIZ)
					blksiz = 0;
				uiop->uio_offset += DIRHDSIZ;
				uiop->uio_resid -= DIRHDSIZ;
				uiop->uio_iov->iov_base =
				    (char *)uiop->uio_iov->iov_base + DIRHDSIZ;
				uiop->uio_iov->iov_len -= DIRHDSIZ;

				/* Copy name */
				nfsm_mtouio(uiop, len);
				p = uiop->uio_iov->iov_base;
				tlen -= len;
				*p = '\0';	/* null terminate */
				/* printf("nfs4_readdirrpc: name: \"%s\" cookie %d\n",
				   p - len, (int) cookie);*/
				uiop->uio_iov->iov_base =
				    (char *)uiop->uio_iov->iov_base + tlen;
				uiop->uio_iov->iov_len -= tlen;
				uiop->uio_offset += tlen;
				uiop->uio_resid -= tlen;

				/* Copy attributes */
				nfsm_v4dissect_attrs(&fattr);

				dp->d_fileno = nfs_v4fileid4_to_fileid(
					fattr.fa4_valid & FA4V_FILEID &&
					    fattr.fa4_fileid ?
					    fattr.fa4_fileid : cookie);

				fty = (u_int)fattr.fa4_type;
				dp->d_type = fattr.fa4_valid & FA4V_TYPE &&
				    (fty < sizeof(fty_to_dty)) ?
				    fty_to_dty[fty] : DT_UNKNOWN;
			} else
				nfsm_adv(nfsm_rndup(len));

			tl = nfsm_dissect(uint32_t *, NFSX_UNSIGNED);
			more_dirs = fxdr_unsigned(int, *tl++);
		}
		/*
		 * If at end of rpc data, get the eof boolean
		 */
		if (!more_dirs) {
			tl = nfsm_dissect(u_int32_t *, NFSX_UNSIGNED);
			more_dirs = (fxdr_unsigned(int, *tl) == 0);
		}

		error = nfs_v4postop(&cp, error);

		m_freem(mrep);
		mrep = NULL;
	}
	/*
	 * Fill last record, iff any, out to a multiple of DIRBLKSIZ
	 * by increasing d_reclen for the last record.
	 */
	if (blksiz > 0) {
		left = DIRBLKSIZ - blksiz;
		dp->d_reclen += left;
		uiop->uio_iov->iov_base =
		    (char *)uiop->uio_iov->iov_base + left;
		uiop->uio_iov->iov_len -= left;
		uiop->uio_offset += left;
		uiop->uio_resid -= left;
	}

	/*
	 * We are now either at the end of the directory or have filled the
	 * block.
	 */
	if (bigenough)
		dnp->n_direofoffset = uiop->uio_offset;
	else {
		if (uiop->uio_resid > 0)
			printf("EEK! readdirrpc resid > 0\n");
		cookiep = nfs4_getcookie(dnp, uiop->uio_offset, 1);
		*cookiep = cookie;
	}
nfsmout:
	if (mrep != NULL)
		m_freem(mrep);
	return (error);
}

int
nfs4_commit(struct vnode *vp, u_quad_t offset, int cnt, struct ucred *cred,
    cthread_t *td)
{
	struct nfsmount *nmp = VFSTONFS(vp->v_mount);
	caddr_t bpos, dpos;
	int error = 0;
	struct mbuf *mreq, *mrep = NULL, *md, *mb;
	struct nfs4_compound cp;
	struct nfs4_oparg_commit commit;

	if ((nmp->nm_state & NFSSTA_HASWRITEVERF) == 0)
		return (0);
	nfsxstats.rpccnt[NFSPROC_COMMIT]++;

	mreq = nfsm_reqhead(vp, NFSV4PROC_COMPOUND, 0);
	mb = mreq;
	bpos = mtod(mb, caddr_t);

	commit.start = offset;
	commit.len = cnt;

	nfs_v4initcompound(&cp);

	nfsm_v4build_compound(&cp, "nfs4_commit()");
	nfsm_v4build_putfh(&cp, vp);
	nfsm_v4build_commit(&cp, &commit);
	nfsm_v4build_finalize(&cp);

	nfsm_request(vp, NFSV4PROC_COMPOUND, td, cred);
	if (error != 0)
		goto nfsmout;

	nfsm_v4dissect_compound(&cp);
	nfsm_v4dissect_putfh(&cp);
	nfsm_v4dissect_commit(&cp, &commit);

	/* XXX */
	/* nfsm_wcc_data(vp, wccflag);*/
	if (bcmp(nmp->nm_verf, commit.verf, NFSX_V4VERF)) {
		bcopy(commit.verf, nmp->nm_verf, NFSX_V4VERF);
		error = NFSERR_STALEWRITEVERF;
	}

nfsmout:
	error = nfs_v4postop(&cp, error);

	if (mrep == NULL)
		m_freem(mrep);
	return (error);
}

