#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <math.h>
#include <sched.h>

/* if defined, call sched_yield between
 * data samples. Results in a slightly
 * slower resolution, but is less intrusive
 * to the rpc.
 */
#define NICE

/* this option removes most of the '0' datapoints
 * from the output data. It saves a lot of disk
 * space.
 */
#define COMPRESS

/* nfs procedure names */
char *procnames[] = {
	"null",
	"getattr",
	"setattr",
	"lookup",
	"access",
	"readlink",
	"read",
	"write",
	"create",
	"mkdir",
	"symlink",
	"mknod",
	"remove",
	"rmdir",
	"rename",
	"link",
	"readdir",
	"readdirplus",
	"fsstat",
	"fsinfo",
	"pathconf",
	"commit",
	"all"
};

/* output files */
FILE *files[23][6];
/* initialized flags */
int initd[23];

/* more output files */
FILE *ftimeo;
FILE *fcong;
FILE *fcwnd;
FILE *fbacklog;
FILE *fpending; 

/* t0 is the time at which the program started,
 * lastt is the time that the last sample was taken
 */
struct timeval t0, lastt;

/* subtract one timeval from another. Assumes a positive result */
void subtime(struct timeval*, struct timeval*, struct timeval*);

/* initialize one of the output files */
void init_file(char *suffix, int proc, int type);

/* used to hold stats for one procedure type */
struct stat_s {
	unsigned int newreq;
	unsigned int released;
	unsigned int sent;
	unsigned int ackd;
};

#ifdef COMPRESS
#define printstat(type, i) 							\
	if (!prev_stat_structs[proc].##type && stat_structs[proc].##type)	\
		fprintf(files[proc][i], "%u.%06u\t%u\n",			\
			t.tv_sec, t.tv_usec,					\
			0);							\
	if (prev_stat_structs[proc].##type || stat_structs[proc].##type)	\
		fprintf(files[proc][i], "%u.%06u\t%u\n", 			\
			t.tv_sec, t.tv_usec,					\
			stat_structs[proc].##type)
#else
#define printstat(type, i) 						\
	fprintf(files[proc][i], "%u.%06u\t%u\n", 			\
		t.tv_sec, t.tv_usec,					\
		stat_structs[proc].##type)
#endif

/* Get stats from the proc files rpcstat and rpcmisc,
 * and record them in various files.
 */
void getstats()
{
	unsigned int proc, newreq, released, sent, ackd;
	FILE *stats = fopen("/proc/rpcstat", "r+");
	FILE *misc = fopen("/proc/rpcmisc", "r+");
	unsigned int timeo;
	unsigned int numread;
	unsigned int cong, cwnd;
	unsigned int backlog, pending;
	struct timeval t, dt;
	struct stat_s stat_structs[23];
	static struct stat_s prev_stat_structs[23];

	memset(stat_structs, 0, sizeof(stat_structs));

	if (stats == NULL) {
		printf("Couldn't open /proc/rpcstats\n");
		exit(1);
	}

	gettimeofday(&t, NULL);
	subtime(&t, &t, &t0);
	subtime(&dt, &t, &lastt);
	memcpy(&lastt, &t, sizeof(t));

	while((numread = fscanf(stats, "%u %u %u %u %u",
			&proc,
			&newreq,
			&released,
			&sent,
			&ackd)) != EOF) {

		if (numread != 5) {
			printf("Warning, bad data in /proc/rpcstat\n");
			continue;
		}

		stat_structs[proc].newreq = newreq / (dt.tv_usec * (double)(1e-4));
		stat_structs[proc].released = released / (dt.tv_usec * (double)(1e-4));
		stat_structs[proc].sent = sent / (dt.tv_usec * (double)(1e-4));
		stat_structs[proc].ackd = ackd / (dt.tv_usec * (double)(1e-4));

		stat_structs[22].newreq += stat_structs[proc].newreq;
		stat_structs[22].released += stat_structs[proc].released;
		stat_structs[22].sent += stat_structs[proc].sent;
		stat_structs[22].ackd += stat_structs[proc].ackd;

		if(!initd[proc]) {
			init_file("new", proc, 0);
			init_file("fin", proc, 1);
			init_file("sent", proc, 2);
			init_file("ackd", proc, 3);
			initd[proc] = 1;
		}
	}

	for(proc=0;proc<=22;proc++) {
		if (initd[proc]) {
			printstat(newreq, 0);
			printstat(released, 1);
			printstat(sent, 2);
			printstat(ackd, 3);
/*
			if (prev_stat_structs[proc].newreq || stat_structs[proc].newreq)
				fprintf(files[proc][0], "%u.%06u\t%u\n", 
					t.tv_sec, t.tv_usec,
					stat_structs[proc].newreq);
			if (prev_stat_structs[proc].released || stat_structs[proc].released)
				fprintf(files[proc][1], "%u.%06u\t%u\n", 
					t.tv_sec, t.tv_usec,
					stat_structs[proc].released);
			if (prev_stat_structs[proc].sent || stat_structs[proc].sent)
				fprintf(files[proc][2], "%u.%06u\t%u\n", 
					t.tv_sec, t.tv_usec,
					stat_structs[proc].sent);
			if (prev_stat_structs[proc].ackd || stat_structs[proc].ackd)
				fprintf(files[proc][3], "%u.%06u\t%u\n", 
					t.tv_sec, t.tv_usec,
					stat_structs[proc].ackd);
*/
		} 
		prev_stat_structs[proc] = stat_structs[proc];
	}

	rewind(stats);
	fprintf(stats, "clear!");
	fclose(stats);

	if (misc == NULL) {
		printf("couldn't open /proc/rpc_misc\n");
		exit(1);
	}

	if (fscanf(misc, "timeouts %u\n", &timeo) != 1) {
		printf("warning, couldn't read from /proc/rpc_misc\n");
		goto out;
	}
	if(timeo) {
		fprintf(ftimeo, "%u.%06u\t%u\n",
			t.tv_sec, t.tv_usec,
			timeo);
	}
	if (fscanf(misc, "cong %u\n", &cong) != 1) {
		printf("warning, couldn't read from /proc/rpc_misc\n");
		goto out;
	}
	fprintf(fcong, "%u.%06u\t%u\n",
		t.tv_sec, t.tv_usec,
		cong / 256);
	if (fscanf(misc, "cwnd %u\n", &cwnd) != 1) {
		printf("warning, couldn't read from /proc/rpc_misc\n");
		goto out;
	}
	fprintf(fcwnd, "%u.%06u\t%u\n",
		t.tv_sec, t.tv_usec,
		(unsigned int)((cwnd + 255)/ 256));
	if (fscanf(misc, "backlog %u\n", &backlog) != 1) {
		printf("warning, couldn't read from /proc/rpc_misc\n");
		goto out;
	}
	fprintf(fbacklog, "%u.%06u\t%u\n",
		t.tv_sec, t.tv_usec,
		backlog);
	if (fscanf(misc, "pending %u\n", &pending) != 1) {
		printf("warning, couldn't read from /proc/rpc_misc\n");
		goto out;
	}
	fprintf(fpending, "%u.%06u\t%u\n",
		t.tv_sec, t.tv_usec,
		pending);

out:
	rewind(misc);
	fprintf(misc, "clear!");
	fclose(misc);
}

int main(int argc, char *argv[], char *envp[])
{
	pid_t child;
	int status;
	int i = 0;
	char filename[100];
	FILE *stats = fopen("/proc/rpcstat", "w");
	int proc;

	/* mark all files as uninitialized */
	memset(initd, 0, sizeof(initd));

	/* initialize the special 'all' files */
	proc=22;
	init_file("new", proc, 0);
	init_file("fin", proc, 1);
	init_file("sent", proc, 2);
	init_file("ackd", proc, 3);
	initd[proc] = 1;

	/* open timeout file */
	if((ftimeo = fopen("timeo", "w")) == NULL) {
		printf("couldn't open output file\n");
		exit(1);
	}
	/* open cong file */
	if((fcong = fopen("cong", "w")) == NULL) {
		printf("couldn't open output file\n");
		exit(1);
	}
	/* open cwnd file */
	if((fcwnd = fopen("cwnd", "w")) == NULL) {
		printf("couldn't open output file\n");
		exit(1);
	}
	/* open backlog file */
	if((fbacklog = fopen("backlog", "w")) == NULL) {
		printf("couldn't open output file\n");
		exit(1);
	}
	/* open pending file */
	if((fpending = fopen("pending", "w")) == NULL) {
		printf("couldn't open output file\n");
		exit(1);
	}

	/* execute given command */
	child = fork();
	if(!child) {
		if (execve(argv[1], argv+1, envp)) 
			return 1;
	} 

	/* flush stats */
	if (stats == NULL) {
		printf("couldn't open rpcstat\n");
		exit(1);
	}
	fprintf(stats, "clear!");
	fclose(stats);

	if ((stats = fopen("/proc/rpcmisc", "w")) == NULL) {
		printf("couldn't open rpcmisc\n");
		exit(1);
	}
	fprintf(stats, "clear!");
	fclose(stats);

	/* initialize times */
	gettimeofday(&t0, NULL);
	gettimeofday(&lastt, NULL);

	/* do nothing until child exits.
	 * meanwhile, getstats is executing
	 * every given interval 
	 */
	while(!waitpid(child, &status,WNOHANG)) {
		getstats(0);
#ifdef NICE
		sched_yield();
#endif
	}

	printf("%s ended with status %d\n", argv[1], WEXITSTATUS(status));
}

void subtime(struct timeval *ans, struct timeval *t1, struct timeval *t2)
{
	ans->tv_sec = t1->tv_sec - t2->tv_sec;
	ans->tv_usec = t1->tv_usec - t2->tv_usec;

	if(ans->tv_usec < 0) {
		ans->tv_usec += 1e6;
		ans->tv_sec--;
	}
}

void init_file(char *suffix, int proc, int type)
{
	char filename[100];

	sprintf(filename, "%s.%s", procnames[proc], suffix);
	files[proc][type] = fopen(filename, "w");
	if (files[proc][type] == NULL) {
		printf("Couldn't open output file %s\n", filename);
		exit(1);
	}
}
