/* Intellimouse Wheel Thinger Utilities
 * VERSION 0.9.5
 * Development release
 * Copylefted under the GNU Public License
 * No code here may be used for profit without the permission of the author.
 * Author : Jonathan Atkins <jcatki@most.fw.hac.com> <jcatki@mysolution.com>
 * PLEASE: contact me if you have any improvements, I will gladly code good ones
 */
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <getopt.h>
#include <regex.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>
#define UTIL_C
#include "util.h"
#undef UTIL_C
#include "imwheel.h"

int buttons[NUM_BUTTONS]={Button4, Button5};
int statebits[STATE_MASK+1];
char *wname=NULL;
XClassHint xch={NULL,NULL};
struct WinAction *wa=NULL;
int num_wa=0;
#ifdef DEBUG
int debug=1;
#else
int debug=0;
#endif
struct Trans masktrans[MAX_MASKTRANS]=
{
	{"ShiftMask",	ShiftMask},
	{"LockMask",	LockMask},
	{"ControlMask",	ControlMask},
	{"Mod1Mask",	Mod1Mask}, //Alt
	{"Mod2Mask",	Mod2Mask},
	{"Mod3Mask",	Mod3Mask},
	{"Mod4Mask",	Mod4Mask},
	{"Mod5Mask",	Mod5Mask},
};
const int reps[1<<NUM_STATES]=
{
	1,			//None
	1,			//ShiftMask
	2,			//          ControlMask
	5,			//ShiftMask|ControlMask
	10,			//                      Mod1Mask
	1,			//ShiftMask|            Mod1Mask
	20,			//          ControlMask|Mod1Mask
	50			//ShiftMask|ControlMask|Mod1Mask
};
const char *keys[NUM_BUTTONS][1<<NUM_STATES]=
{
	{
		"Page_Up",		//None
		"Up",			//ShiftMask
		"Page_Up",		//          ControlMask
		"Page_Up",		//ShiftMask|ControlMask
		"Left",			//                      Mod1Mask
		"Left",			//ShiftMask|            Mod1Mask
		"Left",			//          ControlMask|Mod1Mask
		"Left"			//ShiftMask|ControlMask|Mod1Mask
	},
	{
		"Page_Down",	//None
		"Down",			//ShiftMask
		"Page_Down",	//          ControlMask
		"Page_Down",	//ShiftMask|ControlMask
		"Right",		//                      Mod1Mask
		"Right",		//ShiftMask|            Mod1Mask
		"Right",		//          ControlMask|Mod1Mask
		"Right"			//ShiftMask|ControlMask|Mod1Mask
	}
};

/*----------------------------------------------------------------------------*/

void setupstatebits()
{
	statebits[None                          ]=0;
	statebits[ShiftMask                     ]=1;
	statebits[          ControlMask         ]=2;
	statebits[ShiftMask|ControlMask         ]=3;
	statebits[                      Mod1Mask]=4;
	statebits[ShiftMask            |Mod1Mask]=5;
	statebits[          ControlMask|Mod1Mask]=6;
	statebits[ShiftMask|ControlMask|Mod1Mask]=7;
}

/*----------------------------------------------------------------------------*/

void exitParent(int num)
{
	exit(0);
}

/*----------------------------------------------------------------------------*/

void getOptions(int argc, char **argv, char *opts, const struct option *options)
{
	int ch,i,killold=False,restart=False;

	while((ch=getopt_long(argc,argv,opts,options,&i))>=0)
	{
		switch(ch)
		{
			case 0:
				printf("ch=%c\n\toption[%d]=\"%s\"\n",ch,i,options[i].name);
				if(optarg)
					printf("\toptarg=\"%s\"\n",optarg);
			case '4':
				buttonFlip=True;
				break;
			case 'W':
				useFifo=True;
				if(optarg)
				{
					fifoName=malloc(strlen(optarg)+1);
					strcpy(fifoName,optarg);
				}
				break;
			case 'd':
				detach=False;
				break;
			case 'p':
				pid_check=False;
				break;
			case 'D':
				debug=True;
				break;
			case 'q':
				quit=True;
				break;
			case 'k':
				killold=True;
				break;
			case 'R':
				restart=True;
				break;
			case 'h':
			case '?':
				printUsage(argv[0],options,optionusage);
				exit(0);
				break;
			default:
				printf("option -%c is unknown!\n",ch);
		}
	}
	if(!restart)
	{
		if(killold)
			KillIMWheel();
		if(quit)
			exit(0);
		if(detach)
		{
			if(fork())
			{
				signal(SIGUSR1,exitParent);
				wait(NULL);
				exit(0);
			}
		}
		else
			fprintf(stderr,"imwheel is not running as a daemon.\n");
		setsid();
		if(pid_check)
			WritePID();
		else
			fprintf(stderr,"imwheel is not checking/writing a pid file, BE CAREFUL!\nAn imwheel may be running already, two or more imwheel processes\nwill not operate as expected!\n");
		if(detach)
			kill(getppid(),SIGUSR1);
		fprintf(stderr,"imwheel started (pid=%d)\n",getpid());
	}
	if(useFifo)
		openFifo();
}

/*----------------------------------------------------------------------------*/

void WritePID()
{
	FILE *f;
	int pid;
	
	if((f=fopen(PIDFILE,"r")))
	{
		if(!fscanf(f,"%d",&pid))
			pid=-1;
		fclose(f);
		if(pid>0)
		{
			fprintf(stderr,"imwheel is already running or there is a stale pid file\ncheck on pid %d, or run with -k to kill running imwheel.\nOr clean up pid file: %s\n",pid,PIDFILE);
			exit(1);
		}
	}
	if((f=fopen(PIDFILE,"w")))
	{
		fprintf(f,"%d",getpid());
		fflush(f);
	}
	else
	{
		perror("Couldn't write pid to pid file: ");
		exit(1);
	}
}

/*----------------------------------------------------------------------------*/

void KillIMWheel()
{
	FILE *f;
	pid_t pid;

	if((f=fopen(PIDFILE,"r")))
	{
		if(!fscanf(f,"%d",&pid))
			pid=-1;
		fclose(f);
		if(pid>1)
		{
			if(killpg(pid,SIGTERM)<0)
				perror("imwheel not killed: ");
			else
				fprintf(stderr,"imwheel(pid=%d) killed.\n",pid);
		}
		if(unlink(PIDFILE)<0)
			perror("imwheel pid file could not be removed: ");
	}
	else
		fprintf(stderr,"imwheel is not running.  No process killed.\n");
}

/*----------------------------------------------------------------------------*/

void printKeymap(Display *d, char km[32])
{
	int i;
	
	Printf("Keymap status:\n");
	for(i=0;i<32*8;i++)
		if(getbit(km,i))
			Printf("\t[%d,%d] %d 0x%x \"%s\"\n",i/8,i%8,i,i,
				XKeysymToString(XKeycodeToKeysym(d,i,0)));
	Printf("\n");
}

/*----------------------------------------------------------------------------*/

void printXModifierKeymap(Display *d, XModifierKeymap *xmk)
{
	int i,j;

	Printf("XModifierKeymap:\n");
	for(j=0; j<8; j++)
		for(i=0; i<xmk->max_keypermod && xmk->modifiermap[(j*xmk->max_keypermod)+i]; i++)
			Printf("\t[%d,%02d] %d 0x%x \"%s\"\n",
				j,i,
				xmk->modifiermap[(j*xmk->max_keypermod)+i],
				xmk->modifiermap[(j*xmk->max_keypermod)+i],
				XKeysymToString(XKeycodeToKeysym(d,xmk->modifiermap[(j*xmk->max_keypermod)+i],0)));
}

/*----------------------------------------------------------------------------*/

void setbit(char *buf, int n, Bool val)
{
	Printf("setbit:n=%d,%d val=%d\n",n/8,n%8,val);
	if(val)
		buf[n/8]|=(1<<(n%8));
	else
		buf[n/8]&=0xFF^(1<<(n%8));
}
/*----------------------------------------------------------------------------*/

int getbit(char *buf, int n)
{
	if(buf[n/8]&(1<<(n%8)))
		Printf("getbit:n=%d,%d\n",n/8,n%8);
	return(buf[n/8]&(1<<(n%8)));
}

/*----------------------------------------------------------------------------*/

void printXEvent(XEvent *e)
{
	Printf("type=%d\n",e->type);
	Printf("serial    =%lu\n",e->xany.serial);
	Printf("send_event=%d\n",e->xany.send_event);
	Printf("display   =%p\n",e->xany.display);
	Printf("window    =0x%08x\n",(unsigned)e->xany.window);
	if(e->type&(ButtonRelease|ButtonPress))
	{
		Printf("root         =0x%08x\n",(unsigned)e->xbutton.root);
		Printf("subwindow    =0x%08x\n",(unsigned)e->xbutton.subwindow);
		Printf("time         =0x%08x\n",(unsigned)e->xbutton.time);
		Printf("     x,y     =%d,%d\n",e->xbutton.x,e->xbutton.y);
		Printf("x_root,y_root=%d,%d\n",e->xbutton.x_root,e->xbutton.y_root);
		Printf("state        =0x%x\n",(unsigned)e->xbutton.state);
		Printf("button       =0x%x\n",(unsigned)e->xbutton.button);
		Printf("same_screen  =0x%x\n",(unsigned)e->xbutton.same_screen);
		Printf("\n");
	}
}

/*----------------------------------------------------------------------------*/
/* Returns true if there is a window name
 */
char *windowName(Display *d, Window w)
{
	char *wname;
	
	Printf("0x%08x ",(unsigned)w);
	XFetchName(d,w,&wname);
	if(wname)
	{
		Printf("\"%s\"\n",wname);
		return(wname);
	}
	else
		Printf("(null)\n");
	return(NULL);
}

/*----------------------------------------------------------------------------*/

void printUsage(char *pname, const struct option options[], const char *usage[][2])
{
	int i,maxa=0,maxb=0,len;
	char str[80];

	printf("%s:",pname);
	for(i=0;options[i].name;i++)
	{
		len=0;
		if(options[i].val)
			len+=2;
		if(options[i].val && options[i].name)
			len++;
		if(options[i].name)
			len+=2+strlen(options[i].name);
		if(maxa<len)
			maxa=len;
		len=0;
		if(usage[i][0])
			len=strlen(usage[i][0]);
		if(maxb<len)
			maxb=len;
	}
		
	for(i=0;options[i].name;i++)
	{
		printf(" [");
		if(options[i].name)
			printf("--%s",options[i].name);
		if(options[i].val && options[i].name)
			printf("|");
		if(options[i].val)
			printf("-%c",options[i].val);
		if(usage[i][0])
			printf(" %s",usage[i][0]);
		printf("]");
	}
	printf("\n");
	for(i=0;options[i].name;i++)
	{
		*str=0;
		if(options[i].name)
			sprintf(str,"--%s",options[i].name);
		if(options[i].val && options[i].name)
			sprintf(str,"%s|",str);
		if(options[i].val)
			sprintf(str,"%s-%c",str,options[i].val);
		printf("%-*.*s",maxa,maxa,str);
		if(usage[i][0])
			printf(" %-*.*s",maxb,maxb,usage[i][0]);
		else
			printf(" %-*.*s",maxb,maxb,"");
		if(usage[i][1])
		{
			len=strlen(usage[i][1]);
			if(maxa+maxb+len+2>80)
				printf("\n%80.80s%s","",usage[i][1]);
			else
				printf(" %s",usage[i][1]);
		}
		printf("\n");
	}
}

/*----------------------------------------------------------------------------*/

void Printf(char *fmt, ...)
{
	va_list ap;

	if(debug)
	{
		va_start(ap,fmt);
		vprintf(fmt,ap);
		va_end(ap);
	}
}

/*----------------------------------------------------------------------------*/

void delay(unsigned long micros)
{
	/*struct timeval tv[2];*/
	
	usleep(micros);
	/*
	gettimeofday(&tv[0],NULL);
	do
		gettimeofday(&tv[1],NULL);
	while ((tv[1].tv_sec*1000000-tv[0].tv_sec*1000000)+(tv[1].tv_usec-tv[0].tv_usec)<micros);
	*/
}

/*----------------------------------------------------------------------------*/
//returns Modifier Index if found.  -1 if not.

int isMod(XModifierKeymap *xmk, int kc)
{
	int i,j;

	for(i=0;i<8;i++)
		for(j=0;j<xmk->max_keypermod;j++)
			if(xmk->modifiermap[(i*xmk->max_keypermod)+j]==kc)
				return(i);
	return(-1);
}

/*----------------------------------------------------------------------------*/

unsigned int makeModMask(XModifierKeymap *xmk, char km[32])
{
	int i,b;
	unsigned int mask=0;
	
	for(i=0;i<32*8;i++)
		if(getbit(km,i) && (b=isMod(xmk,i))>=0)
		{
			Printf("makeModMask: or to mask %d\n",(1<<b));
			mask|=(1<<b);
		}
	Printf("makeModMask: 0x%x\n",mask);
	return(mask);
}

/*----------------------------------------------------------------------------*/

void printfState(int state)
{
	int i;

	Printf("State Mask:\n");
	for(i=0;i<MAX_MASKTRANS;i++)
		if(state&masktrans[i].val)
			Printf("\t%s",masktrans[i].name);
	Printf("\n");
}

/*----------------------------------------------------------------------------*/

struct WinAction *addRC(struct WinAction *wa, struct WinAction *wap)
{
	struct WinAction *newwa;

	newwa=realloc(wa, sizeof(struct WinAction)*(num_wa+1));
	memcpy(&newwa[num_wa],wap,sizeof(struct WinAction));
	num_wa++;
	return(newwa);
}

/*----------------------------------------------------------------------------*/

struct WinAction *getRC()
{
	char fname[2][1024]={"","/etc/imwheelrc"}, line[1024], *p, *q, winid[1024];
	int fi,i;
	struct WinAction *newwa=NULL;
	FILE *f=NULL;
	char *home;

	home=getenv("HOME");
	if(home)
		sprintf(fname[0],"%s/.imwheelrc",home);
	num_wa=0;
	for(fi=(home?0:1);fi<2 && !f;fi++)
	{
		Printf("getRC:filename=\"%s\"\n",fname[fi]);
		f=fopen(fname[fi],"r");
	}
	if(!f)
		return(NULL);
	while(fgets(line, 1024, f))
	{
		Printf("getRC:pre :line:\n%s",line);
		q=strchr(line,'"');
		if(q)
		{
			p=strchr(line,'#');
			if(p && p<q)
				*p=0;
		}
		q=strrchr(line,'"');
		if(q)
		{
			p=strchr(q,'#');
			if(p)
				*p=0;
		}
		else
		{
			p=strchr(line,'#');
			if(p)
				*p=0;
		}
		while(strlen(line) && strchr(" \t\n",line[strlen(line)-1]))
			line[strlen(line)-1]=0;
		while(strlen(line) && strchr(" \t",line[0]))
			memmove(line,line+1,strlen(line+1)+1);
		if(!strlen(line))
			continue;
		if(line[0]=='"')
		{	//new window
			Printf("getRC:win:line:\"%s\"\n",line);

			if(!(p=strchr(line+1,'"')))
				exitString("Missing closing quote in config:\n%s\n",line);
			*p=0;
			strcpy(winid,line+1);
			Printf("id=\"%s\"\n",winid);
		}
		else
		{	//strip whitespace
			for(i=0;i<strlen(line);i++)
				while(strlen(line) && strchr(" \t",line[i]))
					memmove(line+i,line+i+1,strlen(line+i+1)+1);
			if(!strlen(line))
				continue;
			Printf("getRC:mod:line:\"%s\"\n",line);
			//new mod pod
			newwa=(struct WinAction*)realloc(newwa,sizeof(struct WinAction)*(num_wa+2));
			newwa[num_wa].id=(char*)malloc(strlen(line+1)+1);
			newwa[num_wa+1].id=NULL;
			strcpy(newwa[num_wa].id,winid);
			newwa[num_wa].reps=1;
			newwa[num_wa].delay=0;
			newwa[num_wa].delayup=0;
			num_wa++;
			//Get Command (if any)
			if(line[0]=='@')
			{
				if(!strcasecmp(line+1,"Exclude"))
					newwa[num_wa-1].button=UNGRAB;
				else
					exitString("Unrecognized command : \"%s\"\n",line+1);
				continue;
			}
			//Get Keysym mask
			p=strchr(line,',');
			if(p)
				*p=0;
			else
				exitString("expected 3 args, got 1, in config.\n%s\n",line);
			Printf("Keysym mask: \"%s\"\n",line);
			newwa[num_wa-1].in=getPipeArray(line);
			memmove(line,p+1,strlen(p+1)+1);
			//Get Button
			p=strchr(line,',');
			if(p)
				*p=0;
			else
				exitString("Expected 3 args, got 2, in config.\n%s\n",line);
			Printf("Button: \"%s\"\n",line);
			if(!strcasecmp(line,"Up"))
				newwa[num_wa-1].button=4;
			else
			{
				if(!strcasecmp(line,"Down"))
					newwa[num_wa-1].button=5;
				else
					exitString("Unrecognized wheel action in config.\n%s\n",line);
			}
			Printf("\t=%d\n",newwa[num_wa-1].button);
			memmove(line,p+1,strlen(p+1)+1);
			//Get Keysym Out
			p=strchr(line,',');
			if(p)
				*p=0;
			if(*line && strlen(line))
			{
				Printf("Keysyms Out: \"%s\"\n",line);
				newwa[num_wa-1].out=getPipeArray(line);
			}
			else
				exitString("Unrecognized or missing Keysym Outs (arg 3) in config.\n%s\n",line);
			if(p)
				memmove(line,p+1,strlen(p+1)+1);
			else
				continue;
			//Get Reps
			p=strchr(line,',');
			if(p)
				*p=0;
			if(*line && strlen(line))
			{
				Printf("Reps: \"%s\"\n",line);
				newwa[num_wa-1].reps=strtol(line,NULL,10);
			}
			if(p)
				memmove(line,p+1,strlen(p+1)+1);
			else
				continue;
			//Get Delay
			p=strchr(line,',');
			if(p)
				*p=0;
			if(*line && strlen(line))
			{
				Printf("Delay: \"%s\"\n",line);
				newwa[num_wa-1].delay=strtol(line,NULL,10);
			}
			if(p)
				memmove(line,p+1,strlen(p+1)+1);
			else
				continue;
			//Get Delay Up
			if(*line && strlen(line))
			{
				Printf("Delay Up: \"%s\"\n",line);
				newwa[num_wa-1].delayup=strtol(line,NULL,10);
			}
		}
	}
	fclose(f);
	//sorting removes order
	//qsort(newwa,num_wa,sizeof(struct WinAction),wacmp);
	for(i=0;newwa && newwa[i].id;i++)
		printWA(&newwa[i]);
	//writeRC(newwa); //this was for debugging rc file writing
	return(newwa);
}

/*----------------------------------------------------------------------------*/

int wacmp(const void *a, const void *b)
{
	return(strcmp(((struct WinAction*)b)->id,((struct WinAction*)a)->id));
}

/*----------------------------------------------------------------------------*/

char **getPipeArray(char *str)
{
	int i;
	char *s,**a=NULL,*t;

	if(!strlen(str))
	{
		a=malloc(sizeof(char*));
		a[0]=NULL;
		return(a);
	}
	for(i=0,s=str; s; i++)
	{
		a=(char**)realloc(a,sizeof(char*)*(i+1));
		t=strsep(&s, "|");
		a[i]=malloc(strlen(t)+1);
		strcpy(a[i],t);
		Printf("%d) \"%s\" \"%s\"\n",i,a[i],s);
	}
	a=(char**)realloc(a,sizeof(char*)*(i+1));
	a[i]=NULL;
	return(a);
}

/*----------------------------------------------------------------------------*/

void exitString(char *format, char *arg)
{
	fprintf(stderr,format,arg);
	exit(1);
}

/*----------------------------------------------------------------------------*/

void printWA(struct WinAction *wa)
{
	int i;

	if(!wa)
		return;
	Printf("WinAction (%p):\n",wa);
	Printf("\tWindow Regex     : \"%s\"\n",wa->id);
	if(wa->button<6)
	{
		Printf("\tKeysyms Mask (%p):\n",wa->in);
		for(i=0; wa->in[i]; i++)
			Printf("\t\t\"%s\"\n",wa->in[i]);
		Printf("\tButton           : %d\n",wa->button);
		Printf("\tKeysyms Out (%p) :\n",wa->out);
		for(i=0; wa->out[i]; i++)
			Printf("\t\t\"%s\"\n",wa->out[i]);
		Printf("\tReps: %d\n",wa->reps);
		Printf("\tRep Delay: %d\n",wa->delay);
		Printf("\tKey Up Delay: %d\n",wa->delayup);
	}
	else
		switch(wa->button)
		{
			case UNGRAB:
				Printf("Command: Exclude (UNGRAB)\n");
				break;
		}
}

/*----------------------------------------------------------------------------*/

void writeRC(struct WinAction *wa)
{
	int i;
	struct WinAction *p,*cur;
	FILE *f=stdout;
	char fname[1024];
	char *home;
	
	if(!wa)
		return;
	home=getenv("HOME");
	if(home)
		sprintf(fname,"%s/.imwheelrc",home);
	else
	{
		perror("imwheel:writeRC: ");
		return;
	}
	if(!(f=fopen(fname,"w")))
	{
		perror("imwheel:writeRC: ");
		return;
	}
	fprintf(f,"# IMWheel Configuration file (%s)\n# (C)Jon Atkins <jcatki@mysolution.com>\n#\n# Generated by imwheel\n# Any extra comments will be lost on reconfiguration\n# However order will be maintained\n# Order!  ORDER, I SAY!!\n",fname);
	for(cur=NULL,p=wa;p->id;p=&p[1])
	{
		if(!cur || strcmp(cur->id,p->id))
		{
			fprintf(f,"\n\"%s\"\n",p->id);
			cur=p;
		}
		if(p->button<6)
		{
			for(i=0; p->in[i]; i++)
				fprintf(f,"%s%s",(i?"|":""),p->in[i]);
			fprintf(f,",\t%s,\t",(p->button==4?"Up":"Down"));
			for(i=0; p->out[i]; i++)
				fprintf(f,"%s%s",(i?"|":""),p->out[i]);
			if(p->delayup>0||p->delay>0||p->reps>1)
				fprintf(f,",\t%d",p->reps);
			if(p->delayup>0||p->delay>0)
				fprintf(f,",\t%d",p->delay);
			if(p->delayup>0)
				fprintf(f,",\t%d",p->delayup);
			fprintf(f,"\n");
		}
		else
		{
			switch(p->button)
			{
				case UNGRAB:
					fprintf(f,"@Exclude\n");
					break;
			}
		}
	}
	fclose(f);
}

/*----------------------------------------------------------------------------*/
char *regerrorstr(int err, regex_t *preg)
{
	static char *str=NULL;
	int len;

	len=regerror(err,preg,NULL,0);
	str=realloc(str,len);
	regerror(err,preg,str,len);
	return(str);
}

/*----------------------------------------------------------------------------*/

int regex(char *expression, char *str)
{
	regex_t preg;
	regmatch_t pmatch;
	int err;

	if(!str || !expression)
		return(0);
	Printf("Testing: \"%s\" ?= \"%s\"\n",expression,str);
	if((err=regcomp(&preg,expression,REG_EXTENDED|REG_NOSUB)) && err!=REG_NOMATCH)
	{
		fprintf(stderr,"RegComp Error On Expression:\n\"%s\"\n%s\nRemoving the offending expression...\n",expression,regerrorstr(err,&preg));
		expression[0]=0;
		regfree(&preg);
		return(1);
	}
	if((err=regexec(&preg,str,1,&pmatch,0)) && err!=REG_NOMATCH)
	{
		fprintf(stderr,"RegExec Error On Expression:\n\"%s\"\n%s\n",expression,regerrorstr(err,&preg));
		regfree(&preg);
		return(2);
	}
	regfree(&preg);
	if(err!=REG_NOMATCH)
		Printf("Match found:\"%s\" = \"%s\"\n",expression,str);
	return(err!=REG_NOMATCH);
}

/*----------------------------------------------------------------------------*/
struct WinAction *findWA(Display		 *d,
						 int			  button   ,
						 char			 *winname  ,
						 char			 *resname  ,
						 char			 *classname,
						 XModifierKeymap *xmk      ,
						 char			  KM[32]   )
{
	int i,j,ok,kc;
	struct WinAction *wap;
	char km[32];

	if(!wa)
		return(NULL);
	wap=NULL;
	for(i=0; !wap && wa[i].id; i++)
	{
		// skip removed or blank id's
		if(wa[i].id[0]==0)
			continue;
		if(button != wa[i].button)
			continue;
		if(regex(wa[i].id,winname  ) ||
		   regex(wa[i].id,resname  ) ||
		   regex(wa[i].id,classname) )
			wap=&wa[i];
		if(!KM || !xmk || wa[i].button>5)
			continue;
		ok=1;
		memcpy(km,KM,32);
		if(wa[i].in[0] && wa[i].in[0][0])
		{
			if(strcasecmp(wa[i].in[0],"None"))
				for(j=0; ok && wa[i].in[j]; j++)
				{
					if(!wa[i].in[j][0])
						continue;
					kc=XKeysymToKeycode(d, XStringToKeysym(wa[i].in[j]));
					ok=getbit(km,kc);
					setbit(km,kc,0);
				}
			for(kc=0;ok && kc<32*8;kc++)
				if(isMod(xmk,kc)>-1)
					ok=!getbit(km,kc);
		}
		if(!ok)
			wap=NULL;
	}
	if(!wap)
		Printf("No Match for : win=\"%s\" res=\"%s\" class=\"%s\"\n",winname,resname,classname);
	return(wap);
}
/*----------------------------------------------------------------------------*/
void doWA(Display          *d     ,
		  XButtonEvent     *e     ,
		  XModifierKeymap  *xmk   ,
		  char              km[32],
		  struct WinAction *wap   )
{
	int i,kc,rep;
	
	Printf("doWA:\n");
	printWA(wap);
	
	for(rep=0;rep<wap->reps;rep++)
	{
		Printf("rep=%d\n",rep);
		//XSetInputFocus(d,e->subwindow,RevertToParent,CurrentTime);
		modMods(d,km,xmk,False,0);
		for(i=0; wap->out[i]; i++)
		{
			kc=XKeysymToKeycode(d, XStringToKeysym(wap->out[i]));
			XTestFakeKeyEvent(d,kc,True,0);
		}
		Printf("doWA: keyup delay=%d\n",wap->delayup);
		delay(wap->delayup);
		for(i--;i>=0; i--)
		{
			kc=XKeysymToKeycode(d, XStringToKeysym(wap->out[i]));
			XTestFakeKeyEvent(d,kc,False,0);
		}
		XSync(d,False);
		modMods(d,km,xmk,True,0);
		XSync(d,False);
		Printf("doWA: nextkey delay=%d\n",wap->delay);
		delay(wap->delay);
	}
}
/*----------------------------------------------------------------------------*/

void modMods(Display *d, char *km, XModifierKeymap *xmk, Bool on, int delay_time)
{
	int i;

	for(i=0;i<32*8;i++)
		if(getbit(km,i) && isMod(xmk,i)>-1)
		{
			Printf("XTestFakeKeyEvent 0x%x=%s delay=%d\n",i,(on?"On":"Off"),delay_time);
			XTestFakeKeyEvent(d,i,on,delay_time);
		}
}

/*----------------------------------------------------------------------------*/

void flushFifo()
{
	char junk;
	
	closeFifo();
	if((fifofd=open(fifoName, O_RDWR|O_NONBLOCK))<0)
	{
		perror("Open GPM Wheel Fifo:");
		exit(1);
	}
	errno=0;
	while(read(fifofd,&junk,1) && !errno)
		Printf("Flushed : 0x%02x\n",junk);
	close(fifofd);
}

/*----------------------------------------------------------------------------*/

void closeFifo()
{
	if(fifofd<0)
		return;
	close(fifofd);
	fifofd=-1;
}

/*----------------------------------------------------------------------------*/

void openFifo() //also will close,flush, and reopen if fifo is already open!
{
	// preflush, or else!
	flushFifo();
	// open for real now!
	if((fifofd=open(fifoName, O_RDONLY))<0)
	{
		perror("Open GPM Wheel Fifo:");
		exit(1);
	}
}

/*----------------------------------------------------------------------------*/
/* vim:ts=4:shiftwidth=4
"*/
