
/*
 * mice.c - mouse definitions for gpm-Linux
 *
 * Copyright 1993        ajh@gec-mrc.co.uk (Andrew Haylett)
 * Copyright 1994-1998   rubini@linux.it (Alessandro Rubini)
 * Copyright (C) 1998    Ian Zimmerman <itz@rahul.net>
 *
 * Modified on 9/8/1998 by Jonathan Atkins for use with wheel mice
 *                         <jcatki@most.fw.hac.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

static char rcsid[] = "$Id: mice.c,v 1.8 1998/09/29 20:11:03 itz Exp $";

/*
 * This file is part of the mouse server. The information herein
 * is kept aside from the rest of the server to ease fixing mouse-type
 * issues. Each mouse type is expected to fill the `buttons', `dx' and `dy'
 * fields of the Gpm_Event structure and nothing more.
 *
 * Absolute-pointing devices (support by Marc Meis), are expecting to
 * fit `x' and `y' as well. Unfortunately, to do it the window size must
 * be accessed. The global variable "win" is available for that use.
 *
 * The `data' parameter points to a byte-array with event data, as read
 * by the mouse device. The mouse device should return a fixed number of
 * bytes to signal an event, and that exact number is read by the server
 * before calling one of these functions.
 *
 * The conversion function defined here should return 0 on success and -1
 * on failure.
 *
 * Refer to the definition of Gpm_Type to probe further.
 */

#include <termios.h>
#include <fcntl.h>
#include <termios.h>
#include <errno.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h> /* stat() */
#include <sys/time.h> /* select() */

#include <linux/fs.h> /* MAJOR */
#include <linux/keyboard.h>

#ifdef HAVE_LINUX_JOYSTICK_H
#include <linux/joystick.h>
#endif


#include "gpmInt.h"
#include "twiddler.h"

/*========================================================================*/
/*
 * Ok, here we are: first, provide the functions.
 * The return value is the number of unprocessed bytes
 */

static int M_ms(Gpm_Event *state,  unsigned char *data)
{
  /*
   * some devices report a change of middle-button state by
   * repeating the current button state  (patch by Mark Lord)
   */
  static unsigned char prev=0;

  if (data[0] == 0x40 && !(prev|data[1]|data[2]))
    state->buttons = GPM_B_MIDDLE; /* third button on MS compatible mouse */
  else
    state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  prev = state->buttons;
  state->dx=      (signed char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (signed char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));

  return 0;
}

static int M_ms_plus(Gpm_Event *state, unsigned char *data)
{
  static unsigned char prev=0;

  state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  state->dx=      (signed char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (signed char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));

  /* Allow motion *and* button change (Michael Plass) */

  if ((state->dx==0) && (state->dy==0) 
      && (state->buttons == (prev&~GPM_B_MIDDLE)))
    state->buttons = prev^GPM_B_MIDDLE;  /* no move or change: toggle middle */
  else
    state->buttons |= prev&GPM_B_MIDDLE;    /* change: preserve middle */

  prev=state->buttons;

  return 0;
}

static int M_ms_plus_lr(Gpm_Event *state,  unsigned char *data)
{
  /*
   * Same as M_ms_plus but with an addition by Edmund GRIMLEY EVANS
   */
  static unsigned char prev=0;

  state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  state->dx=      (signed char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (signed char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));

  /* Allow motion *and* button change (Michael Plass) */

  if ((state->dx==0) && (state->dy==0) 
      && (state->buttons == (prev&~GPM_B_MIDDLE)))
    state->buttons = prev^GPM_B_MIDDLE;  /* no move or change: toggle middle */
  else
    state->buttons |= prev&GPM_B_MIDDLE;    /* change: preserve middle */

  /* Allow the user to reset state of middle button by pressing
     the other two buttons at once (Edmund GRIMLEY EVANS) */

  if (!((~state->buttons)&(GPM_B_LEFT|GPM_B_RIGHT)) &&
      ((~prev)&(GPM_B_LEFT|GPM_B_RIGHT)))
    state->buttons &= ~GPM_B_MIDDLE;

  prev=state->buttons;

  return 0;
}

/* 'Genitizer' (kw@dtek.chalmers.se 11/12/97) */
static int M_geni(Gpm_Event *state,  unsigned char *data)
{
  /* this is a little confusing. If we use the stylus, we
   * have three buttons (tip, lower, upper), and if
   * we use the puck we have four buttons. (And the
   * protocol i a little mangled if several of the buttons
   * on the puck are pressed simultaneously. 
   * I don't use the puck, hence I try to decode three buttons
   * only. tip = left, lower = middle, upper = right
   */
  state->buttons = 
    (data[0] & 0x01)<<2 | 
    (data[0] & 0x02) |
    (data[0] & 0x04)>>2;

  state->dx = ((data[1] & 0x3f) ) * ((data[0] & 0x10)?1:-1);   
  state->dy = ((data[2] & 0x3f) ) * ((data[0] & 0x8)?-1:1);

  return 0;
}

/* m$ 'Intellimouse' (steveb 20/7/97) */
static int M_ms3(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= ((data[0] & 0x20) >> 3)	/* left */
		| ((data[3] & 0x10) >> 3)	/* middle */
		| ((data[0] & 0x10) >> 4);	/* right */
  state->dx=      (signed char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (signed char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
  /* wheel (dz??) is (data[3] & 0x0f) */

  return 0;
}

/* M_brw is a variant of m$ 'Intellimouse' the middle button is different */
static int M_brw(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= ((data[0] & 0x20) >> 3)	/* left */
		| ((data[3] & 0x20) >> 4)	/* middle */
		| ((data[0] & 0x10) >> 4);	/* right */
  state->dx=      (signed char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (signed char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
  if (((data[0]&0xC0) != 0x40)||
      ((data[1]&0xC0) != 0x00)||
      ((data[2]&0xC0) != 0x00)||
      ((data[3]&0xC0) != 0x00)) {
	  gpm_debug_log(LOG_DEBUG, "Skipping a data packet: Data %02x %02x %02x %02x", 
			data[0], data[1], data[2], data[3]);
	  return -1;
  }
  /* wheel (dz) is (data[3] & 0x0f) */
  /* where is the side button? I can sort of detect it at 9600 baud */
  /* Note this mouse is very noisy */

  return 0;
}

static int M_bare(Gpm_Event *state,  unsigned char *data)
{
  /* a bare ms protocol */
  state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  state->dx=      (signed char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (signed char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
  return 0;
}

static int M_sun(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (signed char)(data[1]);
  state->dy=     -(signed char)(data[2]);
  return 0;
}

static int M_msc(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (signed char)(data[1]) + (signed char)(data[3]);
  state->dy=     -((signed char)(data[2]) + (signed char)(data[4]));
  return 0;
}

static int M_logimsc(Gpm_Event *state,  unsigned char *data) /* same as msc */
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (signed char)(data[1]) + (signed char)(data[3]);
  state->dy=     -((signed char)(data[2]) + (signed char)(data[4]));
  return 0;
}

static int M_mm(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= data[0] & 0x07;
  state->dx=      (data[0] & 0x10) ?   data[1] : - data[1];
  state->dy=      (data[0] & 0x08) ? - data[2] :   data[2];
  return 0;
}

static int M_logi(Gpm_Event *state,  unsigned char *data) /* equal to mm */
{
  state->buttons= data[0] & 0x07;
  state->dx=      (data[0] & 0x10) ?   data[1] : - data[1];
  state->dy=      (data[0] & 0x08) ? - data[2] :   data[2];
  return 0;
}

static int M_bm(Gpm_Event *state,  unsigned char *data) /* equal to sun */
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (signed char)data[1];
  state->dy=     -(signed char)data[2];
  return 0;
}

static int M_ps2(Gpm_Event *state,  unsigned char *data)
{
  static int tap_active=0; /* there exist glidepoint ps2 mice */

  state->buttons=
    !!(data[0]&1) * GPM_B_LEFT  |
    !!(data[0]&2) * GPM_B_RIGHT |
    !!(data[0]&4) * GPM_B_MIDDLE|
    ((data[3]==0xff)?GPM_B_UP:0) |
    ((data[3]==0x01)?GPM_B_DOWN:0);
#if 0
  gpm_debug_log(LOG_DEBUG,"Buttons: %d %d %d %d %d\n",
		(state->buttons&GPM_B_LEFT),
		(state->buttons&GPM_B_MIDDLE),
		(state->buttons&GPM_B_RIGHT),
		(state->buttons&GPM_B_UP),
		(state->buttons&GPM_B_DOWN));
  gpm_debug_log(LOG_DEBUG,"wheel? %d\n",opt_wheel);
#endif
  if (data[0]==0 && opt_glidepoint_tap) /* by default this is false */
    state->buttons = tap_active = opt_glidepoint_tap;
  else if (tap_active)
    if (data[0]==8 && !opt_wheel)
      state->buttons = tap_active = 0;
    else state->buttons = tap_active;

 /* Some PS/2 mice send reports with negative bit set in data[0]
  * and zero for movement.  I think this is a bug in the mouse, but
  * working around it only causes artifacts when the actual report is -256;
  * they'll be treated as zero. This should be rare if the mouse sampling
  * rate is set to a reasonable value; the default of 100 Hz is plenty.
  * (Stephen Tell)
  */
  if(data[1] != 0)
    state->dx=   (data[0] & 0x10) ? data[1]-256 : data[1];
  else
    state->dx = 0;
  if(data[2] != 0)
    state->dy= -((data[0] & 0x20) ? data[2]-256 : data[2]);
  else
    state->dy = 0;
  return 0;
}

static int M_netmouse(Gpm_Event *state,  unsigned char *data)
{
  /* Avoid this beasts if you can.  They connect to normal PS/2 port,
     but their protocol is one byte longer... So if you have notebook
     (like me) with internal PS/2 mouse, it will not work
     together. They have four buttons, but two middle buttons can not
     be pressed simultaneously, and two middle buttons do not send
     'up' events (however, they autorepeat...)

     Still, you might want to run this mouse in plain PS/2 mode -
     where it behaves correctly except that middle 2 buttons do
     nothing.

     Protocol is
        3 bytes like normal PS/2
	4th byte: 0xff button 'down', 0x01 button 'up'
	   [this is so braindamaged that it *must* be some kind of
	   compatibility glue...]

	                             Pavel Machek <pavel@ucw.cz>
  */

  state->buttons=
    !!(data[0]&1) * GPM_B_LEFT +
    !!(data[0]&2) * GPM_B_RIGHT +
    !!(data[3]) * GPM_B_MIDDLE;

  if(data[1] != 0)
    state->dx=   (data[0] & 0x10) ? data[1]-256 : data[1];
  else
    state->dx = 0;
  if(data[2] != 0)
    state->dy= -((data[0] & 0x20) ? data[2]-256 : data[2]);
  else
    state->dy = 0;
  return 0;
}

static Gpm_Type *I_netmouse(int fd, unsigned short flags, struct Gpm_Type *type)
{
  unsigned char magic[6] = { 0xe8, 0x03, 0xe6, 0xe6, 0xe6, 0xe9 };
  int i;
  for (i=0; i<6; i++) {
    unsigned char c = 0;
    write( fd, magic+i, 1 );
    read( fd, &c, 1 );
    if (c != 0xfa) {
      fprintf( stderr, "netmouse: No acknowledge (got %d)\n", c );
      return NULL;
    }
  }
  {
    unsigned char rep[3] = { 0, 0, 0 };
    read( fd, rep, 1 );
    read( fd, rep+1, 1 );
    read( fd, rep+2, 1 );
    if (rep[0] || (rep[1] != 0x33) || (rep[2] != 0x55)) {
      fprintf( stderr, "netmouse: Invalid reply magic (got %d,%d,%d)\n", rep[0], rep[1], rep[2] );
      return NULL;
    }
  }
  return type;
}

#define GPM_B_BOTH (GPM_B_LEFT|GPM_B_RIGHT)
static int M_mman(Gpm_Event *state,  unsigned char *data)
{
  /*
   * the damned MouseMan has 3/4 bytes packets. The extra byte 
   * is only there if the middle button is active.
   * I get the extra byte as a packet with magic numbers in it.
   * and then switch to 4-byte mode.
   */
  static unsigned char prev=0;
  static Gpm_Type *mytype=mice; /* it is the first */
  unsigned char b;

  if (data[1]==GPM_EXTRA_MAGIC_1 && data[2]==GPM_EXTRA_MAGIC_2)
    {
    /* got unexpected fourth byte */
    gpm_debug_log(LOG_NOTICE,"Extra byte = %02x",*data);
    if ((b=(*data>>4)) > 0x3) return -1;  /* just a sanity check */
    state->dx=state->dy=0;
    
    mytype->packetlen=4;
    mytype->getextra=0;
    }
  else
    {
    /* got 3/4, as expected */
    
    /* motion is independent of packetlen... */
    state->dx= (signed char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
    state->dy= (signed char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
    
    prev= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
    if (mytype->packetlen==4) b=data[3]>>4;
    }
	  
  if(mytype->packetlen==4) 
    {
    if(b == 0) 
      {
      mytype->packetlen=3;
      mytype->getextra=1;
      }
    else
      {
      if (b & 0x2) prev |= GPM_B_MIDDLE;
      if (b & 0x1) prev |= opt_glidepoint_tap;
      }
    }
  state->buttons=prev;

  /* This "chord-middle" behaviour was reported by David A. van Leeuwen */
  if ( ((prev^state->buttons) & GPM_B_BOTH)==GPM_B_BOTH )
    state->buttons = state->buttons ? GPM_B_MIDDLE : 0;
  prev=state->buttons;
  
  return 0;
}

/* wacom tablet, in native mode */
static int M_wacom(Gpm_Event *state, unsigned char *data)
{
  static int ox=-1, oy, threshold=8;
  int x, y, z;

  if (data[0]==0xa0) /* leave -- but sometimes it's not decoded */
    { ox=-1; state->buttons=0; state->dx=state->dy=0; return 0;}

  
  x = (data[1]<<7) | (data[2]&0x7f);
  y = (data[4]<<7) | (data[5]&0x7f);
  z = data[6]-64 + 128*(data[6]<64);
  if (ox==-1 || abs(x-ox)>18 || abs(y-oy)>18) { ox=x; oy=y;} /* enter */

  state->buttons = (GPM_B_MIDDLE * !!(data[3]&0x10)); /* the b. means middle */
  if (!state->buttons)
    if (data[3]&0x20) state->buttons = (z>threshold) * GPM_B_RIGHT;
    else              state->buttons = (z>threshold) * GPM_B_LEFT;

  if (z>threshold) threshold=1;
  else threshold=8;

  state->dx = (x-ox)/5; 
  state->dy = (y-oy)/5; /* divide by 5, 'cause it's too much sensitive */

  ox=x; oy=y;
  return 0;
}

/* ncr pen support (Marc Meis) */

#define NCR_LEFT_X     40
#define NCR_RIGHT_X    2000

#define NCR_BOTTOM_Y   25
#define NCR_TOP_Y      1490

#define NCR_DELTA_X    (NCR_RIGHT_X - NCR_LEFT_X)
#define NCR_DELTA_Y    (NCR_TOP_Y - NCR_BOTTOM_Y)

static int M_ncr(Gpm_Event *state,  unsigned char *data)
{
  int x,y;

  state->buttons= (data[0]&1)*GPM_B_LEFT +
                !!(data[0]&2)*GPM_B_RIGHT;

  state->dx = (signed char)data[1]; /* currently unused */
  state->dy = (signed char)data[2];

  x = ((int)data[3] << 8) + (int)data[4];
  y = ((int)data[5] << 8) + (int)data[6];

  /* these formulaes may look curious, but this is the way it works!!! */

  state->x = x < NCR_LEFT_X
             ? 0
             : x > NCR_RIGHT_X
               ? win.ws_col+1
               : (long)(x-NCR_LEFT_X) * (long)(win.ws_col-1) / NCR_DELTA_X+2;

  state->y = y < NCR_BOTTOM_Y
             ? win.ws_row + 1
             : y > NCR_TOP_Y
	       ? 0
	       : (long)(NCR_TOP_Y-y) * (long)win.ws_row / NCR_DELTA_Y + 1;

  return 0;
}

static int M_twid(Gpm_Event *state,  unsigned char *data)
{
  unsigned long message=0UL; int i,h,v;
  static int lasth, lastv, lastkey, key, lock=0, autorepeat=0;

  /* build the message as a single number */
  for (i=0; i<5; i++)
    message |= (data[i]&0x7f)<<(i*7);
  key = message & TW_ANY_KEY;

  if ((message & TW_MOD_M) == 0) /* manage keyboard */
    {
    if (((message & TW_ANY_KEY) != lastkey) || autorepeat)
      autorepeat = twiddler_key(message);
    lastkey = key;
    lock = 0; return -1; /* no useful mouse data */
    } 

  switch (message & TW_ANY1)
    {
    case TW_L1: state->buttons = GPM_B_RIGHT;   break;
    case TW_M1: state->buttons = GPM_B_MIDDLE; break;
    case TW_R1: state->buttons = GPM_B_LEFT;  break;
    case     0: state->buttons = 0;            break;
    }
  /* also, allow R1 R2 R3 (or L1 L2 L3) to be used as mouse buttons */
  if (message & TW_ANY2)
	  state->buttons |= GPM_B_MIDDLE;
  if (message & TW_L3)
	  state->buttons |= GPM_B_LEFT;
  if (message & TW_R3)
	  state->buttons |= GPM_B_RIGHT;

  /* put in modifiers information */
  {
  struct {unsigned long in, out;} *ptr, list[] = {
      { TW_MOD_S,  1<<KG_SHIFT },
      { TW_MOD_C,  1<<KG_CTRL  },
      { TW_MOD_A,  1<<KG_ALT   },
      { 0,         0}
  };
  for (ptr = list; ptr->in; ptr++)
    if  (message & ptr->in)
      state->modifiers |= ptr->out;
  }

  /* now extraxt H/V */
  h = (message >> TW_H_SHIFT) & TW_M_MASK;
  v = (message >> TW_V_SHIFT) & TW_M_MASK;
  if (h & TW_M_BIT) h = -(TW_M_MASK + 1 - h);
  if (v & TW_M_BIT) v = -(TW_M_MASK + 1 - v);
  
#ifdef TWIDDLER_STATIC
  /* static implementation: return movement */
  if (!lock)
    { lasth = h; lastv = v; lock = 1; } 
  state->dx = -(h-lasth); lasth = h;
  state->dy = -(v-lastv); lastv = v; 

#elif defined(TWIDDLER_BALLISTIC)
  {
  /* in case I'll change the resolution */
  static int tw_threshold = 5;  /* above this it moves */
  static int tw_scale = 5;      /* every 5 report one */

  if (h > -tw_threshold && h < tw_threshold)
    state->dx=0;
  else 
    {
    h = h - (h<0)*tw_threshold +lasth;
    lasth = h%tw_scale;
    state->dx = -(h/tw_scale);
    }
  if (v > -tw_threshold && v < tw_threshold)
    state->dy=0;
  else
    {
    v = v - (v<0)*tw_threshold +lastv;
    lastv = v%tw_scale;
    state->dy = -(v/tw_scale);
    }
  }

#else /* none defined: use mixed approach */
  {
  /* in case I'll change the resolution */
  static int tw_threshold = 60;    /* above this, movement is ballistic */
  static int tw_scale = 10;         /* ball: every 6 units move one unit */
  static int tw_static_scale = 3;  /* stat: every 3 units move one unit */
  static int lasthrest, lastvrest; /* integral of small motions uses rest */

  if (!lock)
    {
    lasth = h; lasthrest = 0;
    lastv = v; lastvrest = 0;
    lock = 1;
    } 

  if (h > -tw_threshold && h < tw_threshold)
    {
    state->dx = -(h-lasth+lasthrest)/tw_static_scale;
    lasthrest =  (h-lasth+lasthrest)%tw_static_scale;
    }
  else /* ballistic */
    {
    h = h - (h<0)*tw_threshold + lasthrest;
    lasthrest = h%tw_scale;
    state->dx = -(h/tw_scale);
    }
  lasth = h;
  if (v > -tw_threshold && v < tw_threshold)
    {
    state->dy = -(v-lastv+lastvrest)/tw_static_scale;
    lastvrest =  (v-lastv+lastvrest)%tw_static_scale;
    }
  else /* ballistic */
    {
    v = v - (v<0)*tw_threshold + lastvrest;
    lastvrest = v%tw_scale;
    state->dy = -(v/tw_scale);
    }
  lastv = v;
  }
#endif

  /* fprintf(stderr,"%4i %4i -> %3i %3i\n",h,v,state->dx,state->dy); */
  return 0;
}

#ifdef HAVE_LINUX_JOYSTICK_H
/* Joystick mouse emulation (David Given) */

static int M_js(Gpm_Event *state,  unsigned char *data)
{
  struct JS_DATA_TYPE *jdata = (void*)data;
  static int centerx = 0;
  static int centery = 0;
  static int oldbuttons = 0;
  static int count = 0;
  int dx;
  int dy;

  count++;
  if (count < 200)
  {
    state->buttons = oldbuttons;
    state->dx = 0;
    state->dy = 0;
    return 0;
  }
  count = 0;

  if (centerx == 0)
  {
    centerx = jdata->x;
    centery = jdata->y;
  }

  state->buttons = ((jdata->buttons & 1) * GPM_B_LEFT) |
                   ((jdata->buttons & 2) * GPM_B_RIGHT);
  oldbuttons     = state->buttons;

  dx             = (jdata->x - centerx) >> 6;
  dy             = (jdata->y - centery) >> 6;

  if (dx > 0)
    state->dx = dx * dx;
  else
    state->dx = -(dx * dx);
  state->dx >>= 2;

  if (dy > 0)
    state->dy = dy * dy;
  else
    state->dy = -(dy * dy);
   state->dy >>= 2;

  /* Prevent pointer drift. (PC joysticks are notoriously inaccurate.) */

  if ((state->dx >= -1) && (state->dx <= 1))
    state->dx = 0;

  if ((state->dy >= -1) && (state->dy <= 1))
    state->dy = 0;

  return 0;
}
#endif /* have joystick.h */


/*========================================================================*/
/* Then, mice should be initialized */

static int
setspeed(int fd, int old, int new, int needtowrite, unsigned short flags)
{
  struct termios tty;
  char *c;

  tcgetattr(fd, &tty);
  
  tty.c_iflag = IGNBRK | IGNPAR;
  tty.c_oflag = 0;
  tty.c_lflag = 0;
  tty.c_line = 0;
  tty.c_cc[VTIME] = 0;
  tty.c_cc[VMIN] = 1;

  switch (old)
    {
    case 9600:	tty.c_cflag = flags | B9600; break;
    case 4800:	tty.c_cflag = flags | B4800; break;
    case 2400:	tty.c_cflag = flags | B2400; break;
    case 1200:
    default:	tty.c_cflag = flags | B1200; break;
    }

  tcsetattr(fd, TCSAFLUSH, &tty);

  switch (new)
    {
    case 9600:	c = "*q";  tty.c_cflag = flags | B9600; break;
    case 4800:	c = "*p";  tty.c_cflag = flags | B4800; break;
    case 2400:	c = "*o";  tty.c_cflag = flags | B2400; break;
    case 1200:
    default:	c = "*n";  tty.c_cflag = flags | B1200; break;
    }

  if (needtowrite) write(fd, c, 2);
  usleep(100000);
  tcsetattr(fd, TCSAFLUSH, &tty);
  return 0;
}



static struct {
    int sample; char code[2];
    }
  sampletab[]={
    {  0,"O"},
    { 15,"J"},
    { 27,"K"},
    { 42,"L"},
    { 60,"R"},
    { 85,"M"},
    {125,"Q"},
    {1E9,"N"},
};

static Gpm_Type*
I_serial(int fd, unsigned short flags, struct Gpm_Type *type)
{
  int i;
#ifndef DEBUG
  unsigned char c;
  fd_set set; struct timeval timeout={0,0}; /* used when not debugging */


  /* flush any pending input (thanks, Miguel) */
  FD_ZERO(&set);
  for(i=0; /* always */ ; i++)
    {
      FD_SET(fd,&set);
      switch(select(fd+1,&set,(fd_set *)NULL,(fd_set *)NULL,&timeout/* zero */))
        {
        case 1:  if (read(fd,&c,1)==0) break;
        case -1: continue;
        }
      break;
    }

  if (type->fun==M_logimsc)
    write(fd, "QU", 2 );

#if 0 /* Did this ever work? */
  if (type->fun==M_ms && i==2 && c==0x33) /* Aha.. a mouseman... */
    {
      gpm_debug_log(LOG_INFO,"MouseMan detected");
      return mice; /* it is the first */
    }
#endif
#endif

  /* Non mman: change from any available speed to the chosen one */
  for (i=9600; i>=1200; i/=2)
      setspeed(fd, i, opt_baud, (type->fun != M_mman) /* write */, flags);

  /*
   * reset the MouseMan/TrackMan to use the 3/4 byte protocol
   * (Stephen Lee, sl14@crux1.cit.cornell.edu)
   * Changed after 1.14; why not having "I_mman" now?
   */
  if (type->fun==M_mman)
    {
      setspeed(fd, 1200, 1200, 0, flags); /* no write */
      write(fd, "*X", 2);
      setspeed(fd, 1200, opt_baud, 0, flags); /* no write */
      return type;
    }

  if(type->fun==M_geni)
    {
      puts("initializing genitizer");
      setspeed(fd, 1200, 9600, 1, flags); /* write */
      write(fd, ":" ,1); 
      write(fd, "E" ,1); /* setup tablet. relative mode, resolution... */
      write(fd, "@" ,1); /* setup tablet. relative mode, resolution... */

    }

  return type;
}

static Gpm_Type*
I_logi(int fd, unsigned short flags, struct Gpm_Type *type)
{
  int i;
  struct stat buf;
  int busmouse;

  /* is this a serial- or a bus- mouse? */
  if (fstat(fd,&buf)==-1) oops("fstat()");
  i=MAJOR(buf.st_rdev);
  if (stat("/dev/ttyS0",&buf)==-1) oops("stat()");
  busmouse=(i != MAJOR(buf.st_rdev));

  /* fix the howmany field, so that serial mice have 1, while busmice have 3 */
  type->howmany = busmouse ? 3 : 1;

  /* change from any available speed to the chosen one */
  for (i=9600; i>=1200; i/=2)
    setspeed(fd, i, opt_baud, 1 /* write */, flags);

  /* this stuff is peculiar of logitech mice, also for the serial ones */
  write(fd, "S", 1);
  setspeed(fd, opt_baud, opt_baud, 1 /* write */,
	   CS8 |PARENB |PARODD |CREAD |CLOCAL |HUPCL);

  /* configure the sample rate */
  for (i=0;opt_sample<=sampletab[i].sample;i++)
    ;
  write(fd,sampletab[i].code,1);
  return type;
}

static Gpm_Type *I_wacom(int fd, unsigned short flags, struct Gpm_Type *type)
{
  int i;

  if (opt_baud == 1200) opt_baud=9600; /* default to 9600 */
  /* change from any available speed to the chosen one */
  for (i=9600; i>=1200; i/=2)
    setspeed(fd, i, opt_baud, 1 /* write */, flags);
  return type;
}

static Gpm_Type *I_pnp(int fd, unsigned short flags, struct Gpm_Type *type)
{  
  struct termios tty;

  /*
   * Just put the device to 1200 baud. Thanks to Francois Chastrette
   * for his great help and debugging with his own pnp device.
   */
  tcgetattr(fd, &tty);
    
  tty.c_iflag = IGNBRK | IGNPAR;
  tty.c_oflag = 0;
  tty.c_lflag = 0;
  tty.c_line = 0;
  tty.c_cc[VTIME] = 0;
  tty.c_cc[VMIN] = 1;
  tty.c_cflag = flags | B1200;
  tcsetattr(fd, TCSAFLUSH, &tty); /* set parameters */

  /*
   * Don't read the silly initialization string. I don't want to see
   * the vendor name: it is only propaganda, with no information.
   */
  opt_wheel=type->wheel;
  return type;
}

/* intellimouse, ps2 version: Ben Pfaff and Colin Plumb */
static Gpm_Type *I_imps2(int fd, unsigned short flags, struct Gpm_Type *type)
{
  static unsigned char s1[] = { 243, 200, 243, 100, 243, 80, };
  static unsigned char s2[] = { 246, 230, 244, 243, 100, 232, 3, };
  write (fd, s1, sizeof (s1));
  usleep (30000);
  write (fd, s2, sizeof (s2));
  usleep (30000);
  tcflush (fd, TCIFLUSH);
  opt_wheel=1;
  return type;
}

static Gpm_Type *I_twid(int fd, unsigned short flags, struct Gpm_Type *type)
{
  if (twiddler_key_init() != 0) return NULL;
  /*
   * the twiddler is a serial mouse: just drop dtr
   * and run at 2400 (unless specified differently) 
   */
  if(opt_baud==DEF_BAUD) opt_baud = 2400;
  opt_toggle = TIOCM_DTR;
  return I_serial(fd, flags, type);
}



/*========================================================================*/
/* Finally, the table */
#define STD_FLG (CREAD|CLOCAL|HUPCL)

/*
 * Note that mman must be the first, and ms the second (I use this info
 * in mouse-test.c, as a quick and dirty hack
 */

 /*
  * For those who are trying to add a new type, here some docs:
  *
  * The first three strings are the name, an help line, a long name (if any)
  * Then come the function: the decoder and the initializazion function
  * Follows an array of four bytes: it is the protocol-identification, based
  *     on the first two bytes of a packet: if
  *     "((byte0 & proto[0]) == proto[1]) && ((byte1 & proto[2]) == proto[3])"
  *     then we are at the beginning of a packet.
  * The last numbers are: bytes-per-packet bytes-to-read() does-get-extra-byte
  *     and is-absolute-coordinates. The last two are booleans
  */

Gpm_Type mice[]={
  {"mman", "The \"MouseMan\" and similar devices (3/4 bytes per packet).",
           "Mouseman", M_mman, I_serial, CS7 | STD_FLG, /* first */
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 1, 0, 0},
  {"ms",   "The original ms protocol, with a middle-button extension.",
           "", M_ms, I_serial, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 0, 0, 0},
  {"ms+", "Like 'ms', but allows dragging with the middle button.",
           "", M_ms_plus, I_serial, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 0, 0, 0},
  {"ms+lr", "'ms+', but you can reset m by pressing lr (see man page).",
           "", M_ms_plus_lr, I_serial, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 0, 0, 0},
  {"bare", "Unadorned ms protocol. Needed with some 2-buttons mice.",
           "Microsoft", M_bare, I_serial, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 0, 0, 0},
  {"msc",  "Mouse-Systems-Compatible (5bytes). Most 3-button mice.",
           "MouseSystems", M_msc, I_serial, CS8 | CSTOPB | STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00}, 5, 1, 0, 0, 0},
  {"sun",  "'msc' protocol, but only 3 bytes per packet.",
           "", M_sun, I_serial, CS8 | CSTOPB | STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00}, 3, 1, 0, 0, 0},
  {"mm",   "MM series. Probably an old protocol...",
           "MMSeries", M_mm, I_serial, CS8 | PARENB|PARODD | STD_FLG,
                                {0xe0, 0x80, 0x80, 0x00}, 3, 1, 0, 0, 0},
  {"logi", "Used in some Logitech devices (only serial).",
           "Logitech", M_logi, I_logi, CS8 | CSTOPB | STD_FLG,
                                {0xe0, 0x80, 0x80, 0x00}, 3, 3, 0, 0, 0},
  {"bm",   "Micro$oft busmice and compatible devices.",
           "BusMouse", M_bm, NULL, STD_FLG, /* bm is sun */
                                {0xf8, 0x80, 0x00, 0x00}, 3, 3, 0, 0, 0},
  {"ps2",  "Busmice of the ps/2 series. Most busmice, actually.",
           "PS/2", M_ps2, NULL, STD_FLG,
                                {0xc0, 0x00, 0x00, 0x00}, 3, 1, 0, 0, 0},
  {"ncr",  "Ncr3125pen, found on some laptops",
           "", M_ncr, NULL, STD_FLG,
                                {0x08, 0x08, 0x00, 0x00}, 7, 7, 0, 1, 0},
  {"wacom",  "Wacom tablet",
           "", M_wacom, I_wacom, STD_FLG,
                                {0x80, 0x80, 0x80, 0x00}, 7, 7, 0, 0, 0},
  {"genitizer", "\"Genitizer\" tablet, in relative mode.",
           "", M_geni, I_serial, CS8|PARENB|PARODD,
                                {0x80, 0x80, 0x00, 0x00}, 3, 1, 0, 0, 0},
  {"logim",  "Turn logitech into Mouse-Systems-Compatible.",
           "", M_logimsc, I_serial, CS8 | CSTOPB | STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00}, 5, 1, 0, 0, 0},
  {"pnp",  "Plug and pray. New mice may not run with '-t ms'.",
           "", M_bare, I_pnp, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 0, 0, 0},
  {"imps2",   "Microsoft Intellimouse (ps2) - 3 buttons, wheel unused",
           "", M_ps2, I_imps2, STD_FLG,
                                {0xc0, 0x00, 0x00, 0x00}, 4, 1, 0, 0, 1},
  {"ms3", "Microsoft Intellimouse (serial) - 3 buttons, wheel unused",
           "", M_ms3, I_pnp, CS7 | STD_FLG,
                                {0xc0, 0x40, 0xc0, 0x00}, 4, 1, 0, 0, 1},
  {"netmouse",  "Genius NetMouse - 2 buttons and 2 buttons 'up'/'down'.",      
           "", M_netmouse, I_netmouse, CS7 | STD_FLG,
                                {0xc0, 0x00, 0x00, 0x00}, 4, 1, 0, 0, 0}, 
  {"twid", "Twidddler keyboard",
           "", M_twid, I_twid, CS8 | STD_FLG,
                                {0x80, 0x00, 0x80, 0x80}, 5, 1, 0, 0, 0},

#ifdef HAVE_LINUX_JOYSTICK_H
  {"js",   "Joystick mouse emulation",
           "Joystick", M_js, NULL, 0,
                              {0xFC, 0x00, 0x00, 0x00}, 12, 12, 0, 0, 0},
#endif

  {"brw",  "Fellowes Browser - 4 buttons (and a wheel) (dual protocol?)",
           "", M_brw, I_pnp, CS7 | STD_FLG,
                                {0xc0, 0x40, 0xc0, 0x00}, 4, 1, 0, 0, 0},

  {"",     "",
           "", NULL, NULL, 0,
                                {0x00, 0x00, 0x00, 0x00}, 0, 0, 0, 0, 0}
};

/*------------------------------------------------------------------------*/
/* and the help */

int M_listTypes(void)
{
  Gpm_Type *type;

  printf("\n" GPM_NAME " " GPM_RELEASE ", " GPM_DATE "\n");
  printf("Available mouse types are:\n\n");
  printf("  name   synonym         description\n\n");
  for (type=mice; type->fun; type++)
    printf("  %-6s %-12s %s\n",type->name, type->syn, type->desc);

  putchar('\n');

  return 1; /* to exit() */
}


/* Local Variables: */
/* c-indent-level: 2 */
/* End: */
