/*
              Copyright (C) 1998 Hewlett-Packard Company
                         ALL RIGHTS RESERVED.
 
  The enclosed software and documentation includes copyrighted works
  of Hewlett-Packard Co. For as long as you comply with the following
  limitations, you are hereby authorized to (i) use, reproduce, and
  modify the software and documentation, and to (ii) distribute the
  software and documentation, including modifications, for
  non-commercial purposes only.
      
  1.  The enclosed software and documentation is made available at no
      charge in order to advance the general development of
      high-performance networking and computing products.
 
  2.  You may not delete any copyright notices contained in the
      software or documentation. All hard copies, and copies in
      source code or object code form, of the software or
      documentation (including modifications) must contain at least
      one of the copyright notices.
 
  3.  The enclosed software and documentation has not been subjected
      to testing and quality control and is not a Hewlett-Packard Co.
      product. At a future time, Hewlett-Packard Co. may or may not
      offer a version of the software and documentation as a product.
  
  4.  THE SOFTWARE AND DOCUMENTATION IS PROVIDED "AS IS".
      HEWLETT-PACKARD COMPANY DOES NOT WARRANT THAT THE USE,
      REPRODUCTION, MODIFICATION OR DISTRIBUTION OF THE SOFTWARE OR
      DOCUMENTATION WILL NOT INFRINGE A THIRD PARTY'S INTELLECTUAL
      PROPERTY RIGHTS. HP DOES NOT WARRANT THAT THE SOFTWARE OR
      DOCUMENTATION IS ERROR FREE. HP DISCLAIMS ALL WARRANTIES,
      EXPRESS AND IMPLIED, WITH REGARD TO THE SOFTWARE AND THE
      DOCUMENTATION. HP SPECIFICALLY DISCLAIMS ALL WARRANTIES OF
      MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
  
  5.  HEWLETT-PACKARD COMPANY WILL NOT IN ANY EVENT BE LIABLE FOR ANY
      DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
      (INCLUDING LOST PROFITS) RELATED TO ANY USE, REPRODUCTION,
      MODIFICATION, OR DISTRIBUTION OF THE SOFTWARE OR DOCUMENTATION.
 
*/

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <httperf.h>
#include <call.h>
#include <conn.h>
#include <event.h>
#include <object.h>
#include <sess.h>

#define ALIGN(s)	(((s) + sizeof (double) - 1) & ~(sizeof (double) - 1))

static size_t type_size[OBJ_NUM_TYPES] =
  {
    ALIGN (sizeof (Conn)),
    ALIGN (sizeof (Call)),
    ALIGN (sizeof (Sess))
  };

struct free_list_el
  {
    struct free_list_el *next;
  };

static struct free_list_el *free_list[OBJ_NUM_TYPES];

static void
object_destroy (Object *obj)
{
  Object_Type type = obj->type;
  struct free_list_el *el;
  Event_Type event = 0;
  Any_Type arg;

  switch (type)
    {
    case OBJ_CALL:
      call_deinit ((Call *) obj);
      event = EV_CALL_DESTROYED;
      break;

    case OBJ_CONN:
      conn_deinit ((Conn *) obj);
      event = EV_CONN_DESTROYED;
      break;

    case OBJ_SESS:
      sess_deinit ((Sess *) obj);
      event = EV_SESS_DESTROYED;
      break;

    default:
      assert (0);
      break;
    }
  arg.l = 0;
  event_signal (event, obj, arg);

  /* Each object must be at least the size and alignment of "struct
     free_list_el".  Malloc takes care of returning properly aligned
     objects.  */
  el = (struct free_list_el *) obj;
  el->next = free_list[type];
  free_list[type] = el;
}

size_t
object_expand (Object_Type type, size_t size)
{
  size_t offset = type_size[type];
  type_size[type] += ALIGN (size);
  return offset;
}

Object *
object_new (Object_Type type)
{
  struct free_list_el *el;
  Event_Type event = 0;
  size_t obj_size;
  Any_Type arg;
  Object *obj;

  obj_size = type_size[type];

  if (free_list[type])
    {
      el = free_list[type];
      free_list[type] = el->next;
      obj = (Object *) el;
    }
  else
    {
      obj = malloc (obj_size);
      if (!obj)
	{
	  fprintf (stderr, "%s.object_new: %s\n", prog_name, strerror (errno));
	  return 0;
	}
    }
  memset (obj, 0, obj_size);
  obj->ref_count = 1;
  obj->type = type;
  switch (type)
    {
    case OBJ_CALL:
      call_init ((Call *) obj);
      event = EV_CALL_NEW;
      break;

    case OBJ_CONN:
      conn_init ((Conn *) obj);
      event = EV_CONN_NEW;
      break;

    case OBJ_SESS:
      sess_init ((Sess *) obj);
      event = EV_SESS_NEW;
      break;

    default:
      panic ("object_new: bad object type %d\n", type);
      break;
    }
  arg.l = 0;
  event_signal (event, obj, arg);
  return obj;
}

void
object_dec_ref (Object *obj)
{
  assert (obj->ref_count > 0);

  if (--obj->ref_count == 0)
    object_destroy (obj);
}
