/*
 
              Copyright (C) 1998 Hewlett-Packard Company
                         ALL RIGHTS RESERVED.
 
  The enclosed software and documentation includes copyrighted works
  of Hewlett-Packard Co. For as long as you comply with the following
  limitations, you are hereby authorized to (i) use, reproduce, and
  modify the software and documentation, and to (ii) distribute the
  software and documentation, including modifications, for
  non-commercial purposes only.
      
  1.  The enclosed software and documentation is made available at no
      charge in order to advance the general development of
      high-performance networking and computing products.
 
  2.  You may not delete any copyright notices contained in the
      software or documentation. All hard copies, and copies in
      source code or object code form, of the software or
      documentation (including modifications) must contain at least
      one of the copyright notices.
 
  3.  The enclosed software and documentation has not been subjected
      to testing and quality control and is not a Hewlett-Packard Co.
      product. At a future time, Hewlett-Packard Co. may or may not
      offer a version of the software and documentation as a product.
  
  4.  THE SOFTWARE AND DOCUMENTATION IS PROVIDED "AS IS".
      HEWLETT-PACKARD COMPANY DOES NOT WARRANT THAT THE USE,
      REPRODUCTION, MODIFICATION OR DISTRIBUTION OF THE SOFTWARE OR
      DOCUMENTATION WILL NOT INFRINGE A THIRD PARTY'S INTELLECTUAL
      PROPERTY RIGHTS. HP DOES NOT WARRANT THAT THE SOFTWARE OR
      DOCUMENTATION IS ERROR FREE. HP DISCLAIMS ALL WARRANTIES,
      EXPRESS AND IMPLIED, WITH REGARD TO THE SOFTWARE AND THE
      DOCUMENTATION. HP SPECIFICALLY DISCLAIMS ALL WARRANTIES OF
      MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
  
  5.  HEWLETT-PACKARD COMPANY WILL NOT IN ANY EVENT BE LIABLE FOR ANY
      DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
      (INCLUDING LOST PROFITS) RELATED TO ANY USE, REPRODUCTION,
      MODIFICATION, OR DISTRIBUTION OF THE SOFTWARE OR DOCUMENTATION.
 
*/
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include <httperf.h>
#include <event.h>
#include <object.h>
#include <rate.h>
#include <timer.h>

/* By pushing the random number generator state into the caller via
   the xsubi array below, we gain some test repeatability.  For
   example, let us say one generator was starting sessions, and a
   different generator was controlling requests within a session.  If
   both processes were sharing the same random number generator, then
   a premature session termination would change the subsequent session
   arrival spacing.  */

static Time
next_arrival_time_det (Rate_Generator *rg)
{
  return rg->rate->mean_iat;
}

static Time
next_arrival_time_uniform (Rate_Generator *rg)
{
  Time lower, upper;

  lower = rg->rate->min_iat;
  upper = rg->rate->max_iat;

  return lower + (upper - lower)*erand48 (rg->xsubi);
}

static Time
next_arrival_time_exp (Rate_Generator *rg)
{
  Time mean = rg->rate->mean_iat;

  return -mean*log (1.0 - erand48 (rg->xsubi));
}

static void
tick (Timer *t, Any_Type arg)
{
  Time delay, now = timer_now ();
  Rate_Generator *rg = arg.vp;

  rg->timer = 0;
  if (rg->done)
    return;

  while (now > rg->next_time)
    {
      delay = (*rg->next_interarrival_time) (rg);
      if (verbose > 2)
	fprintf (stderr, "next arrival delay = %.4f\n", delay);
      rg->next_time += delay;
      rg->done = ((*rg->tick) (rg->arg) < 0);
      if (rg->done)
	return;
    }
  rg->timer = timer_schedule ((Timer_Callback) tick, arg, rg->next_time - now);
}

static void
done (Event_Type type, Object *obj, Any_Type reg_arg, Any_Type call_arg)
{
  Rate_Generator *rg = reg_arg.vp;

  if (rg->done)
    return;
  rg->done = ((*rg->tick) (rg->arg) < 0);
}

void
rate_generator_start (Rate_Generator *rg, Event_Type completion_event)
{
  Time (*func) (struct Rate_Generator *rg);
  Any_Type arg;
  Time delay;

  /* Initialize random number generator with the init values here, all
     rate generators (although independent) will follow the same
     sequence of random values.  We factor in the client's id to make
     sure no two machines running httperf generate identical random
     numbers.  May want to pass these values as args to
     rate_generator_start in the future.  */
  rg->xsubi[0] = 0x1234 ^ param.client.id;
  rg->xsubi[1] = 0x5678 ^ (param.client.id << 8);
  rg->xsubi[2] = 0x9abc ^ ~param.client.id;

  arg.vp = rg;
  if (rg->rate->rate_param > 0.0)
    {
      switch (rg->rate->dist)
	{
	case DETERMINISTIC: func = next_arrival_time_det; break;
	case UNIFORM:	    func = next_arrival_time_uniform; break;
	case EXPONENTIAL:   func = next_arrival_time_exp; break;
	default:
	  fprintf (stderr, "%s: unrecognized interarrival distribution %d\n",
		   prog_name, rg->rate->dist);
	  exit (-1);
	}
      rg->next_interarrival_time = func;
      delay = (*func) (rg);
      /* bias `next time' so that timeouts are rounded to the closest
         tick: */
      rg->next_time = timer_now () + delay;
      rg->timer = timer_schedule ((Timer_Callback) tick, arg, delay);
    }
  else
    /* generate callbacks sequentially: */
    event_register_handler (completion_event, done, arg);

  rg->start = timer_now ();
  rg->done = ((*rg->tick) (rg->arg) < 0);
}

void
rate_generator_stop (Rate_Generator *rg)
{
  if (rg->timer)
    {
      timer_cancel (rg->timer);
      rg->timer = 0;
    }
  rg->done = 1;
}
