/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#include "xfs_locl.h"

#define XFS_FS_NAME	L"\\XFS"


XFSDevData XFSGlobalData;

/*
 * Set up the IRP table for `driver'
 */

void
InitDevice (PDRIVER_OBJECT driver)
{
    driver->MajorFunction[IRP_MJ_CREATE]		= xfs_create;
    driver->MajorFunction[IRP_MJ_CLOSE]			= xfs_close;
    driver->MajorFunction[IRP_MJ_READ]			= xfs_read;
    driver->MajorFunction[IRP_MJ_WRITE]			= xfs_write;
    driver->MajorFunction[IRP_MJ_QUERY_INFORMATION]	= xfs_fileinfo;
    driver->MajorFunction[IRP_MJ_SET_INFORMATION]	= xfs_fileinfo;


    driver->MajorFunction[IRP_MJ_FLUSH_BUFFERS]		= xfs_flush;
    driver->MajorFunction[IRP_MJ_DIRECTORY_CONTROL]	= xfs_dirctl;

    driver->MajorFunction[IRP_MJ_DEVICE_CONTROL]	= xfs_devctl;
    driver->MajorFunction[IRP_MJ_SHUTDOWN]		= xfs_shutdown;
    
    driver->MajorFunction[IRP_MJ_CLEANUP]		= xfs_cleanup;
    
    /* not implemented yet ... */

#if 0

    /* volume junk */
    driver->MajorFunction[IRP_MJ_QUERY_VOLUME_INFORMATION] = foo;
    driver->MajorFunction[IRP_MJ_SET_VOLUME_INFORMATION] = foo;

    /* filesystem IOCTL's */
    driver->MajorFunction[IRP_MJ_FILE_SYSTEM_CONTROL] = SFsdFSControl;

    /* byte range locks */
    // driver->MajorFunction[IRP_MJ_LOCK_CONTROL]		= SFsdLockControl;

    /* extended security */
    driver->MajorFunction[IRP_MJ_QUERY_SECURITY]		= SFsdSecurity;
    driver->MajorFunction[IRP_MJ_SET_SECURITY]		= SFsdSecurity;

    /* extended attributes */
    driver->MajorFunction[IRP_MJ_QUERY_EA]				= SFsdExtendedAttr;
    driver->MajorFunction[IRP_MJ_SET_EA]				= SFsdExtendedAttr;

    /*  fast-io */
    PtrFastIoDispatch = driver->FastIoDispatch = &(SFsdGlobalData.SFsdFastIoDispatch);
    
    PtrFastIoDispatch->SizeOfFastIoDispatch	= sizeof(FAST_IO_DISPATCH);
    PtrFastIoDispatch->FastIoCheckIfPossible	= SFsdFastIoCheckIfPossible;
    PtrFastIoDispatch->FastIoRead					= SFsdFastIoRead;
    PtrFastIoDispatch->FastIoWrite				= SFsdFastIoWrite;
    PtrFastIoDispatch->FastIoQueryBasicInfo	= SFsdFastIoQueryBasicInfo;
    PtrFastIoDispatch->FastIoQueryStandardInfo	= SFsdFastIoQueryStdInfo;
    PtrFastIoDispatch->FastIoLock					= SFsdFastIoLock;
    PtrFastIoDispatch->FastIoUnlockSingle		= SFsdFastIoUnlockSingle;
    PtrFastIoDispatch->FastIoUnlockAll			= SFsdFastIoUnlockAll;
    PtrFastIoDispatch->FastIoUnlockAllByKey	= SFsdFastIoUnlockAllByKey;
    PtrFastIoDispatch->AcquireFileForNtCreateSection = SFsdFastIoAcqCreateSec;
    PtrFastIoDispatch->ReleaseFileForNtCreateSection = SFsdFastIoRelCreateSec;
    
#if(_WIN32_WINNT >= 0x0400)
    PtrFastIoDispatch->FastIoQueryNetworkOpenInfo = SFsdFastIoQueryNetInfo;
    PtrFastIoDispatch->AcquireForModWrite		= SFsdFastIoAcqModWrite;
    PtrFastIoDispatch->ReleaseForModWrite		= SFsdFastIoRelModWrite;
    PtrFastIoDispatch->AcquireForCcFlush		= SFsdFastIoAcqCcFlush;
    PtrFastIoDispatch->ReleaseForCcFlush		= SFsdFastIoRelCcFlush;
    
    // MDL functionality
    PtrFastIoDispatch->MdlRead						= SFsdFastIoMdlRead;
    PtrFastIoDispatch->MdlReadComplete			= SFsdFastIoMdlReadComplete;
    PtrFastIoDispatch->PrepareMdlWrite			= SFsdFastIoPrepareMdlWrite;
    PtrFastIoDispatch->MdlWriteComplete			= SFsdFastIoMdlWriteComplete;
    
    // although this FSD does not support compressed read/write functionality,
    //	NTFS does, and if you design a FSD that can provide such functionality,
    //	you should consider initializing the fast io entry points for reading
    //	and/or writing compressed data ...
#endif	// (_WIN32_WINNT >= 0x0400)

    // last but not least, initialize the Cache Manager callback functions
    //	which are used in CcInitializeCacheMap()
    SFsdGlobalData.CacheMgrCallBacks.AcquireForLazyWrite = SFsdAcqLazyWrite;
    SFsdGlobalData.CacheMgrCallBacks.ReleaseFromLazyWrite = SFsdRelLazyWrite;
    SFsdGlobalData.CacheMgrCallBacks.AcquireForReadAhead = SFsdAcqReadAhead;
    SFsdGlobalData.CacheMgrCallBacks.ReleaseFromReadAhead = SFsdRelReadAhead;
#endif    

    return;
}


/*
 * The entrypoint for the driver
 */


NTSTATUS
DriverEntry(PDRIVER_OBJECT driver,
	    PUNICODE_STRING	RegistryPath)
{
    NTSTATUS		RC = STATUS_SUCCESS;
    UNICODE_STRING	devname;
    BOOLEAN		RegShutdown = FALSE;

    try {

	try {

	    RtlZeroMemory(&XFSGlobalData, sizeof(XFSGlobalData));
	    XFSGlobalData.magic = XFS_DEV_DATA_MAGIC;

	    /* init resource */
	    RC = ExInitializeResourceLite(&(XFSGlobalData.GlobalLock));
	    ASSERT(NT_SUCCESS(RC));
	    XFS_SETFLAGS (XFSGlobalData.flags, DEVDATA_FLAGS_GLOBALLOCK);

	    
	    /* save entry for later use */
	    XFSGlobalData.DriverObject = driver;

	    /* init logical volume block structure */
	    InitializeListHead(&(XFSGlobalData.NextVCB));


	    /* Create DNLC
	     *        Locking
	     */

	    /* set up function pointer */
	    InitDevice (driver);

	    /* Create device */
	    RtlInitUnicodeString(&devname, XFS_FS_NAME);
	    RC = IoCreateDevice(driver,
				0,
				&devname,
				FILE_DEVICE_NETWORK_FILE_SYSTEM,
				0,
				FALSE,
				&XFSGlobalData.DeviceObject);
	    
	    if (!NT_SUCCESS(RC))
		try_return(RC);
	    
	    RC = IoRegisterShutdownNotification(XFSGlobalData.DeviceObject);
	    if (!NT_SUCCESS(RC))
		try_return (RC);

	    RegShutdown = TRUE;


	    xfs_log (XFSGlobalData.DeviceObject,
		     IRP_MJ_DEVICE_CONTROL,
		     0,
		     0,
		     STATUS_NO_MEDIA_IN_DEVICE,
		     2,
		     xfs_log_new_seq(),
		     0,
		     0,
		     NULL,
		     0,
		     NULL);

	try_exit: NOTHING ;


	} except (EXCEPTION_EXECUTE_HANDLER) {
	    RC = GetExceptionCode();
	} 
    } finally {

	/* Did we fail ? */

	if (!NT_SUCCESS(RC)) {

	    if (RegShutdown)
		IoUnregisterShutdownNotification(XFSGlobalData.DeviceObject);

	    /* delete device */
	    if (XFSGlobalData.DeviceObject) {
		IoDeleteDevice(XFSGlobalData.DeviceObject);
		XFSGlobalData.DeviceObject = NULL;
	    }

	    /* FreeDNLC & co*/

	    if ( XFS_TESTFLAGS (XFSGlobalData.flags, DEVDATA_FLAGS_GLOBALLOCK))
		ExDeleteResourceLite(&XFSGlobalData.GlobalLock);
	}
    }
    return RC;
}




