/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 *
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_msg_locl.h>
#include <xfs/xfs_syscalls.h>
#include <xfs/xfs_vfsops.h>
#include <xfs/xfs_vnodeops.h>
#include <xfs/xfs_dev.h>

RCSID("$Id: xfs_message.c,v 1.40 1999/08/01 10:23:45 lha Exp $");

int
xfs_message_installroot(int fd,
			struct xfs_message_installroot * message,
			u_int size,
			struct proc *p)
{
    int error = 0;

    XFSDEB(XDEBMSG, ("xfs_message_installroot\n"));

    if (xfs[fd].root != NULL) {
	printf("XFS PANIC WARNING! xfs_message_installroot: called again!\n");
	error = EBUSY;
    } else {
	error = new_xfs_node(&xfs[fd], &message->node, &xfs[fd].root, p);
	if (error)
	    return error;
	xfs[fd].root->vn->v_flag |= VROOT;
    }
    return error;
}

int
xfs_message_installnode(int fd,
			struct xfs_message_installnode * message,
			u_int size,
			struct proc *p)
{
    int error = 0;
    struct xfs_node *n, *dp;

    XFSDEB(XDEBMSG, ("xfs_message_installnode\n"));

retry:
    dp = xfs_node_find(&xfs[fd], &message->parent_handle);
    if (dp) {
	struct vnode *t_vnode = XNODE_TO_VNODE(dp);

	if (xfs_do_vget (t_vnode, LK_INTERLOCK|LK_SHARED, p))
		goto retry;

	error = new_xfs_node(&xfs[fd], &message->node, &n, p);
	if (error) {
	    vrele (t_vnode);
	    return error;
	}

	xfs_dnlc_enter_name(t_vnode,
			    message->name,
			    XNODE_TO_VNODE(n));
	vrele(XNODE_TO_VNODE(n));
	vput (t_vnode);
    } else {
	printf("XFS PANIC WARNING! xfs_message_installnode: no parent\n");
	error = ENOENT;
    }
    XFSDEB(XDEBMSG, ("return: xfs_message_installnode: %d\n", error));

    return error;
}

int
xfs_message_installattr(int fd,
			struct xfs_message_installattr * message,
			u_int size,
			struct proc *p)
{
    int error = 0;
    struct xfs_node *t;

    t = xfs_node_find(&xfs[fd], &message->node.handle);
    if (t != 0) {
	t->tokens = message->node.tokens;
	xfs_attr2vattr(&message->node.attr, &t->attr);
#ifdef UVM
	uvm_vnp_setsize(XNODE_TO_VNODE(t), t->attr.va_size);
#elif HAVE_KERNEL_VNODE_PAGER_SETSIZE
	vnode_pager_setsize(XNODE_TO_VNODE(t), t->attr.va_size);
#endif
	bcopy(message->node.id, t->id, sizeof(t->id));
	bcopy(message->node.rights, t->rights, sizeof(t->rights));
	t->anonrights = message->node.anonrights;
    } else {
	printf("XFS PANIC WARNING! xfs_message_installattr: no node!\n");
	error = ENOENT;
    }

    return error;
}

int
xfs_message_installdata(int fd,
			struct xfs_message_installdata * message,
			u_int size,
			struct proc *p)
{
    struct xfs_node *t;
    int error = 0;

    XFSDEB(XDEBMSG, ("xfs_message_installdata\n"));

retry:
    t = xfs_node_find(&xfs[fd], &message->node.handle);
    if (t != NULL) {
	struct xfs_fhandle_t *fh = (struct xfs_fhandle_t *)&message->cache_handle;
	struct vnode *vp;
	struct vnode *t_vnode = XNODE_TO_VNODE(t);

	XFSDEB(XDEBMSG, ("cache_name = '%s'\n", message->cache_name));

	if (xfs_do_vget(t_vnode, LK_INTERLOCK|LK_SHARED, p))
		goto retry;

	error = xfs_fhlookup (p,
			      fh,
			      &vp);

	if (error != 0) {
#ifdef __osf__
	    struct nameidata *ndp = &u.u_nd;
#else
	    struct nameidata nd;
	    struct nameidata *ndp = &nd;
#endif

	    XFSDEB(XDEBMSG,
		   ("xfs_message_installdata: fhlookup failed: %d, "
		    "opening by name\n", error));

	    NDINIT(ndp, LOOKUP, FOLLOW, UIO_SYSSPACE,
		   message->cache_name, p);
	    error = namei(ndp);
	    vp = ndp->ni_vp;
	}

	if (error == 0) {
#ifndef __osf__
	    xfs_vfs_unlock(vp, p);
#endif
	    if (DATA_FROM_XNODE(t))
		vrele(DATA_FROM_XNODE(t));
	    DATA_FROM_XNODE(t) = vp;

	    XFSDEB(XDEBMSG, ("xfs_message_installdata: t = %p;"
			     " tokens = %x\n",
			     t, message->node.tokens));

	    t->tokens = message->node.tokens;
	    xfs_attr2vattr(&message->node.attr, &t->attr);
#ifdef HAVE_KERNEL_VNODE_PAGER_SETSIZE
	    vnode_pager_setsize(XNODE_TO_VNODE(t), t->attr.va_size);
#endif
	    if (XNODE_TO_VNODE(t)->v_type == VDIR
		&& (message->flag & XFS_INVALID_DNLC))
		cache_purge (XNODE_TO_VNODE(t));
	    bcopy(message->node.id, t->id, sizeof(t->id));
	    bcopy(message->node.rights, t->rights, sizeof(t->rights));
	    t->anonrights = message->node.anonrights;
	} else {
	    printf("XFS PANIC WARNING! xfs_message_installdata failed!\n");
	    printf("Reason: lookup failed on cache file '%s', error = %d\n",
		   message->cache_name, error);
	}
	vput (t_vnode);
    } else {
	printf("XFS PANIC WARNING! xfs_message_installdata failed\n");
	printf("Reason: No node to install the data into!\n");
	error = ENOENT;
    }

    return error;
}

int
xfs_message_invalidnode(int fd,
			struct xfs_message_invalidnode * message,
			u_int size,
			struct proc *p)
{
    int error = 0;
    struct xfs_node *t;

    XFSDEB(XDEBMSG, ("xfs_message_invalidnode\n"));

    t = xfs_node_find(&xfs[fd], &message->handle);
    if (t != 0) {
	/* XXX Really need to put back dirty data first. */
	if (DATA_FROM_XNODE(t)) {
	    vrele(DATA_FROM_XNODE(t));
	    DATA_FROM_XNODE(t) = (struct vnode *) 0;
	}
	XFS_TOKEN_CLEAR(t, ~0,
			XFS_OPEN_MASK | XFS_ATTR_MASK |
			XFS_DATA_MASK | XFS_LOCK_MASK);
	cache_purge(XNODE_TO_VNODE(t));
    } else {
	printf("XFS PANIC WARNING! xfs_message_invalidnode: no node!\n");
	error = ENOENT;
    }

    return error;
}

int
xfs_message_updatefid(int fd,
		      struct xfs_message_updatefid * message,
		      u_int size,
		      struct proc *p)
{
    int error = 0;
    struct xfs_node *t;

    XFSDEB(XDEBMSG, ("xfs_message_updatefid\n"));
    t = xfs_node_find (&xfs[fd], &message->old_handle);
    if (t != NULL) {
	t->handle = message->new_handle;
    } else {
	printf ("XFS PANIC WARNING! xfs_message_updatefid: no node!\n");
	error = ENOENT;
    }
    return error;
}

/*
 * Try to clean out nodes for the userland daemon
 */

static void
gc_vnode (struct vnode *vp,
	  struct proc *p)
{
    simple_lock(&vp->v_interlock);
    
    /* This node is on the freelist */
    if (vp->v_usecount <= 0) {
	
	/*  DIAGNOSTIC */
	if (vp->v_usecount < 0 || vp->v_writecount != 0) {
		    vprint("vrele: bad ref count", vp);
		    panic("vrele: ref cnt");
	}
	
	XFSDEB(XDEBMSG, ("xfs_message_gc: success\n"));
	
#ifdef HAVE_KERNEL_FUNC_VGONEL
	vgonel (vp, p);
#else
	simple_unlock(&vp->v_interlock); 
	vgone (vp);
#endif

    } else {
	XFSDEB(XDEBMSG, ("xfs_message_gc: used\n"));
    }

}

int
xfs_message_gc_nodes(int fd,
		     struct xfs_message_gc_nodes *message,
		     u_int size,
		     struct proc *p)
{
    XFSDEB(XDEBMSG, ("xfs_message_gc\n"));

    if (message->len == 0) {
	struct vnode *vp;

	/* XXX see comment in xfs_node_find */
	/* XXX osf */
	/* XXXSMP do gone[l] need to get mntvnode_slock ? */

	for(vp = XFS_TO_VFS(&xfs[fd])->mnt_vnodelist.lh_first;
	    vp != NULL; 
	    vp = vp->v_mntvnodes.le_next) {

	    gc_vnode (vp, p);
	}

    } else {
	struct xfs_node *t;
	int i;

	for (i = 0; i < message->len; i++) {
	    t = xfs_node_find (&xfs[fd], &message->handle[i]);
	    if (t == NULL)
		continue;

	    gc_vnode(XNODE_TO_VNODE(t), p);
	}
    }

    return 0;
}


/*
 * Probe what this xfs support
 */

int
xfs_message_version(int fd,
		    struct xfs_message_version *message,
		    u_int size,
		    struct proc *p)
{
    struct xfs_message_wakeup msg;
    int ret;

    switch (message->probe) {
    case XFS_VERSION_VERSION:
	ret = XFS_VERSION_YES;
	break;
    case XFS_VERSION_FHGET:
	ret = XFS_VERSION_YES;
	break;
#if defined(HAVE_GETFH) && defined(HAVE_FHOPEN)
    case XFS_VERSION_FHGET_NATIVE:
	ret = XFS_VERSION_YES;
	break;
#endif	
    case XFS_VERSION_GC_NODES:
	ret = XFS_VERSION_YES;
	break;
    default:
	ret = XFS_VERSION_NO;
	break;
    }

    msg.header.opcode = XFS_MSG_WAKEUP;
    msg.sleepers_sequence_num = message->header.sequence_num;
    msg.error = ret;

    return xfs_message_send(fd, (struct xfs_message_header *) &msg, sizeof(msg));
}




