/*
 * Copyright (c) 1995, 1996, 1997, 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <rx/rx.h>
#include <rx/rx_null.h>

#include <ports.h>
#include <ko.h>

#ifdef KERBEROS
#include <des.h>
#include <krb.h>
#include <rxkad.h>
#endif

#include <err.h>
#include <assert.h>
#include <ctype.h>

#ifndef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <service.h>
#include <msecurity.h>
#include <netinit.h>

#include <getarg.h>

#include "rx/rxgencon.h"
#include "vldb.h"
#include "vldb.ss.h"
#include "ubik.ss.h"

RCSID("$Id: vldbserver.c,v 1.18 1999/07/18 19:47:09 map Exp $");

int vl_database;
vlheader vl_header;
off_t file_length;

char vl_header_ydr[VLHEADER_SIZE];

static int debug = 0;

static void
vlservdebug (char *fmt, ...)
    __attribute__ ((format (printf, 1, 2)));

static void
vlservdebug (char *fmt, ...)
{
    va_list args;
    if (!debug)
	return ;

    va_start (args, fmt);
    vfprintf (stderr, fmt, args);
    va_end(args);
}

void
write_header()
{
    off_t pos;
    int length = VLHEADER_SIZE;

    if (ydr_encode_vlheader(&vl_header, vl_header_ydr, &length) == NULL)
        err(1, "write_header");

    pos = lseek(vl_database, 0, SEEK_SET);
    assert(pos == 0);

    length = write(vl_database, vl_header_ydr, VLHEADER_SIZE);
    assert (length == VLHEADER_SIZE);
}

void
read_header()
{
    char vl_header_ydr[VLHEADER_SIZE];
    int length = VLHEADER_SIZE;

    if (lseek(vl_database, 0, SEEK_SET) == -1)
        err(1, "lseek");

    length = read(vl_database, vl_header_ydr, VLHEADER_SIZE);
    if (length == -1)
        err(1, "read");
    if (length != VLHEADER_SIZE)
        errx(1, "read_header read failed");

    if (ydr_decode_vlheader(&vl_header, vl_header_ydr, &length) == NULL)
        err(1, "read_header");
}

void
get_file_length()
{
    file_length = lseek(vl_database, 0, SEEK_END);
    if (file_length == -1) {
        err(1, "lseek");
    }
}

off_t
find_first_free()
{
    off_t pos;

    if (vl_header.vital_header.freePtr == 0) { /* if there are no free entries */
	pos = lseek(vl_database, 0, SEEK_END);
	if (pos == -1)
	    err(1, "lseek");
	if (ftruncate(vl_database, pos + DISK_VLENTRY_SIZE) == -1)
	    err(1, "ftruncate");
	return pos;
    } else { /* there are free entries */
	/* Not implemented yet */
	assert(0);
    }
    return 0;
}

int
write_entry(off_t offset, disk_vlentry *vl_entry)
{
    off_t pos;
    char vl_entry_ydr[DISK_VLENTRY_SIZE];
    int length = DISK_VLENTRY_SIZE;

    if (ydr_encode_disk_vlentry((disk_vlentry *) vl_entry, vl_entry_ydr, &length) == NULL)
	err(1, "write_entry");

    pos = lseek(vl_database, offset, SEEK_SET);
    assert(pos == offset);

    length = write(vl_database, vl_entry_ydr, DISK_VLENTRY_SIZE);
    assert (length == DISK_VLENTRY_SIZE);

    return 0;
}

int
read_entry(off_t offset, disk_vlentry *vl_entry)
{
    off_t pos;
    char vl_entry_ydr[DISK_VLENTRY_SIZE];
    int length = DISK_VLENTRY_SIZE;

    pos = lseek(vl_database, offset, SEEK_SET);
    assert(pos == offset);

    length = read(vl_database, vl_entry_ydr, DISK_VLENTRY_SIZE);
    assert (length == DISK_VLENTRY_SIZE);

    if (ydr_decode_disk_vlentry((disk_vlentry *) vl_entry, vl_entry_ydr, &length) == NULL)
	err(1, "write_entry");

    return 0;
}

void
create_database()
{
    int i;

    vl_header.vital_header.vldbversion = 0;
    vl_header.vital_header.headersize = VLHEADER_SIZE;
    vl_header.vital_header.freePtr = 0;
    vl_header.vital_header.eofPtr = VLHEADER_SIZE;
    vl_header.vital_header.allocs = 0;
    vl_header.vital_header.frees = 0;
    vl_header.vital_header.MaxVolumeId = 0xA0000000 - 1;
    for (i = 0; i < MAXTYPES; i++)
	vl_header.vital_header.totalEntries[i] = 0;

    for (i = 0; i < MAXSERVERID+1; i++)
	vl_header.IpMappedAddr[i] = 0;

    memset(vl_header.VolnameHash, 0, HASHSIZE * sizeof(int32_t));
    memset(vl_header.VolidHash, 0, HASHSIZE * MAXTYPES * sizeof(int32_t));
    write_header();
    get_file_length();
}

unsigned long
get_id_hash(long id)
{
    return ((unsigned long) id) % HASHSIZE;
}

unsigned long
get_name_hash(const char *name)
{
    int i;
    unsigned long hash = 0x47114711;

    for (i = 0; name[i] && i < 32; i++)
	hash *= name[i];

    return hash % HASHSIZE;
}

int
get_first_id_entry(unsigned long hash_id, long type, disk_vlentry *vl_entry)
{
    off_t offset = vl_header.VolidHash[type][hash_id];
    int status;

    vlservdebug ("  get_first_id_entry hash_id: %lu type: %ld offset: %d\n", 
		 hash_id, type, (int) offset);

    if (offset == 0)
	return VL_NOENT;

    status = read_entry(offset, vl_entry);

    return status;
}

int
get_first_name_entry(unsigned long hash_name, disk_vlentry *vl_entry)
{
    off_t offset = vl_header.VolnameHash[hash_name];
    int status;

    vlservdebug ("  get_first_name_entry hash_name: %lu offset: %d\n", 
		 hash_name, (int) offset);
    if (offset == 0)
	return VL_NOENT;

    status = read_entry(offset, vl_entry);

    return status;
}

int
insert_entry(disk_vlentry *vl_entry)
{
    off_t offset;
    int status;
    unsigned long hash_id, hash_name;
    disk_vlentry first_id_entry;
    disk_vlentry first_name_entry;
    
    /* Allokera plats i filen */
    offset = find_first_free();

    /* Allocate new volume id? */
    /*id = vl_header.vital_header.MaxVolumeId++;*/

    /* Hitta plats i hashtabellerna */
    /* XXX At present, only RW is handled */
    hash_id = get_id_hash(vl_entry->volumeId[RWVOL]);
    hash_name = get_name_hash(vl_entry->name);

    status = get_first_id_entry(hash_id, vl_entry->volumeType, &first_id_entry);

/* XXX    vl_entry->nextIDHash[vldb_entry->type] = status ? 0 : first_id_entry.nextID;*/
    vl_entry->nextIdHash[vl_entry->volumeType] = status ? 0 : vl_header.VolidHash[vl_entry->volumeType][hash_id];

    status = get_first_name_entry(hash_name, &first_name_entry);
/* XXX    pr_entry->nextName = status ? 0 : first_name_entry.nextName;*/
    vl_entry->nextNameHash = status ? 0 : vl_header.VolnameHash[hash_name];

    /* XXX: uppdatera owned och nextOwned */

    /* Lgg in entryt i filen */
    status = write_entry(offset, vl_entry);
    if (status)
	return status;
    
    /* Uppdatera hashtabell */
    vl_header.VolidHash[vl_entry->volumeType][hash_id] = offset;
    vl_header.VolnameHash[hash_name] = offset;
    write_header();
    return 0;
}

void
make_vldb_from_vl(struct vldbentry *vldb_entry, struct disk_vlentry *vl_entry)
{
    int i;

    strncpy(vldb_entry->name, vl_entry->name, VLDB_MAXNAMELEN);
    vldb_entry->volumeType = vl_entry->volumeType;
    vldb_entry->nServers = 1; /* XXX What is this supposed to be? */

    for (i = 0; i < MAXNSERVERS; i++) {
	vldb_entry->serverNumber[i] = vl_entry->serverNumber[i];
	vldb_entry->serverPartition[i] = vl_entry->serverPartition[i];
	vldb_entry->serverFlags[i] = vl_entry->serverFlags[i];
    }

    for (i = 0; i < MAXTYPES; i++)
	vldb_entry->volumeId[i] = vl_entry->volumeId[i];

    vldb_entry->cloneId = vl_entry->cloneId;
    vldb_entry->flags = vl_entry->flags;
}

/*
 * The rpc - calls
 */

int
VL_CreateEntry(struct rx_call *call, 
	       const vldbentry *newentry) 
{
    struct disk_vlentry vl_entry;
    int i;

    if (!sec_is_superuser(call))
	return VL_PERM;

    vlservdebug ("VL_CreateEntry (name=%s, type=%d, ids=%d,%d,%d flags=%d)\n",
		 newentry->name, newentry->volumeType,
		 newentry->volumeId[RWVOL],
		 newentry->volumeId[ROVOL],
		 newentry->volumeId[BACKVOL],
		 newentry->flags);

    /* XXX Should see to it that name is null terminated too */
    strncpy(vl_entry.name, newentry->name, VLDB_MAXNAMELEN);
    vl_entry.volumeType = newentry->volumeType;

    /* XXX All fields mustn't be set */
    for (i = 0; i < MAXNSERVERS; i++) {
	vl_entry.serverNumber[i] = newentry->serverNumber[i];
	vl_entry.serverPartition[i] = newentry->serverPartition[i];
	vl_entry.serverFlags[i] = newentry->serverFlags[i];
    }

    for (i = 0; i < MAXTYPES; i++)
	vl_entry.volumeId[i] = newentry->volumeId[i];

    vl_entry.cloneId = newentry->cloneId;
    vl_entry.flags = newentry->flags;

    insert_entry(&vl_entry);
    return 0;
}

int 
VL_DeleteEntry(struct rx_call *call, 
	       const int32_t Volid, 
	       const int32_t voltype)
{
    vlservdebug ("VL_DeleteEntry\n") ;

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM ;
}

/*
 *
 */

int
VL_GetEntryByID(struct rx_call *call,
		const int32_t Volid, 
		const int32_t voltype, 
		vldbentry *entry) 
{
    struct disk_vlentry vl_entry;

    vlservdebug ("VL_GetEntryByID (Volid=%d,Voltype=%d)\n", 
		  Volid, voltype);
    
    if (get_first_id_entry(get_id_hash(Volid), voltype, &vl_entry) != 0)
	return VL_NOENT;

    while (1) {
	/* Return entry if match found */
	if (vl_entry.volumeId[voltype] == Volid) {
	    make_vldb_from_vl(entry, &vl_entry);
	    return 0;
	}
	
	if (vl_entry.nextIdHash[voltype] == 0)
	    break;
	
	read_entry(vl_entry.nextIdHash[voltype], &vl_entry);
    }
    return VL_NOENT;
}

/*
 *
 */

int
VL_GetEntryByName(struct rx_call *call, 
		  const char *volumename, 
		  vldbentry *entry) 
{
    struct disk_vlentry vl_entry;

    vlservdebug ("VL_GetEntryByName %s\n", volumename) ;

    if (isdigit(volumename[0])) {
	return VL_GetEntryByID(call, atol(volumename), 0 /* XXX */, entry);
    }

    if (get_first_name_entry(get_name_hash(volumename), &vl_entry) == 0) {
	while (1) {
	    /* Return entry if match found */
	    if (strcmp(vl_entry.name, volumename) == 0) {
		make_vldb_from_vl(entry, &vl_entry);
		return 0;
	    }
	    
	    if (vl_entry.nextNameHash == 0)
		break;
	    
	    read_entry(vl_entry.nextNameHash, &vl_entry);
	}
    } 

    return VL_NOENT;
}

/*
 *
 */

int 
VL_GetNewVolumeId (struct rx_call *call, 
		   const int32_t bumpcount,
		   int32_t *newvolumid)
{
    vlservdebug ("VL_GetNewVolumeId(bumpcount=%d)\n", bumpcount) ;
    
    if (!sec_is_superuser(call))
	return VL_PERM;

    *newvolumid = vl_header.vital_header.MaxVolumeId;
    vlservdebug ("   returning low volume id = %d\n", *newvolumid);
    vl_header.vital_header.MaxVolumeId += bumpcount;
    write_header();

    return 0;
}

/*
 *
 */

int
VL_ReplaceEntry (struct rx_call *call, 
		 const int32_t Volid, 
		 const int32_t voltype,
		 const vldbentry *newentry,
		 const int32_t ReleaseType) 
{
    vlservdebug ("VL_ReplaceEntry\n") ;

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM ;
}

/*
 *
 */

int
VL_UpdateEntry (struct rx_call *call, 
		const int32_t Volid, 
		const int32_t voltype, 
		const VldbUpdateEntry *UpdateEntry,
		const int32_t ReleaseType)
{
    vlservdebug ("VL_UpdateEntry\n") ;

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM ;
}

/*
 *
 */

int 
VL_SetLock (struct rx_call *call, 
	    const int32_t Volid, 
	    const int32_t voltype,
	    const int32_t voloper) 
{
    vlservdebug ("VL_SetLock\n") ;

    if (!sec_is_superuser(call))
	return VL_PERM;

    return 0;
}

/*
 *
 */

int
VL_ReleaseLock (struct rx_call *call, 
		const int32_t volid,
		const int32_t voltype, 
		const int32_t ReleaseType) 
{
    vlservdebug ("VL_ReleaseLock\n") ;

    if (!sec_is_superuser(call))
	return VL_PERM;


    return 0;
}

/*
 *
 */

int
VL_ListEntry (struct rx_call *call, 
	      const int32_t previous_index, 
	      int32_t *count,
	      int32_t *next_index, 
	      vldbentry *entry) 
{
    vlservdebug ("VL_ListEntry\n") ;
    return VL_PERM ;
}

/*
 *
 */

int
VL_ListAttributes (struct rx_call *call, 
		   const VldbListByAttributes *attributes,
		   int32_t *nentries,
		   bulkentries *blkentries) 
{
    vlservdebug ("VL_ListAttributes\n") ;
    return VL_PERM ;
}

/*
 *
 */

int
VL_GetStats (struct rx_call *call, 
	     vldstats *stats,
	     vital_vlheader *vital_header) 
{
    vlservdebug ("VL_GetStats") ;
    return VL_PERM ;
}

/*
 *
 */

int 
VL_Probe(struct rx_call *call)
{
    vlservdebug ("VL_Probe\n") ;
    return 0;
}

/*
 *
 */

int
VL_CreateEntryN(struct rx_call *call,
		const nvldbentry *entry)
{
    int i;
    struct vldbentry vldb_entry;

    vlservdebug ("VL_CreateEntryN\n") ;

    if (!sec_is_superuser(call))
	return VL_PERM;

    strncpy(vldb_entry.name, entry->name, VLDB_MAXNAMELEN);
    vldb_entry.volumeType = RWVOL;
    vldb_entry.nServers = entry->nServers;

    for (i = 0; i < MAXNSERVERS; i++) {
	vldb_entry.serverNumber[i] = entry->serverNumber[i];
	vldb_entry.serverPartition[i] = entry->serverPartition[i];
	vldb_entry.serverFlags[i] = entry->serverFlags[i];
    }

    for (i = 0; i < MAXTYPES; i++)
	vldb_entry.volumeId[i] = entry->volumeId[i];

    vldb_entry.cloneId = entry->cloneId;
    vldb_entry.flags = entry->flags;

    return VL_CreateEntry(call, &vldb_entry);
}

/*
 *
 */

int
VL_GetEntryByIDN(struct rx_call *call,
		 int32_t Volid,
		 int32_t voltype,
		 nvldbentry *entry)
{
    struct vldbentry vldb_entry;
    int status, i;

    vlservdebug ("VL_GetEntryByIDN (Volid=%d,Voltype=%d)\n", Volid, voltype);

    if (voltype == -1)
	voltype = RWVOL;

    status = VL_GetEntryByID(call, Volid, voltype, &vldb_entry);

    if (status)
	return status;

    strncpy(entry->name, vldb_entry.name, VLDB_MAXNAMELEN);
    entry->nServers = vldb_entry.nServers;
    for (i = 0; i < MAXNSERVERS; i++) {
	entry->serverNumber[i] = vldb_entry.serverNumber[i];
	entry->serverPartition[i] = vldb_entry.serverPartition[i];
	entry->serverFlags[i] = vldb_entry.serverFlags[i];
    }

    for (i = 0; i < MAXTYPES; i++)
	entry->volumeId[i] = vldb_entry.volumeId[i];

    entry->cloneId = vldb_entry.cloneId;
    entry->flags = vldb_entry.flags;

    return 0;
}

/*
 *
 */

int
VL_GetEntryByNameN(struct rx_call *call, 
		   const char *volumename, 
		   nvldbentry *entry) 
{
    struct vldbentry vldb_entry;
    int status, i;

    vlservdebug ("VL_GetEntryByNameN(volumename=%s)\n", volumename) ;
    status = VL_GetEntryByName(call, volumename, &vldb_entry);

    if (status)
	return status;

    memset (entry, 0, sizeof(*entry));
    strncpy(entry->name, vldb_entry.name, VLDB_MAXNAMELEN);
    entry->nServers = vldb_entry.nServers;
    for (i = 0; i < MAXNSERVERS; i++) {
	entry->serverNumber[i] = vldb_entry.serverNumber[i];
	entry->serverPartition[i] = vldb_entry.serverPartition[i];
	entry->serverFlags[i] = vldb_entry.serverFlags[i];
    }

    for (i = 0; i < MAXTYPES; i++)
	entry->volumeId[i] = vldb_entry.volumeId[i];

    entry->cloneId = vldb_entry.cloneId;
    entry->flags = vldb_entry.flags;

    return 0;
}

/*
 *
 */

int
VL_GetEntryByNameU(struct rx_call *call, 
		   const char *volumename, 
		   uvldbentry *entry) 
{
    vlservdebug ("VL_GetEntryByNameU %s\n", volumename);
    memset(entry, 0, sizeof(*entry));
    
    return RXGEN_OPCODE;
}

/*
 *
 */

int
VL_ListAttributesN (struct rx_call *call, 
		   const VldbListByAttributes *attributes,
		   int32_t *nentries,
		   nbulkentries *blkentries) 
{
    vlservdebug ("VL_ListAttributesN\n");
    vlservdebug ("  attributes: Mask=(%d=", attributes->Mask);

    if (attributes->Mask & VLLIST_SERVER)
	vlservdebug  ("SERVER ");
    if (attributes->Mask & VLLIST_PARTITION)
	vlservdebug  ("PARTITION ");
    if (attributes->Mask & VLLIST_VOLUMETYPE)
	vlservdebug  ("VOLUMETYPE ");
    if (attributes->Mask & VLLIST_VOLUMEID)
	vlservdebug  ("VOLUMEID ");
    if (attributes->Mask & VLLIST_FLAG)
	vlservdebug  ("FLAG");

    vlservdebug (") server=%d partition=%d volumetype=%d volumeid=%d flag=%d\n",
	   attributes->server,
	   attributes->partition,
	   attributes->volumetype,
	   attributes->volumeid,
	   attributes->flag);

    *nentries = 1;

    blkentries->len = 0;
    blkentries->val = NULL;

    return VL_PERM;
}

/*
 *
 */

int
VL_ListAttributesU(struct rx_call *call, 
		   const VldbListByAttributes *attributes,
		   int32_t *nentries,
		   ubulkentries *blkentries) 
{
    vlservdebug ("VL_ListAttributesU\n") ;
    *nentries = 0;
    blkentries->len = 0;
    blkentries->val = NULL;
    return 0;
}

/*
 *
 */

int
VL_UpdateEntryByName(struct rx_call *call,
		     const char volname[65],
		     const struct VldbUpdateEntry *UpdateEntry,
		     const int32_t ReleaseType)
{
    vlservdebug ("VL_UpdateEntryByName (not implemented)\n");

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM;
}

/*
 *
 */

int
VL_GetAddrsU(struct rx_call *call,
	     const struct ListAddrByAttributes *inaddr,
	     struct afsUUID *uuid,
	     int32_t *uniq,
	     int32_t *nentries,
	     bulkaddrs *addrs)
{
    vlservdebug ("VL_GetAddrsU (not implemented)\n");
    return VL_PERM;
}

/*
 *
 */

int
VL_RegisterAddrs(struct rx_call *call,
		 const struct afsUUID *uid,
		 const int32_t spare,
		 const bulkaddrs *addrs)
{
    vlservdebug ("VL_RegistersAddrs (not implemented)\n");

    if (!sec_is_superuser(call))
	return VL_PERM;

    return 0;
}

/*
 *
 */

int
VL_CreateEntryU(struct rx_call *call,
		const struct uvldbentry *newentry)
{
    vlservdebug ("VL_CreateEntryU (not implemented)\n");

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM;
}

/*
 *
 */

int
VL_ReplaceEntryU(struct rx_call *call)
{
    vlservdebug ("VL_ReplaceEntryU (not implemented)\n");

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM;
}

/*
 *
 */

int
VL_ReplaceEntryN(struct rx_call *call,
		 const int32_t Volid,
		 const int32_t voltype,
		 const struct vldbentry *newentry,
		 const int32_t ReleaseType)
{
    vlservdebug ("VL_ReplaceEntryN (not implemented)\n");

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM;
}

/*
 *
 */

int
VL_ChangeAddrs(struct rx_call *call,
	       const int32_t old_ip,
	       const int32_t new_ip)
{
    vlservdebug ("VL_ChangeAddrs (not implemented)\n");

    if (!sec_is_superuser(call))
	return VL_PERM;

    return VL_PERM;
}

/*
 *
 */

int
VL_GetEntryByIDU(struct rx_call *call)
{
    vlservdebug ("VL_GetEntryByIDU (not implemented)\n");
    return VL_PERM;
}

/*
 *
 */

int
VL_ListEntryN(struct rx_call *call)
{
    vlservdebug ("VL_ListEntryN (not implemented)\n");
    return VL_PERM;
}

/*
 *
 */

int
VL_ListEntryU(struct rx_call *call)
{
    vlservdebug ("VL_ListEntryU (not implemented)\n");
    return VL_PERM;
}

/*
 *
 */

int
VL_GetAddrs(struct rx_call *call,
	    const int32_t handle,
	    const int32_t spare,
	    struct VL_Callback *spare3,
	    const int32_t *nentries,
	    bulkaddrs *blkaddr)
{
    vlservdebug ("VL_GetAddrs (not implemented)\n");
    return VL_PERM;
}

/*
 *
 */

int
VL_LinkedListN(struct rx_call *call)
{
    vlservdebug ("VL_LinkedListN (not implemented)\n");
    return VL_PERM;
}

/*
 *
 */

int
VL_LinkedListU(struct rx_call *call)
{
    vlservdebug ("VL_LinkedListU (not implemented)\n");
    return VL_PERM;
}
 
int
vldbserver_init(char *databaseprefix)
{
    char database[MAXPATHLEN];
    int status;

    if (databaseprefix == NULL)
	databaseprefix = MILKO_SYSCONFDIR;

    snprintf (database, sizeof(database), "%s/vl_database", 
	      databaseprefix);

    vlservdebug ("Loading db from file %s\n", database);

    vl_database = open(database,
                       O_RDWR | O_CREAT,
                       S_IRWXU);

    if (vl_database == -1)
        err(1, "open");

    get_file_length();
    if (file_length == 0) {
        fprintf (stderr, "Database not found, creating a new.\n");
	create_database();
	/* XXX <insert code for making new root.afs or whatever here> */

	status = 0;
        if (status)
            return status;
    } else {
        fprintf(stderr, "Loading database.\n");
        read_header();
        get_file_length();
    }

    return 0;
}

/*
 *
 */

static struct rx_service *vldbservice = NULL;
static struct rx_service *ubikservice = NULL;

static char *cell = NULL;
static char *realm = NULL;
static char *databasedir = NULL;
static char *srvtab_file = NULL;

static struct getargs args[] = {
    {"cell",	0, arg_string,    &cell, "what cell to use"},
    {"realm",	0, arg_string,	  &realm, "what realm to use"},
    {"prefix",'p', arg_string,    &databasedir, "what dir to store the db"},
    {"debug", 'd', arg_flag,      &debug, "output debugging"},
    {"srvtab",'s', arg_string,    &srvtab_file, "what srvtab to use"},
    { NULL, 0, arg_end, NULL }
};

static void
usage(void)
{
    arg_printusage(args, NULL, "", ARG_AFSSTYLE);
}

int
main(int argc, char **argv) 
{
    int ret;
    int optind = 0;
    
    if (getarg (args, argc, argv, &optind, ARG_AFSSTYLE)) {
	usage ();
    }

    argc -= optind;
    argv += optind;

    if (argc) {
	printf("unknown option %s\n", *argv);
	return 0;
    }

    ports_init();
    cell_init(0);

    if (cell)
	cell_setthiscell (cell);

    network_kerberos_init (srvtab_file);
    
    ret = vldbserver_init(databasedir);
    if (ret)
	errx (1, "vldbserver_init: error %d\n", ret);

    ret = network_init(htons(afsvldbport), "vl", VLDB_SERVICE_ID, 
		       VL_ExecuteRequest, &vldbservice, realm);
    if (ret)
	errx (1, "network_init failed with %d", ret);
    
    ret = network_init(htons(afsvldbport), "ubik", VOTE_SERVICE_ID, 
		       Ubik_ExecuteRequest, &ubikservice, NULL);
    if (ret)
	errx (1, "network_init failed with %d", ret);

    printf("Milko vldbserver %s-%s started\n", PACKAGE, VERSION);

    rx_SetMaxProcs(vldbservice,5) ;
    rx_SetMaxProcs(ubikservice,5) ;
    rx_StartServer(1) ;

    abort() ; /* should not get here */
    return 0;
}
